//////////////////////////////////////////////////////////////         
//      $Id: debInterface.cpp,v 1.6 1998/12/15 14:21:04 toivo Exp $ 
//
// Author: Toivo Pedaste
//
#include "../config.h"

#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>		// for O_RDONLY

#include <time.h>		// for localtime
#include <setjmp.h>

#include <qdir.h> 
#include <qfileinf.h> 

#include <kurl.h>

#include "packageInfo.h"
#include "debInterface.h"
#include "updateLoc.h"
#include "kpackage.h"
#include "managementWidget.h"
#include "utils.h"
#include "procbuf.h"
#include "options.h"
#include "cache.h"


extern KApplication *app;
extern Params *params;

procbuf reader;

#define AVAIL          "/var/lib/dpkg/available"
#define STATUS          "/var/lib/dpkg/status"

#define INFODIR         "/var/lib/dpkg/info/"

static param pinstall[] =  {
  {"Allow Downgrade",TRUE,TRUE},
  {"Check Conflicts",TRUE,TRUE},
  {"Check Dependencies",TRUE,TRUE},
  {"Test (do not install)",FALSE,FALSE},
  {0,FALSE,FALSE}
};

static param puninstall[] =  {
  {"Purge Config Files",TRUE,FALSE},
  {"Check Dependencies",TRUE,TRUE},
  {"Test (do not uninstall)",FALSE,FALSE},
  {0,FALSE,FALSE}
};


//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
DEB::DEB():pkgInterface()
{
  head = "DEB";
  icon = "deb.xpm";  

  kfm = 0;

  pict = new QPixmap();
  *pict = globalKIL->loadIcon(icon); 
  bad_pict = new QPixmap();
  *bad_pict = globalKIL->loadIcon("dbad.xpm");
  updated_pict = new QPixmap();
  *updated_pict = globalKIL->loadIcon("dupdated.xpm");
  new_pict = new QPixmap();
  *new_pict = globalKIL->loadIcon("dnew.xpm");

  packagePattern = "*.deb";
  queryMsg = i18n("Querying DEB package list: ");
  typeID = "/deb";

  locatedialog = new Locations(2, 4, 6, this,
  i18n("Locate of Debian package archives"),
  "Deb","Packages available *.deb",
  i18n("Location of directories containg Debian packages"),
  i18n("Location of 'Packages' files for sections of Debian distributions"),
  i18n("Location of base directory of Debian distribution"));
  connect(locatedialog,SIGNAL(returnVal(LcacheObj *)),
	  this,SLOT(setAvail(LcacheObj *)));
  locatedialog->apply_slot();

  pinstall[0].name = i18n("Allow Downgrade");
  pinstall[1].name = i18n("Check Conflicts");
  pinstall[2].name = i18n("Check Dependencies");
  pinstall[3].name = i18n("Test (do not install)");

  puninstall[0].name = i18n("Purge Config Files");
  puninstall[1].name = i18n("Check Dependencies");
  puninstall[2].name = i18n("Test (do not uninstall)");

}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
DEB::~DEB()
{
}

//////////////////////////////////////////////////////////////////////////////
param *DEB::initinstallOptions()
{
  return &(pinstall[0]);
}

param *DEB::inituninstallOptions()
{
  return &(puninstall[0]);
}

// check if debian file
bool DEB::isType(char *buf)
{
  if  (!strcmp(buf,"!<arch>\n")) {
    return true;
  } else if (!strncmp(buf,"0.9",3)) {
    return true;
  } else
    return false;
}

QString DEB::getPackList(cacheObj *cp) 
{
  QString string = "";
  QString tmpf;
  int res;

  QString url = cp->location;

  if ((res = cacheObj::newDCache(url, cp->cacheFile, &tmpf))) {
    if (res < 0)
      return string;

    if ( kfm != 0)   {
      KpMsgE( i18n("KFM already active!"),
	      "",TRUE);
      return string;
    }

    unlink(tmpf.data());
    fetchCache = cp->cacheFile;
    kpkg->kp->setStatus(i18n("Calling KFM"));

    //    printf("getPackList=%s\n",url.data());

    kfm = new KFM;
    kfm->allowKFMRestart(TRUE);
    kpkg->kp->setStatus(i18n("KFM Done"));

    if ( !kfm->isOK() ) {
      KpMsgE(i18n("Could not start or find KFM"),"",TRUE);
      delete kfm;
      string = "";
      return string;
    }
    
    kpkg->kp->setStatus(i18n("Starting Job"));

    connect( kfm, SIGNAL( finished() ), this, SLOT( slotKFMFinished() ) );
    connect( kfm, SIGNAL( error(int, const char*) ), this,
	     SLOT( slotKFMError(int, const char*) ) );
    kpkg->kp->setStatus(i18n("Connected"));
    kfm->copy(url, tmpf.data() );
    kpkg->kp->setStatus(i18n("Waiting..."));
    app->enter_loop();
  } 
  return tmpf;
}

void DEB::slotKFMFinished()
{
  QString string;

  string.sprintf(i18n("Finished '%s'"),fetchCache.data());
  kpkg->kp->setStatus(string.data());

  disconnect( kfm, SIGNAL( finished() ), this, SLOT( slotKFMFinished() ) );
  delete kfm;
  kfm = 0;
 app->exit_loop();
}

void DEB::slotKFMError(int err , const char* text) 
{
  printf ("DE=%d %s\n", err, text);
  disconnect( kfm, SIGNAL( error(int, const char*) ), this, SLOT( slotKFMError(int, const char*) ) );
 delete kfm;
 kfm = 0;
 cacheObj::rmDCache(fetchCache);
 app->exit_loop();
}

bool DEB::parseName(QString name, QString *n, QString *v)
{
  int d1, d2, s1;

  s1 = name.findRev('.');
  if (s1 > 0) {
      d2 = name.findRev('-',s1-1);
      if (d2 > 0) {
	d1 = name.findRev('_',d2-1);
	if (d1 < 0)
	  d1 = d2;
	*n = name.left(d1);
	*v = name.mid(d1+1,s1-d1-1);
	return TRUE;
      }
  }
  return FALSE;
}

void DEB::listPackages(QList<packageInfo> *pki)
{ 
  QString s;
  cacheObj *cp;

  listPackList(pki,STATUS,0);
  if (params->DisplayP != Params::INSTALLED) {
    int n = 0;
    for (cp = packageLoc->first(); cp != 0; cp = packageLoc->next()) {
      if (cp->base != "-") {
	s = getPackList(cp);
	if (s != "") {
	  listPackList(pki,s.data(),cp);
	}
      } else {
	s = getDir(cp);
	if (s != "")
	  listDir(pki,s,cp->location);
      }
      n++;
    }
  }
}

void DEB::listPackList(QList<packageInfo> *pki, char *fname, cacheObj *cp)
{
  int np;
  bool local = FALSE;
  QString vb;
  char linebuf[1024];
  FILE *file;
  packageInfo *p;

  QString sline = i18n("Querying DEB package list: ");
  sline += fname;

  if (cp) {
    KURL *u = new KURL(cp->base.data());
    if ( strcmp( u->protocol(), "file" ) == 0 ) {
      local = TRUE;
    }
    delete u;
  }

  kpkg->kp->setStatus(sline.data());
  kpkg->kp->setPercent(0);

  np = 0;
  file= fopen(fname,"r");
  vb = "";

  if (file) {
    while (fgets(linebuf,sizeof(linebuf),file)) {
      if (strcmp(linebuf,"\n")) {
	vb += linebuf;
      } else {
	p = collectInfo(vb.data());
	if (p) {
	  if (!p->update(pki, typeID, cp == 0)) {
	    delete p;
	  } else if (cp) {
	    p->info->insert("base",new QString(cp->base));
	  }
	}
	vb.truncate(0);
      }
    }
    fclose(file);
  }
  kpkg->kp->setPercent(100);
}


//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
// mode: i = query installed    u = query uninstalled
packageInfo *DEB::getPackageInfo(char mode, const char *name)
{
  packageInfo *pki = 0; 
  QString vb,search;
  char linebuf[1024];
  FILE *file;
  bool found = false;
  
  switch(mode)
    {
      ////////////////////////////////////////////////////////////////////////
      // query an installed package!
    case 'i':
      file= fopen(STATUS,"r");
      vb = "";
      search = "Package: ";
      search += name;
      search += "\n";

      if (file) {
	while (fgets(linebuf,sizeof(linebuf),file)) {
	  if (!found) {
	    if (!strcmp(search.data(),linebuf)) {
	      found = true;
	    } else {
	      continue;
	    }
	  }
	  if (strcmp(linebuf,"\n")) {
	    vb += linebuf;
	  } else {
	    pki = DEB::collectInfo(vb.data());
	    break;	
	  }
	}
      }
      fclose(file);
      break;

      ////////////////////////////////////////////////////////////////////
      // query an uninstalled package      
    case 'u':
      reader.setup("dpkg");
      *reader.proc << "--info"<<name;
      if (reader.start(0)) 
	pki = DEB::collectInfo(reader.buf.data());
      if (pki)
	pki->updated = TRUE;
      break;
    }
  return pki;
}


//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
packageInfo *DEB::collectInfo(char *inp)
{
  QDict<QString> *a = new QDict<QString>;
  a->setAutoDelete(TRUE);

  int have_group;
  char *str, *xstr;
  QString qstr;
  bool bad_install = FALSE;
  bool available = FALSE;

    have_group = 0;

    //   remove prefix output from dpkg
    str = strtok(inp,"\n");
    do {
      xstr = strchr(str,':');
      if (*str == ' ')
	str++;
      if (!strncmp("Package",str,7))
	break;
    } while ((str = strtok(NULL,"\n")));

    // parse 'name: text' elements
    
    if (str) {
      do {
cont:	if (str[0] == 0)
          break;

	xstr = strchr(str,':');
	if (xstr) {
	  *xstr++ = 0;
	  xstr++;
	  
	  char *p = str; // lower case field names
	  while (*p) {
	    *p = tolower(*p);
	    p++;
	  }
	  if (*str == ' ')
	    str++;

	  if (!strcmp("conffiles",str)) {
	  } else if (!strcmp("description",str)) {
	    qstr = xstr;
	    qstr += "\n";
	    while ((str = strtok(NULL,"\n"))) {
	      if (str[0] == ' ') {
		qstr += str;
	      } else {
		a->insert("description", new QString(qstr.data()));
		goto cont;
	      }
	    }
	    a->insert("description", new QString(qstr.data()));

	  } else if (!strcmp("package",str)) {
	    a->insert("name", new QString(xstr));
	  } else if (!strcmp("md5sum",str)) {
	    available = TRUE;
	    bad_install = FALSE;
	  } else if (!strcmp("section",str)) {
	    a->insert("group", new QString(xstr));
	    have_group++;
	  } else if (!strcmp("status",str)) {
	    if (!strncmp(xstr+strlen(xstr)-13,"not-installed",13)) {
	      delete a;
	      return 0;
	    }
	    if (strcmp("install ok installed",xstr) && 
		strcmp("deinstall ok installed",xstr) &&
		strcmp("purge ok installed",xstr)) {
	      bad_install = TRUE;
	    }
	    a->insert(str, new QString(xstr));
	  } else if (!strcmp("size",str)) {
	    a->insert("file-size", new QString(xstr));
	  } else if (!strcmp("installed-size",str)) {
	    QString *stmp = new QString(xstr);
	    *stmp += "000";
	    a->insert("size", stmp);
	  } else {
	    a->insert(str, new QString(xstr));
	  }
	}
      } while ((str = strtok(NULL,"\n")));
    }

    if (!have_group) {
	  a->insert("group", new QString("OTHER"));
    }

    packageInfo *i = new packageInfo(a,this);
    if (bad_install) {
      i->packageState = packageInfo::BAD_INSTALL;
    } else if (available) {
      i->packageState = packageInfo::AVAILABLE;
    } else {
      i->packageState = packageInfo::INSTALLED;
    }
    return i;

}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
QList<char> *DEB::getFileList(packageInfo *p)
{
  char *filelistfile;
  QString vb, fn;
  char *name;
  char mode;
  char linebuf[1024];
  FILE *file;
  char *s, *str, *strp;
  int prlen;

  fn = p->getFilename();
  if(fn != "")
    mode = 'u';
  else
    mode = 'i';

  QList<char> *filelist = new QList<char>;
  filelist->setAutoDelete(TRUE);

  switch(mode)
    {
      ////////////////////////////////////////////////////////////////////////
      // query an installed package!
    case 'i':
      name = p->getProperty("name")->data();

      vb = INFODIR;
      vb += name;
      vb += ".list";
      filelistfile= vb.data();
      file= fopen(filelistfile,"r");

      if (file) {
	while (fgets(linebuf,sizeof(linebuf),file)) {
	  s = strdup(linebuf);
	  s[strlen(s) - 1] = 0;    // remove new line
	  filelist->append(s);
	}
	fclose(file);
      }
      break;

      ////////////////////////////////////////////////////////////////////
      // query an uninstalled package      
    case 'u':
      name = fn.data();
      reader.setup("dpkg");
      *reader.proc << "--contents"<<name;

      if (!reader.start(1))
	return 0;

      str = strtok(reader.buf.data(),"\n");
      strp = strrchr(str,' ');
      if (strp) {
	prlen = strp - str;
      } else {
	prlen = 0;
      }
      do {
	filelist->append(strdup(str+prlen));
      } while ((str = strtok(NULL,"\n")));
      break;
    }
   
  return filelist;
}
//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
QList<char> *DEB::depends(const char *name, int src) {return 0;}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
QList<char> *DEB::verify(packageInfo *package, QList<char> *files)
{
  int  p = 0;
  uint c = 0;
  QList<char> *errorlist = new QList<char>;
  QDir d;

  QListIterator<char> it(*files);
  uint step = (it.count() / 100) + 1;

  kpkg->kp->setStatus(i18n("Verifying"));
  kpkg->kp->setPercent(0);

  char *s;
  for(; (s = it.current()); ++it) {

    // Update the status progress
    c++;
    if(c > step) {
      c=0; p++;
      kpkg->kp->setPercent(p);
    }

    if (!d.exists(s)) {
      errorlist->append(strdup(s));
    }	
  }

  kpkg->kp->setPercent(100);
  return errorlist;
}

//////////////////////////////////////////////////////////////////////////////
// Use the ~/.dpkg_ok file to tell if dpkg has worked or failed
// file created by the install/uninstall scripts
//////////////////////////////////////////////////////////////////////////////
int DEB::dpkgChange(int del)
{
    QDateTime st;
    st.setTime_t(0);

    char *tmpd = getenv("HOME");
    if (tmpd) {
      QString tfile;
      tfile.sprintf("%s/.dpkg_ok",tmpd);
      QFileInfo *fileInfo = new QFileInfo(tfile.data());
      if (fileInfo->exists()) {
	return st.secsTo(fileInfo->lastModified());
	if (del)
	  unlink(tfile.data());
      } else {
	return 0;
      }
    } else
      return 0;
}

//////////////////////////////////////////////////////////////////////////////
// Call the script to uninstall packages setting parameters
// to dpkg dependent on flags, returning whether everyting worked
//////////////////////////////////////////////////////////////////////////////
int DEB::doUninstall(int uninstallFlags, QString packs)
{
  int test = 0, ctime, nctime;

  ctime = dpkgChange(0);

  reader.setup("kvt");
  *reader.proc << "-e" << "kpackage_dpkg_rm";

  *reader.proc <<  i18n("'Delete this window to continue'");

  if ((uninstallFlags>>0 & 1) ^ puninstall[0].invert)
    *reader.proc << "--purge";
  else
    *reader.proc << "--remove";
  if ((uninstallFlags>>1 & 1) ^ puninstall[1].invert)
    *reader.proc << "--force-depends";
  if ((uninstallFlags>>2 & 1) ^ puninstall[2].invert) {
    *reader.proc << "--no-act";
    test = 1;
  }

  *reader.proc << packs.data();

  reader.start(1);
  nctime = dpkgChange(1);
  
  return (nctime == ctime) || test;
}

//////////////////////////////////////////////////////////////////////////////
int DEB::uninstall(int uninstallFlags, packageInfo *p)
{
  QString packs;
  packs = p->getProperty("name")->data();

  return doUninstall(uninstallFlags, packs);
}

//////////////////////////////////////////////////////////////////////////////
int DEB::uninstall(int uninstallFlags, QList<packageInfo> *p)
{
  QString packs = "";
  packageInfo *i;

  for (i = p->first(); i!= 0; i = p->next())  {
    packs += *i->getProperty("name");
    packs += " ";
  }
  return doUninstall( uninstallFlags, packs);
}

//////////////////////////////////////////////////////////////////////////////
// Call the script to install packages setting parameters
// to dpkg dependent on flags, returning whether everyting worked
//////////////////////////////////////////////////////////////////////////////

int DEB::install(int installFlags, packageInfo *p)
{
  QString fname = p->fetchFilename();

  return doInstall(installFlags, fname);
}

//////////////////////////////////////////////////////////////////////////////
int DEB::install(int installFlags, QList<packageInfo> *p)
{
  QString packs = "";
  packageInfo *i;

  for (i = p->first(); i!= 0; i = p->next())  {
    QString fname = i->fetchFilename();
    if (fname != "") {
      packs += fname;
      packs += " ";
    }
  }
  return doInstall(installFlags, packs);
}

//////////////////////////////////////////////////////////////////////////////
int DEB::doInstall(int installFlags, QString packs)
{
  int test = 0, ctime, nctime;

  ctime = dpkgChange(0);

  reader.setup("kvt");
  *reader.proc << "-e" << "kpackage_dpkg_ins";
  *reader.proc <<  i18n("'Delete this window to continue'");

  if ((installFlags>>0 & 1) ^ pinstall[0].invert)
    *reader.proc << "--refuse-downgrade";
  if ((installFlags>>1 & 1) ^ pinstall[1].invert)
    *reader.proc << "--force-conflicts";
  if ((installFlags>>2 & 1) ^ pinstall[2].invert)
    *reader.proc << "--force-depends";
  if ((installFlags>>3 & 1) ^ pinstall[3].invert) {
    *reader.proc << "--no-act";
    test = 1;
  }

  *reader.proc <<  packs.data();

  reader.start(1);
  nctime = dpkgChange(1);
  
  return (nctime == ctime) || test;
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
QString DEB::FindFile(const char *name)
{
  reader.setup("dpkg");
  *reader.proc << "-S"<<name;

  if (!reader.start(1))
    return 0;

  int p = 0,lines = 0;
  while ((p = reader.buf.find(':',p)) >= 0) {
    lines++;
    reader.buf.replace(p,1,"\t");
    p = reader.buf.find('\n',p);
  }

  if (lines == 1) {
    if (reader.buf.find("not found") >= 0)
      reader.buf.truncate(0);
  }

  return reader.buf;
}


//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
void DEB::setLocation()
{
    locatedialog->restore();
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
void DEB::setAvail(LcacheObj *slist)
{
  if (packageLoc)
    delete packageLoc;
  packageLoc = slist;
}

//////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
