/*
                                 MAP2Q3

           Program to convert Quake/Quake2 maps to Q3A format

                   Copyright (C) Laszlo Menczel 2006
                          menczel@invitel.hu

        This is free software without warranty. See 'licence.txt.
*/

#if defined _WIN32
  #include <windows.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <math.h>

#include "mutil.h"

//=================================================================================

#define MAX_CLASSNAME   64
#define MAX_BLK_LINES   256

#define MAX_NAME_LEN    64

#define MIN_ENT_LINES   2
#define MAX_ENT_LINES   MAX_BLK_LINES

#define MIN_BRUSH_LINE  3
#define MAX_BRUSH_LINE  32

#define LEFT_BRACE      '{'
#define RIGHT_BRACE     '}'
#define RIGHT_PAREN     ')'
#define QUOTE           '\"'

#define IGNORED_CLASS   -1
#define NO_CLASS        -2
#define NO_TOKEN        -2

#define DEF_LIGHT_SCALE 1.0
#define DEF_SIZE_SCALE  1.0
#define DEF_LIGHT_VALUE 300

enum
{
  OK,
  END_OF_FILE,
  TOO_MANY_LINES,
  NOT_ENOUGH_LINES,
  NO_DELIMITER,
  CLASS_ERROR,
  TOKEN_ERROR,
  ENT_FILE_ERR,
  BAD_LIGHT_VAL,
  ENT_BRUSH_ERR,
  TOO_MANY_BRUSHES,
  COORD_ERROR,
  ORIGIN_ERROR,
  DISCARD_OBJECT
};

#define MAX_BRUSH_BLK  32

#define NO_BRUSH  -1

enum
{
  NORMAL_BRUSH,
  PLATFORM_BRUSH,
  DOOR_BRUSH,
  TRIGGER_BRUSH,
  NUM_BRUSH_TYPES
};

//=================================================================================

static char progname[MUT_ARG_MAXLEN];

static char map_name[MUT_ARG_MAXLEN];
static char new_name[MUT_ARG_MAXLEN];
static char ent_table[MUT_ARG_MAXLEN];
static char light_scale_val[MUT_ARG_MAXLEN];
static char size_scale_val[MUT_ARG_MAXLEN];

static int quake2 = 0;
static int ent_number_only = 0;

static arglist_t arglist[] =
{
  { MUT_ARG_SWITCH, MUT_ARG_OPTIONAL, "q2",   (void *) &quake2              }, 
  { MUT_ARG_SWITCH, MUT_ARG_OPTIONAL, "enum", (void *) &ent_number_only     }, 

  { MUT_ARG_PAIRED, MUT_ARG_REQUIRED, "s",    (void *) &map_name[0]         }, 
  { MUT_ARG_PAIRED, MUT_ARG_OPTIONAL, "d",    (void *) &new_name[0]         }, 
  { MUT_ARG_PAIRED, MUT_ARG_OPTIONAL, "e",    (void *) &ent_table[0]        }, 
  { MUT_ARG_PAIRED, MUT_ARG_OPTIONAL, "l",    (void *) &light_scale_val[0]  }, 
  { MUT_ARG_PAIRED, MUT_ARG_OPTIONAL, "g",    (void *) &size_scale_val[0]   }, 

  { MUT_ARG_END }
};

static FILE *infile, *outfile;

static char *logfile = "map2q3.log";

static char *q1_ent_table = "q1.ent";
static char *q2_ent_table = "q2.ent";

static char *texture_def[NUM_BRUSH_TYPES] =
{
  "base_floor/concrete 0 0 0 0.5 0.5 0 0 0",
  "base_floor/clang_floor3blava 0 0 0 0.5 0.5 0 0 0",
  "base_floor/clang_floor3blava 0 0 0 0.5 0.5 0 0 0",
  "common/trigger -4 0 0 0.5 0.5 0 0 0",
};

static double light_scale = DEF_LIGHT_SCALE;
static double size_scale = DEF_SIZE_SCALE;

static char buf[MAX_BLK_LINES][MUT_MAX_LINE_LEN];

static char *conv_error[] =
{
  "",
  "end of file",
  "too many lines",
  "not enough lines",
  "no delimiter (right parenthesis) in line",
  "no classname found",
  "token extraction error",
  "cannot open the file",
  "bad light value",
  "invalid brush definition in entity",
  "too many brushes in an entity",
  "bad coordinate in entity or brush definition",
  "no origin defined for entity"
};

static int obj_discarded = 0;

/* entity class info */

typedef struct classinfo_s
{
  char old_name[MAX_NAME_LEN];
  char new_name[MAX_NAME_LEN];
  int brushtype;
} classinfo_t;

static classinfo_t clinfo[MAX_CLASSNAME];

static int class_count = 0;

static int curr_class = NO_CLASS;

static char *curr_class_str = NULL;
static char *curr_class_ori = NULL;

/* brush definition block info */

typedef struct brdef_s
{
  int used;     // contains valid info
  int start;    // starting line in buffer
  int end;      // ending line in buffer
} brdef_t;

static brdef_t brdef[MAX_BRUSH_BLK];

//=================================================================================

static void clear_buf(void)
{
  int i;

  for (i = 0; i < MAX_BLK_LINES; i++)
    buf[i][0] = 0;
}

//=================================================================================

static void clear_brush_info(void)
{
  int i;

  for (i = 0; i < MAX_BRUSH_BLK; i++)
  {
    brdef[i].used = 0;
    brdef[i].start = -1;
    brdef[i].end = -1;
  }
}

//=================================================================================

static int init_ent_table(void)
{
  FILE *f;
  int i;
  char old[MAX_NAME_LEN], new[MAX_NAME_LEN], *cp;

  f = fopen(ent_table, "rt");
  if (f == NULL)
    return ENT_FILE_ERR;
    
  strcpy(clinfo[0].old_name, "\"worldspawn\"");
  strcpy(clinfo[0].new_name, "\"worldspawn\"");
  clinfo[0].brushtype = NORMAL_BRUSH;
  
  i = 1;
  while (1)
  {
    if (i == MAX_CLASSNAME)
    {
      printf("Warning: too many classnames in entity replacement table\n");
      break;
    }
    
    cp = fgets(buf[0], MUT_MAX_LINE_LEN, f);
    if (cp == NULL)
      break;
     
    mut_stripn_eol(buf[0], MUT_MAX_LINE_LEN);    

    cp = mut_skip_space(buf[0], MUT_MAX_LINE_LEN);
    if (cp == NULL)
      break;

    cp = mut_get_token(old, cp, NULL, MAX_NAME_LEN);
    if (cp == NULL)
    {
      fclose(f);
      return TOKEN_ERROR;
    }
    else
      sprintf(clinfo[i].old_name, "\"%s\"", old);
    
    if (strcmp(old, "func_plat") == 0)
      clinfo[i].brushtype = PLATFORM_BRUSH;
      
    else if (strcmp(old, "func_door") == 0)
      clinfo[i].brushtype = DOOR_BRUSH;

    else if (strncmp(old, "trigger_", 8) == 0)
      clinfo[i].brushtype = TRIGGER_BRUSH;

    else
      clinfo[i].brushtype = NO_BRUSH;

    cp = mut_get_token(new, cp, NULL, MAX_NAME_LEN);
    if (cp == NULL)
    {
      fclose(f);
      return TOKEN_ERROR;
    }
    else
    {
      if (new[0] == '*')
        sprintf(clinfo[i].new_name, "\"%s\"", old);
      else
        sprintf(clinfo[i].new_name, "\"%s\"", new);
    }

    i++;
  }
    
  fclose(f);
  class_count = i;
  return OK;
}

//=================================================================================

static void buf2log(void)
{
  FILE *f;
  int n;
  
  f = fopen(logfile, "at");
  if (f != NULL)
  {
    n = 0;
    while (buf[n][0] != 0)
    {
      mut_logprintf("%s\n", buf[n]);
      n++;
    }

    mut_logprintf("===================================================\n");
    fclose(f);
  }
}

//=================================================================================

static int is_comment(char *s)
{
  if (*s == '/' && *(s + 1) == '/')
    return 1;
  else
    return 0;
}

//=================================================================================

static int brush_info_ok(void)
{
  int i;

  while (brdef[i].used)
  {
    if (
         brdef[i].start < 0 ||
         brdef[i].end < 0 ||
         brdef[i].end < brdef[i].start ||
         brdef[i].end - brdef[i].start + 1 < MIN_BRUSH_LINE ||
         brdef[i].end - brdef[i].start + 1 > MAX_BRUSH_LINE
       )
       return 0;
    
    i++;
  }

  return 1;
}

//=================================================================================

static int find_class(char *s)
{
  int i;
  
  for (i = 0; i < class_count; i++)
    if (strcmp(s, clinfo[i].old_name) == 0)
      return i;
  
  return IGNORED_CLASS;
}

//=================================================================================

static int scale_origin(void)
{
  int i, c[3];
  double val;
  char *cp, token[MAX_NAME_LEN];
  
  cp = curr_class_ori;
  
  for (i = 0; i < 3; i++)
  {
    cp = mut_get_token(token, cp, " \t\"", MAX_NAME_LEN);
    if (cp == NULL)
      return TOKEN_ERROR;
    
    if (mut_stof(token, &val) < 0)
      return COORD_ERROR;
    
    c[i] = (int) floor(val * size_scale);
  }
  
  sprintf(curr_class_ori, "\"%d %d %d\"", c[0], c[1], c[2]);
      
  return OK;
}

//=================================================================================

int scale_light_intensity(void)
{
  int n;
  double val;
  char token[MAX_NAME_LEN], *val_str;

  n = 0;
  while (buf[n][0] != 0)
  {
    val_str = mut_get_token(token, buf[n], NULL, MAX_NAME_LEN);
    if (val_str == NULL)
      return TOKEN_ERROR;

    if (strcmp(token, "\"light\"") == 0)
    {
      if (mut_get_token(token, val_str, NULL, MAX_NAME_LEN) == NULL)
        return TOKEN_ERROR;
        
      mut_strip_quote(token, MAX_NAME_LEN);

      if (mut_stof(token, &val) < 0)
         return BAD_LIGHT_VAL;
 
      val *= light_scale;
      if (val < 1.0)
        val = 1.0;
              
      val_str = mut_skip_space(val_str, MUT_MAX_LINE_LEN);
      sprintf(val_str, "\"%d\"", (int) val);

      return OK;
    }
    n++;
  }
  
  // no light value, insert default (300)
  
  sprintf(buf[n], "\"light\" \"%d\"", (int) (light_scale * DEF_LIGHT_VALUE));

  return OK;
}

//=================================================================================

static int scale_face_coordinates(char *s, char *new)
{
  int i, c[9];
  double val;
  char *cp, token[MAX_NAME_LEN];

  // decode & adjust coordinate values

  cp = s;
  for (i = 0; i < 9; i++)
  {
    cp = mut_get_token(token, cp, " \t()", MAX_NAME_LEN);
    if (cp == NULL)
      return TOKEN_ERROR;
      
    if (mut_stof(token, &val) < 0)
      return COORD_ERROR;
    else
      c[i] = (int) floor(val * size_scale);
  }

  // write new values

  cp = mut_skip_delimiter(cp, " \t()", MAX_NAME_LEN);

  sprintf
  (
    new, "( %d %d %d ) ( %d %d %d ) ( %d %d %d ) %s",
    c[0], c[1], c[2], c[3], c[4], c[5], c[6], c[7], c[8], cp
  ); 
  
  return OK;
}      

//=================================================================================

static int convert_brushes(void)
{
  int i, res, line;
  char *cp, tmp[MUT_MAX_LINE_LEN];

  i = 0;
  while (brdef[i].used)
  {
    line = brdef[i].start;
    while (line <= brdef[i].end)
    {
      if (size_scale != 1.0)
      {
        res = scale_face_coordinates(buf[line], tmp);
        if (res > 0)
          return res;
        else
          strcpy(buf[line], tmp);
      }
      
      cp = mut_str_last(buf[line], RIGHT_PAREN, MUT_MAX_LINE_LEN);
      if (cp == NULL)
        return NO_DELIMITER;
      cp++;
      sprintf(cp, " %s", texture_def[clinfo[curr_class].brushtype]);

      line++;
    }

    i++;
  }
  
  return OK;
}

//=================================================================================

/*
   Assumes that the file pointer is at the start of 'f'. Reads consecutive lines
   from file 'f' until a left brace '{' is found. (This is interpreted to be the
   start of the 'worldspawn' section of the map.) Returns 1 if OK, zero if the
   brace is not found. Ignores empty lines and comments.
*/

static int find_worldspawn(FILE *f)
{
  char *res;

  clear_buf();

  while (1)
  {
    res = fgets(buf[0], MUT_MAX_LINE_LEN, f);
    if (res == NULL)
      return 0;

    res = mut_skip_space(buf[0], MUT_MAX_LINE_LEN);
    if (res == NULL)
      continue;
    
    if (is_comment(res))
      continue;
      
    if (*res == LEFT_BRACE)
      return 1;
    else
      return 0;
  }
}

//=================================================================================

/*
   Assumes that the program is currently processing the worldspawn section. Finds
   the next left brace '{' delimiter and loads into 'buf' conscutive lines until
   a right brace '}' is found. Return 0 if OK, -1 if end of the worldspawn section
   is detected. In case of error returns:
   
   END_OF_FILE = end of file, no left brace found
   TOO_MANY_LINES = line count > 6
   NOT_ENOUGH_LINES = line count < 4
*/

static int read_brush(FILE *f)
{
  int n;
  char *res;

  clear_brush_info();
  clear_buf();

  // find start of brush definiton block

  while (1)
  {
    res = fgets(buf[0], MUT_MAX_LINE_LEN, f);
    if (res == NULL)
      return END_OF_FILE;
    
    mut_stripn_eol(buf[0], MUT_MAX_LINE_LEN);

    res = mut_skip_space(buf[0], MUT_MAX_LINE_LEN);
    if (res == NULL)
      continue;

    if (is_comment(res))
      continue;
 
    if (*res == LEFT_BRACE)
      break;
    else if (*res == RIGHT_BRACE)  // end of worldspawn section
      return -1;
    else
      continue;
  }
    
  // read lines of brush definition

  n = 0;
  while (1)
  {
    res = fgets(buf[n], MUT_MAX_LINE_LEN, f);
    if (res == NULL)
      return END_OF_FILE;
      
    mut_stripn_eol(buf[n], MUT_MAX_LINE_LEN);

    res = mut_skip_space(buf[n], MUT_MAX_LINE_LEN);
    if (res == NULL)
      continue;

    if (is_comment(res))
      continue;
      
    if (*res == RIGHT_BRACE)
    {
      buf[n][0] = 0;
      break;
    }
    
    n++;
    if (n > MAX_BRUSH_LINE)
      return TOO_MANY_LINES;
  }
  
  if (ent_number_only)
    return OK;

  if (n < MIN_BRUSH_LINE)
    return DISCARD_OBJECT;

  brdef[0].start = 0;
  brdef[0].end = n - 1;
  brdef[0].used = 1;
  
  return OK;
}

//=================================================================================

/*
   Assumes that the program is currently processing the entity section. Finds
   the next left brace '{' delimiter and loads into 'buf' consecutive lines until
   a right brace '}' is found. If the entity definiton contains enclosed brush
   definitions, the starting and ending lines are stored in the global array
   'brdef'.
   
   Return 0 if OK, -1 if end of file is detected. In case of error returns:
   
   TOO_MANY_LINES = line count > MAX_ENT_LINES
   NOT_ENOUGH_LINES = line count < MIN_ENT_LINES
   ENT_BRUSH_ERR = bad brush definiton
*/

static int read_entity(FILE *f)
{
  int n, brush_flag, brush_index;
  char *res, token[MAX_NAME_LEN];
  
  clear_brush_info();
  clear_buf();
  curr_class = NO_CLASS;
  curr_class_str = NULL;
  curr_class_ori = NULL;

  // find start of entity definiton block

  while (1)
  {
    res = fgets(buf[0], MUT_MAX_LINE_LEN, f);
    if (res == NULL)
      return -1;
    
    mut_stripn_eol(buf[0], MUT_MAX_LINE_LEN);

    res = mut_skip_space(buf[0], MUT_MAX_LINE_LEN);
    if (res == NULL)
      continue;

    if (is_comment(res))
      continue;
 
    if (*res == LEFT_BRACE)
      break;
  }
    
  // read entity definition lines

  n = brush_flag = brush_index = 0;
  while (1)
  {
    if (n == MAX_ENT_LINES)
      return TOO_MANY_LINES;

    if (brush_index == MAX_BRUSH_BLK)
      return TOO_MANY_BRUSHES;
      
    res = fgets(buf[n], MUT_MAX_LINE_LEN, f);
    if (res == NULL)
      return END_OF_FILE;
      
    mut_stripn_eol(buf[n], MUT_MAX_LINE_LEN);

    res = mut_skip_space(buf[n], MUT_MAX_LINE_LEN);
    if (res == NULL)
      continue;

    if (is_comment(res))
      continue;
      
    if (*res == LEFT_BRACE)
    {
      if (brush_flag)
        return ENT_BRUSH_ERR;
      else
      {
        brdef[brush_index].start = n + 1;
        brush_flag = 1;
      }
    }

    else if (*res == RIGHT_BRACE)
    {
      if (brush_flag)
      {
        brdef[brush_index].end = n - 1;
        brdef[brush_index].used = 1;
        brush_flag = 0;
        brush_index++;
      }
      else
      {
        buf[n][0] = 0;
        break;
      }
    }

    else
    {
      res = mut_get_token(token, buf[n], NULL, MAX_NAME_LEN);
      if (res == NULL)
        return TOKEN_ERROR;
        
      if (strcmp(token, "\"classname\"") == 0)
      {
        if (mut_get_token(token, res, NULL, MAX_NAME_LEN) == 0)
          return TOKEN_ERROR;
        else
        {
          curr_class = find_class(token);
          curr_class_str = mut_skip_space(res, MUT_MAX_LINE_LEN);
        }      
      }

      else if (strcmp(token, "\"origin\"") == 0)
      {
        if (mut_get_token(token, res, NULL, MAX_NAME_LEN) == 0)
          return TOKEN_ERROR;
        else
          curr_class_ori = mut_skip_space(res, MUT_MAX_LINE_LEN);
      }
    }

    n++;
  }
 
  if (ent_number_only)
    return OK;

  if (n < MIN_ENT_LINES)
    return DISCARD_OBJECT;

  if (curr_class == NO_CLASS)
    return CLASS_ERROR;

  if (brdef[0].used)
  {
    if (! brush_info_ok()) 
      return ENT_BRUSH_ERR;
  }
  else
  {
    if (curr_class_ori == NULL)
      return ORIGIN_ERROR;
  }
    
  return OK;
}

//=================================================================================

static int write_brush(FILE *f, int num)
{
  int n;
  
  if (! ent_number_only)
  {
    n = convert_brushes();
    if (n > 0)
      return n;
  }

  fprintf(f, "  // brush %d\n  {\n", num);    

  n = 0;
  while (buf[n][0] != 0)
  {
    fprintf(f, "    %s\n", buf[n]);
    n++;
  }
       
  fprintf(f, "  }\n");    

  return OK;
}

//=================================================================================

static int write_ent(FILE *f, int num)
{
  int n;

  if (! ent_number_only)
  {
    sprintf(curr_class_str, "%s", clinfo[curr_class].new_name);

    if (strcmp(clinfo[curr_class].old_name, "\"light\"") == 0 && light_scale != 1.0)
    {
      n = scale_light_intensity();
      if (n > 0)
        return n;
    }

    if (brdef[0].used)
    {
      n = convert_brushes();
      if (n > 0)
        return n;
    }
    else
    {
      if (size_scale != 1.0)
      {
        n = scale_origin();
        if (n > 0)
          return n;
      }
    }
  }
  
  fprintf(f, "// entity %d\n", num);
  fprintf(f, "  {\n");
  
  n = 0;
  while (buf[n][0] != 0)
  {
    fprintf(f, "    %s\n", buf[n]);
    n++;
  }
  
  fprintf(f, "  }\n");

  return OK;
}

//=================================================================================

int main(int argc, char *argv[])
{
  int res, count, replaced;
    
  printf("MAP2Q3 map converter utility\n");
  printf("Copyright (C) L.Menczel, 2006\n");
  
  if (mut_file_exist(logfile))
    unlink(logfile);

  mut_set_logname(logfile);

  //================ read command line arguments

  res = mut_parse_cmd_line(argc, argv, arglist, progname);
  
  if (res < 0)
  {
    printf("%s\n", mut_errmsg(mut_last_error()));
    printf("Press ENTER...");
    getchar();
    return 1;
  }
  
  //================ set defaults if no argument specified

  if (new_name[0] == 0)  // no output file name
  {
    strcpy(new_name, map_name);
    mut_strip_ext(new_name);
    if (quake2)
    {
      if (ent_number_only)
        strcat(new_name, ".q2n");
      else
        strcat(new_name, ".2q3");
    }
    else
    {
      if (ent_number_only)
        strcat(new_name, ".q1n");
      else
        strcat(new_name, ".1q3");
    }
  }
  
  if (! ent_number_only)     // full conversion requested
  {    
    if (ent_table[0] == 0)   // no entity replacement table
    {
      if (quake2)
        strcpy(ent_table, q2_ent_table);
      else
        strcpy(ent_table, q1_ent_table);
    }
  
    if (light_scale_val[0] != 0)  // light scale specified
    {
      if (mut_stof(light_scale_val, &light_scale) < 0)
        printf("Bad light scale value, using default (%f)\n", DEF_LIGHT_SCALE);
    }

    if (size_scale_val[0] != 0)  // size scale specified
    {
      if (mut_stof(size_scale_val, &size_scale) < 0)
        printf("Bad size scale value, using default (%f)\n", DEF_SIZE_SCALE);
    }

    //================= initalize entity replacement table

    res = init_ent_table();
    if (res > 0)
    {
      printf("Entity table error: %s", conv_error[res]);
      if (res == ENT_FILE_ERR)
        printf(" '%s'", ent_table);
      printf("\n");
      return 1;
    }
  
    printf("%d classnames in '%s'\n", class_count, ent_table);

  }  /* END IF (! ent_number_only) */

  //================= open files

  infile = fopen(map_name, "rt");
  if (infile == NULL)
  {
    printf("Error when opening %s\n", map_name);
    printf("Press ENTER...");
    getchar();
    return 1;
  }
 
  outfile = fopen(new_name, "wt");
  if (outfile == NULL)
  {
    printf("Error when creating %s\n", new_name);
    printf("Press ENTER...");
    getchar();
    fclose(infile);
    return 1;
  }
  
  if (! ent_number_only)
  {
    if (quake2)
      fprintf(outfile, "// Quake2 ");
    else
      fprintf(outfile, "// Quake ");
      
    fprintf(outfile, "map '%s' converted to Q3A format\n", map_name);
    fprintf(outfile, "// Converter: MAP2Q3\n");
    fprintf(outfile, "// Copyright (C) L.Menczel, 2006\n");
  }

  fprintf(outfile, "// entity 0\n{\n");
  fprintf(outfile, "  \"classname\" \"worldspawn\"\n");
  
  //=================== process brush definitions (worldspawn section)

  printf("Processing '%s'\n", map_name);

  if (! find_worldspawn(infile))
  {
    printf("No 'worldspawn' section in %s\n", map_name);
    goto err_exit;
  }

  curr_class = 0;  // brushes in 'worldspawn', type NORMAL
  
  count = 0;
  while (1)    
  {
    res = read_brush(infile);
    if (res == -1)
      break;

    count++;

    if (res > 0)
    {
      if (res == DISCARD_OBJECT)
      {
        mut_logprintf("Discarded invalid brush #%d:\n", count - 1);
        buf2log();
        obj_discarded = 1;
        continue;
      }
      else
      {
        printf("Brush #%d load error: %s\n", count - 1, conv_error[res]);
        buf2log();
        goto err_exit;
      }
    }
      
    res = write_brush(outfile, count);
    if (res != 0)
    {
      printf("Brush #%d write error: %s\n", count - 1, conv_error[res]);
      buf2log();
      goto err_exit;
    }
  }

  if (count == 0)
  {
    printf("No brush found\n");
    goto err_exit;
  }

  fprintf(outfile, "}\n");
  printf("%d brushes\n", count);
  
  //================ process entity definitons

  count = replaced = 0;
  while (1)    
  {
    res = read_entity(infile);
    if (res == -1)
      break;

    count++;

    if (res > 0)
    {
      if (res == DISCARD_OBJECT)
      {
        mut_logprintf("Discarded invalid entity %d:\n", count);
        buf2log();
        obj_discarded = 1;
        continue;
      }
      else
      {        
        printf("Ent #%d load error (%d): %s\n", count, res, conv_error[res]);
        buf2log();
        goto err_exit;
      }
    }
          
    if (! ent_number_only && curr_class == IGNORED_CLASS)
      continue;

    res = write_ent(outfile, count);
    if (res > 0)
    {
      printf("Ent #%d write error: %s\n", count, conv_error[res]);
      buf2log();
      continue;
    }
    
    replaced++;
  }

  if (count == 0)
    printf("Warning: no entity\n");
  else
  {
    if (ent_number_only)
      printf("%d entities numbered\n", count);
    else
      printf("%d entities (%d replaced)\n", count, replaced);
  }
    
  if (obj_discarded)
  {
    printf("WARNING: invalid objects have been discarded,\n");
    printf("see 'map2q3.log' for details.\n");
  }
  
  fclose(infile);
  fclose(outfile);

  return 0;

err_exit:

  fclose(infile);
  fclose(outfile);
  unlink(new_name);

  return 1;
}
