/*
Lucifer Burn In Utility 1.0
By Peter Todd (c) 1999
*/

#include <time.h>
#include <sys/signal.h>

#include "main.h"
#include "memory.h"
#include "storage.h"
#include "cpu.h"
#include "fpu.h"

extern char version[];

void DisplayHelp();
void BurnLoop();
void UpdateDisplay();
void DisplayVersion();
int  signalhandler(int sig);

int stop;
int error = 0;
int loopsdone;

// default counts for each type of burn loop
int memorycount = 1;
int storagecount = 1;
int cpucount = 30000;
int fpucount = 30000;

// options
int check = 1;
int stoponerror = 0;
int burnmem = 0;
int burnstorage = 0;
int burncpu = 1;
int burnfpu = 1;
size_t mem2burn = 0;
int loops = 0; // if this is 0 we will run forever, if nonzero we will loop
               // loops times
double displayinterval = 1;
int nodisplay = 0;
size_t blocksize = 1024;

int numberofburndirs;
char **burndirlist;
size_t *storage2burn;

// displays help if needed and processes command line
// doesn't do much eh?
int main(int argc, char *argv[])
{
  int i = 1;

  // these are the different command line options
  char mem_op[] = "-m\0";
  char check_op[] = "-c\0";
  char stop_op[] = "-s\0";
  char nomem[] = "--nomem\0";
  char nostorage[] = "--nostorage\0";
  char nocpu[] = "--nocpu\0";
  char nofpu[] = "--nofpu\0";
  char loops_op[] = "-l\0";
  char displayinterval_op[] = "-t\0";
  char nodisplay_op[] = "--nodisplay\0";

  char memorycount_op[] = "--memc\0";
  char storagecount_op[] = "--storc\0";
  char cpucount_op[] = "--cpuc\0";
  char fpucount_op[] = "--fpuc\0";

  char burndirlist_op[] = "--dirlist\0";
  char blocksize_op[] = "-b\0";

  char help_op[] = "-h\0";
  char help_oplong[] = "--help\0";

  char version_op[] = "-v\0";
  char version_oplong[] = "--version\0";

  /* signal handler */
  int signalhandler(int sig);

  /* setup the signal handler */
  signal(SIGINT, signalhandler);

  // process the command line
  for (i = 0; i < argc; i++) {

    if ((!strcmp(argv[i], help_op)) && (argc = 2)){
      DisplayHelp();
      exit(EXIT_SUCCESS);
    }

    if ((!strcmp(argv[i], help_oplong)) && (argc = 2)){
      DisplayHelp();
      exit(EXIT_SUCCESS);
    }

    if ((!strcmp(argv[i], version_op)) && (argc = 2)){
      DisplayVersion();
      exit(EXIT_SUCCESS);
    }

    if ((!strcmp(argv[i], version_oplong)) && (argc = 2)){
      DisplayVersion();
      exit(EXIT_SUCCESS);
    }
 
    // the amount of memory to burn
    if ((!strcmp(argv[i], mem_op)) && (i < (argc - 1))){
      i++;
      mem2burn = (unsigned long int)atol(argv[i]);
    }
    // number of loops to go through
    else if ((!strcmp(argv[i], loops_op)) && (i < (argc - 1))){
      i++;
      loops = atoi(argv[i]);
    }
    // the display interval
    else if ((!strcmp(argv[i], displayinterval_op)) && (i < (argc - 1))){
      i++;
      displayinterval = (double)atoi(argv[i]);
    }

    // check results?
    else if (!strcmp(argv[i], check_op)){
      check = 1;
    }
    // stop on errors?
    else if (!strcmp(argv[i], stop_op)){
      stoponerror = 1;
    }

    // no display?
    else if (!strcmp(argv[i], nodisplay_op)){
      nodisplay = 1;
    }

    // these args turn burn functions off
    else if (!strcmp(argv[i], nomem)){
      burnmem = 0;
    }
    else if (!strcmp(argv[i], nostorage)){
      burnstorage = 0;
    }
    else if (!strcmp(argv[i], nocpu)){
      burncpu = 0;
    }
    else if (!strcmp(argv[i], nofpu)){
      burnfpu = 0;
    }

    // these args decide the count rates of the different burn functions
    else if ((!strcmp(argv[i], memorycount_op)) && (i < (argc - 1))){
      i++;
      memorycount = atoi(argv[i]);
      burnmem = 1;
    }
    else if ((!strcmp(argv[i], storagecount_op)) && (i < (argc - 1))){
      i++;
      storagecount = atoi(argv[i]);
      burnstorage = 1;
    }
    else if ((!strcmp(argv[i], cpucount_op)) && (i < (argc - 1))){
      i++;
      cpucount = atoi(argv[i]);
      burncpu = 1;
    }
    else if ((!strcmp(argv[i], fpucount_op)) && (i < (argc - 1))){
      i++;
      fpucount = atoi(argv[i]);
      burnfpu = 1;
    }

    // the blocksize arg
    else if ((!strcmp(argv[i], blocksize_op)) && (i < (argc - 1))){
      i++;
      blocksize = (long unsigned int)atol(argv[i]);
    }

    // this must be the list of dir's for BurnStorage
    else if ((!strcmp(argv[i], burndirlist_op)) && (i < (argc - 2))){

    // allocate the burndirlist array
    burndirlist = (char **)malloc(sizeof(char **) * ((argc - i) / 2));

    // allocate the storage2burn array
    storage2burn = (size_t *)malloc(sizeof(size_t) * ((argc - i) / 2));

      // now go and extract all of the directories and sizes
      for (;i < (argc - 1);) {

        i++;
        // the first arg is the directory to burn
        // we have to allocate storage in the array first
        // you have to add one because strlen cuts only prints the number of
        // printable characters, it cuts off the ending \0
        *(burndirlist + numberofburndirs) = (char *)malloc(strlen(argv[i]) + 1);

        // copy the string over
        *(burndirlist + numberofburndirs) = strcpy(*(burndirlist + numberofburndirs),argv[i]);

        i++;

        // check if i < argc
        if (!(i < argc)){
          // user probably didn't give us a valid dir-storage2burn combo
          printf("Error! Invalid -dirlist argument\n");
          exit(EXIT_FAILURE);
          }

        // the second arg is the amount of storage to burn
        // we don't need to allocate storage for it because we have already
        // done so

        // and copy the size over
        *(storage2burn + numberofburndirs) = (size_t)atol(argv[i]);

        numberofburndirs++; // we're done with the allocating so increment this

      }
    }

  }

  // get on with the rest of the program :)
  BurnLoop();

  return 0;
}

// displays the command line help
// nothing more
void DisplayHelp()
{

  printf("Lucifer V1.0 - A burn-in utility\n\n");

  printf("General Options:\n\n");
  printf("-c     Turn off checking\n");
  printf("-s     Stop on errors\n");
  printf("-t x   Display interval in seconds\n");
  printf("-l x   Loop x times and exit\n");
  printf("--nodisplay\n");
  printf("       Don't display anything but errors\n\n");

  printf("Burning Routine control:\n\n");
  printf("--nomem      Don't burn the memory\n");
  printf("--nocpu      Don't burn the cpu\n");
  printf("--nofpu      Don't burn the fpu\n");
  printf("--nostorage  Don't burn the storage\n\n");

  printf("How often to run the routines:\n\n");
  printf("--memc x     Run the memory routine x times per loop\n");
  printf("--cpuc x     Run the cpu routine x times per loop\n");
  printf("--fpuc x     Run the fpu routine x times per loop\n");
  printf("--storc x    run the storage routine x times per loop\n\n");

  printf("Routine specific options:\n\n");
  printf("-m x                  The amount of memory to burn\n");
  printf("-dirlist dir size...  Directory list for the storage burning routine.  This\n");
  printf("                      option *must* go at the end of the option list.\n");
  printf("-b blocksize          The blocksize used\n\n");

  printf("All bug reports should go to pc_toilet@hotmail.com\n");

}

void DisplayVersion()
{

  printf("%s\n",version);

}

// the main burn-in loop
void BurnLoop()
{

  time_t lasttime;

  loopsdone = 0;

  // get the time
  if (displayinterval) lasttime = time(NULL);

  do {

    // should we only run for x loops?
    if (loops != 0) loopsdone++;

    if (check){
      if (burnmem) BurnMemory(memorycount);
      if (burnstorage) BurnStorage(storagecount);
      if (burncpu) BurnCPU(cpucount);
      if (burnfpu) BurnFPU(fpucount);
    }
    else {
      if (burnmem) BurnMemoryNoCheck(memorycount);
      if (burnstorage) BurnStorageNoCheck(storagecount);
      if (burncpu) BurnCPUNoCheck(cpucount);
      if (burnfpu) BurnFPUNoCheck(fpucount);
    }

    if (stoponerror && error)
      stop = 1;

    if (loops && loopsdone >= loops) stop = 1;

    // display only if display interval = 0 or if more then display interval
    // seconds has passed between now and the last time the display was
    // updated or if this is the last loop
    if (!nodisplay) {
      if (!displayinterval || (loopsdone >= loops && loops))
        UpdateDisplay();
      else if (difftime(time(NULL),lasttime) > displayinterval) {
        UpdateDisplay();
        // get the time
        lasttime = time(NULL);
      }
    }

  } while (!stop);

}

// updates the display with new Important Info
void UpdateDisplay()
{
  long unsigned int i;

  // display options

  printf("Current Options:\n");

  printf("Check - %d Stop on Error - %d Loops - %d Display Interval - %d\n"
  ,check,stoponerror,loops,(int)displayinterval);

  printf("\n");

  if (!burnmem)
    printf("Memory - Off\n");
  else{
    printf("Memory - %d Mem2Burn - %lu\n",memorycount,(long unsigned int)mem2burn);
  }

  printf("\n");

  if (!burnstorage)
    printf("Storage - Off\n");
  else{
    printf("Storage - %d Block Size - %lu\n\n",storagecount,(long unsigned int)blocksize);

    printf("DirList:\n");

    // print the dirlist
    for (i = 0; i < numberofburndirs; i++){

      printf("%s - %lu\n",*(burndirlist + i),(long unsigned int)*(storage2burn + i));
    }
  }

  printf("\n");

  if (!burncpu)
    printf("CPU - Off\n");
  else
    printf("CPU - %d\n",cpucount);

  printf("\n");

  if (!burnfpu)
    printf("FPU - Off\n");
  else
    printf("FPU - %d\n",fpucount);

  printf("\n");

  printf("Current Status:\n");

  if (loops) 
    printf("Loops Completed - %d Errors - %d\n",loopsdone,error);
  else 
    printf("Errors - %d\n",error);

  printf("\n");
}

/* signal handler, just sets stop to 1 */
int signalhandler(int sig)
{

  stop = 1;

  return 0;
}



