/***********************************************************************
*                                                                      *
* This program is free software; you can redistribute it and/or modify *
* it under the terms of the GNU General Public License as published by *
* the Free Software Foundation; either version 2 of the License, or    *
* (at your option) any later version.                                  *
*                                                                      *
* This program is distributed in the hope that it will be useful,      *
* but WITHOUT ANY WARRANTY; without even the implied warranty of       *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
* GNU General Public License for more details.                         *
*                                                                      *
* You should have received a copy of the GNU General Public License    *
* along with this program; if not, write to the Free Software          *
* Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*                                                                      *
***********************************************************************/

/**********************************************************************/
/*								      */
/*	whatfor - description wrapper for directory program 'ls'      */
/*								      */
/*	Version 0.3.02						      */
/*								      */
/* Created by   Craig Van Degrift April 3, 1999		  	      */
/*		KanjiFlash@CompuServe.com			      */
/*								      */
/* Last Revision: August 13, 1999				      */
/*								      */
/* 'whatfor' is designed to help newbies understand what the hundreds */
/* of files and directories on their system are there for.	      */
/*								      */
/* whatfor is run like the GNU directory listing utility 'ls' with    */
/* nearly all of the same options. It produces single column listings */
/* which have appended file description information gleened from a    */
/* database, 'whatfor_db', nominally placed in /usr/local/share/misc, */
/* but optionally placed anywhere an environmental variable           */
/* WHATFOR_DB dictates.						      */
/*								      */
/* A complete description of options, etc., is given by an associated */
/* man page, "whatfor.1".					      */
/*								      */
/* Color, type indicators (as in 'ls -F'), and long formats are	      */
/* are supported, but quoted names ('ls -Q') are not.  		      */
/*								      */
/* Please send all bug reports, enhancement suggestions, and database */
/* additions to the author at the above e-mail address.		      */
/*								      */
/* 'whatfor' actually runs your 'ls' program in a pipe and then tacks */
/* on the descriptions.  If your system's 'ls' is not the GNU 'ls' or */
/* has some unusual options set under an alias, 'whatfor' may not run */
/* correctly. 'whatfor' was developed on a Linux i386 system so its   */
/* database may not be so well matched to other systems.  Edit as     */
/* appropriate.							      */
/*								      */
/*								      */
/* Database format:						      */
/*								      */
/* The human-readable, ascii database format closely resembles the    */
/* output from "ls -R1p" so that "ls" might be helpful in creating    */
/* additional database entries.	Each directory name or filename entry */
/* is separated by at least one tab or space from a following         */
/* optional description entry.	Newlines '\n' terminate each entry.   */
/*								      */
/* Directory name entries must begin with a '/' and end with a ':'.   */
/* They must not have a '/' immediately before the terminating ':'.   */
/*								      */
/* Filename entries can only contain a '/' at the end to indicate     */
/* that they are subdirectories as is done by 'ls -p'.	They must not */
/* begin with a '/' or contain one.				      */
/*								      */
/* File and directory names sometimes have version numbers attached,  */
/* particularly the names of libraries and source directories.  The   */
/* search rules used by whatfor ignore target endings that are formed */
/* by '-', '.', and digits unless the database entries contain exact  */
/* matches for the entire name, including those endings.  Since this  */
/* rule causes /dev/tty10 to match /dev/tty1, and these situations    */
/* are common and difficult to anticipate in /dev, the rule is not    */
/* used in /dev directory.  There is a danger that this rule will     */
/* still cause erroneous matches in other directories.		      */
/*								      */
/* Filename entries may also be terminated with '@', '|', '*', or '=' */
/* with the same meanings as defined by 'ls -F'.  When these marks    */
/* are present, the matching test will use them.		      */
/*								      */
/* The internal format for the database is a linked list of           */
/* directories with each specified directory entry associated with a  */
/* linked list of filename entries.  The top pointer of the directory */
/* list is dbf.first_dir[0]. Long directory names are linked ahead of */
/* shorter names except that a default directory "/*:" is linked      */
/* last, after "/:".						      */
/*								      */
/* dbf.first_dir[] is a 64 element array where elements 1..63 hold    */
/* pointers into the linked list of directory names based on the name */
/* lengths. This shortcut table is used to bypass names longer than   */
/* the target name.  It is built by the function fill_dir_table().    */
/* See comments before that function for details.		      */
/*								      */
/* The default directory specified as "/*:" contains files that might */
/* be found in any directory.  This wildcard file list is linked to   */
/* the final file in the filename list of each of the regular	      */
/* directory entries.  Thus, after the regular filenames are checked, */
/* the wildcard list is then also checked.  The first match ends the  */
/* search.  This extra link is only added to directories that are     */
/* read from "whatfor_db" after the "/*:" directory has been read.    */
/* Any directories read earlier are not linked to the wildcard        */
/* filename entries.						      */
/* 								      */
/* In addition, the wildcard directory entry itself is linked to the  */
/* end of the directory list so that it is used only if none of the   */
/* other specific directories match a directory being listed.	      */
/*								      */
/* The wildcard directory entry "/*:" may contain wildcard filename   */
/* entries like "*.c" which will be used for all filenames that end   */
/* in ".c" and have not already been matched.			      */
/*								      */
/* Only the ends of the directories are compared, but longer direct-  */
/* ories are checked before shorter ones.  If the directory 	      */
/* "/usr/bin:" is being displayed, the "whatfor_db" directory entry   */
/* "/bin:" will be considered a match and will be used if the more    */
/* specific entry "/usr/bin:" is not present.			      */
/*								      */
/* The info fields of the dir_info and file_info structures hold the  */
/* file/dir name followed by a '\0', and then the description         */
/* followed by another '\0'.  The offset of the description in the    */
/* info field is specified by the "desc_offset" fields.		      */
/*								      */
/* Any whitespace at the beginning of the description is discarded    */
/* as is the '\n' at its end.					      */
/*								      */
/*								      */
/* Additional notes concerning relevant characteristics of 'ls':      */
/*								      */
/* If a filename ends in a colon, 'ls' will present it in the same    */
/* manner as a directory whose name is the same without the colon.    */
/*								      */
/* This ambiguity is resolved by invisibly using the -Q option which  */
/* puts quote marks around the filename.  All directory name headings */
/* are listed with the ':" after the quote closing the quoted         */
/* filename, but any filename ending with a ":" will have the colon   */
/* inside of the quote pair.					      */
/*								      */
/* When 'ls' is given specific filenames alone without a path,	      */
/* it does not report the directory name (pwd). 'whatfor' invokes     */
/* the program 'pwd' to findout the present working directory.	      */
/*								      */
/* When 'ls' is given specific filenames with partial or full paths,  */
/* it uses those (relative to the pwd) and lists the file with the    */
/* partial or full paths, but does not report the directory name      */
/* separately.							      */
/*								      */
/* Also, when 'ls' is given a single directory name, it lists the     */
/* files in the specified directory without giving the directory      */
/* name or adding the directory name to the filename.		      */
/*								      */
/* 'whatfor' sorts this out by first running 'ls -Fd' with the        */
/* specified argument list and seeing which specified items were      */
/* were files and which were directories.  It then runs 'ls -FQ'      */
/* with the argument list to get the desired results and correctly    */
/* interpret them.						      */
/*								      */
/**********************************************************************/

#include <stdio.h>	/* printf, fprintf, putchar, sscanf, popen,  */
			/* pclose, fgetc, fgets, perror, stderr,     */
			/* fopen, fclose			     */

#include <stdlib.h>	/* malloc, free, exit, atexit, getenv	     */

#include <string.h>	/* strlen, strdup, strcpy, strchr, strrchr,  */
			/* strcmp, strncmp, strcat, strpbrk 	     */

#include <ctype.h>	/* isdigit 				     */

#include <unistd.h>	/* isatty, STDOUT_FILENO		     */

#define BUF_SIZE	8192
#define TRUE		1
#define FALSE		0
#define PATH_ENTRY	0
#define FILENAME_ENTRY	1
#define DIR_TABLE_SIZE	64

struct file_info {
	struct file_info* next; /* pointer to next filename         */
	int desc_offset;	/* offset of description in info    */
	char info[0];		/* filename+null+description+null   */
};				/* sizeof(struct file_info) = 8	    */

struct dir_info {
	struct dir_info* next;  /* pointer to next directory        */
	struct file_info* first_file;	/* pointer to first file    */
	int desc_offset;	/* offset of description in info    */
	char info[0];		/* significant part of dir path     */
};				/*  + null + description + null     */
				/* sizeof(struct dir_info) = 12     */

/* When binary storage of the whatfor database is implemented, the  */
/* following structure must contain all necessary information to    */
/* use the binary data.						    */

struct db_file {
	const char magic[20];
	int size;	/* size of allocation needed for entire db */
	const int table_size;
	struct dir_info* first_dir[DIR_TABLE_SIZE]; /* pointer to directory links */
} dbf={"binary whatfor db\0\0\0",sizeof(struct db_file),DIR_TABLE_SIZE,{(struct dir_info*)NULL}};

char buf[BUF_SIZE];
char* pwd	 = NULL;	/* pwd + '/' at end */
char* db_path 	 = NULL;
char* ls_command = NULL;
const char* environ_key	    = "WHATFOR_DB";
const char* default_db_path = "/usr/local/share/misc/whatfor_db";

const char* version	    = "whatfor 0.3.02\n GPL August 13, 1999\n by Craig Van Degrift\n KanjiFlash@CompuServe.com";

const char* help 	    = \
"Usage: whatfor [--gap=COLS] [options and file specs of \"ls\" command]\n\n"
"Runs \"ls\" command and appends additional descriptive information from a\n"
"database \"whatfor_db\" located in \"/usr/local/share/misc\" or in a \n"
"directory specified by an environmental variable \"WHATFOR_DB\".\n\n" \
"See \"whatfor\" and \"whatfor_db\" man pages for more information.\n";

struct dir_info* default_dir = NULL;  /* "/*:" entry in "whatfor_db" */

int verbose	   = 0;		/* --verbose options increment this  */
int gap		   = 12;	/* --gap=cols option resets this     */
int show_types     = FALSE;	/* -F option of 'ls' */
int long_format    = FALSE;	/* -l option of 'ls' */
int show_files     = FALSE;	/* -p option of 'ls' */
int illegal_option = FALSE;	/* -x option of 'ls' */


/* The following function prototypes are listed in order according   */
/* to the appearance of their definitions.			     */

void		 process(void);
char*		 uncolor(char*);
char*		 unquote(char*);
void		 print_file_info(const char*, struct dir_info*, char*);
struct dir_info* print_dir_info(char*);
struct dir_info* get_cur_dir(char*);
char*		 merge_paths(const char*, char*);
void		 get_pwd(void);
void		 get_db(FILE* db);
void		 load_db(void);
void		 load_ascii_db(FILE*, char*);
void		 no_memory(void);
void		 load_binary_db(FILE*);
void		 save_db(void);
void		 insert_new_dir(const struct dir_info*, struct dir_info*);
void		 show_links(void);
void		 fill_dir_table(void);
void		 free_storage(void);



int main(int argc, char* argv[])
{
	int   i;
	char* cp;


	/* The following is not necessary, but it makes me feel good */

	atexit(free_storage);

	get_pwd();

	strcpy(buf,"ls ");
	for (i=1; i < argc; i++) {
		if (argv[i][0] == '-') {
			if (!strcmp(argv[i],"--version")) {
				printf("%s\n",version);
				exit(0);
			}
			else if (!strcmp(argv[i],"--help")) {
				printf("%s\n",help);
				exit(0);
			}
			else if (!strcmp(argv[i],"--verbose")) {
				verbose++;
				continue;
			}
			else if ((cp=strstr(argv[i],"--gap="))) {
				sscanf(cp+6,"%d",&gap);
				continue;
			}
			else if (!strcmp(argv[i],"--color=auto") && isatty(STDOUT_FILENO)) {
				argv[i] = "--color=always";
			}
			else if (argv[i][1] != '-') {
				show_types |= (int)strchr(argv[i],'F');
				long_format |= (int)strchr(argv[i],'l');
				show_files |= (int)strchr(argv[i],'p');
				illegal_option |= (int)strpbrk(argv[i],"Qmx");
				if ((cp=strchr(argv[i],'l')))
					*cp='F'; /* Delete the 'l' for now */	
			}
		}

		if (strlen(buf)+strlen(argv[i])+4 > sizeof(buf) ) {
			fprintf(stderr,"\awhatfor: Command line argument too long - Max is %d\n", i=sizeof(buf));
			exit(1);
		}

		strcat(buf,argv[i]);
		strcat(buf," ");
	}

	if (illegal_option) {
		fprintf(stderr,"\am,x & Q options of 'ls' are incompatible with whatfor.\n");
		exit(1);
	}

	strcat(buf,"-Fd"); /* -Fd is useful to resolve ambiguities */
			   /* Use of these options here is made    */
			   /* transparent to the user.	           */
 
	ls_command = strdup(buf);

	if (verbose)
		fprintf(stderr,"Verbose messaging set\n" \
		"nominal gap from start of name to start of description = %d\n",gap);

	load_db();
	process();

	exit(0);
}

void process(void)
{
	FILE* pipein;
	int buf_len;
	struct dir_info* cur_dir;
	char* assumed_path = NULL;

	if (verbose)
		fprintf(stderr,"Invoking:%s\n",ls_command);

	pipein = popen(ls_command,"r");

	/* "ls" doesn't list the path unless it is printing more */
	/* than one directory, so we have first used the 'ls -d' */
	/* option to figure out the assumed path first.          */

	while (fgets(buf, sizeof(buf), pipein)) {
		uncolor(buf);
		buf_len = strlen(buf);
		if (buf_len < 2)
			continue;
		if (buf[buf_len-2] == '/') {
			if (!assumed_path) {
				buf[strlen(buf)-1] = ':';
				assumed_path = strdup(buf);
			}
		}
		else if (assumed_path && strcmp(assumed_path,".:")) {
			free(assumed_path);
			assumed_path = strdup(".:");
		}
		else if (!assumed_path)
			assumed_path = strdup(".:");
	}

	pclose(pipein);

	/* This is for when 'ls' bombs - invalid filename, etc. */
	if (!assumed_path)
		exit(0);

	strcpy(buf,assumed_path);
	free(assumed_path);
	assumed_path=merge_paths(pwd, buf);

	if (verbose)
		fprintf(stderr,"assumed_path=%s\n",assumed_path);

	/* Now replace d option with Q option for the real thing */

	ls_command[strlen(ls_command)-1] = 'Q';

	/* Reinsert the 'l' option where the 'F' option is */
	/* since an 'F' option was earlier placed where    */
	/* the 'l' option was to prevent 'l' during first run of 'ls' */

	if (long_format)
		ls_command[strlen(ls_command)-2] = 'l';
	
	pipein = popen(ls_command,"r");

	while (fgets(buf, sizeof(buf), pipein)) {
		buf_len = strlen(buf);
		if (buf_len < 2)
			printf("%s",buf);
		else if (buf[buf_len-2] == ':') {
			free(assumed_path);
			assumed_path = NULL;
			cur_dir = print_dir_info(unquote(buf));
		}
		else
			print_file_info(assumed_path,cur_dir,unquote(buf));
	}
	pclose(pipein);

	free(assumed_path);
	assumed_path = NULL;
}


/* For color, the entry is prefixed by				   */
/*      '\033' '[' {attribute and color codes} 'm' 		   */ 
/* then the text is given, and then the following suffix is passed */
/*	'033' '[' 'm'						   */
/* The input string is changed (not a copy) and a pointer to it    */
/* returned.							   */

char* uncolor(char* s)
{	
	char* pin  = s;
	char* pout = s;

	while (*pin) {
		while (*pin && *pin != '\033')
			*pout++ = *pin++;
		if (!*pin)
			break;
		pin++;
		while (*pin && *pin != 'm')
			pin++;
		if (!*pin)
			break;
		pin++;
	}
	*pout = '\0';
	return s;
}
	

/* The following removes all double quotes, including escaped ones  */
/* from a string and returns a pointer to the changed input string. */
/* Are there other escapes that need to be handled when undoing     */
/* 'ls -Q' output?						    */

char* unquote(char* s)
{
	char* pin  = s;
	char* pout = s;

	do {
		if (*pin == '\"' || (*pin == '\\' && (*(pin+1) == '\"' || *(pin+1) == ':')))
			pin++;
		*pout++ = *pin;
	} while (*pin++);

	return s;
}
		

/* Function to locate and print the file description	*/

void print_file_info(const char* assumed_path, struct dir_info* cur_dir, char* text)
{
	struct file_info* cur_file;
	char last_c;
	char* filename_pos;
	char* file_path;
	char* link_pos;
	char* textp;
	char* infop;
	int   full_name_len = 0;

	/* Ignore blank lines from ls */

	if (*text == '\n') {
		printf("%s",text);
		return;
	}


	/* Drop terminal \n or return if there is none. The latter */
	/* case can happen when colored filenames are present.     */

	if (strlen(text) && *(textp = text + strlen(text)-1) == '\n') {
		*textp = '\0';
	}
	else {
		printf("%s",text);
		return;
	}


	/* If "ls -l" and line has block totals, print them and return */

	if(long_format && !strncmp(text,"total ",6)) {
		textp = text+6;
		while (*textp && isdigit(*textp))
			textp++;
		if (!*textp) {
			printf("%s\n",text);
			return;
		}
	}

	/* Extract file type from "ls -F" and replace it with a \0 */
	/* Use space to signify no file type (ordinary file)       */

	last_c = text[strlen(text)-1];
	if (!strchr("*/@|=",last_c))
		last_c = ' ';
	else
		text[strlen(text)-1] = '\0';


	printf("%s",text);

	if (show_types || (show_files && last_c == '/')) {
		full_name_len++;
		printf("%c",last_c);
	}

	uncolor(text);

	if (!strlen(text)) {
		fprintf(stderr,"\aEmpty buffer in print_file_info()\n");
		exit(1);
	}

	
	/* Locate start of filename */

	filename_pos = strrchr(text,' ');
	if (filename_pos)
		filename_pos++;
	else
		filename_pos=text;


	/* If "ls -l" and we have a link, find */
	/* start of link name and set last_c   */

	if (long_format && *(filename_pos-3)=='-' && *(filename_pos-2)=='>') {
		link_pos = filename_pos-5;
		while(*link_pos != ' ')
			link_pos--;
		link_pos++;
		last_c = '@';
	}
	else
		link_pos = filename_pos;	

	full_name_len += strlen(link_pos);


	if (assumed_path) {

		/* When 'ls' does not explicitly give directory path, */
		/* it may be partly from pwd and partly included in   */
		/* the filename so we must sort them out.	      */

		file_path = merge_paths(assumed_path, filename_pos);
		textp = strrchr(file_path,'/');

		if (!textp) {
			fprintf(stderr,"\aImpossible file_path in print_file_info():%s\n",file_path);
			exit(1);
		}

		/* Add ":\0" while preserving initial '/', */
		/* but otherwise replace final '/'         */

		if (textp == file_path)
			*++textp = ':';
		else
			*textp = ':';
		*++textp = '\0';

		cur_dir = get_cur_dir(file_path);

		free(file_path);
		file_path = NULL;

		textp=filename_pos;

/* Each new slash must restart a version check */

		while(*textp)
			if (*textp++=='/')
				filename_pos=textp;

	}


	/* Find description in database and print with specified gap */
	/* between start of name and description                     */

	if (cur_dir) {
		cur_file=cur_dir->first_file;

		while (cur_file) {
			textp=filename_pos;
			infop=cur_file->info;

			/* '*' wild card at start of whatfor_db file */
			/* entry signifies just test remainder       */

			if(*infop=='*' && *++infop!='\0') {
				textp += strlen(textp) - strlen(infop);
				if (*infop && strchr("*/@|=",infop[strlen(infop)-1]))
					textp++;
			}

			while (*textp && *infop && *textp==*infop) {
				textp++,infop++;
			}

			/* If the matched portion does not end in a   */
			/* digit and the unmatched portion is like a  */
			/* version number and we are not in /dev,     */
			/* then ignore the version number.	      */

			if (!strstr(cur_dir->info,"/dev:") && (textp==filename_pos || !isdigit(*(textp-1))))
				while (*textp && (isdigit(*textp) || *textp == '-' || *textp == '.'))
					textp++;

			/* Respect type spec if present in database */

			if (!*textp && (!*infop || last_c == *infop)) {
				while (full_name_len++ < gap - 1)
					putchar(' ');
				if (last_c == '@' && !(long_format || show_types))
					printf(" ->");
				printf(" %s",cur_file->info+cur_file->desc_offset);
				break;  /* Match found! */
			}

			cur_file = cur_file->next;
		}
	}
	printf("\n");
}


/* Function to print directory and its description.  It returns a  */
/* pointer to the matching directory entry in the database.	   */

struct dir_info* print_dir_info(char* name)
{
	int   filename_len = strlen(name);
	char* full_path;
	struct dir_info* cur_dir;

	name[--filename_len] = '\0';  /* remove terminal \n  */

	printf("%s",name);

	uncolor(name);
	full_path = merge_paths(pwd, name);
	full_path[strlen(full_path)-1] = ':'; /* replace terminal '/' by ':' */
	cur_dir = get_cur_dir(full_path);
	free(full_path);
	full_path = NULL;
	

	/* Find description in database and print with specified gap */
	/* between start of name and description                     */

	if (cur_dir && cur_dir->info[cur_dir->desc_offset] != '\0') {
		while (filename_len++ < gap - 1)
			putchar(' ');
		printf(" %s",cur_dir->info+cur_dir->desc_offset);
	}
	printf("\n");
	return cur_dir;
}


/* Find and return pointer that points to the dir_info structure  */
/* of the database that corresponds to the name	parameter.	  */
/* name must begin with a '/' and end with a \:'.		  */
/*								  */
/* Version number endings do not disqualify a match from being    */
/* true.  If a default directory exists, it will be used if no    */
/* if no other match is found.					  */

struct dir_info* get_cur_dir(char* name)
{
	struct dir_info* cur_dir = NULL;
	int name_len = strlen(name);
	int i,j;

	if (name_len < DIR_TABLE_SIZE)
		cur_dir = dbf.first_dir[name_len];
	else
		cur_dir = dbf.first_dir[0];

	if (name_len <= 1 || name[0] != '/' || name[name_len-1] != ':') {
		fprintf(stderr,"\aget_cur_dir: Invalid name:%s\n",name);
		exit(1);
	}


	while(cur_dir && !strstr("/*:",cur_dir->info)) {
//printf("name=%s cur_dir->info=%s\n",name,cur_dir->info);
		i = name_len-1;
		j = cur_dir->desc_offset-2;

		while (j >= 0) {
		
			while (j >= 0 && name[i] == cur_dir->info[j]) { 
				i--;
				j--;
			}

			if (j < 0 || (name[i+1] != '/' && name[i+1] != ':'))
				break;

			/* Any mismatch at this point might be   */
			/*    caused by a version number so      */
			/*    ignore any possible version number */
			/*    at end of each section of name,    */
			/*    the listing's path.		 */

			if (isdigit(name[i]) || name[i] == '-' || name[i] == '.')
				while (isdigit(name[--i]) || name[i] == '-' || name[i] == '.');
			else
				break;
		}
		if (j < 0)
			break;

		cur_dir = cur_dir->next;
	}

	if (verbose > 1) {
		fprintf(stderr,"get_cur_dir: ");
		if (cur_dir)
			fprintf(stderr,"Found =%s ",cur_dir->info);
		else
			fprintf(stderr,"No match ");
		fprintf(stderr,"for %s\n",name);
	}
	return cur_dir;	
}


/* The following function merges a specified abbreviated dir    */
/* argument (from the command argument or from the "ls" output) */
/* with the present working directory to form a full directory  */
/* name for the target directory.	 			*/
/* Directories ending with a ":" should be handled ok even      */
/* though "ls" also uses a ":" to terminate directory names on  */
/* its output.							*/
/*								*/
/* The path_start string must begin with a '/'.  Its final      */
/* character will be replaced by a '/'.				*/
/*								*/
/* The returned string must be eventually freed.		*/

char* merge_paths(const char* path_start, char* path_end)
{
	char* cp0;
	char* cp1;
	int   l_end = strlen(path_end);

	/* Remove terminating '\n' if present */

	if (l_end && path_end[l_end-1] == '\n')
		path_end[l_end-1] = '\0';
	
	cp0 = strdup(path_end);
	strcpy(buf,path_start);
	buf[strlen(path_start)-1] = '/';
	strcat(buf,cp0);
	free(cp0);

	cp1 = cp0 = buf;
	while (*cp0) {
//fprintf(stderr,"cp0=%s   cp1=%s   buf=%s\n",cp0,cp1,buf);
		if (!strncmp("/../",cp0,4)) {
			if (cp1 > buf) {
				cp1--;
				while (*cp1 !='/')
					cp1--;
			}
			cp0 += 3;
//fprintf(stderr,"Did /../\n");
			continue;
		}
		else if (!strncmp("/..:\0",cp0,5)) {
			if (cp1 > buf) {
				cp1--;
				while (*cp1 !='/')
					cp1--;
			}
			if (cp1 == buf)
				cp1++;
			cp0 += 3;
//fprintf(stderr,"Did /..:0\n");
		}
		else if (!strncmp("/.:\0",cp0,4)) {
			if (cp1 == buf)
				cp1++;
			cp0 += 2;
//fprintf(stderr,"Did /.:0\n");
		}
		else if (cp1 != buf
			  && (!strncmp("/:\0",cp0,3)
			  || !strncmp(".:\0",cp0,3))) {
			cp0++;
//fprintf(stderr,"Did .:0\n");
		}
		else if (!strncmp("/./",cp0,3)) {
			cp0 += 2;
//fprintf(stderr,"Did /./\n");
			continue;
		}
		else if (!strncmp("//",cp0,2)) {
			cp1=buf;
			cp0++;
//fprintf(stderr,"Did //\n");
			continue;
		}
		*cp1++ = *cp0++;
	}
	*cp1 = '\0';

	if (verbose > 1)
		fprintf(stderr,"merged path=%s\n",buf);

	return strdup(buf);
}


/* Runs "pwd" program in pipe to get present working directory and   */
/* appends a "/" to its end unless it is the top directory.	     */
/* Result saved in global variable "pwd". 			     */

void get_pwd(void)
{
	FILE* pipein;
	int err;
	int len;

	pipein = popen("pwd","r");
	err = fgets(buf, sizeof(buf), pipein) == NULL;
	pclose(pipein);

	if (err) {
		perror("Problem getting pwd");
		exit(1);
	}


	/* Append '/' if not top dir and delete '\n' at end  */

	len = strlen(buf);
	if (len == 2 )
		buf[1] = '\0';
	else
		buf[len-1] = '/';


	pwd = strdup(buf);

	if (verbose)
		fprintf(stderr,"pwd is %s\n",pwd);

}


/* The following function locates and opens the whatfor_db, sets    */
/* the db_path global variable, and returns a file pointer to the   */
/* open database.						    */
/*								    */
/* It first checks for an environment variable "WHATFOR_DB" and,    */
/* if found, will use the database file specified there. 	    */
/* Otherwise, it will look for & use 				    */
/*                "/usr/local/share/misc/whatfor_db". 		    */

void load_db(void)
{

	FILE* db = NULL;	
	char* env;

	if ((env = getenv(environ_key))) {
		strcpy(buf,env);
		db = fopen(buf,"r");
		if (db == NULL) {
			fprintf(stderr,"\a\"whatfor\" database file \"%s\" specified by\n"
			" environment variable \"%s\" could not be opened.\n"
			" Will try \"%s\".\n",buf,environ_key, default_db_path);
		}
	}

	if (db == NULL) {
		strcpy(buf,default_db_path);
		db = fopen(buf,"r");
	}

	if (db == NULL) {
		fprintf(stderr,"\aUnable to open whatfor database %s\n", buf);
		exit(1);
	}
	else {
		db_path=strdup(buf);
		if (verbose) {
			fprintf(stderr,"Successfully opened database %s\n",db_path);
		}
	}

	get_db(db);

	fclose(db);
}


/* The following function determines whether the database is an       */
/* ascii-formatted file or a digested and compressed binary file and  */
/* calls either load_ascii_db or load_binary_db.		      */
/*								      */
/* The binary format has not yet been implemented.		      */

void get_db(FILE* db)
{
	int c;
	int db_is_ascii;
	char* cp;

/* Now to put the first line or string into buf and figure out */
/*   whether the database file is a binary file or an ascii file */

	cp=buf;
	while((c=fgetc(db))!=EOF && (*cp++=c)!='\n' && c!='\0' && cp<buf+sizeof(buf))
		;

	*cp='\0'; /* Ensure termination by '\0' */

	db_is_ascii=strcmp(buf,dbf.magic);

	if (verbose) {
		fprintf(stderr,"Whatfor database is ");
		if (db_is_ascii)
			fprintf(stderr,"an ASCII source\n");
		else
			fprintf(stderr,"a binary source\n");
	}

	if (db_is_ascii) 
		load_ascii_db(db, buf);
	else
		load_binary_db(db);
}


/* If a digested and compressed binary database file is implemented, */
/* this function will save it.					     */

void save_db(void)
{
}


/* Read the ASCII whatfor database and construct linked lists of     */
/* directory entries, each in turn with a linked list of filename    */
/* entries.							     */
/*								     */
/* If a default directory "/*:" is present, it is linked to the end  */
/* of the directory list and its files are linked to the ends of the */
/* filename lists of all subsequently entered directories.  The      */
/* default directory is linked to the end of the directory list.     */

void load_ascii_db(FILE* db, char* first_line)
{

	char* cp;
	int entry;
	int info_len;
	int name_len;
	int size_inc;
	struct dir_info* new_dir=NULL;
	struct file_info* last_file=NULL;
	struct file_info* new_file=NULL;

	if (first_line != buf)
		strcpy(buf,first_line);

	do {	/*  database string is in buf */

		buf[strlen(buf)-1] = '\0'; /* remove terminal '\n' */

		if (*buf=='#' || *buf=='!') {
			if (verbose > 3)
				fprintf(stderr,"Comment: %s\n",buf);
			continue;
		}

		cp = buf;
		while (*cp && *cp != '\n' && *cp != '\t' && *cp != ' ')
			cp++;
		*cp='\0';  /* terminate name with '\0' */

		name_len = (int)(cp - buf);

		if (name_len > 1 && buf[0] == '/' && *(cp-1) == ':') {
			entry = PATH_ENTRY;
			if (verbose > 2)
				fprintf(stderr,"Dir:%s",buf);
		}
		else if(name_len > 0) {
			entry = FILENAME_ENTRY;
			if (verbose > 2)
				fprintf(stderr,"   File:%s",buf);

			if (new_dir == NULL) {
				fprintf(stderr,"\aError in whatfor database %s.\n First entry (%s) is not a directory path.\n",db_path,buf);
				exit(1);
			}

		}
		else
			continue;

		while ((*++cp) && (*cp == '\t' || *cp == ' ' || *cp == '\n'))
			;
		info_len = strlen(cp);

		if (verbose > 3)
			if (!info_len)
				fprintf(stderr,"\n");
			else
				fprintf(stderr," INFO:%s\n",cp);
			

		if (entry == PATH_ENTRY) {
			size_inc = sizeof(struct dir_info)+name_len+info_len+2;
			new_dir = (struct dir_info*)malloc(size_inc);
			if (!new_dir)
				no_memory();
			dbf.size += size_inc;

			new_dir->desc_offset = name_len+1;
			strcpy(new_dir->info,buf);
			strcpy(new_dir->info+new_dir->desc_offset,cp);

			if (default_dir)
				new_dir->first_file = default_dir->first_file;
			else
				new_dir->first_file = NULL;
			last_file = NULL;

			if (!strcmp(buf,"/*:"))
				if (default_dir) {
					fprintf(stderr,"\aError in whatfor database %s.\n It has more than one default directory (/*:).\n",db_path);
					exit(1);
				}
				else
					default_dir = new_dir;

			insert_new_dir(default_dir,new_dir);
		}
		else {
			/* entry is a FILENAME_ENTRY */

			size_inc = sizeof(struct file_info)+name_len+info_len+2;
			new_file = (struct file_info*)malloc(size_inc);
			if (!new_file)
				no_memory();
			dbf.size += size_inc;

			new_file->desc_offset = name_len+1;
			strcpy(new_file->info,buf);
			strcpy(new_file->info+new_file->desc_offset,cp);

			if (last_file)
				last_file->next = new_file;
			else
				new_dir->first_file = new_file;
			last_file = new_file;

			if (default_dir && new_dir != default_dir)
				new_file->next = default_dir->first_file;
			else 
				new_file->next = NULL;
		}
	} while (fgets(buf, sizeof(buf), db));

	fill_dir_table();

	if (verbose>3)
		show_links();

	if (verbose)
		fprintf(stderr,"End of reading of database. dbf.size=%d\n",dbf.size);

	save_db();
}


/* Routine to handle malloc failure */

void no_memory(void)
{
	fprintf(stderr,"Out of memory (malloc returned NULL\n");
	exit(1);
}


/* The following diagnostic routine that prints the complete set of  */
/* linked filename and directory lists (without descriptions).	     */

void show_links(void)
{
	struct dir_info* d;
	struct file_info* f;

	fprintf(stderr,"\nThe completed link tree is:\n");
	d = dbf.first_dir[0];
	while (d) {
		fprintf(stderr,"%s\n",d->info);
		f = d->first_file;
		while (f) {
			fprintf(stderr,"    %s\n",f->info);
			f = f->next;
		}
		d = d->next;
	}

}


/* This routine inserts a directory entry in the directory linked  */
/* list with longer named ones first; earlier entries are included */
/* before later ones of the same length.  The default entry, if    */
/* present, is placed at the end.				   */

void insert_new_dir(const struct dir_info* default_dir, struct dir_info* new_dir)
{
	struct dir_info* d;
	int name_len = new_dir->desc_offset - 1;

	if (dbf.first_dir[0] == NULL) {
		dbf.first_dir[0] = new_dir;
		new_dir->next = NULL;
		return;
	}
	if (dbf.first_dir[0] == default_dir || name_len > strlen(dbf.first_dir[0]->info)) {
		new_dir->next = dbf.first_dir[0];
		dbf.first_dir[0] = new_dir;
		return;
	}

	d = dbf.first_dir[0];
	while (d->next != NULL && d->next != default_dir && (new_dir == default_dir || name_len <= strlen(d->next->info))) {
		d = d->next;
	}

	new_dir->next = d->next;
	d->next = new_dir;
}


/* The directory table contains shortcut pointers into the directory */
/* linked list for different ranges of directory name length.  When  */
/* the target directory, for example, is 9 bytes long (including     */
/* initial '/' and final ':'), one only needs to check database      */
/* filename entries of that length or shorter. That way, for	     */
/* example, "/usr/bin:" is quickly matched with "/usr/bin:" in the   */
/* database or, if "/usr/bin:" is not there, it can work its way     */
/* down to "/bin:" or to the default directory entry, "/*:".	     */
/*								     */
/* dbf.first_dir[0] points to the entire list and must be used when  */
/* the target length is longer than DIR_TABLE_SIZE-1.		     */
/*								     */
/* dbf.first_dir[n] points to the earliest entry of length n and is  */
/* a good entry for directory names of length n (and shorter).	     */

void fill_dir_table(void)
{
	int i;
	int len;
	struct dir_info* d = dbf.first_dir[0];

	for (i=DIR_TABLE_SIZE-1; i>0; i--) {
		while (d->next && (i < (len = strlen(d->info)) || len > DIR_TABLE_SIZE-1))
			d = d->next;
		
		dbf.first_dir[i] = d;
	}

	if (verbose > 2) {
		fprintf(stderr,"\nDirectory pointer shortcut table follows:\n");
		for (i=0; i<DIR_TABLE_SIZE; i++)
			fprintf(stderr,"%d:%s\n",i,dbf.first_dir[i]->info);
		fprintf(stderr,"\n");
	}
}


/* This is to load the unimplemented binary format database file    */

void load_binary_db(FILE* db)
{
}


/* This is run by "exit()" and assumes that it is ok to free NULL    */
/* pointers and that the order of freeing of unlinked variables is   */
/* not critical. Since it appears that Linux maintains a local heap  */
/* for each executable which is automatically freed upon exiting,    */
/* this routine is probably overkill in Linux.  Still, it makes me   */
/* feel virtuous...						     */

void free_storage(void)
{
	struct dir_info* cur_dir = dbf.first_dir[0];
	struct dir_info* next_dir;
	struct file_info* cur_file;
	struct file_info* next_file;
	
	while(cur_dir) {
		cur_file = cur_dir->first_file;
		while(cur_file) {
			next_file = cur_file->next;
			if (default_dir && cur_dir !=default_dir && cur_file == default_dir->first_file)
				break;
			free(cur_file);
			cur_file = next_file;
		}

		next_dir = cur_dir->next;
		free(cur_dir);
		cur_dir = next_dir;

	}

	free(pwd);
	free(db_path);
	free(ls_command);

	if (verbose)
		fprintf(stderr,"\nStorage freed...  Bye-bye!\n");
}
