/* Copyright 2000 Servaas Vandenberghe, Leuven, Belgium, All rights reserved.
 This software is licensed under the GNU GENERAL PUBLIC LICENSE, version 2, or
 a later version at your option.

 Redistribution and use of this source code, with or without modification, is
 permitted provided that the following condition is met:

 1. Redistributions of this source code must retain the above copyright
    notice, this condition, and the following disclaimer.

  THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED
  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO
  EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
  OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
  OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
  ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  */

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>   /* getopt */
#include <stdio.h>
#include <string.h>
#include <errno.h> 

#ifndef VERBOSE
#define VERBOSE 0
#endif

#ifndef VERSION
#define VERSION "0.0"
#endif
static char version[]= VERSION ", " __DATE__ ", " __TIME__ ;

#define PROGLEN 1025
static char progname[PROGLEN]="toascii";
static char cname[]="toascii";

static void help(void) {
  printf("usage: %s [options] [from-file [to-file]] \n", progname);
  printf("%s substitutes control characters by spaces and replaces ",cname);
  printf("cr,lf[,ff] by lf. \n");
  printf("-B   remove 0x80 from the input stream \n"
         "-b   remove control characters from the stream, except for 0x80 \n"
	 "-i   pass iso-8859 range [0xA0..0xFF] \n"
	 "-F   flush the output stream per processed character \n"
	 "-p   preserve control characters; do not substitute by spaces \n"
	 "-R   reverse the cr,lf processing; replace lf by cr,lf \n"
	 "-w   wp5 to iso-8859 mapping\n"
	 );
}

/*
 *			M A I N
 *
 *	This routine opens the file and reads date and attributes.
 *
 */
#define DEBUG 0
#define BUFSIZE 128
int main(int argc, char *argv[]) {
  char where[]="toascii main";
  FILE *fpout=stdout, *fpin=stdin;
  char *fromfile=NULL, *tofile=NULL;
  unsigned ins, inread, outwrite;
  unsigned char inc[BUFSIZE], *inpnt, inprev;
  int err=0;
  char newlines[]="\x0d\x0c\x0a";
  char wpin[] ="\x08\x0F\x11\x12\x80\x84\xCC";
  char wpout[]="\xE0\xE9\xE8\xEA\x20\x2D\x0A";
  char *wp_in[]={ "\xF0\x1C\x04\xF0", "\xF0\x33\x01\xF0", 
		  "\xF0\x3D\x01\xF0", 
		  "\xF0\x20\x04\xF0", /* bold on */
		  "\xF0\x1F\x04\xF0", /* bold off */
		  NULL};
  char *wp_out[]={ "'", "\xEE", "\xF4", "*-", "-*"};
  unsigned wp_inlen[]={4, 4, 4, 4,4};
  unsigned wp_outlen[]={1,1, 1, 2,2};
  char *xlatin=NULL, *xlatout, **xlat_in=NULL, **xlat_out;
  unsigned *xlat_inlen, *xlat_outlen;
  int opt, files;
  extern char *optarg;   /* pointer to the argument of the option */
  extern int optind;     /*, opterr, optopt; */
  int opt_noblank=0;     /* do not substitute a ctrl char by a space */
  int opt_noxblank=0;    /* do not substitute 0x80 by a space */
  int opt_pres=0;        /* 0x80 is a breakable space in word perfect */
  int opt_todos=0;       /* map lf to cr,lf */
  int opt_flush=0;
  int opt_wp=0;          /* wp5 to iso-8859 char mapping */
  int opt_iso=0;         /* pass iso-8859 range 0xA0..0xFF */

  optarg=NULL;  /* init getopt */
  optind=0;

  if(argc) strncpy(progname,argv[0],PROGLEN-1);
  progname[PROGLEN-1]=0;

  while ( (opt=getopt(argc, argv, "BbiFhpRwV")) != EOF ) {
    int this_optind=optind-1;
    switch(opt) {
    case 'B':
      opt_noxblank^=1;
      break;
    case 'b':
      opt_noblank^=1;
      break;
    case 'i':
      opt_iso^=1;
      break;
    case 'F':
      opt_flush^=1;
      break;
    case 'p':
      opt_pres^=1;
      break;
    case 'h':
      help(); goto end_error;
    case 'R':
      opt_todos^=1;
      break;
    case 'w':
      opt_wp^=1;
      break;
    case 'V':
      printf("%s %s\n", cname,version); goto end_error;
    case ':':
      fprintf(stderr,"%s: Expected parameter at option %i\n",where,
	      this_optind);
      err=10; goto end_error;	
    case '?':
      fprintf(stderr,"%s: Getopt returned error at option %i\n",
	      where,this_optind);
      err=11; goto end_error;
      break;
    default:
      fprintf(stderr,"%s: ?? getopt returned character code 0x%x\n",where,opt);
      err=12; goto end_error;
    }
  }

  if(opt_wp) {
    xlatin=wpin;
    xlatout=wpout;
    xlat_in=wp_in;
    xlat_out=wp_out;
    xlat_inlen=wp_inlen;
    xlat_outlen=wp_outlen;
  }
  /*** check if file name present ***/
  files=argc-optind;
  if (files>=1) {
    fromfile=argv[optind];
    if(files>=2) tofile=argv[optind+1];
    if(files>2) {
      fprintf(stderr,"%s: Extra garbage at command line\n",where);
      err=1; goto end_error;
    }
  }

  if(fromfile && strlen(fromfile)) {  /* open the fromfile */
    if(!(fpin=fopen(fromfile, "r"))) {
      fprintf(stderr,"%s: fopen %s: %u %s\n",where,fromfile,errno,
	      strerror(errno));
      err=2; goto end_error;
    } 
  } /* if(fromfile) */

  if(tofile && strlen(tofile)) {  /* open the tofile */
    if(fromfile && !strcmp(fromfile,tofile)) {
      fprintf(stderr,"%s: fopen %s: equal files\n",where,tofile);
      err=3; goto end_error;
    }
    if(!(fpout=fopen(tofile, "w"))) {
      fprintf(stderr,"%s: fopen %s: %u %s\n",where,tofile,errno,
	      strerror(errno));
      err=4; goto end_error;
    } 
  } /* if(tofile) */

  /*** process ***/
  inprev=0;
  ins=inread=fread(inc, sizeof(inc[0]), BUFSIZE, fpin);
  inpnt=inc;
  for(; ins; ) { /* inpnt[0..ins-1] needs processing */
    unsigned char outc[3]={*inpnt, ' ', ' '};
    unsigned outs=1, inrem=0;

    /* printf("*%02X*", bytein); */
    if(*inpnt==0) {
      inrem=1;
      if(!opt_pres) outc[0]=' ';
      if(opt_noblank) outs=0;
    } else if(strchr(newlines, *inpnt)) { /* got cr,ff,lf */
      inrem=1;
      outs=0;
      if(inprev==0 || !strchr(newlines, inprev) || inprev==*inpnt) {
	/* prev char was no cr,ff,lf  or  equal */
	/* removes  cr lf  or  lf cr  sequence */
	if(opt_todos) outc[outs++]='\x0d';
	outc[outs++]='\x0a';
      }
    } else if(xlatin) {
      char *tmp;
      if((tmp=strchr(xlatin, *inpnt))) { /* xlat single char */
	unsigned off=(unsigned )tmp-(unsigned )xlatin;
	inrem=1;
	outc[0]=xlatout[off];
      }
    }

    if(!inrem && xlat_in) {
      unsigned u;
      for(u=0; xlat_in[u]; u++) 
	if(ins>=xlat_inlen[u] && !strncmp(inpnt, xlat_in[u], xlat_inlen[u])) {
	  inrem=xlat_inlen[u];
	  strncpy(outc, xlat_out[u], xlat_outlen[u]);
	  outs=xlat_outlen[u];
	  break;
	}
    }
    if(!inrem) {
      if(*inpnt==128) {
	if(!opt_pres) outc[0]=' ';
	if(opt_noxblank) outs=0;
      } else if(*inpnt<32 || (*inpnt>=127 && (!opt_iso || *inpnt<160))) {
	if(!opt_pres) outc[0]=' ';
	if(opt_noblank) outs=0;
      }
    }

    if(!inrem) inrem=1;

    if(outs) {
      if((outwrite=fwrite(outc, sizeof(outc[0]), outs, fpout))<outs) break;
      if(opt_flush) 
	if(fflush(fpout)) break;
    }

    inprev=*inpnt;
    ins  -=inrem;
    inpnt =&inpnt[inrem];

    /* fill inc[] buffer */
    if(inread && ins<BUFSIZE/2) {
      unsigned infree=BUFSIZE-ins;

      memmove(inc, inpnt, sizeof(inc[0])*ins);
      inpnt=inc;

      inread=fread(&inc[ins], sizeof(inc[0]), infree, fpin);
      ins+=inread;
    }
  } /* for */

  if(ferror(fpin)) {
    fprintf(stderr,"%s: fread: %u %s\n",where,errno,strerror(errno));
    err=10;
  }
  if(ferror(fpout)) { /* error on write */
    fprintf(stderr,"%s: fwrite: %u %s\n",where,errno,strerror(errno));
    err=11;
  }

end_error:
  if(fpout && fpout!=stdout) 
    if(fclose(fpout)) {
      fprintf(stderr,"%s: fclose: %u %s\n",where,errno,strerror(errno));
      if(!err) err=98;
    }

  if(fpin && fpin!=stdin) 
    if(fclose(fpin)) {
      fprintf(stderr,"%s: fclose: %u %s\n",where,errno,strerror(errno));
      if(!err) err=99;
    }

  return(err);
}
#undef BUFSIZE
#undef DEBUG

#undef PROGLEN
#undef VERBOSE
#undef VERSION
