/* released under the terms and conditions of the GPL */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <dirent.h>
#include <ctype.h>

#include <sys/stat.h>
#include <sys/types.h>

#define SLS_NULL  0x00

#define SLS_NODIR 0x01		/* no directory dereference */
#define SLS_CHILD 0x02		/* indent by one */

#define SLS_ALL   0x04		/* also show .files */
#define SLS_AALL  0x08		/* almost all - no . .. */

#define SLS_BRIEF 0x10		/* list files as single line */
#define SLS_TAG   0x20		/* tag files *@/|= */

uid_t myuid;
gid_t mygid;
time_t mytime;

#define LINKB 1024
#define DIRB  1024

int perfile(char *fname, int opts)
{
  struct stat s;
  struct tm p, q, r, *t;
  time_t td;
  int result = 0;
  off_t o, m;
  char b[LINKB];
  int bl;
  DIR *d;
  struct dirent *de;
  char c;
  int a;

  if (lstat(fname, &s)) {
    fprintf(stderr, "Can't stat file <%s>\n", fname);
    result = 1;
  } else {

    if (opts & SLS_BRIEF) {
      if (opts & SLS_CHILD) {
	putchar(' ');
      }
      printf("%s", fname);
      if (opts & SLS_TAG) {
	c = '\0';
	switch (s.st_mode & S_IFMT) {
	case S_IFSOCK:
	  c = '=';
	  break;
	case S_IFLNK:
	  c = '@';
	  break;
	case S_IFBLK:
	case S_IFCHR:
	  c = '#';
	  break;
	case S_IFDIR:
	  if (opts & (SLS_CHILD | SLS_NODIR)) {
	    c = '/';
	  } else {
	    c = ':';
	  }
	  break;
	case S_IFIFO:
	  c = '|';
	  break;
	}
	if (c != '\0') {
	  putchar(c);
	}
      }
    } else {

      if (opts & SLS_CHILD) {
	putchar(' ');
      }
      switch (s.st_mode & S_IFMT) {
      case S_IFSOCK:
	putchar('s');
	break;
      case S_IFLNK:
	putchar('l');
	break;
      case S_IFREG:
	putchar('-');
	break;
      case S_IFBLK:
	putchar('b');
	break;
      case S_IFDIR:
	if (opts & (SLS_CHILD | SLS_NODIR)) {
	  putchar('d');
	} else {
	  putchar('D');
	}
	break;
      case S_IFCHR:
	putchar('c');
	break;
      case S_IFIFO:
	putchar('f');
	break;
      default:
	putchar('?');
	break;
      }

      if (mygid == s.st_gid) {
	if (myuid == s.st_uid) {
	  putchar('b');
	  a = (S_IRUSR | S_IWUSR | S_IXUSR);
	} else {
	  putchar('g');
	  a = (S_IRGRP | S_IWGRP | S_IXGRP);
	}
      } else {
	if (myuid == s.st_uid) {
	  putchar('o');
	  a = (S_IRUSR | S_IWUSR | S_IXUSR);
	} else {
	  putchar('-');
	  a = (S_IROTH | S_IWOTH | S_IXOTH);
	}
      }

      switch (s.st_mode & (S_ISUID | S_ISGID | S_ISVTX)) {
      case (S_ISUID | S_ISGID | S_ISVTX):
	putchar('C');		/* christmas tree */
	break;
      case (S_ISUID | S_ISGID):
	putchar('B');		/* both suid and gid */
	break;
      case (S_ISUID | S_ISVTX):
	putchar('S');		/* sticky suid */
	break;
      case (S_ISGID | S_ISVTX):
	putchar('G');		/* sticky gid */
	break;
      case S_ISUID:
	putchar('s');
	break;
      case S_ISGID:
	putchar('g');
	break;
      case S_ISVTX:
	putchar('t');
	break;
      default:
	putchar('-');
	break;
      }

      switch (s.st_mode & (S_IRUSR | S_IRGRP | S_IROTH)) {
      case (S_IRUSR | S_IRGRP | S_IROTH):
	c = 'a';		/* all/anybody */
	break;
      case (S_IRUSR | S_IRGRP):
	c = 'b';		/* both owner and group */
	break;

      case (S_IRUSR | S_IROTH):
	c = 'q';		/* all except group */
	break;
      case (S_IRGRP | S_IROTH):
	c = 'e';		/* all except owner */
	break;

      case (S_IRUSR):
	c = 'o';		/* owner */
	break;
      case (S_IRGRP):
	c = 'g';		/* group */
	break;
      case (S_IROTH):
	c = 'n';		/* neither owner nor group */
	break;

      default:
	c = '-';		/* nothing at all */
	break;
      }
      if (s.st_mode & ((S_IRUSR | S_IRGRP | S_IROTH) & a)) {
	c = toupper(c);
      }
      putchar(c);

      switch (s.st_mode & (S_IWUSR | S_IWGRP | S_IWOTH)) {
      case (S_IWUSR | S_IWGRP | S_IWOTH):
	c = 'a';		/* all/anybody */
	break;
      case (S_IWUSR | S_IWGRP):
	c = 'b';		/* both owner and group */
	break;

      case (S_IWUSR | S_IWOTH):
	c = 'q';		/* all except group */
	break;
      case (S_IWGRP | S_IWOTH):
	c = 'e';		/* all except owner */
	break;

      case (S_IWUSR):
	c = 'o';		/* owner */
	break;
      case (S_IWGRP):
	c = 'g';		/* group */
	break;
      case (S_IWOTH):
	c = 'n';		/* neither owner nor group */
	break;

      default:
	c = '-';		/* nothing at all */
	break;
      }
      if (s.st_mode & ((S_IWUSR | S_IWGRP | S_IWOTH) & a)) {
	c = toupper(c);
      }
      putchar(c);

      switch (s.st_mode & (S_IXUSR | S_IXGRP | S_IXOTH)) {
      case (S_IXUSR | S_IXGRP | S_IXOTH):
	c = 'a';		/* all/anybody */
	break;
      case (S_IXUSR | S_IXGRP):
	c = 'b';		/* both owner and group */
	break;

      case (S_IXUSR | S_IXOTH):
	c = 'q';		/* all except group */
	break;
      case (S_IXGRP | S_IXOTH):
	c = 'e';		/* all except owner */
	break;

      case (S_IXUSR):
	c = 'o';		/* owner */
	break;
      case (S_IXGRP):
	c = 'g';		/* group */
	break;
      case (S_IXOTH):
	c = 'n';		/* neither owner nor group */
	break;

      default:
	c = '-';		/* nothing at all */
	break;
      }
#ifdef TRACE
      fprintf(stderr, "perfile(): fmode<%02o>,<x<%02o>, a<%02o>\n", s.st_mode, (S_IXUSR | S_IXGRP | S_IXOTH), a);
#endif
      if (s.st_mode & ((S_IXUSR | S_IXGRP | S_IXOTH) & a)) {
	c = toupper(c);
      }
      putchar(c);

      putchar(' ');
      t = localtime(&s.st_mtime);
      memcpy(&p, t, sizeof(struct tm));
      t = localtime(&mytime);
      memcpy(&q, t, sizeof(struct tm));

      if (mytime < s.st_mtime) {
	printf("future");
      } else {

	if (p.tm_year == q.tm_year) {	/* same year */
	  if (p.tm_mon == q.tm_mon) {	/* same month */
	    if (p.tm_mday == q.tm_mday) {	/* same day */
	      if (p.tm_hour == q.tm_hour) {	/* same hour */
		td = mytime - s.st_mtime;
		t = gmtime(&td);
		memcpy(&r, t, sizeof(struct tm));
		printf("%02d.%02d", r.tm_min, r.tm_sec);
	      } else {		/* this day */
		printf("%02d:%02d", p.tm_hour, p.tm_min);
	      }
	    } else {		/* this month */
	      switch (p.tm_wday) {
	      case 0:
		printf("Sun");
		break;
	      case 1:
		printf("Mon");
		break;
	      case 2:
		printf("Tue");
		break;
	      case 3:
		printf("Wed");
		break;
	      case 4:
		printf("Thu");
		break;
	      case 5:
		printf("Fri");
		break;
	      case 6:
		printf("Sat");
		break;
	      default:
		printf("Und");
		break;
	      }
	      printf("%02d", p.tm_mday);
	    }
	  } else {		/* last month or older */
	    printf("%02d", p.tm_mday);
	    switch (p.tm_mon) {
	    case 0:
	      printf("Jan");
	      break;
	    case 1:
	      printf("Feb");
	      break;
	    case 2:
	      printf("Mar");
	      break;
	    case 3:
	      printf("Apr");
	      break;
	    case 4:
	      printf("May");
	      break;
	    case 5:
	      printf("Jun");
	      break;
	    case 6:
	      printf("Jul");
	      break;
	    case 7:
	      printf("Aug");
	      break;
	    case 8:
	      printf("Sep");
	      break;
	    case 9:
	      printf("Oct");
	      break;
	    case 10:
	      printf("Nov");
	      break;
	    case 11:
	      printf("Dec");
	      break;
	    default:
	      printf("Unm");
	      break;
	    }
	  }
	} else {		/* last year or older */
	  printf(" %04d", p.tm_year + 1900);
	}
      }

      putchar(' ');

      o = s.st_size;
      m = 0;

      if (o > 9999) {		/* bigger than 9999 bytes */
	if (o % 1024 != 0)
	  m = 1;
	o = m + (o / 1024);
	if (o > 9999) {		/* bigger than 9999k */
	  if (o % 1024 != 0)
	    m = 1;
	  o = m + (o / 1024);
	  if (o > 9999) {	/* bigger than 9999M - fs limit */
	    if (o % 1024 != 0)
	      m = 1;
	    o = m + (o / 1024);
	    printf("%4ldG", o);
	  } else {
	    printf("%4ldM", o);
	  }
	} else {
	  printf("%4ldk", o);
	}
      } else {
	printf("%4ld ", o);
      }

      putchar(' ');

      puts(fname);

      if ((s.st_mode & S_IFMT) == S_IFLNK) {
	bl = readlink(fname, b, LINKB - 1);
	if (bl > 0) {
	  b[bl] = '\0';
	  if (opts & SLS_CHILD) {
	    putchar(' ');
	  }
	  printf("     -> %s\n", b);
	} else {
	  perror("Readlink failed");
	}
      }
    }

    if (((s.st_mode & S_IFMT) == S_IFDIR) && (!(opts & (SLS_CHILD | SLS_NODIR)))) {
      if (getcwd(b, LINKB - 1) == NULL) {
	perror("Getcwd failed");
      } else {
	if (chdir(fname)) {
	  perror("Chdir failed");
	} else {
	  d = opendir(".");
	  if (d != NULL) {
	    do {
	      de = readdir(d);
	      if (de != NULL) {
		if (de->d_name[0] == '.') {
		  if (opts & SLS_ALL) {

		    perfile(de->d_name, opts | SLS_CHILD);
		  } else if (opts & SLS_AALL) {
		    if (strcmp(de->d_name, ".") && strcmp(de->d_name, "..")) {
		      perfile(de->d_name, opts | SLS_CHILD);
		    }
		  }
		} else {
		  perfile(de->d_name, opts | SLS_CHILD);
		}
	      }
	    } while (de != NULL);
	  } else {
	    perror("Opendir failed");
	  }
	  closedir(d);
	  if (chdir(b)) {
	    perror("Chdir back failed");
	  }
	}
      }
    }
    if (opts & SLS_BRIEF) {
      if (!(opts & SLS_CHILD)) {
	putchar('\n');
      }
    }
  }

  return result;
}

int main(int argc, char **argv)
{
  int j, i;
  int opts, dopts;
  char *ptr;
  char b[DIRB];

  mygid = getgid();
  myuid = getuid();
  mytime = time(NULL);

  dopts = 0;
  opts = SLS_NULL;
  i = 1;

  while (i < argc) {
    ptr = argv[i];

    if (ptr[0] == '-' && dopts == 0) {
      j = 1;
      while (ptr[j] != '\0') {
	switch (ptr[j]) {
	case 'a':
	  opts = opts | SLS_ALL;
	  break;
	case 'A':
	  opts = opts | SLS_AALL;
	  break;
	case 'B':
	  opts = opts | SLS_TAG;
	case 'b':
	  opts = opts | SLS_BRIEF;
	  break;
	case 'c':
	  fprintf(stderr, "(c) 1999 mgw - GPL - free software\n");
	  exit(1);
	  break;
	case 'd':
	  opts = opts | SLS_NODIR;
	  break;
	case 'h':
	  fprintf(stderr, "Usage: %s [-aAbBcdv-] [files]\n", argv[0]);
	  fprintf(stderr, "Output format (4 fields):\n");
	  fprintf(stderr, " <mode> <time> <size> <filename>:\n");
	  fprintf(stderr, "Mode (6 character flags):\n");
	  fprintf(stderr, " <type><owner><set><read><write><exec>\n");
	  fprintf(stderr, "Time (format varies with age):\n");
	  fprintf(stderr, " <minutes>.<seconds>\n");
	  fprintf(stderr, " <hours>:<minutes>\n");
	  fprintf(stderr, " <weekday><monthday>\n");
	  fprintf(stderr, " <monthday><month>\n");
	  fprintf(stderr, " <year>\n");
	  fprintf(stderr, "Size (4 most significant digits):\n");
	  fprintf(stderr, " <bytes>\n");
	  fprintf(stderr, " <kbytes>k\n");
	  fprintf(stderr, " <Mbytes>M\n");
	  fprintf(stderr, " <Gbytes>G\n");
	  exit(1);
	  break;
        case 'l':
          /* ignore -l for those used to typing ls -la */
        break;
        case 'v':
	  fprintf(stderr, "sls %s\n",VERSION);
	  exit(1);
        break;
	case '-':
	  if (ptr[j + 1] == '\0') {
	    dopts = 1;
	  }
	  break;
	default:
	  fprintf(stderr, "%s: Unknown option <-%c>: Try -h\n", argv[0], ptr[j]);
	  exit(1);
	  break;
	}
	j++;
      }
      i++;
    } else {
#ifdef TRACE
      fprintf(stderr, "main(): doing <%s> with <%02x>\n", ptr, opts);
#endif
      perfile(ptr, opts);
      i++;
      dopts = 1;
    }
  }

  if (dopts == 0) {
    if (getcwd(b, DIRB - 1) == NULL) {
      perror("Getcwd failed");
    } else {
#ifdef TRACE
      fprintf(stderr, "main(): doing cwd <%s> with opts <%02x>\n", b, opts);
#endif
      perfile(".", opts);
    }
  }
  return 0;
}
