/*
 * bltText.c --
 *
 *	This module implements multi-line, rotate-able text for the BLT toolkit.
 *
 * Copyright 1993-1997 Bell Labs Innovations for Lucent Technologies.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that the
 * copyright notice and warranty disclaimer appear in supporting documentation,
 * and that the names of Lucent Technologies any of their entities not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 *
 * Lucent Technologies disclaims all warranties with regard to this software,
 * including all implied warranties of merchantability and fitness.  In no event
 * shall Lucent Technologies be liable for any special, indirect or
 * consequential damages or any damages whatsoever resulting from loss of use,
 * data or profits, whether in an action of contract, negligence or other
 * tortuous action, arising out of or in connection with the use or performance
 * of this software.  
 */

#include "bltInt.h"
#include <X11/Xutil.h>

static GC bitmapGC;

static void
DrawCompoundText(display, drawable, gc, x, y, extsPtr)
    Display *display;
    Drawable drawable;
    GC gc;
    register int x, y;			/* Origin of text */
    TextExtents *extsPtr;
{
    register TextSegment *textPtr = extsPtr->textSegArr;
    register int i;

    for(i = 0; i < extsPtr->numSegments; i++, textPtr++) {
	XDrawString(display, drawable, gc, x + textPtr->x, y + textPtr->y, 
	    textPtr->text, textPtr->numChars);
    }
}

/*
 * -----------------------------------------------------------------
 *
 * Blt_GetCompoundTextExtents --
 *
 *	Get the extents of a possibly multiple-lined text string.
 *
 * Results:
 *	Returns via *widthPtr* and *heightPtr* the dimensions of
 *	the text string.
 *
 * -----------------------------------------------------------------
 */
TextExtents *
Blt_GetCompoundTextExtents(font, string, leader, justify)
    Tk_Font font;
    char string[];
    int leader;
    Tk_Justify justify;
{
    int maxHeight, maxWidth;
    int count;			/* Count the number of character on each line */
    int numSegments;
    int width;			/* Running dimensions of the text */
    TextSegment *textPtr;
    TextExtents *extsPtr;
    int lineHeight;
    int size;
    register char *p;
    register int i;
    Tk_FontMetrics fontMetrics;

    Tk_GetFontMetrics(font, &fontMetrics);
    lineHeight = fontMetrics.linespace +  leader;
    numSegments = 0;
    for (p = string; *p != '\0'; p++) {
	if (*p == '\n') {
	    numSegments++;
	}
    }
    if (*(p - 1) != '\n') {
	numSegments++;
    }
    size = sizeof(TextExtents) + (sizeof(TextSegment) * (numSegments - 1));
    extsPtr = (TextExtents *)calloc(1, size);
    extsPtr->numSegments = numSegments;

    numSegments = count = 0;
    width = maxWidth = maxHeight = 0;
    textPtr = extsPtr->textSegArr;
    for (p = string; *p != '\0'; p++) {
	if (*p == '\n') {
	    if (count > 0) {
		width = Tk_TextWidth(font, string, count);
		if (width > maxWidth) {
		    maxWidth = width;
		}
	    }
	    textPtr->width = width;
	    textPtr->numChars = count;
	    textPtr->y = maxHeight + fontMetrics.ascent;
	    textPtr->text = string;
	    textPtr++;
	    numSegments++;
	    maxHeight += lineHeight;
	    string = p + 1;	/* Start the string on the next line */
	    count = 0;		/* Reset to indicate the start of a new line */
	    continue;
	}
	count++;
    }
    if (numSegments < extsPtr->numSegments) {
	width = Tk_TextWidth(font, string, count);
	if (width > maxWidth) {
	    maxWidth = width;
	}
	textPtr->width = width;
	textPtr->numChars = count;
	textPtr->y = maxHeight + fontMetrics.ascent;
	textPtr->text = string;
	maxHeight += lineHeight;
	numSegments++;
    }
    textPtr = extsPtr->textSegArr;
    for (i = 0; i < numSegments; i++, textPtr++) {
	switch (justify) {
	default:
	case TK_JUSTIFY_LEFT:
	    textPtr->x = 0;	/* No offset for left justified text strings */
	    break;
	case TK_JUSTIFY_RIGHT:
	    textPtr->x =  (maxWidth - textPtr->width);
	    break;
	case TK_JUSTIFY_CENTER:
	    textPtr->x = (maxWidth - textPtr->width) / 2;
	    break;
	}
    }
    extsPtr->width = maxWidth;
    extsPtr->height = maxHeight - leader;
    return (extsPtr);
}

/*
 * -----------------------------------------------------------------
 *
 * Blt_GetTextExtents --
 *
 *	Get the extents of a possibly multiple-lined text string.
 *
 * Results:
 *	Returns via *widthPtr* and *heightPtr* the dimensions of
 *	the text string.
 *
 * -----------------------------------------------------------------
 */
Dimension
Blt_GetTextExtents(font, string, leader)
    Tk_Font font;
    char string[];
    int leader;
{
    Dimension textDim;		/* Running dimensions of the text */
    int count;			/* Count the number of character on each line */
    int w, lineHeight;
    register char *p;
    Tk_FontMetrics fontMetrics;

    Tk_GetFontMetrics(font, &fontMetrics);
    lineHeight = fontMetrics.linespace +  leader;
    count = 0;
    textDim.width = textDim.height = 0;
    for (p = string; *p != '\0'; p++) {
	if (*p == '\n') {
	    if (count > 0) {
		w = Tk_TextWidth(font, string, count);
		if (w > textDim.width) {
		    textDim.width = w;
		}
	    }
	    textDim.height += lineHeight;
	    string = p + 1;	/* Start the string on the next line */
	    count = 0;		/* Reset to indicate the start of a new line */
	    continue;
	}
	count++;
    }
    if ((count > 0) && (*(p - 1) != '\n')) {
	textDim.height += lineHeight;
	w = Tk_TextWidth(font, string, count);
	if (w > textDim.width) {
	    textDim.width = w;
	}
    }
    textDim.height -= leader;
    return textDim;
}

/*
 * -----------------------------------------------------------------
 *
 * Blt_GetBoundingBox
 *
 * 	Computes the size the bounding box of a rotated rectangle, given
 *	by the width, height, and rotation.  The rectangle corners are
 *	simply rotated and the min and max x,y coordinates are determined.
 *      The rectangle is centered at 0,0.  Point 0 is at -w/2, -h/2.
 *	Point 1 is at w/2, -h/2, etc.
 *
 *  		0 ------- 1
 *  		|         |
 *  		|    x    |
 *  		|         |
 *  		3 ------- 2
 *
 * Results:
 *	The width and height of the bounding box containing the
 *	rotated rectangle are returned.
 *
 * -----------------------------------------------------------------
 */
void
Blt_GetBoundingBox(width, height, theta, bmWidthPtr, bmHeightPtr, pointArr)
    int width, height;		/* Unrotated region */
    double theta;		/* Rotation of box */
    int *bmWidthPtr, *bmHeightPtr;	/* Rotated bounding box region */
    XPoint *pointArr;		/* Points of the rotated box */
{
    register int i;
    double sinTheta, cosTheta;
    double maxX, maxY;
    register double x, y;
    Point2D corner[4];

    /* Set the four corners of the rectangle whose center is the origin */

    corner[1].x = corner[2].x = (double)width * 0.5;
    corner[0].x = corner[3].x = -corner[1].x;
    corner[2].y = corner[3].y = (double)height * 0.5;
    corner[0].y = corner[1].y = -corner[2].y;

    theta = (-theta / 180.0) * M_PI;
    sinTheta = sin(theta), cosTheta = cos(theta);
    maxX = maxY = 0.0;

    /* Rotate the four corners and find the maximum X and Y coordinates */

    for (i = 0; i < 4; i++) {
	x = (corner[i].x * cosTheta) - (corner[i].y * sinTheta);
	y = (corner[i].x * sinTheta) + (corner[i].y * cosTheta);
	if (x > maxX) {
	    maxX = x;
	}
	if (y > maxY) {
	    maxY = y;
	}
	if (pointArr != NULL) {
	    pointArr[i].x = ROUND(x);
	    pointArr[i].y = ROUND(y);
	}
    }

    /*
     * By symmetry, the width and height of the bounding box are
     * twice the maximum x and y coordinates.
     */
    *bmWidthPtr = (int)((maxX + maxX) + 0.5);
    *bmHeightPtr = (int)((maxY + maxY) + 0.5);
}

/*
 * -----------------------------------------------------------------
 *
 * Blt_TranslateBoxCoords --
 *
 * 	Translate the coordinates of a given bounding box based
 *	upon the anchor specified.  The anchor indicates where
 *	the given xy position is in relation to the bounding box.
 *
 *  		nw --- n --- ne
 *  		|            |
 *  		w   center   e
 *  		|            |
 *  		sw --- s --- se
 *
 * 	The coordinates returned are translated to the origin of the
 * 	bounding box (suitable for giving to XCopyArea, etc.)
 *
 * Results:
 *	The translated coordinates of the bounding box are returned.
 *
 * -----------------------------------------------------------------
 */
Point2D
Blt_TranslateBoxCoords(x, y, width, height, anchor)
    int x, y;			/* Window coordinates of anchor */
    int width, height;		/* Extents of the bounding box */
    Tk_Anchor anchor;		/* Direction of the anchor */
{
    Point2D trans;
    double w, h;

    w = (double)width, h = (double)height;

    trans.x = (double)x, trans.y = (double)y;
    switch (anchor) {
    case TK_ANCHOR_NW:		/* Upper left corner */
	break;
    case TK_ANCHOR_W:		/* Left center */
	trans.y -= (h * 0.5);
	break;
    case TK_ANCHOR_SW:		/* Lower left corner */
	trans.y -= h;
	break;
    case TK_ANCHOR_N:		/* Top center */
	trans.x -= (w * 0.5);
	break;
    case TK_ANCHOR_CENTER:	/* Centered */
	trans.x -= (w * 0.5);
	trans.y -= (h * 0.5);
	break;
    case TK_ANCHOR_S:		/* Bottom center */
	trans.x -= (w * 0.5);
	trans.y -= h;
	break;
    case TK_ANCHOR_NE:		/* Upper right corner */
	trans.x -= w;
	break;
    case TK_ANCHOR_E:		/* Right center */
	trans.x -= w;
	trans.y -= (h * 0.5);
	break;
    case TK_ANCHOR_SE:		/* Lower right corner */
	trans.x -= w;
	trans.y -= h;
	break;
    }
    return (trans);
}

/*
 * -----------------------------------------------------------------
 *
 * Blt_RotateBitmap --
 *
 *	Creates a new bitmap containing the rotated image of the given
 *	bitmap.  We also need a special GC of depth 1, so that we do
 *	not need to rotate more than one plane of the bitmap.
 *
 * Results:
 *	Returns a new bitmap containing the rotated image.
 *
 * -----------------------------------------------------------------
 */
Pixmap
Blt_RotateBitmap(display, draw, srcBitmap, srcWidth, srcHeight, theta, 
	destWidthPtr, destHeightPtr)
    Display *display;		/* X display */
    Drawable draw;		/* Root window drawable */
    Pixmap srcBitmap;		/* Source bitmap to be rotated */
    int srcWidth, srcHeight;	/* Width and height of the source bitmap */
    double theta;		/* Right angle rotation to perform */
    int *destWidthPtr, *destHeightPtr;
{
    XImage *src, *dest;
    Pixmap destBitmap;
    int destWidth, destHeight;
    register int dx, dy;	/* Destination bitmap coordinates */
    register int sx, sy;	/* Source bitmap coordinates */
    unsigned long pixel;

    /* Create a bitmap and image big enough to contain the rotated text */

    Blt_GetBoundingBox(srcWidth, srcHeight, theta, &destWidth, &destHeight,
	(XPoint *)NULL);
    destBitmap = Tk_GetPixmap(display, draw, destWidth, destHeight, 1);

    XSetForeground(display, bitmapGC, 0x0);
    XFillRectangle(display, destBitmap, bitmapGC, 0, 0, destWidth, destHeight);

    src = XGetImage(display, srcBitmap, 0, 0, srcWidth, srcHeight, 1, ZPixmap);
    dest = XGetImage(display, destBitmap, 0, 0, destWidth, destHeight, 1,
	ZPixmap);

    theta = FMOD(theta, 360.0);
    if (FMOD(theta, (double)90.0) == 0.0) {
	int quadrant;

	/* Handle right-angle rotations specially */

	/*
	 * FIXME: Should take switch statement out of the inner loop 
	 * and handle each of the right angle cases separately. 
	 */

	quadrant = (int)(theta / 90.0);
	for (dx = 0; dx < destWidth; dx++) {
	    for (dy = 0; dy < destHeight; dy++) {
		switch (quadrant) {
		case 3:	/* 270 degrees */
		    sx = dy, sy = destWidth - dx - 1;
		    break;
		case 2:	/* 180 degrees */
		    sx = destWidth - dx - 1, sy = destHeight - dy - 1;
		    break;
		case 1:	/* 90 degrees */
		    sx = destHeight - dy - 1, sy = dx;
		    break;
		default:
		    /* The calling routine should never let this happen. */
		case 0:	/* 0 degrees */
		    sx = dx, sy = dy;
		    break;
		}
		pixel = XGetPixel(src, sx, sy);
		if (pixel) {
		    XPutPixel(dest, dx, dy, pixel);
		}
	    }
	}

    } else {
	double radians, sinTheta, cosTheta;
	double srcX, srcY;	/* Center of source rectangle */
	double destX, destY;	/* Center of destination rectangle */
	double transX, transY;
	double rx, ry;		/* Angle of rotation for x and y coordinates */

	radians = (theta / 180.0) * M_PI;
	sinTheta = sin(radians), cosTheta = cos(radians);

	/*
	 * Coordinates of the centers of the source and destination rectangles
	 */
	srcX = srcWidth * 0.5;
	srcY = srcHeight * 0.5;
	destX = destWidth * 0.5;
	destY = destHeight * 0.5;

	/*  Rotate each pixel of dest image, placing results in source image */

	for (dx = 0; dx < destWidth; dx++) {
	    for (dy = 0; dy < destHeight; dy++) {

		/* Translate origin to center of destination image */
		transX = dx - destX;
		transY = dy - destY;

		/* Rotate the coordinates about the origin */
		rx = (transX * cosTheta) - (transY * sinTheta);
		ry = (transX * sinTheta) + (transY * cosTheta);

		/* Translate back to the center of the source image */
		rx += srcX;
		ry += srcY;

		sx = ROUND(rx);
		sy = ROUND(ry);

		/*
		 * Verify the coordinates, since the destination image can be
		 * bigger than the source
		 */

		if ((sx >= srcWidth) || (sx < 0) || (sy >= srcHeight) ||
		    (sy < 0)) {
		    continue;
		}
		pixel = XGetPixel(src, sx, sy);
		if (pixel) {
		    XPutPixel(dest, dx, dy, pixel);
		}
	    }
	}
    }
    /* Write the rotated image into the destination bitmap */
    XPutImage(display, destBitmap, bitmapGC, dest, 0, 0, 0, 0, destWidth,
	destHeight);

    /* Clean up temporary resources used */
    XDestroyImage(src), XDestroyImage(dest);
    *destWidthPtr = destWidth;
    *destHeightPtr = destHeight;
    return (destBitmap);
}

/*
 * -----------------------------------------------------------------
 *
 * Blt_CreateTextBitmap --
 *
 *	Draw a bitmap, using the the given window coordinates
 *	as an anchor for the text bounding box.
 *
 * Results:
 *	Returns the bitmap representing the text string.
 *
 * Side Effects:
 *	Bitmap is drawn using the given font and GC in the
 *	drawable at the given coordinates, anchor, and rotation.
 *
 * -----------------------------------------------------------------
 */
Pixmap
Blt_CreateTextBitmap(tkwin, text, attrPtr, widthPtr, heightPtr, 
	     bmWidthPtr, bmHeightPtr)
    Tk_Window tkwin;
    char *text;			/* Text string to draw */
    TextAttributes *attrPtr;	/* Text attributes: rotation, color, font,
				 * linespacing, justification, etc. */
    int *widthPtr, *heightPtr;	/* Width and height of text bounding box */
    int *bmWidthPtr;
    int *bmHeightPtr;		/* Extents of rotated text string */
{
    int width, height;		
    Pixmap bitmap;
    Display *display;
    TextExtents *extsPtr;
    Window root;

    extsPtr = Blt_GetCompoundTextExtents(attrPtr->font, text, attrPtr->leader, 
	attrPtr->justify);

    display = Tk_Display(tkwin);

    /* Create a temporary bitmap to contain the text string */

    width = extsPtr->width + PADDING(attrPtr->padX);
    height = extsPtr->height + PADDING(attrPtr->padY);

    root = RootWindow(Tk_Display(tkwin), Tk_ScreenNumber(tkwin));
    bitmap = Tk_GetPixmap(display, root, width, height, 1);
    /* Clear the pixmap and draw the text string into it */

    XSetForeground(display, bitmapGC, 0);
    XFillRectangle(display, bitmap, bitmapGC, 0, 0, width, height);
    XSetFont(display, bitmapGC, Tk_FontId(attrPtr->font));
    XSetForeground(display, bitmapGC, 1);
    DrawCompoundText(display, bitmap, bitmapGC, attrPtr->padLeft, attrPtr->padTop,
	extsPtr);
    free((char *)extsPtr);


    if (attrPtr->theta != 0.0) {
	Pixmap rotBitmap;

	/* Replace the text pixmap with a rotated one */

	rotBitmap = Blt_RotateBitmap(display, root, bitmap, width, height, 
		attrPtr->theta, bmWidthPtr, bmHeightPtr);
	Tk_FreePixmap(display, bitmap);
	bitmap = rotBitmap;
    } else {
	*bmWidthPtr = width, *bmHeightPtr = height;
    }
    *widthPtr = width;
    *heightPtr = height;
    return (bitmap);
}


/*
 * -----------------------------------------------------------------------
 *
 * Blt_ScaleBitmapRegion --
 *
 *	Creates a new scaled bitmap from another bitmap. The new bitmap
 *	is bounded by a specified region. Only this portion of the bitmap
 *	is scaled from the original bitmap.
 *
 *	By bounding scaling to a region we can generate a new bitmap
 *	which is no bigger than the specified viewport.
 *
 * Results:
 *	The new scaled bitmap is returned.
 *
 * Side Effects:
 *	A new pixmap is allocated. The caller must release this.
 *
 * -----------------------------------------------------------------------
 */
Pixmap
Blt_ScaleBitmapRegion(display, draw, srcBitmap, srcWidth, srcHeight, scaledWidth, 
	scaledHeight, regionX, regionY, regionWidth, regionHeight)
    Display *display;
    Drawable draw;
    Pixmap srcBitmap;
    int srcWidth, srcHeight, scaledWidth, scaledHeight;
    int regionX, regionY, regionWidth, regionHeight;
{
    XImage *src, *dest;
    Pixmap destBitmap;
    double xScale, yScale;
    register int dx, dy;	/* Destination bitmap coordinates */
    register int sx, sy;	/* Source bitmap coordinates */
    unsigned long pixel;
    double tmp;


    /* Create a new bitmap the size of the region and clear it */

    destBitmap = Tk_GetPixmap(display, draw, regionWidth, regionHeight, 1);
    XSetForeground(display, bitmapGC, 0x0);
    XFillRectangle(display, destBitmap, bitmapGC, 0, 0, regionWidth, regionHeight);

    src = XGetImage(display, srcBitmap, 0, 0, srcWidth, srcHeight, 1, ZPixmap);
    dest = XGetImage(display, destBitmap, 0, 0, regionWidth, regionHeight, 1,
	ZPixmap);

    /* Scale each pixel of destination image from results of source image */

    xScale = (double)srcWidth / (double)scaledWidth;
    yScale = (double)srcHeight / (double)scaledHeight;

    for (dx = 0; dx < regionWidth; dx++) {
	for (dy = 0; dy < regionHeight; dy++) {
	    
	    tmp = (double)(dx + regionX) * xScale;
	    sx = ROUND(tmp);
	    tmp = (double)(dy + regionY) * yScale;
	    sy = ROUND(tmp);
	    
	    /*
	     * Verify the coordinates, since the destination image can be
	     * bigger than the source
	     */
	    if ((sx >= srcWidth) || (sx < 0) || (sy >= srcHeight) || (sy < 0)) {
		continue;
	    }
	    pixel = XGetPixel(src, sx, sy);
	    if (pixel) {
		XPutPixel(dest, dx, dy, pixel);
	    }
	}
    }

    /* Write the rotated image into the destination bitmap */

    XPutImage(display, destBitmap, bitmapGC, dest, 0, 0, 0, 0, regionWidth,
	regionHeight);
    XDestroyImage(src), XDestroyImage(dest);
    return (destBitmap);
}

/*
 * -----------------------------------------------------------------------
 *
 * Blt_ScaleBitmap --
 *
 *	Same as Blt_ScaleBitmapRegion, except that the region is unbounded.
 *	The scaled bitmap will be a fully scaled version of the original,
 *	not a portion of it.
 *
 * Results:
 *	The new scaled bitmap is returned.
 *
 * Side Effects:
 *	A new pixmap is allocated. The caller must release this.
 *
 * -----------------------------------------------------------------------
 */
Pixmap
Blt_ScaleBitmap(display, draw, srcBitmap, srcWidth, srcHeight, scaledWidth, 
	scaledHeight)
    Display *display;
    Drawable draw;
    Pixmap srcBitmap;
    int srcWidth, srcHeight, scaledWidth, scaledHeight;
{
    return Blt_ScaleBitmapRegion(display, draw, srcBitmap, srcWidth, srcHeight,
	scaledWidth, scaledHeight, 0, 0, scaledWidth, scaledHeight); 
}

/*LINTLIBRARY*/
void
Blt_InitTextAttributes(attrPtr)
    TextAttributes *attrPtr;
{
    /* Initialize these attributes to zero */
    attrPtr->leader = 0;
    attrPtr->padLeft = attrPtr->padRight = 0;
    attrPtr->padTop = attrPtr->padBottom = 0;
    attrPtr->bgColorPtr = attrPtr->fgColorPtr = (XColor *)NULL;
    attrPtr->theta = 0.0;
    attrPtr->anchor = TK_ANCHOR_CENTER;
    attrPtr->justify = TK_JUSTIFY_CENTER;
}

void
Blt_SetTextAttributes(attrPtr, colorPtr, font, theta, anchor, justify)
    TextAttributes *attrPtr;
    XColor *colorPtr;
    Tk_Font font;
    double theta;
    Tk_Anchor anchor;
    Tk_Justify justify;
{
    Blt_InitTextAttributes(attrPtr);
    attrPtr->fgColorPtr = colorPtr;
    attrPtr->font = font;
    attrPtr->theta = theta;
    attrPtr->anchor = anchor;
    attrPtr->justify = justify;
}




/*
 * -----------------------------------------------------------------
 *
 * DrawText --
 *
 *	Draw a text string, possibly rotated, using the the given
 *	window coordinates as an anchor for the text bounding box.
 *
 * Results:
 *	Returns the x-coordinate to the right of the text.
 *
 * Side Effects:
 *	Text string is drawn using the given font and GC at the
 *	the given window coordinates.
 *
 *      The Stipple, FillStyle, and TSOrigin fields of the GC are
 *      modified for rotated text.  This assumes the GC is private,
 *      *not* shared (via Tk_GetGC)
 *
 * -----------------------------------------------------------------
 */
void
Blt_DrawText2(tkwin, draw, text, attrPtr, x, y, areaPtr)
    Tk_Window tkwin;
    Drawable draw;
    char text[];
    TextAttributes *attrPtr;	/* Text attribute information */
    int x, y;			/* Window coordinates to draw text */
    Dimension *areaPtr;
{
    Point2D p;
    double theta;
    Display *display;
    
    if ((text == NULL) || (*text == '\0')) {	
	return;		/* Empty string, do nothing */
    }
    display = Tk_Display(tkwin);
    theta = FMOD(attrPtr->theta, (double)360.0);
    if (theta < 0.0) {
	theta += 360.0;
    }

    if (theta == 0.0) {
	TextExtents *extsPtr;
	int width, height;

	/*
	 * This is the easy case of no rotation. Simply draw the text
	 * using the standard drawing routines.  If a background color
	 * was specified (both bgColorPtr and fillGC fields will be
	 * set), then fill a rectangle with the background color.
	 */
	extsPtr = Blt_GetCompoundTextExtents(attrPtr->font, text, attrPtr->leader,
		attrPtr->justify);

	width = extsPtr->width + PADDING(attrPtr->padX);
	height = extsPtr->height + PADDING(attrPtr->padY);

	p = Blt_TranslateBoxCoords(x, y, width, height, attrPtr->anchor);
	x = (int)p.x, y = (int)p.y;
	if (attrPtr->bgColorPtr != NULL) {
	    XFillRectangle(display, draw, attrPtr->fillGC, x, y, width - 1, 
		height - 1);
	}
	DrawCompoundText(display, draw, attrPtr->textGC, x + attrPtr->padLeft, 
		y + attrPtr->padTop, extsPtr);
	free((char *)extsPtr);
	if (areaPtr != NULL) {
	    areaPtr->width = width, areaPtr->height = height;
	}
	return;

    } else {
	int width, height;
	int bmWidth, bmHeight;
	Pixmap bitmap;

	/*
	 * Rotate the text by writing the text into a bitmap and rotating
	 * the bitmap.  If the text is at a right angle and a background
	 * color is set, we can just call XCopyPlane operation. Otherwise, 
	 * set the clip mask and origin in the GC first.  Make sure we
	 * restore the GC because it may be shared.
	 */
	attrPtr->theta = theta;
	bitmap = Blt_CreateTextBitmap(tkwin, text, attrPtr, &width, &height,
		      &bmWidth, &bmHeight);
	p = Blt_TranslateBoxCoords(x, y, bmWidth, bmHeight, attrPtr->anchor);
	x = (int)p.x, y = (int)p.y;
	
	theta = FMOD(theta, (double)90.0);

	if ((attrPtr->bgColorPtr == NULL) || (theta != 0.0)) {

	    /* 
	     * Non-right angle rotation or empty background
	     */

	    if (attrPtr->bgColorPtr != NULL) {
		XPoint pointArr[4];
		register int i;
		int cx, cy;
		int dummy;

		/*
		 * Simulate the rotated background of the bitmap by 
		 * computing the bounding polygon and filling it with 
		 * the background color.
		 */
		Blt_GetBoundingBox(width, height, attrPtr->theta, &dummy, 
			&dummy, pointArr);

		/* Translate the anchor to the center of the bitmap */

		p.x += (bmWidth * 0.5);
		p.y += (bmHeight * 0.5);
		cx = ROUND(p.x), cy = ROUND(p.y);

		for (i = 0; i < 4; i++) {
		    pointArr[i].x += cx;
		    pointArr[i].y += cy;
		}
		XFillPolygon(display, draw, attrPtr->fillGC, pointArr, 4, 
			Convex, CoordModeOrigin);
	    }
	    XSetClipMask(display, attrPtr->textGC, bitmap);
	    XSetClipOrigin(display, attrPtr->textGC, x, y);
	    XCopyPlane(display, bitmap, draw, attrPtr->textGC, 0, 0, bmWidth,
		   bmHeight, x, y, 1);
	    XSetClipMask(display, attrPtr->textGC, None);
	} else {
	    XCopyPlane(display, bitmap, draw, attrPtr->textGC, 0, 0, bmWidth,
		       bmHeight, x, y, 1);
	}
	Tk_FreePixmap(display, bitmap);
	if (areaPtr != NULL) {
	    areaPtr->width = bmWidth, areaPtr->height= bmHeight;
	}
	return;
    }
}

void
Blt_DrawText(tkwin, draw, text, attrPtr, x, y)
    Tk_Window tkwin;
    Drawable draw;
    char text[];
    TextAttributes *attrPtr;	/* Text attribute information */
    int x, y;			/* Window coordinates to draw text */
{
    Blt_DrawText2(tkwin, draw, text, attrPtr, x, y, (Dimension *)NULL);
}

void
Blt_InitBitmapGC(tkwin)
    Tk_Window tkwin;
{
    Pixmap bitmap;
    XGCValues gcValues;
    unsigned int gcMask;
    Window root;

    root = RootWindow(Tk_Display(tkwin), Tk_ScreenNumber(tkwin));
    bitmap = Tk_GetPixmap(Tk_Display(tkwin), root, 1, 1, 1);
    gcValues.foreground = gcValues.background = 0;
    gcMask = (GCForeground | GCBackground);
    bitmapGC = Blt_GetPrivateGCFromDrawable(tkwin, bitmap, gcMask, &gcValues);
    Tk_FreePixmap(Tk_Display(tkwin), bitmap);
}
