/*

	rmsafe, rmrest - file deletion recover utilities

	June 11, 1998 - VERSION 0.2

	Matt Kressel	(matty@inch.com)
	
	This program is protected under the GNU public license.
	If you don't know what that is please read the file
	COPYING that should have been included with this document.

	I make no claims about the stability nor rubustness of
	the software.  Should you find a bug.  Please record the
	exact circumstance in which you encountered it and
	mail it to: matty@inch.com. And please, NO MORE SPAM!

	Happy Recovering!

	PS: Look for version 0.3 and X11 support coming soon...

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include <unistd.h>
#include <pwd.h>
#include <errno.h>
#include <fcntl.h>
#include <dirent.h>
#include <limits.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "rmsafe.h"


int	RMSAFE_DIR = 	0;
int	RMSAFE_FORCE =	0;
int	RMSAFE_INTERACTIVE = 	0;
int	RMSAFE_RECURSIVE = 	0;
int	RMSAFE_VERBOSE =	0;
char	progname[256];

void rmsafe_help()
{
	printf("Usage:	%s [OPTION]... PATH...\n",progname);
	printf("\n");
	printf("  -d, --directory\tmove contents of directory to the trash\n");
	printf("  -f, --force\t\tignore nonexistant files, never prompt\n");
	printf("  -i, --interactive\tprompt before any removal\n");
	printf("  -v, --verbose\t\texplain what is being done\n");
	printf("  -r, -R, --recursive\tmove contents of directories recusively\n");
	printf("  -h, --help\t\tdisplay this help and exit\n");
	printf("      --version\t\toutput version information and exit\n");
	exit(0);
}

void rmsafe_version()
{
	printf("%s: version 0.2, matty@inch.com (Matt Kressel)\n",progname);
	exit(0);
}

	
void rmsafe_file(char *filename, char *trash_dir, struct stat *stbuf)
{
	char full_pathname[FILENAME_MAX];
	char index_pathname[FILENAME_MAX];
	char *just_filename;
	char buf[2048];		/* 2K buffer */
	FILE *index = NULL;
	char yna[5];
	int read_fd;
	int write_fd;
	int has_error = 0;
	ssize_t	amount;

	just_filename = get_filename(filename);

	sprintf(full_pathname,"%s/%s",trash_dir,just_filename);
	sprintf(index_pathname,"%s/rmsafe.index",trash_dir);


	while(1)
	{
		if(RMSAFE_INTERACTIVE)
		{
			printf("%s: store file %s ? (y/n/a): ",progname,just_filename);
			scanf("%s",yna);
			if((yna[0] == 'n') || (yna[0] == 'N'))
				break;
			if((yna[0] == 'a') || (yna[0] == 'A'))
				RMSAFE_INTERACTIVE = 0;
		}
		if((read_fd = open(filename,O_RDONLY)) < 0)
		{
			rmsafe_error(filename);
			break;
		}
		if((write_fd = open(full_pathname,(O_EXCL | O_CREAT | O_WRONLY),stbuf->st_mode)) < 0)
		{
			rmsafe_error(full_pathname);
			break;
		}
		if((index = fopen(index_pathname,"a"))==NULL)
		{
			rmsafe_error(index_pathname);
			break;
		}

		/* read/write loop */

		while((amount = read(read_fd,buf,2048)))
		{
			if(amount == -1)	/* error */
			{
				rmsafe_error(filename);
				has_error = 1;
				break;
			}
			if(amount == 0)		/* EOF */
				break;
	
			if(write(write_fd,buf,amount) < 0)
			{
				rmsafe_error(full_pathname);
				has_error = 1;
				break;
			}
		}


		if(has_error == 0)
		{
			close(read_fd);
			if(remove(filename) < 0)
			{
				rmsafe_error(filename);
				if((errno == EPERM) || (errno == EACCES))
					remove(full_pathname);
				has_error = 1;
			}
		}
				
		

		if(has_error == 0)
		{
			fprintf(index,"%s\t%s\n",filename,full_pathname);
			if(RMSAFE_VERBOSE)
				printf("%s -> %s\n",filename,full_pathname);
		}

		break;
		
	}

	
	if (read_fd) close(read_fd);
	if (write_fd) close(write_fd);
	if (index) fclose(index);		
}

/* iterates through the files in a directory and recurses subdirs */

void rmsafe_directory(char *dirname, char *trash_dir)
{
	DIR *the_dir;
	FILE *index;
	char yna[25];
	struct stat stbuf;
	struct dirent *the_dirent;
	char pathname[FILENAME_MAX];
	char index_pathname[FILENAME_MAX];


	while(1)
	{
		if(RMSAFE_INTERACTIVE)
		{
			printf("%s: descend directory %s ? (y/n/a): ",progname,dirname);
			scanf("%s",yna);
			if((yna[0] == 'n') || (yna[0] == 'N'))
				break;
			if((yna[0] == 'a') || (yna[0] == 'A'))
				RMSAFE_INTERACTIVE = 0;
		}
		if((the_dir=opendir(dirname))==NULL)
		{
			rmsafe_error(dirname);
			break;
		}

		while((the_dirent=readdir(the_dir)) != NULL)
		{
			/* skip "." and ".." */

			if((strcmp(the_dirent->d_name,".") == 0) ||
				(strcmp(the_dirent->d_name,"..") == 0))
					continue;

			/* get the full pathname of the file/dir */

			sprintf(pathname,"%s/%s",dirname,the_dirent->d_name);

			stat(pathname,&stbuf);

			/* recurse if directory */

			if(S_ISDIR(stbuf.st_mode))
				rmsafe_directory(pathname,trash_dir);

			/* do normal if file */
			if(S_ISREG(stbuf.st_mode))
				rmsafe_file(pathname,trash_dir,&stbuf);

			if(S_ISLNK(stbuf.st_mode))
				printf("%s: %s: is a symbolic link\n",progname,the_dirent->d_name);

		}
		closedir(the_dir);
	
		/* write special entry for directory in index file */
	
		sprintf(index_pathname,"%s/rmsafe.index",trash_dir);
		if((index = fopen(index_pathname,"a"))==NULL)
		{
			rmsafe_error(index_pathname);
			break;
		}


		if(RMSAFE_INTERACTIVE)
		{
			printf("%s: store directory %s ? (y/n/a): ",progname,dirname);
			scanf("%s",yna);
			if((yna[0] == 'n') || (yna[0] == 'N'))
				break;
			if((yna[0] == 'a') || (yna[0] == 'A'))
				RMSAFE_INTERACTIVE = 0;
		}
	
		fprintf(index,"%s/\tDIR\n",dirname);
		if(RMSAFE_VERBOSE)
			printf("saved directory: %s/\n",dirname);


		fclose(index);
		
		
		if(rmdir(dirname) < 0)
			rmsafe_error(dirname);
		
		break;
	}
	
}

	
int main(int argc, char *argv[])
{

	int c;

	char trash_dir[FILENAME_MAX];	/* trash directory */
	char pathname[FILENAME_MAX];	/* trash directory */
	struct passwd *the_user;		/* the user info */
	struct stat stbuf;

	strcpy(progname,argv[0]);

	while(1)
        {
		int option_index = 0;
		static struct option long_options[] =
		{
			{"directory",0,0,'d'},
			{"force",0,0,'f'},
			{"interactive",0,0,'i'},
			{"recursive",0,0,'r'},
			{"Recursive",0,0,'R'},
			{"help",0,0,'h'},
			{"version",0,0,'n'},
			{"verbose",0,0,'v'},
			{0,0,0,0}
		};
			
		c = getopt_long(argc,argv,"dfirhv",long_options, &option_index);
		if (c == -1) break;

		switch (c)
		{
			case 'd':
				RMSAFE_DIR = 1;
			break;
			case 'f':
				RMSAFE_FORCE = 1;
			break;
			case 'i':
				RMSAFE_INTERACTIVE = 1;
			break;
			case 'r': 
				RMSAFE_RECURSIVE = 1;
			break;
			case 'R': 
				RMSAFE_RECURSIVE = 1;
			break;
			case 'h':
				rmsafe_help();
			break;
			case 'v':
				RMSAFE_VERBOSE = 1;
			break;
			case 'n':
				rmsafe_version();
			break;
			default:
				printf("Try \"%s --help\" for more information.\n",progname);
				exit(1);
		}
	}
	if (optind < argc)
	{
		/* main loop */

		the_user = get_user();	/* get user struct */
		read_conf(trash_dir,the_user->pw_name);	/* read configuration */
		check_trash_dir(trash_dir);
		


		while (optind < argc)
		{
			/* remove last / from pathname */
			if((strlen(argv[optind])==1) && (argv[optind][0]=='/'))
			{
				printf("%s: Command Not Allowed!\n",progname);
				exit(1);
			}

			if(argv[optind][strlen(argv[optind])-1] == '/')
				argv[optind][strlen(argv[optind])-1] = '\0';

			if(argv[optind][0] != '/')
			{
				getcwd(pathname,sizeof(pathname));
				sprintf(pathname,"%s/%s",pathname,argv[optind]);
			} else strcpy(pathname,argv[optind]);


			/* stat the file to see if it is a directory */

			if(stat(pathname,&stbuf) < 0)
				if(errno == ENOENT)
				{
					if(RMSAFE_FORCE == 0)
						rmsafe_error(argv[optind]);
					optind++;
					continue;
				}

			if(S_ISDIR(stbuf.st_mode))
			{
				if (RMSAFE_DIR || RMSAFE_RECURSIVE)
				{
					rmsafe_directory(pathname,trash_dir);
					optind++;
					continue;
				}
				else 
				{
					printf("%s: %s: is a directory\n",progname,argv[optind]); 
					optind++;
					continue;
				}
			}
	
			if(S_ISREG(stbuf.st_mode))
			{
				rmsafe_file(pathname,trash_dir,&stbuf);
				optind++;
				continue;
			}
			
			if(S_ISLNK(stbuf.st_mode))
			{
				printf("%s: %s: is a symbolic link\n",progname,argv[optind]);
				optind++;
				continue;
			}
				

			printf("%s: %s: unknown/unhandled file type. Skipping...\n",progname,argv[optind]);
		
			optind++;
		}
	} else
	{
		if(RMSAFE_FORCE == 0)
		{
			printf("%s: missing filename and/or directory.\n",progname);
			printf("Try \"%s --help\" for more information.\n",progname);
		}
		exit(1);
	}
	exit(0);
}
					
