/*

	rmsafe, rmrest - file deletion recover utilities

	June 11, 1998 - VERSION 0.2

	Matt Kressel	(matty@inch.com)
	
	This program is protected under the GNU public license.
	If you don't know what that is please read the file
	COPYING that should have been included with this document.

	I make no claims about the stability nor rubustness of
	the software.  Should you find a bug.  Please record the
	exact circumstance in which you encountered it and
	mail it to: matty@inch.com. And please, NO MORE SPAM!

	Happy Recovering!

	PS: Look for version 0.3 and X11 support coming soon...

*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include <unistd.h>
#include <pwd.h>
#include <errno.h>
#include <fcntl.h>
#include <dirent.h>
#include <limits.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "rmsafe.h"


int	RMREST_DIR = 	0;
int	RMREST_INTERACTIVE = 	0;
int	RMREST_VERBOSE =	0;
int	RMREST_CLEAN =	0;
int	RMREST_SUBSTRING =	0;
int	RMREST_LIST =	0;
int 	RMREST_ALL = 0;
int restnum = 0;
int cleannum = 0;
char 	progname[FILENAME_MAX];
char    trash_dir[FILENAME_MAX];	/* trash directory */
char	*substring;


file_node *tail = NULL;		/* tail of file list always */




void rmrest_help()
{
	printf("Usage:	%s [OPTION]... PATH...\n",progname);
	printf("\n");
	printf("  -a, --all\t\trestore all files, no prompting\n");
	printf("  -s, --substring\trestores/cleans all files matching substring\n");
	printf("  -d, --directory\tautomatically attempt to restore directories\n");
	printf("  -i, --interactive\tprompt before any restoring\n");
	printf("  -c, --clean\t\tremove files permanently\n");
	printf("  -v, --verbose\t\texplain what is being done\n");
	printf("  -l, --list\t\tlist the contents of the trash directory\n");
	printf("  -h, --help\t\tdisplay this help and exit\n");
	printf("      --version\t\toutput version information and exit\n");
	exit(0);
}

void rmrest_version()
{
	printf("rmrest: version 0.2, matty@inch.com (Matt Kressel) \n");
	exit(0);
}


void empty_dir_check(char *dirname)
{
	DIR *the_dir;
	struct dirent *the_dirent;
	int file_count = 0;

	while(1)
	{
		if((the_dir=opendir(dirname))==NULL)
		{
			rmsafe_error(dirname);
			break;
		}

		while((the_dirent=readdir(the_dir)) != NULL)
			file_count++;

		if(file_count == 2)
			printf("%s: No files in trash dir %s.\n",progname,dirname);

		closedir(the_dir);	
		break;
	}
}

	

/* builds a two way linked list from the index file */
void build_file_list(char *trash_dir)
{

	FILE *index = NULL;
	char aLine[FILENAME_MAX*2+1];
	char index_pathname[FILENAME_MAX];
	file_node *new_node = NULL;

	while(1)
	{
		sprintf(index_pathname,"%s/rmsafe.index",trash_dir);

		if((index=fopen(index_pathname,"r"))==NULL)
		{
			rmsafe_error(index_pathname);
			if(errno != ENOENT)
				printf("%s: Error opening index file!\n",progname);
			else empty_dir_check(trash_dir);
			break;
		}
	
		while(fgets(aLine,(FILENAME_MAX*2+1),index))
		{
			new_node = (file_node *) malloc(sizeof(file_node));
			sscanf(aLine,"%s\t%s",new_node->real_name,new_node->trash_name);

			/* if not directory and in DIR only mode */
			if((strcmp(new_node->trash_name,"DIR")!=0) && (RMREST_DIR == 1))
				new_node->skip = 1;
			else new_node->skip = 0;

			new_node->next = NULL;
			new_node->prev = tail;
			if(tail)
				tail->next = new_node;
			tail = new_node;
		}
		fclose(index);
		break;
	}
}	

void print_file_list()
{
	file_node *file_ptr;
	int namelen;

	file_ptr = tail;

	if(file_ptr)
	{	
		while(file_ptr)
		{
			if(file_ptr->skip == 0)
				if(RMREST_VERBOSE)
					printf("%s\n",file_ptr->real_name);
				else printf("%s\n",get_filename(file_ptr->real_name));
			file_ptr = file_ptr->prev;
		}
	} else printf("%s: No files in trash directory.\n",progname);
}

/* removes instance of a file from the linked list */

void remove_file_node(char *trash_name,char *real_name)
{

	file_node *file_ptr,*temp_ptr;

	file_ptr = tail;
	while(file_ptr)
	{
		if((strcmp(trash_name,file_ptr->trash_name)==0) &&
			(strcmp(real_name,file_ptr->real_name)==0))
				break;

		file_ptr = file_ptr->prev;
	}

	if(file_ptr)
	{
		temp_ptr = file_ptr;

		if(file_ptr->prev)
			file_ptr->prev->next = temp_ptr->next;

		if(file_ptr->next)
			file_ptr->next->prev = temp_ptr->prev;

		if(tail == temp_ptr)
			tail = temp_ptr->prev;

		free(temp_ptr);
			

	}
}		

/* writes the index liked list file out to disk */
void write_index_file()
{

	FILE *index;
	char index_pathname[FILENAME_MAX];
	file_node *file_ptr;

	sprintf(index_pathname,"%s/rmsafe.index",trash_dir);
	file_ptr = tail;


	while(1)
	{
		if(file_ptr != NULL)
		{	

			/* move to head of list */	
			while(file_ptr->prev)
				file_ptr = file_ptr->prev;
		}
	
		if((index=fopen(index_pathname,"w"))==NULL)
		{
			rmsafe_error(index_pathname);
			printf("%s: Error writing index file!\n",progname);
			break;
		}

		while(file_ptr)
		{
			fprintf(index,"%s\t%s\n",file_ptr->real_name,file_ptr->trash_name);
			file_ptr = file_ptr->next;
		}
		
		fclose(index);
		break;
	}
	
}	

/* returns pointer if file node exists in list, NULL otherwise */
file_node *in_trash(char *filename)
{
	
	/* NOTE: filename is just the filename of the file, not the pathname */

	file_node *file_ptr;
	file_node *retval;

	file_ptr = tail;

	while(file_ptr)
	{
		if(RMREST_SUBSTRING)
		{
			if((strstr(get_filename(file_ptr->real_name),filename)) &&
				(file_ptr->skip == 0))
			break;
		} else	if((strcmp(filename,get_filename(file_ptr->real_name))==0) &&
				(file_ptr->skip == 0))
			break;

		file_ptr = file_ptr->prev;
	}	
	

	return file_ptr;
}


/* removes a file from trash permanently */
void rmrest_clean(char *filename)
{
	file_node *file_ptr;
	int has_error = 0;
	int namelen;
	int isdir = 0;
	char yna[5];

	while(1)
	{
		if((file_ptr = in_trash(filename)) == NULL)
		{
			if(!RMREST_SUBSTRING && !RMREST_ALL)
				if(RMREST_DIR)
					printf("%s: Directory \"%s\" not in trash basket.\n",progname,filename);
				else printf("%s: File \"%s\" not in trash basket.\n",progname,filename);

			break;
		}
			
		if(RMREST_INTERACTIVE)
		{
			printf("%s: remove forever %s ? (y/n/a): ",progname,file_ptr->real_name);
			scanf("%s",yna);
			if((yna[0] == 'n') || (yna[0] == 'N'))
			{
				file_ptr->skip = 1;
				break;
			}
			if((yna[0] == 'a') || (yna[0] == 'A'))
				RMREST_INTERACTIVE = 0;
		}

		if(strcmp(file_ptr->trash_name,"DIR"))
		{
			if(remove(file_ptr->trash_name) < 0)
			{
				rmsafe_error(file_ptr->trash_name);
				has_error = 1;
			}
		} else isdir = 1;

		if(has_error == 0)
		{
			if(RMREST_VERBOSE)
			{
				if(isdir)
					printf("deleted directory %s\n",file_ptr->real_name); 
				else printf("deleted file %s\n",file_ptr->real_name); 
			}
			remove_file_node(file_ptr->trash_name,file_ptr->real_name);
			cleannum++;
		}
		break;
	}
}



/* restores file to its original directory */
void rmrest_file(file_node *file_ptr,char *t_file, char *r_file)
{
	char buf[2048];		/* 2K buffer */
	char yna[5];
	char trash_file[FILENAME_MAX];
	char real_file[FILENAME_MAX];
	char trash_dir[FILENAME_MAX];	/* trash directory */
	char dummy[FILENAME_MAX];	/* dummy var */
	int read_fd;
	int write_fd;
	int has_error = 0;
	int filelen;
	int isdir = 0;
	struct stat stbuf;
	ssize_t	amount;


	strcpy(trash_file,t_file);
	strcpy(real_file,r_file);

	if(strcmp(trash_file,"DIR")==0)
	{
		isdir = 1;
	}
	

	while(1)
	{
		/* skip file if in dir mode */
		if((isdir == 0) && (RMREST_DIR == 1))
			break;


		if(RMREST_INTERACTIVE)
		{
			printf("%s: \"restore\" %s ? (y/n/a): ",progname,real_file);
			scanf("%s",yna);
			if((yna[0] == 'n') || (yna[0] == 'N'))
			{
				file_ptr->skip = 1;
				break;
			}
			if((yna[0] == 'a') || (yna[0] == 'A'))
				RMREST_INTERACTIVE = 0;
		}

		if(isdir == 0)
		{
			if(stat(trash_file,&stbuf) < 0)
			{
				rmsafe_error(trash_file);
				break;
			}

			if((read_fd = open(trash_file,O_RDONLY)) < 0)
			{
				rmsafe_error(trash_file);
				break;
			}
			if((write_fd = open(real_file,(O_EXCL | O_CREAT | O_WRONLY),stbuf.st_mode)) < 0)
			{
				if(errno == ENOENT)
				{
					printf("\n%s: Create and/or Restore missing directories first!!\n",progname,real_file);
					has_error = 2;
					break;
				}

				rmsafe_error(real_file);
				break;
			}

			/* read/write loop */

			while((amount = read(read_fd,buf,2048)))
			{
				if(amount == -1)	/* error */
				{
					rmsafe_error(trash_file);
					has_error = 1;
					break;
				}
				if(amount == 0)		/* EOF */
				break;
	
				if(write(write_fd,buf,amount) < 0)
				{
					rmsafe_error(real_file);
					has_error = 1;
					break;
				}
			}


			if(has_error == 0)
			{
				close(read_fd);
				
				if(remove(trash_file) < 0)
				{
					rmsafe_error(trash_file);
					if((errno == EPERM) || (errno == EACCES))
						remove(real_file);
					has_error = 1;
				}
			}
		} else /* isdir == 1 */
		{
			/* create directory with 0700 perms */

			if(mkdir(real_file,S_IRUSR | S_IWUSR | S_IXUSR)<0)
			{
				if(errno == ENOENT)
				{
					printf("\n%s: Create and/or Restore missing directories first!!\n",progname,real_file);
					has_error = 2;
					break;
				}
				rmsafe_error(real_file);
				has_error = 1;
			}
		}
	
		if(has_error == 0)
		{
			remove_file_node(trash_file,real_file);
			if(RMREST_VERBOSE)
				printf("%s -> %s\n",trash_file,real_file);
			restnum++;
		}

		break;
		
	}

	if (has_error == 2)  /* fatal error */
	{
		write_index_file();
		exit(1);
	}
	if (read_fd) close(read_fd);
	if (write_fd) close(write_fd);
}

void rmrest_substring()
{
	file_node *file_ptr;

	if(!(file_ptr = in_trash(substring)))
		if(RMREST_DIR)
			printf("%s: Directory substring \"%s\" not found in trash basket.\n",progname,substring);
		else printf("%s: Substring \"%s\" not found in trash basket.\n",progname,substring);
	while((file_ptr = in_trash(substring)))
	{
		if (RMREST_CLEAN)
			rmrest_clean(substring);
		else rmrest_file(file_ptr,file_ptr->trash_name,file_ptr->real_name);
	}
}

void rmrest_all()
{
	file_node *file_ptr = tail;

	while(file_ptr)
	{
		if (RMREST_CLEAN)
			rmrest_clean(get_filename(file_ptr->real_name));
		else rmrest_file(file_ptr,file_ptr->trash_name,file_ptr->real_name);
		if(file_ptr->skip == 1)
			file_ptr = file_ptr->prev;
		else file_ptr = tail;
	}
}

int main(int argc, char *argv[])
{

	int c;

	char filename[FILENAME_MAX];
	struct passwd *the_user;		/* the user info */
	file_node *file_ptr;

	strcpy(progname,argv[0]);

	while(1)
        {
		int option_index = 0;
		static struct option long_options[] =
		{
			{"substring",1,0,'s'},
			{"all",1,0,'a'},
			{"directory",0,0,'d'},
			{"interactive",0,0,'i'},
			{"clean",0,0,'c'},
			{"list",0,0,'l'},
			{"help",0,0,'h'},
			{"version",0,0,'n'},
			{"verbose",0,0,'v'},
			{0,0,0,0}
		};
			
		c = getopt_long(argc,argv,"s:adiclhv",long_options, &option_index);
		if (c == -1) break;

		switch (c)
		{
			case 'a':
				RMREST_ALL = 1;
			break;
			case 's':
				if(optarg == 0)
				{
					printf("%s: missing argument to 's' option\n",progname);
					exit(1);
				}
				substring = optarg;
				RMREST_SUBSTRING = 1;
				break; 
			case 'd':
				RMREST_DIR = 1;
			break;
			case 'i':
				RMREST_INTERACTIVE = 1;
			break;
			case 'v':
				RMREST_VERBOSE = 1;
			break;
			case 'c':
				RMREST_CLEAN = 1;
			break;
			case 'l':
				RMREST_LIST = 1;
			break;
			case 'h':
				rmrest_help();
			break;
			case 'n':
				rmrest_version();
			break;
			default:
				printf("Try \"%s --help\" for more information.\n",progname);
				exit(1);
		}
	}
	if (optind < argc)
	{
		/* main loop */

		the_user = get_user();	/* get user struct */
		read_conf(trash_dir,the_user->pw_name);	/* read configuration */
		check_trash_dir(trash_dir);
		build_file_list(trash_dir);

		
		while (optind < argc)
		{
			
			sprintf(filename,get_filename(argv[optind]));
			if(RMREST_DIR)
			{
				if(filename[strlen(filename)-1]!='/')
					sprintf(filename,"%s/",filename);
			}


			if (RMREST_CLEAN)
			{
				rmrest_clean(filename);
				optind++;
				continue;
			}

			if((file_ptr = in_trash(filename)))
				rmrest_file(file_ptr,file_ptr->trash_name,file_ptr->real_name);
			else if(RMREST_DIR)
				printf("%s: Directory \"%s\" not in trash basket.\n",progname,filename);
			else {
				printf("%s: File \"%s\" not in trash basket.\n",progname,filename);
			}
		
			optind++;
		}
		write_index_file();
	} else
	{
		if (RMREST_LIST)
		{
			the_user = get_user();	/* get user struct */
			read_conf(trash_dir,the_user->pw_name);	/* read configuration */
			check_trash_dir(trash_dir);
			build_file_list(trash_dir);
			print_file_list();	
		}
		else if(RMREST_ALL)
		{
			the_user = get_user();	/* get user struct */
			read_conf(trash_dir,the_user->pw_name);	/* read configuration */
			check_trash_dir(trash_dir);
			build_file_list(trash_dir);
			rmrest_all();
			write_index_file();
		}
		else if(RMREST_SUBSTRING)
		{
			the_user = get_user();	/* get user struct */
			read_conf(trash_dir,the_user->pw_name);	/* read configuration */
			check_trash_dir(trash_dir);
			build_file_list(trash_dir);
			rmrest_substring();
			write_index_file();
		}
		else 
		{
			printf("rmrest: missing filename and/or directory.\n");
			printf("Try \"%s --help\" for more information.\n",progname);
			exit(1);
		}
	}
	if(RMREST_VERBOSE)
	{
		if(RMREST_LIST == 0)
		{
			if(RMREST_CLEAN)
				printf("%s: cleaned %d file(s).\n",progname,cleannum);
			else printf("%s: restored %d file(s).\n",progname,restnum);	
		}
	}
	exit(0);
}
					
