/*
 * PAK : split files in multi-volume archive.
 * Version 1.0
 *
 * Copyright (c) 1998, Emanuele Assenza (eassenza@lep.polito.it).
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.                      
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 *
 */ 
 
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

/* #define DEBUG */
/* #define INVERT_BYTES */

#define VER          "1.0"
#define PROG_NAME    "pak"
#define EXT          ".pak"
#define DEF_SIZE     1422
#define DEF_BLOCK    1024
#define CMD_CREATE   'c'
#define CMD_EXTRACT  'e'
#define CMD_LIST     'l'
#define OPT_SIZE     "-s" 
#define OPT_ALL      "-a"

struct main_pac_header {
  char prog_name[16];
  unsigned char n_vol;
} main_header;

struct pac_header {
  unsigned long int data_size;
  unsigned short int fname_size;
} header;

struct pac_tail {
  char end;
  char end_file;
} tail;

void split_pac(unsigned long int, int);
void merge_pac(char *, int);
void list_pac(char *, int);
FILE *pac_open(int *);
void read_file_name(char *, FILE *);
unsigned long int read_size(char **);
unsigned long int invert_long(unsigned long int);
unsigned short int invert_short(unsigned short int);
void pac_usage(void), print_msg(void);

char **name_in;                 /* Input filenames */
char *name_out;                 /* Output filename */
int pos = 1;          /* Current argument position */

/* ===================================================================== */

void main(int argc, char *argv[])
{
  unsigned long int filesize;
  char *p;

  if(argc < 3)
    pac_usage();
                              /* Parse command line */
  switch(argv[1][0]){
    
  case CMD_CREATE:
    filesize =  read_size(argv);
    if( argv[++pos] && argv[++pos] ){
      name_out = argv[pos-1]; 
      name_in = argv+pos;
#ifdef DEBUG
      fprintf(stderr, "pos: %d, nfiles: %d, in: %s, out: %s\n",
	      pos, argc-pos, *name_in, name_out);
#endif
      split_pac( filesize * DEF_BLOCK, argc - pos );
    } else 
      pac_usage();
    break;
    
  case CMD_EXTRACT:
    if( argv[++pos] ){
      if( !strcmp(argv[pos], OPT_ALL) && argv[pos+1] ){
	name_in = argv + (++pos);
	if( (p = strstr(*name_in, EXT)) )         
	  *p = '\0';
#ifdef DEBUG
      fprintf(stderr,"extract: file %s, pos %d\n", *name_in, pos) ;
#endif
	merge_pac( *name_in, 1 );
      } else if( argv[pos][0] != '-' ){
	name_in = argv+pos;
#ifdef DEBUG
	fprintf(stderr,"extract: file %s, pos %d\n", *name_in, pos) ;
#endif
	merge_pac( *name_in, 0 );
      } else 
	pac_usage();
    }
    break;

  case CMD_LIST:
    if( argv[++pos] ){
      if( !strcmp(argv[pos], OPT_ALL) && argv[pos+1] ){
	name_in = argv + (++pos);
	if( (p = strstr(*name_in, EXT)) )         
	  *p = '\0';
#ifdef DEBUG
	fprintf(stderr,"list: file %s, pos %d\n", *name_in, pos) ;
#endif
	list_pac( *name_in, 1 );
      } else if( argv[pos][0] != '-' ){
	name_in = argv+pos;
#ifdef DEBUG
	fprintf(stderr,"list: file %s, pos %d\n", *name_in, pos) ;
#endif
	list_pac( *name_in, 0 );
      } else 
	pac_usage();
      
    }
    break;

  default:
    pac_usage();
    
  }
}

/* ===================================================================== */

void list_pac(char *arcname, int all)
{
  FILE *fin;
  char tmpname[257], *outname;
  unsigned long int jump;
  short int MH_size = sizeof(main_header), H_size = sizeof(header), 
    T_size = sizeof(tail), vol_counter = 0;

  print_msg();

  tail.end_file = 1;
  if( all )
    sprintf(tmpname, "%s%s", arcname, EXT);      /* Add extension */
  else
    strcpy(tmpname, arcname);

  if( !(fin = fopen(tmpname, "rb")) ){             /* Open volume */
    fprintf(stderr, "%s: Unable to open %s\n",PROG_NAME, tmpname);
    exit(1);
  } 

  fread(&main_header, MH_size, 1, fin);       /* Read main header */
 
  if( strcmp(main_header.prog_name, PROG_NAME) ){
    printf("This isn't a %s archive!", PROG_NAME);
    exit(1);
  }

  fprintf(stderr, "Volume %s n. %u\n---------------------------------\n",
	  tmpname, main_header.n_vol);

  while(1){                                         /* Main Loop */

    fread(&header, H_size, 1, fin);               /* Read header */ 
#ifdef INVERT_BYTES
    header.data_size = invert_long( header.data_size );
    header.fname_size = invert_short( header.fname_size );
#endif
    outname = (char *) malloc( header.fname_size+1 );
    read_file_name( outname, fin );   /* Read name of archived file */
#ifdef DEBUG
    fprintf(stderr, "header: data_size %lu, name size %u, file: %s.\n", 
	    header.data_size, header.fname_size, outname );
#endif  

    fprintf(stderr, "file: %s \t %lu bytes ... ", outname, header.data_size);
    fseek(fin, header.data_size, SEEK_CUR);
    fread(&tail, T_size, 1, fin);                      /* Read tail */ 

#ifdef DEBUG
    fprintf(stderr, "\ntail: end %d, endfile %d\n-------\n", 
	    tail.end, tail.end_file );
#endif  
 
    switch( tail.end ){
                                         /* End of archived file */
    case 0:
      fprintf(stderr, "Ok.\n");
      break;
                                                /* End of volume */ 
    case 1:
      fclose(fin);
      if(tail.end_file){
	fprintf(stderr, "Ok.\n");
      } else 
	fprintf(stderr, "continue.\n---------------------------------\n");

      if( !all )
	exit(0);

      sprintf(tmpname, "%s.p%.2u", arcname, ++vol_counter);

      if( !(fin = fopen(tmpname, "rb")) ){
	fprintf(stderr, "%s: Unable to open %s !\n",PROG_NAME, tmpname);
	exit(1);
      }

      fread(&main_header, MH_size, 1, fin);
      fprintf(stderr, "Volume %s n. %u\n---------------------------------\n",
	      tmpname, main_header.n_vol);
      break;
                                 /* End of archive (last volume) */
    case 2: 
      fclose(fin);
      fprintf(stderr, "Ok.\n\nEnd.\n\n");
      exit(1);

    }
  }
  
}

/* ===================================================================== */

void merge_pac(char *arcname, int all)
{
  
  FILE *fin, *fout;
  char *outname, tmpname[257];
  register unsigned long int nByte;
  short int MH_size = sizeof(main_header), H_size = sizeof(header), 
    T_size = sizeof(tail), vol_counter = 0;

  print_msg();

  tail.end_file = 1;
  if( all )
    sprintf(tmpname, "%s%s", arcname, EXT);      /* Add extension */
  else
    strcpy(tmpname, arcname);

  if( !(fin = fopen(tmpname, "rb")) ){             /* Open volume */
    fprintf(stderr, "%s: Unable to open %s\n",PROG_NAME, tmpname);
    exit(1);
  } 

  fread(&main_header, MH_size, 1, fin);       /* Read main header */
 
  if( strcmp(main_header.prog_name, PROG_NAME) ){
    printf("This isn't a %s archive!", PROG_NAME);
    exit(1);
  }

  fprintf(stderr, "Volume %s n. %u\n---------------------------------\n",
	  tmpname, main_header.n_vol);

  while(1){                                         /* Main Loop */

    fread(&header, H_size, 1, fin);               /* Read header */ 
#ifdef INVERT_BYTES
    header.data_size = invert_long( header.data_size );
    header.fname_size = invert_short( header.fname_size );
#endif
    outname = (char *) malloc( header.fname_size+1 );
    read_file_name( outname, fin );   /* Read name of archived file */
#ifdef DEBUG
    fprintf(stderr, "header: data_size %lu, name size %u, file: %s.\n", 
	    header.data_size, header.fname_size, outname );
#endif  
 
    if( tail.end_file ){                   /* Open a new output file */
      if( !(fout = fopen( outname, "wb" )) ){
	fprintf(stderr, "%s: Unable to open %s\n",PROG_NAME, outname);
	exit(1);
      }
    }
    fprintf(stderr, "Extracting %s ... ", outname);

    for(nByte=0 ; nByte<header.data_size ; nByte++)   /* Write data */
      putc(getc(fin),fout);

    fread(&tail, T_size, 1, fin);                      /* Read tail */ 

#ifdef DEBUG
    fprintf(stderr, "bytes: %lu, tail: end %d, endfile %d\n---------\n", 
	    nByte, tail.end, tail.end_file );
#endif  
    
    switch( tail.end ){
                                         /* End of archived file */
    case 0:
      fclose(fout);
      fprintf(stderr, "Ok.\n");
      break;
                                                /* End of volume */ 
    case 1:
      fclose(fin);
      if(tail.end_file){
	fclose(fout);
	fprintf(stderr, "Ok.\n");
      } else 
	fprintf(stderr, "continue.\n---------------------------------\n");

      if( !all )
	exit(0);

      sprintf(tmpname, "%s.p%.2u", arcname, ++vol_counter);

      if( !(fin = fopen(tmpname, "rb")) ){
	fprintf(stderr, "%s: Unable to open %s !\n",PROG_NAME, tmpname);
	exit(1);
      }

      fread(&main_header, MH_size, 1, fin);
      fprintf(stderr, "Volume %s n. %u\n---------------------------------\n",
	      tmpname, main_header.n_vol);
      break;
                                 /* End of archive (last volume) */
    case 2: 
      fclose(fin);
      fclose(fout);
      fprintf(stderr, "Ok.\n\nEnd.\n\n");
      exit(1);

    }
  }
}

/* ===================================================================== */

void split_pac(unsigned long int vol_size, int Nfiles) 
{
  
  FILE *fin, *fout;
  unsigned long int space = vol_size, MINSIZE;
  register unsigned long int nByte;
  long int fileIndex;
  short int MH_size = sizeof(main_header), H_size = sizeof(header), 
    T_size = sizeof(tail), HT_size = H_size+T_size;
  int INPUT_END = 0;
  char buf, *p, outname[257];
#ifdef INVERT_BYTES
  struct pac_header inv_header;
#endif

  print_msg();

  if( (p = strstr(name_out, EXT)) )          /* Remove extension */
    *p = '\0';       
  sprintf(outname, "%s%s",name_out, EXT);    /* Create file name */

#ifdef DEBUG
  fprintf(stderr, "split_pac (sizes), MH: %d, H: %d, T: %d\n",
	  MH_size, H_size, T_size);
#endif

  space -= MH_size;

  if( !(fout = fopen( outname, "wb")) ){
    fprintf(stderr, "%s: Unable to open %s\n", PROG_NAME, outname);
    exit(1);
  }

  printf("Volume: %s\n------------------------------\n",outname);

  strcpy(main_header.prog_name, PROG_NAME);
  main_header.n_vol = 1;
  fwrite(&main_header, MH_size, 1, fout);
  fileIndex = MH_size;
  fin = pac_open( &Nfiles );
                                                              /* Main loop */
  while(1){      
    
    space -= HT_size + strlen(*name_in);/* Update available space for data */  
    if( *(name_in + 1) )                /* Set the minimun necessary space */
      MINSIZE = HT_size + strlen( *(name_in + 1) );
    else
      MINSIZE = 0;

    nByte = 0;
    fwrite(&header, H_size, 1, fout);         /* Write header and filename */
    fwrite(*name_in, strlen(*name_in), 1, fout);
    fprintf(stderr, "Adding %s ... ", *name_in);    

    while(nByte < space){   
      buf = getc(fin);                                       /* Write data */
      if( (INPUT_END = feof(fin)) )
	break;
      putc(buf, fout);
      nByte++;
    }   
  
    space -= nByte;
    header.data_size = nByte;                    /* Set header informations */
    header.fname_size = strlen(*name_in);
#ifdef INVERT_BYTES
    inv_header.data_size = invert_long( header.data_size );
    inv_header.fname_size = invert_short( header.fname_size );
#endif
#ifdef DEBUG
    fprintf(stderr, "data: %lu bytes, fname: %d, space: %lu\n", 
	    nByte, strlen(*name_in), space);
#endif

    fseek(fout, fileIndex, SEEK_SET);                   /* Return to header */

#ifdef DEBUG
    fprintf(stderr, "fpos index %lu\n", fileIndex);
#endif
#ifdef INVERT_BYTES
    fwrite(&inv_header, H_size, 1, fout);                   /* Write header */ 
#else
    fwrite(&header, H_size, 1, fout); 
#endif

    fseek(fout, 0, SEEK_END);
                                                   /* Set tail informations */
    if( INPUT_END ) {         
      tail.end_file = 1;
      
      if( space < MINSIZE )   
	tail.end = 1;
      else 
	tail.end = 0;    

      if( Nfiles == 1 )         
	tail.end = 2;

    } else {                   

      tail.end_file = 0;
      tail.end = 1;

    }
 
    fwrite(&tail, T_size, 1, fout);                          /* Write tail */

#ifdef DEBUG
    fprintf(stderr, "Tail: endfile %d, end %d\n", tail.end_file, tail.end );
#endif
 
    switch(tail.end){
                                                            /* End of file */
    case 0:                               
      fclose(fin);    
      fprintf(stderr, "Ok.\n");
      fileIndex += nByte + HT_size + header.fname_size;     
      Nfiles--;
      name_in++;	     
      fin = pac_open( &Nfiles );
      break;
                                                           /* End of volume */
    case 1:
      fclose(fout);
      fileIndex = MH_size;
      space = vol_size - MH_size;
      sprintf(outname, "%s.p%.2u", name_out, main_header.n_vol++);
                                                             /* End of file */
      if( tail.end_file ){
	fclose(fin);    
	fprintf(stderr, "Ok.\n");
	Nfiles--;
	name_in++;
	fin = pac_open( &Nfiles );
#ifdef DEBUG
	fprintf(stderr, "file: %s , n: %d\n", *name_in, Nfiles);
#endif
      } else 
	fprintf(stderr, "continue.\n");	
      
      fprintf(stderr, "%lu Bytes.\n------------------------------\n",
	      space+MH_size);

      if( !(fout = fopen( outname, "wb")) ){
	fprintf(stderr, "%s: Unable to open %s\n", PROG_NAME, outname);
	exit(1);
      }
	      
      printf("Volume: %s\n------------------------------\n",outname);
      fwrite(&main_header, MH_size, 1, fout);
      break;
                                            /* End of archive (last volume) */
    case 2:
      fclose(fout);
      fclose(fin);
      fprintf(stderr, "Ok.\n");
      fprintf(stderr, "%lu Bytes.\n------------------------------\nEnd.\n\n",
	      vol_size-space);
      exit(0);
    }
  }
}
  

/* ===================================================================== */

FILE *pac_open(int *num)
{
  FILE *fp;

  while( !(fp = fopen(*name_in, "rb")) ){
    fprintf(stderr, "Unable to open %s ... skipping.\n", *name_in);
    if( !(--(*num)) ){
      fprintf(stderr, "End.\n\n");
      exit(1);
    }
    name_in++;
  }

  return fp;
}

/* ===================================================================== */

unsigned long int read_size(char *option[])
{
  long int value;
  char *p;
                             /* Parse the command line to read the size */
  if( !strcmp(option[2], OPT_SIZE) && option[3] ){
    value = strtol(option[3], &p, 0);
    if(value > 0 && !*p){
      pos += 2;
#ifdef DEBUG
      fprintf(stderr, " size: %ld\n", value);
#endif
      return (unsigned long int) value;
    }
    pac_usage();
  } else if(option[2][0] == '-')
    pac_usage();
                            /* If there is no size return the defaut one */
  return DEF_SIZE;

}

/* ===================================================================== */

void read_file_name(char *p, FILE *fin)
{
  int i;
                                      /* Read the file name from archive */
  for(i=0 ; i<header.fname_size ; ++i)
    *(p+i) = getc(fin);

  *(p+i) = '\0';
}

/* ===================================================================== */

unsigned long int invert_long(unsigned long int lin)
{
  unsigned long int tmp, lout = 0;
  unsigned char i, byte, ldim = (char) sizeof(lin); 

  byte = (char) lin;
  tmp = byte;
  lout = lout | tmp;
  
  for(i=0 ; i<ldim-1 ; i++){
    lin = lin >> 8;
    lout = lout << 8;
    byte = (char) lin;
    tmp = byte;
    lout = lout | tmp;
  }

  return lout;
}

/* ===================================================================== */

unsigned short int invert_short(unsigned short int sin)
{
  unsigned short int tmp, sout = 0;
  unsigned char i, byte, sdim = (char) sizeof(sin); 

  byte = (char) sin;
  tmp = byte;
  sout = sout | tmp;
  
  for(i=0 ; i<sdim-1 ; i++){
    sin = sin >> 8;
    sout = sout << 8;
    byte = (char) sin;
    tmp = byte;
    sout = sout | tmp;
  }

  return sout;
}

/* ===================================================================== */

void print_msg(void)
{
  printf("PAK %s Copyright (c) 1998 Emanuele Assenza\n\n", VER);
}

/* ===================================================================== */

void pac_usage(void)
{
  print_msg();
  printf("Usage: %s [command] <option> ARCHIVE FILE <input files>\n\n", PROG_NAME);
  puts("       command    c : Create new archive ");
  puts("                  e : Extract archive");
  puts("                  l : List the contents of archive\n");

  printf("       options    -s [NUM] : Archive size (%d-blocks, default %d)\n",
	 DEF_BLOCK, DEF_SIZE);
  puts("                  -a       : Extract all archives\n");
  exit(1);
}





