/*
 * mibfm.c - Memory iButton File Manager
 *
 *  Copyright (c) by 	Karl-Martin Skontorp <kms@users.sourceforge.net>
 *			Martin Dahl <iznogood@users.sourceforge.net>
 *
 *  			Anders Semb Hermansen <ahermans@users.sourceforge.net>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>

#include "mibfm.h"

extern int  MLanFirst(int,int);
extern int  MLanNext(int,int);
extern void MLanFamilySearchSetup(int);
extern int  MLanAccess(void);
extern int  MLanSpeed(int);
extern int  MLanReadFile(uchar *, uchar *);
extern int  MLanFormatWriteFile(uchar *, int, uchar *);
extern int  MLanListFiles(void);
extern int  Aquire1WireNet(char *);
extern void Release1WireNet();
extern void MLanSerialNum(uchar *, int);
extern int MLanTouchReset(void);

extern void parse_args(int argc, char **argv);
extern void show_version(void);
extern void print_banner(void);
extern void show_params(void);

void PrintSerialNum(void);
void shutdown_net(void);
void upload_file(void);
void download_file(void);
void check_result(int result);

int main(int argc, char **argv)
{
    uchar SerialNumber[8];

    /* The default settings */
    family = DS1996;
    port = DEFAULT_PORT;
    ib_filename = DEFAULT_FILE;
    filename = DEFAULT_FILE;
    command = 0;
    force = 0;
    quiet = 0;

    /* Parse the command line arguments */
    parse_args(argc, argv);

    /* No command, show parameters and exit */
    if (!command) {
	show_params();
    }

    if (!quiet) {
	print_banner();
    }

    if (!quiet) {
	printf("  => Aquiring 1-Wire net.\n");
    }

    /* Attempt to aquire the 1-Wire net */
    if (!Aquire1WireNet(port)) {  
	if (!quiet) {
	    printf("     "); // Indent on non-quiet mode
	}
	printf("Unable to open %s, or no DS2480-based adapter detected.\n", port);

	exit(1);
    }

    /* Successfully aquired the 1-Wire net */
    if (!quiet) {
	printf("     Opened %s. DS2480-based adapter detected.\n", port);
	printf("\n");
    }

    /* Search for first iButton in desired family */
    if (!quiet) {
	printf("  => Look for first button in the 0x%02X family.\n", family);
    }

    MLanFamilySearchSetup(family);

    while (MLanNext(TRUE, FALSE)) {
	MLanSerialNum(SerialNumber, TRUE);
	if (SerialNumber[0] == family) {
	    break;
	}
    }

    if (SerialNumber[0] != family) {
	if (!quiet) {
	    printf("     "); // Indent on non-quiet mode
	}
	printf("No iButton in this family found.\n");

	shutdown_net();	

	exit(1);
    }

    if (!quiet) {
	/* print the Serial Number of the device just found */
	printf("     iButton found: ");
	PrintSerialNum();
	printf("\n");
    }

    if (!quiet) {
	printf("\n");
	printf("  => Accessing 1-Wire net.\n");
    }

    if (!MLanAccess()) {
	if (!quiet) {
	    printf("     "); // Indent on non-quiet mode
	}
	printf("Unable to access 1-Wire net.\n");

	shutdown_net();	

	exit(1);
    }

    switch (command) {
	case C_UPLOAD:
	    if (!quiet) {
		printf("\n");
		printf("  => Uploading file to iButton.\n");
	    }
	    upload_file();
	    break;
	case C_DOWNLOAD:
	    if (!quiet) {
		printf("\n");
		printf("  => Downloading file from iButton.\n");
	    }
	    download_file();
	    break;
    } 

    shutdown_net();	
    
    exit(0);
}

void PrintSerialNum(void) {
    uchar SerialNumber[8];
    int i;

    /* Read the serialnumber */
    MLanSerialNum(SerialNumber,TRUE);

    /* Print the serial number */
    for (i = 7; i >= 0; i--)
	printf("%02X",SerialNumber[i]);
}

void shutdown_net(void) {
    /* Turn off overdrive */
    MLanSpeed(MODE_NORMAL);
    if (!quiet) {
	printf("\n");
	printf("  => Setting 1-Wire net speed to normal.\n");
    }

    /* Release the 1-wire net */
    if (!quiet) {
	printf("\n");
	printf("  => Releasing 1-Wire net.\n");
	Release1WireNet();
	printf("     Closed port %s.\n", port);
    }
}

void upload_file() {
    int filesize = 0;
    struct stat filestat;
    int i;
    int result;
    uchar *tran_buffer; // File buffer

    /* Uploading, make buffer the size of local file */
    memset(&filestat, 0, sizeof(filestat));
    if(stat(filename, &filestat) < 0) {
	if (!quiet) {
	    printf("     "); // Indent on non-quiet mode
	}
	printf("%s - %s\n", filename, strerror(errno));
	
	shutdown_net();	
	
	exit(1);
    }
    filesize = filestat.st_size;
    tran_buffer = malloc(filesize);
    /* Just in case... */
    memset(tran_buffer, 0, filesize);
    
    /* Open file */    
    local_file = fopen(filename, "r");

    /* Read file into the buffer */
    for (i = 0; i < filesize; i++) {
	tran_buffer[i] = fgetc(local_file);
    }

    /* Close that file */
    fclose(local_file);

    /* Transfer the file to the iButton */
    result = MLanFormatWriteFile(ib_filename, filesize, tran_buffer);

    check_result(result);

    if (!quiet) {
	printf("     Local file: %s\n", filename);
	printf("     iButton file: %s\n",ib_filename);
	printf("     %d byte(s) successfully written to iButton.\n", filesize);
    }

    free(tran_buffer);
}

void download_file() {
    int result = 0;
    int i;
    uchar *tran_buffer; // File buffer

    /* Impossible to know size of file when downloading, make buffer the
     * size of the largest file possible */
    tran_buffer = malloc(MAX_FILESIZE);
    /* Just in case... */
    memset(tran_buffer, 0, MAX_FILESIZE);

    /* Open file */
    local_file = fopen(filename, "w");

    /* Read the file from the iButton */
    result = MLanReadFile(ib_filename, tran_buffer);

    check_result(result);
    
    if (!quiet) {
	printf("     Local file: %s\n", filename);
	printf("     iButton file: %s\n",ib_filename);
	printf("     %d byte(s) successfully read from iButton.\n", result);
    }

    /* Write file from buffer */
    for (i = 0; i < result; i++) {
	fputc(tran_buffer[i], local_file);
    }

    free(tran_buffer);

    /* Close that file */
    fclose(local_file);
}

void check_result(int result) {
    /* Check and see if we got errors in return. Not all of these are actually
     * supported... */
    switch (result) {
	case READ_ERROR:
	    if (!quiet) {
		printf("     "); // Indent on non-quiet mode
	    }
	    printf("Read error.\n");
	    
	    shutdown_net();	
	    
	    exit(1);
	    break;
	case INVALID_DIR:
	    if (!quiet) {
		printf("     "); // Indent on non-quiet mode
	    }
	    printf("Invalid dir.\n");
	    
	    shutdown_net();	
	    
	    exit(1);
	    break;
	case NO_FILE:
	    if (!quiet) {
		printf("     "); // Indent on non-quiet mode
	    }
	    printf("No file.\n");
	   
	    shutdown_net();	
	    
	    exit(1);
	    break;
	case WRITE_ERROR:
	    if (!quiet) {
		printf("     "); // Indent on non-quiet mode
	    }
	    printf("Write error.\n");
	    
	    shutdown_net();	
	    
	    exit(1);
	    break;
	case WRONG_TYPE:
	    if (!quiet) {
		printf("     "); // Indent on non-quiet mode
	    }
	    printf("Wrong type.\n");
	    
	    shutdown_net();	
	    
	    exit(1);
	    break;
	case FILE_TOO_BIG:
	    if (!quiet) {
		printf("     "); // Indent on non-quiet mode
	    }
	    printf("File too big.\n");
	    
	    shutdown_net();	
	    
	    exit(1);
	    break;
	case TRUE:
	    /* Do nothing */
	    break;
    }
}
