
/* logiso_get.c  
   Usage:	logiso_get  [ -v ] cd_mount_point [ log_file ]
      Get the isofs usage log and place it in log_file.  The log is written
      to stdout if log_file is not given.

      cd_mount_point can be any valid file path at or below the mount
      point of the cd.

      If -v is specified, header information is printed, otherwise just
      a list of inode numbers and operation numbers.
*/
/* (C) Copyright 1995 by Michael Coulter.  All rights reserved. */

#include <fcntl.h>
#include <linux/iso_fs.h>
#include "/usr/include/linux/iso_fs.h"
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <unistd.h>

#define FALSE		0
#define TRUE		1

extern void print_usage();
extern void process_args(int argc, 
			 char** argv, 
			 FILE** out_fp_p, 
			 int* verbose_p,
			 char** mount_path_p);

main(int argc, char** argv)
{
   struct iso_log_entry*	entry_p;
   struct iso_log_entry*	entry_limit_p;
   int				file_id;
   char*			mount_path;
   FILE*			out_fp;
   struct iso_log_info*		iso_log_info_p;
   int				result;
   int				verbose;

   process_args(argc, argv, &out_fp, &verbose, &mount_path);

   /* file_id = open("/mnt/system_cd/FileList", O_RDONLY); */
   file_id = open(mount_path, O_RDONLY); 
   if (file_id == -1) {
      fprintf(stderr, "Unable to open file %s.\n", mount_path);
      exit(1);
   }

   result = ioctl(file_id, ISO_IOC_GETLOGSIZE);
   if (verbose) {
      fprintf(out_fp, "The size is %d, 0x%x\n", result, result);
   }
   if (result < 0) {
      fprintf(stderr, 
      	      "Got size %d, aborting.\n", 
      	      result);
      exit(1);
   }
   iso_log_info_p = (struct iso_log_info*) malloc(result);

   result = ioctl(file_id, ISO_IOC_READLOG, iso_log_info_p);
   if (result != 0) {
      fprintf(stderr, "Failure to read log, status %d\n", result);
      exit(1);
   }
   if (iso_log_info_p->version != ISO_LOG_VERSION) {
      fprintf(stderr, "Expecting version %d, got %d\n", 
              ISO_LOG_VERSION, 
              iso_log_info_p->version);
      /* exit(2); */
   }
   if (verbose) {
      fprintf(out_fp, "version %d\nbuf_format %d\nnsize %d\ndevice %d\n"
                       "overflow_count %d\nnbr_entries %d\n",
	      iso_log_info_p->version,
	      iso_log_info_p->buf_format,
	      iso_log_info_p->size,
	      iso_log_info_p->device,
	      iso_log_info_p->overflow_count,
	      iso_log_info_p->nbr_entries);
   }

   entry_p = &iso_log_info_p->log_entry[0];
   entry_limit_p = entry_p + iso_log_info_p->nbr_entries;
   while (entry_p < entry_limit_p) {
      if (entry_p->device == iso_log_info_p->device) {
	 fprintf(out_fp, "%ld		%d	%d\n", 
		 entry_p->inode, 
		 entry_p->operation,
		 entry_p->device);
      }
      entry_p++;
   }
   fclose(out_fp);
   return 0;
}

void print_usage() 
{

char usage_str[] =
"   Usage:	logiso_get  [ -v ] cd_mount_point [ log_file ]\
      Get the isofs usage log and place it in log_file.  The log is written\
      to stdout if log_file is not given.\
\
      cd_mount_point can be any valid file path at or below the mount\
      point of the cd.\
\
      If -v is specified, header information is printed, otherwise just\
      a list of inode numbers and operation numbers.\
";
   fprintf(stderr, "%s\n", usage_str);
} /* end print_usage */

void process_args(int argc, 
		  char** argv, 
		  FILE** out_fp_p, 
		  int* verbose_p, 
		  char** mount_path_p)
{
   argc--; argv++;  /* skip command name */
   *verbose_p = FALSE;
   if (argc >= 1 && strcmp(*argv, "-v") == 0) {
      argc--; argv++;  /* done with -v */
      *verbose_p = TRUE;
   }
   if (argc < 1 ) {
      print_usage();
      fprintf(stderr, "No cd_mount_point was provided.\n");
      exit(1);
   }
   *mount_path_p = *argv;  argc--;
   *out_fp_p = stdout;
   if (argc >= 1) {
      if ((*out_fp_p = fopen(*argv, "w") ) == NULL) {
         fprintf(stderr, "Unable to open %s\n", *argv);
	 argc--; argv++;  /* done with -v */
      }
   }
} /* end process_args */
