/** lft.c -- list contents of directory(s) by file type.
 *
 * Copyright (c) 1997 by Leith S. Young.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**/

/* Written by Leith S. Young (leithy@ma.ultranet.com). */

#ifndef ONE_KILOBYTE
#define ONE_KILOBYTE	1024
#endif

#ifndef ONE_MEGABYTE
#define ONE_MEGABYTE	(1024 * 1024)
#endif

#ifndef ONE_GIGABYTE
#define ONE_GIGABYTE	(1024 * 1024 * 1024)
#endif

#define TTY 1
#define TRACE_IN 1
#define TRACE_OUT 0

#include <config.h>

#include <termios.h>
#ifdef GWINSZ_IN_SYS_IOCTL
#include <sys/ioctl.h>
#endif

#include <stddef.h>
#include <stdio.h>
#include <malloc.h>

#include "xproto.h"

#if HAVE_ERROR_AT_LINE
#include <error.h>
#else
#include "error.h"
#endif

#if HAVE_GETOPT_LONG
#include <getopt.h>
#else
#include "getopt.h"
#endif

#if HAVE_FNMATCH
#include <fnmatch.h>
#else
#include "fnmatch.h"
#endif

#include "system.h"
#include "xstrtol.h"
#include "safe-lstat.h"
#include "safe-stat.h"

#if !HAVE_STRCHR
#define strchr index
#endif

#if !HAVE_STRRCHR
#define strrchr rindex
#endif

#if DEBUG_VERSION
int debug;
#endif

#define INIT_ELEMENT(s1, s2, s3) sizeof (s1) - 1, s1, s2, s3
#define NEWLINE(x) putchar('\n'), x = check_rows(x)

/**
 * Null is a valid character in a color indicator (think about Epson
 * printers, for example) so we have to use a length/buffer string type. 
**/

struct cntrl
  {
    unsigned int len;		/* Length of string */
    char *string;		/* Pointer to the string */
    char *header;		/* Pointer to header */
    char *arg;			/* Pointer to argument for types option */
  };

/* This array of structures and the following enumerations must remain in sync. */
static struct cntrl file_type[] =
{
	/* Directory: bright blue */
  {INIT_ELEMENT ("01;34", "Directory", "directory")},
	/* Executable: bright green */
  {INIT_ELEMENT ("01;32", "Executable", "executable")},
	/* File: white */
  {INIT_ELEMENT ("00;37", "Regular", "regular")},
	/* Block device: yellow */
  {INIT_ELEMENT ("00;33", "Block special", "block-special")},
	/* Char device: bright yellow */
  {INIT_ELEMENT ("01;33", "Character special", "character-special")},
	/* hard link bright white */
  {INIT_ELEMENT ("01;37", "Hard link", "hard-link")},
#ifdef S_ISLNK
	/* Symlink: bright cyan */
  {INIT_ELEMENT ("01;36", "Symbolic link", "symbolic-link")},
	/* orphan Symlink: bright red */
  {INIT_ELEMENT ("01;31", "Orphan link", "orphan-link")},
#endif
#ifdef S_ISMPC
	/* multiplex file: red */
  {INIT_ELEMENT ("00;31", "Multiplex", "multiplex")},
#endif
#ifdef S_ISFIFO
	/* Pipe: magenta */
  {INIT_ELEMENT ("00;35", "Named pipe (FIFO)", "named-pipe")},
#endif
#ifdef S_ISSOCK
	/* Socket: bright magenta */
  {INIT_ELEMENT ("01;35", "Socket", "socket")},
#endif
	/* unknown file: white */
  {INIT_ELEMENT ("00;37", "Unknown", "unknown")},
};

/* These enumerations and the preceding array of structures must remain in sync. */
enum
  {
    DIR_IDX,
    EXEC_IDX,
    REG_IDX,
    BLK_IDX,
    CHR_IDX,
    HRD_IDX,
#ifdef S_ISLNK
    SYM_IDX,
    SYMO_IDX,
#endif
#ifdef S_ISMPC
    MPC_IDX,
#endif
#ifdef S_ISFIFO
    FIFO_IDX,
#endif
#ifdef S_ISSOCK
    SOCK_IDX,
#endif
    UNKN_IDX,
    TOT_TYPES			/* Total number of file types available. */
  }
tindex;
/* End synchronization */

int put_version __P ((char *pname, char *date, char *author));
int lofle __P ((char **array, int nelems));
int get_option __P ((int argc, char **argv, const char *short_opts,
		     struct option const *long_opts, int (*setopt) ()));
char *getuser __P ((uid_t uid));
char *getgroup __P ((gid_t gid));
char *xgetcwd __P ((void));
char *basename __P ((const char *name));
char *dirname __P ((char *path));
char *xstrdup __P ((char *string));
void *xmalloc __P ((size_t n));
void *xrealloc __P ((void *p, size_t n));

static int process_dir __P ((char *parent, char *child, char **argv, int index,
			     int f_spec));
static int set_option __P ((int c, int swt, int fatal));
static int compare_full_name __P ((const char **name1, const char **name2));
static int compare_base_name __P ((const char **name1, const char **name2));
static int compare_ext __P ((const char **name1, const char **name2));
static int compare_size __P ((const char **name1, const char **name2));
static int compare_slack __P ((const char **name1, const char **name2));
static int check_rows __P ((int rows_put));
static int trace_flow __P ((int entering, int rcode, int recursive,
			    char *func_name, char *path));
static long occ_blks __P ((double fsize, int bsize));
static char size_id (double size);
static char **check_env_opts __P ((char *env_var, char *env_opts));
static char **save_names __P ((char **narray, int nindex, char *name,
			       size_t size));
static void list_names __P ((int pindx, struct cntrl file_type[],
			     char **narray, char *path));
static void set_color (const char *string, int length, int action);
static void help __P ((int status));
static float slack __P ((double fsize, unsigned int blocks, int bsize));
static double cvt_size __P ((double size));

/* The following two externals are set by getopt_long */
extern int optind;
extern char *optarg;		/* Also set by check_env_opts if processing
				   option argument in environment variable */
static struct
  {
    int all;			/* list all */
    int bdev;			/* Include block devices (special) */
    int cdev;			/* Include character devices (special) */
    int color;			/* Display output in color if a tty. */
    int dir;			/* Include sub-directories (if any) */
    int Debug;			/* Display debugging info */
    int dref;			/* Dereference symbolic links */
    int empty;			/* display empty directories */
    int exec;			/* Include executable file(s) */
    int find;			/* output raw data */
    int help;			/* display help */
    int hlink;			/* Include hard link(s) */
    int long_format;		/* List names in long format */
    int kilo;			/* Display blocks in 1k byte blocks */
    int mux;			/* Include multiplexors */
    int names;			/* List filenames */
    int odf;			/* only list files that start with '.' */
    int olink;			/* only orphaned symbolic link(s) */
    int pipe;			/* Include FIFO(s) (named pipes) */
    int page;			/* page output (useful for debugging) */
    int pcbs;			/* POSIXLY_CORRECT block size */
    int recurse;		/* search all sub-directories */
    int reg;			/* Include non-executable file(s) */
    int reverse;		/* sort order: 0 = ascending, 1 = descending */
    int search;			/* 0 = best, 1 = top-down or 2 = bottom-up */
    int slink;			/* Include symbolic link(s) */
    int sock;			/* Include socket(s) */
    int tr_flow;		/* trace (debug) */
    int unknown;		/* Include unknown files */
    int vertical;		/* display names vertically */
    int version;		/* display version */
  }
opt;

static struct option const long_opts[] =
{
  {"all", no_argument, NULL, 'a'},
  {"color", no_argument, &opt.color, 1},
  {"dereference", no_argument, NULL, 'L'},
  {"debug", optional_argument, NULL, 11},
  {"empty-dir", no_argument, NULL, 'e'},
  {"find", no_argument, NULL, 'f'},
  {"help", no_argument, &opt.help, 1},
  {"kilobytes", no_argument, NULL, 'k'},
  {"long-format", no_argument, NULL, 'l'},
  {"names", no_argument, NULL, 'n'},
  {"only-dot-files", no_argument, NULL, 'o'},
  {"page", no_argument, &opt.page, 1},
  {"posix-block-size", no_argument, NULL, 'p'},
  {"reverse", no_argument, NULL, 'r'},
  {"recurse", optional_argument, NULL, 'R'},
  {"sort", optional_argument, NULL, 10},
  {"types", required_argument, NULL, 't'},
  {"vertical", required_argument, NULL, 'v'},
  {"version", no_argument, &opt.version, 1},
  {NULL, 0, NULL, 0}
};

/* Sort options */
enum
  {
    FULLNAME,
    BASENAME,
    EXT,
    SIZE,
    SLACK,
    NONE
  }
sort_type;			/* Type of sort to perform; FULLNAME is default. */

/* Trace options */
/* A function needs to call trace_flow at its beginning and end
 * in-order to be traceable. */
enum
  {
    TR_NONE,			/* Do not trace traceable functions */
    TR_RECUR,			/* Only trace traceable recursive functions */
    TR_ALL			/* Trace all traceable functions */
  };

/* Search options */
enum
  {
    BEST,			/* Chose best search method. */
    TOP_DOWN,			/* Search directory tree from the top-down. */
    BOTTOM_UP			/* Search directory tree from the bottom-up. */
  };

/* Color options */
enum
  {
    COLOR_OFF,
    COLOR_ON,
    DEFAULT
  };

/* name program was run with; may include leading path */
char *program_name;
/* name program was run with, stripped of any leading path */
char *prog_name;
/* printf control string (header) */
const char *const pfcontrol1 = "%-19s%6s%12s%12s%12s%8s";
/* printf control string (body) */
const char *const pfcontrol2 = "%-19s%6u%12lu%12lu%11.2f%c%7.1f%%";
/* number of columns available on tty */
int columns;
/* number of rows available on tty */
int rows;
/* number of rows output to tty */
int rows_put = 0;
/* Device type of the device being searched */
static dev_t search_dev;
/* Block size used by occ_blks and slack */
static int block_size;
/* Grand totals */
static unsigned long int grand_total_files = 0;
static unsigned long int grand_total_blocks = 0;
static double grand_total_size = 0.0;

int
main (int argc, char **argv)
{
  short int all_ok = 0;
  short int file_spec = 0;
  char *full_path;

  {				/* Begin prolog */
    int i;
    int n;
    long int tmp_long;
    char *e_ptr;
    char *t_ptr;
    char *lft_opts;
    char **opt_err = NULL;
    char *tmp_path = NULL;
#if DEBUG_VERSION
    char **debug_args;
#endif
    DIR *dir_ptr = NULL;
    static const char *const args_ba =
    "arguments %s processing by %s (10 max), optind = %d.";
    static const char *const errmsg =
    "ignoring invalid %s in environment variable %s: %s";
    static const char *const optn = "_OPTIONS";
    struct dirent *dp;
    struct stat sbuf;

    program_name = argv[0];
    prog_name = basename (argv[0]);

    /* Set default file type switches on (unset --type option.) */
    set_option ('t', 0, 1);

#if DEBUG_VERSION
    debug_args = xmalloc (argc * sizeof (char **));
    for (i = 0; i < argc; debug_args[i++] = xstrdup (argv[i]));
#endif /* DEBUG_VERSION */

    /* Begin processing of environment variables */
    rows = 25;
    columns = 80;
    if ((e_ptr = getenv ("LINES")))
      {
	if (xstrtol (e_ptr, NULL, 0, &tmp_long, NULL) == LONGINT_OK
	    && 0 < tmp_long && tmp_long <= INT_MAX)
	  {
	    rows = (int) tmp_long;
	  }
	else
	  {
	    error (0, 0, errmsg, "length", "LINES", e_ptr);
	    rows_put += 1;	/* In case --page option chosen */
	  }
      }
    if ((e_ptr = getenv ("COLUMNS")))
      {
	if (xstrtol (e_ptr, NULL, 0, &tmp_long, NULL) == LONGINT_OK
	    && 0 < tmp_long && tmp_long <= INT_MAX)
	  {
	    columns = (int) tmp_long;
	  }
	else
	  {
	    error (0, 0, errmsg, "width", "COLUMNS", e_ptr);
	    rows_put += 1;	/* In case --page option chosen */
	  }
      }
#ifdef TIOCGWINSZ
    {
      struct winsize ws;

      if (ioctl (1, TIOCGWINSZ, &ws) != -1 && ws.ws_col != 0)
	columns = ws.ws_col;
    }
#endif
    lft_opts = xmalloc (strlen (optn) + strlen (prog_name) + 1);
    strcpy (lft_opts, prog_name);
    strcat (lft_opts, optn);
    for (i = 0; lft_opts[i] != '_'; ++i)
      lft_opts[i] = toupper (lft_opts[i]);
    if ((e_ptr = getenv (lft_opts)))
      opt_err = check_env_opts (e_ptr, lft_opts);
    /* End processing of environment variables */

    /* Process command-line arguments */
    get_option (argc, argv, "aefklLnoprRv", long_opts, set_option);

    if (opt.Debug)
#if DEBUG_VERSION
      {
	debug = 1;
	if (e_ptr)		/* If true, display environment variable */
	  {
	    printf ("%s = %s", lft_opts, e_ptr);
	    NEWLINE (rows_put);
	  }
      }
#else
      {
	error (0, 0, "Debug code not included. Option ignored.");
	rows_put = check_rows (rows_put);
      }
#endif /* DEBUG_VERSION */

    if (opt_err)		/* display errors found. */
      {
	error (0, 0, opt_err[0]);
	rows_put = check_rows (rows_put);
	free (opt_err[0]);
	for (i = 1; opt_err[i] != NULL; ++i)
	  {
	    fprintf (stderr, "%s", opt_err[i]);
	    NEWLINE (rows_put);
	    free (opt_err[i]);
	  }
	free (opt_err);
      }
    free (lft_opts);

    if (opt.help)
      help (EXIT_SUCCESS);

    if (opt.version)
      {
	put_version (prog_name, "1997", "Leith S. Young");
	exit (EXIT_SUCCESS);
      }
    block_size = 512;		/* POSIXLY_CORRECT block size. */
    if (opt.kilo)
      block_size = 1024;
    else if (!opt.pcbs)
      set_option ('p', 1, 1);

    /* Begin processing of path */
    tmp_path = xmalloc (PATH_MAX + 1);
    if (argv[optind])
      {
	realpath (argv[optind], tmp_path);
	if (!(dir_ptr = opendir (tmp_path)))
	  {
	    if (errno != ENOTDIR && errno != ENOENT && errno != EINVAL)
	      error (1, errno, argv[optind]);
	    else
	      {
		t_ptr = dirname (tmp_path);
		strcpy (tmp_path, t_ptr);
		free (t_ptr);
		if (!(dir_ptr = opendir (tmp_path)))
		  error (1, errno, tmp_path);
		else
		  {
		    for (i = optind; argv[i] != NULL; i++)
		      {
			t_ptr = basename (argv[i]);
			n = strlen (t_ptr);
			memmove (argv[i], t_ptr, n);
			argv[i][n] = '\0';
		      }
		    file_spec = 1;
		  }
	      }
	  }
	else
	  {
	    if (argv[++optind] == NULL)
	      optind = 0;
	    else
	      file_spec = 1;
	  }
      }
    else
      tmp_path = xgetcwd ();

    if (dir_ptr)
      CLOSEDIR (dir_ptr);

    chdir (tmp_path);
    if (!(dir_ptr = opendir (tmp_path)))
      error (1, errno, tmp_path);

    while ((dp = readdir (dir_ptr)) != NULL)
      {
	if (dp->d_name[0] == '.' && (dp->d_name[1] == '\0'
			|| (dp->d_name[1] == '.' && dp->d_name[2] == '\0')))
	  {
	    if (SAFE_LSTAT (dp->d_name, &sbuf) == -1)
	      error (1, errno, " (lstat) %s", dp->d_name);
	    else
	      {
		search_dev = sbuf.st_dev;
		break;
	      }
	  }
      }
    CLOSEDIR (dir_ptr);
    full_path = xstrdup (tmp_path);
    free (tmp_path);
    /* End processing of path */

#if DEBUG_VERSION
    if (debug)
      {
	printf ("argv[0] %s - argc = %d", argv[0], argc);
	NEWLINE (rows_put);
	printf (args_ba, "before", prog_name, optind);
	NEWLINE (rows_put);
	for (i = 1; i < argc; i++)
	  {
	    printf ("arg %d = %s", i, debug_args[i]);
	    NEWLINE (rows_put);
	    if (i == 10)
	      break;
	  }
	printf (args_ba, "after", prog_name, optind);
	NEWLINE (rows_put);
	for (i = 1; i < argc; i++)
	  {
	    printf ("arg %d = %s", i, argv[i]);
	    NEWLINE (rows_put);
	    if (i == 10)
	      break;
	  }
	NEWLINE (rows_put);
	printf ("realpath = %s", full_path);
	NEWLINE (rows_put);
      }
    for (i = 0; i < argc; free (debug_args[i]), ++i);
    free (debug_args);
#endif /* DEBUG_VERSION */

    if (opt.search == BEST)	/* direction not forced */
      {
	if (!file_spec)		/* If we're not looking for specific files */
	  opt.search = TOP_DOWN;	/* search directory tree from top-down */
	else
	  opt.search = BOTTOM_UP;	/* otherwise, search from bottom-up to
					 * insure all files are examined. */
      }
  }				/* End prolog */

  all_ok = process_dir (full_path, full_path, argv, optind, file_spec);

  if (isatty (TTY) && opt.color)
    set_color (NULL, 0, DEFAULT);
  if (!opt.find)
    {
      if (all_ok == 1)
	{
	  if (opt.recurse)	/* skip grand total if not recursive search */
	    {
	      printf (pfcontrol2, "Grand Total", grand_total_files,
		      grand_total_blocks,
		      occ_blks (grand_total_size, block_size),
		    cvt_size (grand_total_size), size_id (grand_total_size),
		      slack (grand_total_size, grand_total_blocks,
			     block_size));
	      NEWLINE (rows_put);
	    }
	}
      else if (file_spec)
	error (1, ENOENT, argv[optind]);
    }
  return EXIT_SUCCESS;
}

static int
process_dir (char *parent, char *child, char **argv, int idx, int f_spec)
{
  static int success;

  DIR *dirp;
  struct dirent *dptr;
  struct stat sn_statf;
  struct stat sl_statf;

  char *next_child;

  int i;
  int cd_or_pd;			/* True if file is the current or parent directory. */
  int orphan_link = 0;
  int file_count = 0;
  int found_something = 0;

  struct
    {
      int found;		/* Number of files found */
      unsigned long blocks;	/* Number of blocks allocated for files */
      double size;		/* Number of bytes occupied by files */
      char **names;		/* pointer to array containing file names */
    }
  f_type[TOT_TYPES];

  unsigned long int dir_total_files = 0;
  unsigned long int dir_total_blocks = 0;
  double dir_total_size = 0.0;

#if DEBUG_VERSION
  if (debug)
    trace_flow (TRACE_IN, 0, 1, "process_dir", child);
#endif /* DEBUG_VERSION */

  for (i = 0; i != TOT_TYPES; i++)
    {
      f_type[i].found = 0;
      f_type[i].blocks = 0L;
      f_type[i].size = 0.0;
      f_type[i].names = NULL;
    }

  chdir (child);

  if (!(dirp = opendir (child)))
    {
      if (errno == EACCES)
	{
	  NEWLINE (rows_put);
	  error (0, 0, "%s: Permission denied", child);
	  rows_put = check_rows (rows_put);
	  chdir (parent);
	  return trace_flow (TRACE_OUT, success, 1, "process_dir", child);
	}
      else
	error (1, errno, "%s", child);
    }
  if (dirp)
    {
      while ((dptr = readdir (dirp)) != NULL)
	{
	  /* Is this a . or .. file? */
	  if (dptr->d_name[0] == '.' && (dptr->d_name[1] == '\0'
		    || (dptr->d_name[1] == '.' && dptr->d_name[2] == '\0')))
	    cd_or_pd = 1;
	  else
	    {
	      cd_or_pd = 0;
	      ++file_count;
	    }
	  if (SAFE_LSTAT (dptr->d_name, &sn_statf) == -1)
	    error (1, errno, "(lstat) %s", dptr->d_name);
	  /* if not 'a' or 'o' option skip files starting with '.'. */
	  if (!opt.all)
	    {
	      if (opt.odf && dptr->d_name[0] != '.')
		{
		  if (!S_ISDIR (sn_statf.st_mode))
		    continue;
		}
	      else if (!opt.odf && dptr->d_name[0] == '.')
		continue;
	    }
	  if (!cd_or_pd && search_dev == sn_statf.st_dev)
	    {
	      if (S_ISDIR (sn_statf.st_mode) && opt.recurse
		  && opt.search == BOTTOM_UP)
		{
		  next_child = xmalloc (PATH_MAX + 1);
		  if (!realpath (dptr->d_name, next_child))
		    error (1, errno, "(realpath) %s", next_child);
		  success = process_dir (child, next_child, argv, idx, f_spec);
		  free (next_child);
		}
	    }
	  if (f_spec)
	    {
	      i = idx;
	      while (fnmatch (argv[i], dptr->d_name, 0))
		{
		  if (argv[++i] == NULL)
		    break;
		}
	      if (argv[i] == NULL)
		continue;
	    }
	  tindex = -1;
	  if (S_ISDIR (sn_statf.st_mode) && opt.dir && !cd_or_pd)
	    {
	      if (opt.odf && dptr->d_name[0] != '.')
		continue;
	      tindex = DIR_IDX;
	    }
	  else if (S_ISREG (sn_statf.st_mode) && (opt.exec || opt.reg))
	    {
	      if (sn_statf.st_mode & S_IXUGO)
		{
		  if (opt.exec)
		    tindex = EXEC_IDX;
		  else
		    continue;
		}
	      else if (opt.reg)
		tindex = REG_IDX;
	      else
		continue;
	    }
#ifdef S_ISLNK
	  else if (S_ISLNK (sn_statf.st_mode) && opt.slink)
	    {
	      if (SAFE_STAT (dptr->d_name, &sl_statf) == -1)
		{
		  if (errno == ENOENT)
		    orphan_link = 1;
		  else
		    error (1, errno, "(stat) %s", dptr->d_name);
		}
	      if (orphan_link)
		{
		  orphan_link = 0;
		  tindex = SYMO_IDX;
		}
	      else
		tindex = SYM_IDX;
	    }
	  else if (S_ISLNK (sn_statf.st_mode) && opt.olink)
	    {
	      if (SAFE_STAT (dptr->d_name, &sl_statf) == -1)
		{
		  if (errno == ENOENT)
		    orphan_link = 1;
		  else
		    error (1, errno, "(stat) %s", dptr->d_name);
		}
	      if (!orphan_link)
		continue;
	      orphan_link = 0;
	      tindex = SYMO_IDX;
	    }
#endif
#ifdef S_ISMPC
	  else if (S_ISMPC (sn_statf.st_mode) && opt.mux)
	    {
	      tindex = MPC_IDX;
	    }
#endif
#ifdef S_ISFIFO
	  else if (S_ISFIFO (sn_statf.st_mode) && opt.pipe)
	    {
	      tindex = FIFO_IDX;
	    }
#endif
#ifdef S_ISSOCK
	  else if (S_ISSOCK (sn_statf.st_mode) && opt.sock)
	    {
	      tindex = SOCK_IDX;
	    }
#endif
	  else if (S_ISCHR (sn_statf.st_mode) && opt.cdev)
	    {
	      tindex = CHR_IDX;
	    }
	  else if (S_ISBLK (sn_statf.st_mode) && opt.bdev)
	    {
	      tindex = BLK_IDX;
	    }
	  else if (opt.unknown && !(S_ISDIR (sn_statf.st_mode)
#ifdef S_ISLNK
				    || S_ISLNK (sn_statf.st_mode)
#endif
#ifdef S_ISMPC
				    || S_ISMPC (sn_statf.st_mode)
#endif
#ifdef S_ISFIFO
				    || S_ISFIFO (sn_statf.st_mode)
#endif
#ifdef S_ISSOCK
				    || S_ISSOCK (sn_statf.st_mode)
#endif
				    || S_ISREG (sn_statf.st_mode)
				    || S_ISCHR (sn_statf.st_mode)
				    || S_ISBLK (sn_statf.st_mode)))
	    {
	      tindex = UNKN_IDX;
	    }
/* if we get here, searching for hard links, all or only '.' files. */

/* Note:
 * Since hard links are not reflected in st_mode field of stat structure,
 * search for hard links should probably be stand-alone option like
 * only_dot_files.
 */
	  else if (opt.hlink)
	    {
	      if (sn_statf.st_nlink > 1)
		{
/* Treat '.' and '..' dirs as hard links for display purposes */
		  if (cd_or_pd)
		    {
		      tindex = HRD_IDX;
		    }
		  else if (!S_ISDIR (sn_statf.st_mode))
		    {
		      tindex = HRD_IDX;
		    }
		  else
		    continue;
		}
	      else
		continue;
	    }
	  else
	    continue;

	  if (tindex >= 0)
	    found_something = 1;
	  /* if true, save name */
	  if (opt.names || (opt.search == TOP_DOWN && tindex == DIR_IDX))
	    f_type[tindex].names = save_names (f_type[tindex].names,
					       f_type[tindex].found,
					       dptr->d_name, NLENGTH (dptr));
	  f_type[tindex].found += 1;
	  f_type[tindex].blocks += ST_NBLOCKS (sn_statf);
	  if (f_type[tindex].blocks)
	    f_type[tindex].size += (double) sn_statf.st_size;
/* end of "while ((dptr = readdir (dirp)) != NULL)" */
	}

      if (found_something)
	{
	  success = 1;
	  if (!opt.find)
	    {
	      NEWLINE (rows_put);
	      printf ("%s:", child);
	      NEWLINE (rows_put);
	    }
	  i = 0;
	  do
	    {
	      if (opt.names && i == 0)
		i = 1;
	      else
		{
		  if (opt.find)
		    break;
		  i = 2;
		  NEWLINE (rows_put);
		  printf (pfcontrol1, "File type/Total", "Found",
			  "Allocated", "Occupied", "Used", "Slack");
		  NEWLINE (rows_put);
		}
	      for (tindex = 0; tindex < TOT_TYPES; tindex++)
		{
		  switch (tindex)
		    {
		    case DIR_IDX:
		      if (opt.dir && f_type[tindex].found)
			break;
		      continue;
		    case EXEC_IDX:
		      if (opt.exec && f_type[tindex].found)
			break;
		      continue;
		    case REG_IDX:
		      if (opt.reg && f_type[tindex].found)
			break;
		      continue;
		    case HRD_IDX:
		      if (opt.hlink && f_type[tindex].found)
			break;
		      continue;
		    case CHR_IDX:
		      if (opt.cdev && f_type[tindex].found)
			break;
		      continue;
		    case BLK_IDX:
		      if (opt.bdev && f_type[tindex].found)
			break;
		      continue;
		    case UNKN_IDX:
		      if (opt.unknown && f_type[tindex].found)
			break;
		      continue;
#ifdef S_ISLNK
		    case SYM_IDX:
		      if (opt.slink && f_type[tindex].found)
			break;
		      continue;
		    case SYMO_IDX:
		      if (opt.olink && f_type[tindex].found)
			break;
		      continue;
#endif
#ifdef S_ISMPC
		    case MPC_IDX:
		      if (opt.mux && f_type[tindex].found)
			break;
		      continue;
#endif
#ifdef S_ISFIFO
		    case FIFO_IDX:
		      if (opt.pipe && f_type[tindex].found)
			break;
		      continue;
#endif
#ifdef S_ISSOCK
		    case SOCK_IDX:
		      if (opt.sock && f_type[tindex].found)
			break;
		      continue;
#endif
		    default:
		      continue;
		    }
		  if (i == 1)
		    list_names (tindex, file_type, f_type[tindex].names, child);
		  else
		    {
		      if (opt.kilo)
			f_type[tindex].blocks = (f_type[tindex].blocks + 1) / 2;
		      if (isatty (TTY) && opt.color)
			set_color (file_type[tindex].string,
				   file_type[tindex].len, COLOR_ON);
		      printf (pfcontrol2, file_type[tindex].header,
			      f_type[tindex].found, f_type[tindex].blocks,
			      occ_blks (f_type[tindex].size, block_size),
			      cvt_size (f_type[tindex].size),
			      size_id (f_type[tindex].size),
			      slack (f_type[tindex].size,
				     f_type[tindex].blocks, block_size));
		      dir_total_files += f_type[tindex].found;
		      dir_total_blocks += f_type[tindex].blocks;
		      dir_total_size += f_type[tindex].size;
		      if (isatty (TTY) && opt.color)
			set_color (NULL, 0, COLOR_OFF);
		      NEWLINE (rows_put);
		    }
		}
	    }
	  while (i != 2);

	  if (!opt.find)
	    {
	      printf (pfcontrol2, "Directory Total",
		      dir_total_files, dir_total_blocks,
		      occ_blks (dir_total_size, block_size),
		      cvt_size (dir_total_size), size_id (dir_total_size),
		      slack (dir_total_size, dir_total_blocks, block_size));
	      NEWLINE (rows_put);

	      grand_total_files += dir_total_files;
	      grand_total_blocks += dir_total_blocks;
	      grand_total_size += dir_total_size;

	      if (opt.recurse && opt.search == TOP_DOWN)
		{
		  if (f_type[DIR_IDX].names != NULL)
		    {
		      next_child = xmalloc (PATH_MAX + 1);
		      for (i = 0; f_type[DIR_IDX].names[i] != NULL; ++i)
			{
			  if (!realpath (f_type[DIR_IDX].names[i], next_child))
			    error (1, errno, "(realpath) %s", next_child);
			  if (SAFE_LSTAT (f_type[DIR_IDX].names[i],
					  &sn_statf) == -1)
			    error (1, errno, " (lstat) %s",
				   f_type[DIR_IDX].names[i]);
			  else if (search_dev != sn_statf.st_dev)
			    continue;
			  /* skip . and .. files */
			  if (f_type[DIR_IDX].names[i][0] == '.'
			      && (f_type[DIR_IDX].names[i][1] == '\0'
				  || (f_type[DIR_IDX].names[i][1] == '.'
				   && f_type[DIR_IDX].names[i][2] == '\0')))
			    continue;
			  success = process_dir (child, next_child, argv, idx,
						 f_spec);
			}
		      free (next_child);
		      for (i = 0; f_type[DIR_IDX].names[i] != NULL; ++i)
			free (f_type[DIR_IDX].names[i]);
		      free (f_type[DIR_IDX].names);
		    }
		}
	    }
	}
      CLOSEDIR (dirp);
      if (!opt.find && !file_count)
	{
	  if (opt.empty)
	    {
	      NEWLINE (rows_put);
	      printf ("%s: [EMPTY]", child);
	      NEWLINE (rows_put);
	      NEWLINE (rows_put);
	    }
	}
      chdir (parent);
      return trace_flow (TRACE_OUT, success, 1, "process_dir", child);
    }
  chdir (parent);
  return trace_flow (TRACE_OUT, 0, 1, "process_dir", child);
}

static int
trace_flow (int entering, int rcode, int recursive, char *func_name, char *path)
{
#if DEBUG_VERSION
  struct tr_info
  {
    char *fname;		/* Pointer to function name */
    int r_level;		/* Recursion level */
  };
  static struct tr_info *tr_func;
  static int last = -1;
  int func_found = 0;
  int x;
  char enter[] = "Entering";
  char leave[] = "Leaving ";
  char *e_or_l;

  if (debug)
    {
      if (opt.tr_flow != TR_NONE)
	{
	  if (last < 0)
	    {
	      last = 0;
	      tr_func = xmalloc (sizeof (struct tr_info));
	      tr_func[last].fname = NULL;
	      tr_func[last].r_level = 0;
	    }
	  for (x = 0; tr_func[x].fname != NULL; ++x)
	    {
	      if (!strcmp (tr_func[x].fname, func_name))
		{
		  func_found = 1;
		  break;
		}
	    }
	  if (!func_found)
	    {
	      tr_func = xrealloc (tr_func,
				  sizeof (struct tr_info) * (last + 2));
	      tr_func[last].fname = xstrdup (func_name);
	      x = last;
	      tr_func[++last].fname = NULL;
	      tr_func[last].r_level = 0;
	    }
	  if (recursive)
	    {
	      if (entering)
		{
		  e_or_l = enter;
		  ++tr_func[x].r_level;
		}
	      else
		e_or_l = leave;
	      printf ("%s %s (%s), level = %d", e_or_l,
		      func_name, path, tr_func[x].r_level);
	      if (!entering)
		--tr_func[x].r_level;
	      NEWLINE (rows_put);
	    }
	  else if (opt.tr_flow == TR_ALL)
	    {
	      if (entering)
		e_or_l = enter;
	      else
		e_or_l = leave;
	      if (path)
		printf ("%s %s (%s)", e_or_l, func_name, path);
	      else
		printf ("%s %s", e_or_l, func_name);
	      NEWLINE (rows_put);
	    }
	}
    }
#endif /* DEBUG_VERSION */
  return rcode;
}

/* process environment variable ..._OPTIONS */
static char **
check_env_opts (char *env_var, char *env_opts)
{
  static const char *const enverr = " %10s: %s";
  static const char *const bad_arg = "invalid argument(s) (ignored)";
  static const char *const err_header =
     "Errors found in environment variable %s:";
  static const char *const missing_arg =
     "optional argument missing (using default)";
  static const char *const required_arg =
     "required argument missing (option ignored)";
  static const char *const no_arg =
     "option does not take an argument (argument ignored)";

  char **env_err = NULL;
  char *err_tmp;
  char *env;
  char *tok;
  char *save_tok;
  int i;
  int x;
  int y;

  if (env_var)
    {
      err_tmp = xmalloc (PATH_MAX + 1);
      sprintf (err_tmp, err_header, env_opts);
      env_err = xmalloc (sizeof (char **));
      env_err[0] = xstrdup (err_tmp);
      env = xstrdup (env_var);
      for (y = 1, tok = strtok (env, "-; "); tok; tok = strtok (NULL, "-; "))
	{
	  save_tok = NULL;
	  if (strchr (tok, '='))
	    {
	      save_tok = tok;
	      tok = xstrdup (save_tok);
	      optarg = strchr (tok, '=');
	      *optarg++ = '\0';
	    }
	  for (i = 0, x = 0; long_opts[i].name != NULL; ++i)
	    {
	      if (!strncmp (long_opts[i].name, tok, strlen (tok)))
		{
		  if (save_tok)
		    {
		      switch (long_opts[i].has_arg)
			{
			case no_argument:
			  sprintf (err_tmp, enverr, tok, no_arg);
			  env_err = xrealloc (env_err,
					      sizeof (char **) * (y + 1));
			  env_err[y++] = xstrdup (err_tmp);
			  break;
			case optional_argument:
			  if (!*optarg)
			    {
			      sprintf (err_tmp, enverr, tok, missing_arg);
			      env_err = xrealloc (env_err,
						sizeof (char **) * (y + 1));
			      env_err[y++] = xstrdup (err_tmp);
			    }
			  break;
			case required_argument:
			  if (!*optarg)
			    {
			      sprintf (err_tmp, enverr, tok, required_arg);
			      env_err = xrealloc (env_err,
						sizeof (char **) * (y + 1));
			      env_err[y++] = xstrdup (err_tmp);
			    }
			  break;
			}
		    }
		  x = 1;
		  if (long_opts[i].flag)
		    *long_opts[i].flag = 1;
		  else if (!set_option (long_opts[i].val, 1, 0))
		    {
		      sprintf (err_tmp, enverr, tok, bad_arg);
		      env_err = xrealloc (env_err, sizeof (char **) * (y + 1));
		      env_err[y++] = xstrdup (err_tmp);
		    }
		  break;
		}
	    }
	  if (!x)
	    {
	      sprintf (err_tmp, enverr, tok,
		       "invalid option (option ignored)");
	      env_err = xrealloc (env_err, sizeof (char **) * (y + 1));
	      env_err[y++] = xstrdup (err_tmp);
	    }
	  if (save_tok)
	    {
	      free (tok);
	      tok = save_tok;
	    }
	}
      free (env);
      free (err_tmp);
    }
  if (y == 1)
    {
      free (env_err[0]);
      free (env_err);
      return NULL;
    }
  env_err[y] = NULL;
  return env_err;
}

/* convert size (in bytes) to kilobytes, megabytes or gigabytes. */
static double
cvt_size (double size)
{
  if (size >= ONE_GIGABYTE)
    return size / ONE_GIGABYTE;
  else
    return (size >= ONE_MEGABYTE ? size / ONE_MEGABYTE : size / ONE_KILOBYTE);
}

/* Return size indicator K = kilobytes; M = megabytes; G = gigabytes. */
static char
size_id (double size)
{
  if (size >= ONE_GIGABYTE)
    return 'G';
  else
    return (size >= ONE_MEGABYTE ? 'M' : 'K');
}

/* compute number of blocks occupied by file(s) */
static long
occ_blks (double fsize, int bsize)
{
  if (fsize == 0.0)
    return 0;
  else
    {
      unsigned long t1;
      double t2;

      t1 = (unsigned long) (fsize / bsize);
      t2 = t1 * bsize;
      return t2 != fsize ? (fsize / bsize) + 1.0 : fsize / bsize;
    }
}

/* compute percentage of difference between 'Allocated' and 'Used' */
static float
slack (double fsize, unsigned int blocks, int bsize)
{
  if (blocks == 0)
    return (float) 0.0;
  else
    return (float) ((((double) (blocks * bsize) - fsize)
		     / (blocks * bsize)) * 100);
}

/* Save file names for later printing */
static char **
save_names (char **narray, int nindex, char *name, size_t size)
{
  int n = sizeof (char **);

#if DEBUG_VERSION
  if (debug)
    trace_flow (TRACE_IN, 0, 0, "save_names", name);
#endif /* DEBUG_VERSION */
  n = (n * nindex) + (n * 2);
  narray = xrealloc (narray, n);
  narray[nindex] = xmalloc (size + 1);
  strncpy (narray[nindex], name, size);
  narray[nindex][size] = '\0';
  narray[++nindex] = NULL;
#if DEBUG_VERSION
  trace_flow (TRACE_OUT, 0, 0, "save_names", NULL);
#endif /* DEBUG_VERSION */
  return narray;
}

/* Print file names */
static void
list_names (int pindx, struct cntrl file_type[], char **narray, char *path)
{
  struct stat sn_statf;
  int (*func) ();
  int i;
  int n;
  int q;
  int nn;
  int x;
  int nx;
  int mfs;
  int lsize;
  int ncols;
  int mcols;
  int nrows;
  int nfiles;
  int *ndx;
  char *pth;
  char **tarray;

#if DEBUG_VERSION
  if (debug)
    trace_flow (TRACE_IN, 0, 0, "list_names", NULL);
#endif /* DEBUG_VERSION */

  switch (sort_type)
    {
    default:
    case FULLNAME:
      func = compare_full_name;
      break;
    case BASENAME:
      func = compare_base_name;
      break;
    case EXT:
      func = compare_ext;
      break;
    case SIZE:
      func = compare_size;
      break;
    case SLACK:
      func = compare_slack;
      break;
    case NONE:
      func = NULL;
      break;
    }
  for (nfiles = 0; narray[nfiles] != NULL; nfiles++);
  mfs = lofle (narray, nfiles);
  if (func && nfiles > 1)
    qsort (narray, nfiles, sizeof (char *), func);
  if (mfs)
    {
      if (!opt.find)
	{
	  NEWLINE (rows_put);
	  printf ("%s:", file_type[pindx].header);
	  NEWLINE (rows_put);
	}
#ifdef S_ISLNK
      if (opt.dref && (pindx == SYM_IDX || pindx == SYMO_IDX))
	{
	  pth = xmalloc (PATH_MAX + 1);
	  for (i = 0, nx = 0; narray[i] != NULL; i++)
	    {
	      lsize = readlink (narray[i], pth, PATH_MAX + 1);
	      if (lsize > nx)
		nx = lsize;
	    }
	  nx += 2;
	  nn = mfs + nx + 4;
	  free (pth);
	}
      else
#endif
	nn = mfs + 2;

      /* Calculate the maximum number of columns that will fit. */
      ncols = columns / nn;
      if (ncols == 0)
	ncols = 1;
      if (opt.vertical)
	{
	  /* Calculate the number of rows that will be in each column
	     except possibly for a short column on the right. */
	  nrows = nfiles / ncols + (nfiles % ncols != 0);
	  /* Recalculate columns based on rows. */
	  ncols = nfiles / nrows + (nfiles % nrows != 0);

	  if (ncols > 1)
	    {
	      if (nfiles != nrows * ncols)
		{
		  x = ((sizeof (char **) * (nrows * ncols))
		       + sizeof (char **));
		  narray = xrealloc (narray, x);
		  for (x = nfiles; x < nrows * ncols + 1; x++)
		    {
		      narray[x] = xmalloc (2 * sizeof (char));
		      narray[x][0] = ' ';
		      narray[x][0] = '\0';
		    }
		  narray[x] = NULL;
		  nfiles = nrows * ncols;
		}
	      ndx = xmalloc (nfiles * sizeof (int));

	      for (q = 0, x = 0; x < nrows; x++)
		{
		  i = x;
		  do
		    {
		      ndx[q++] = i;
		      i += nrows;
		    }
		  while (i < nfiles);
		}
	      tarray = xmalloc ((nfiles + 1) * sizeof (char **));
	      tarray[nfiles] = NULL;
	      for (i = 0; i < nfiles; tarray[i] = narray[ndx[i]], i++);
	      free (ndx);
	      free (narray);
	      narray = tarray;
	    }
	}
      mcols = columns;
      if (nn >= mcols)
	nn = mcols - 2;
      else if (ncols > 1)
	mcols = nn * (ncols + 1);
      for (i = 0, x = 0;;)	/* Infinite for loop */
	{
	  x += nn;
	  if (x < mcols)
	    {
	      if (!opt.find)
		{
		  if (opt.long_format)
		    {
		      unsigned long blks;
		      char tmode[11];

		      SAFE_LSTAT (narray[i], &sn_statf);
		      blks = ST_NBLOCKS (sn_statf);
		      if (opt.kilo)
			blks /= 2;
		      mode_string (sn_statf.st_mode, tmode);
		      tmode[10] = '\0';
		      printf ("%-10s %-8.8s %-8.8s %6lu%9.2f%c%6.1f%% ",
			      tmode, getuser (sn_statf.st_uid),
			      getgroup (sn_statf.st_gid),
			      blks, cvt_size (sn_statf.st_size),
			      size_id (sn_statf.st_size),
			      slack (sn_statf.st_size, blks, block_size));
		      x = mcols;
		    }
		  if (isatty (TTY) && opt.color)
		    set_color (file_type[pindx].string,
			       file_type[pindx].len, COLOR_ON);
#ifdef S_ISLNK
		  if (opt.dref && (pindx == SYM_IDX || pindx == SYMO_IDX))
		    printf ("%-*s", mfs, narray[i]);
		  else
#endif
		    printf ("%-*s", mfs + 2, narray[i]);
#ifdef S_ISLNK
		  if (opt.dref && (pindx == SYM_IDX || pindx == SYMO_IDX))
		    {
		      int xindx;
		      char *pth;

		      pth = xmalloc (PATH_MAX + 1);
		      realpath (narray[i], pth);
		      if (isatty (TTY) && opt.color)
			{
			  if (SAFE_STAT (pth, &sn_statf) == -1)
			    {
			      if (errno != ENOENT)
				error (1, errno, "(stat) %s", pth);
			      else
				SAFE_LSTAT (narray[i], &sn_statf);
			    }
			  if (S_ISDIR (sn_statf.st_mode))
			    xindx = DIR_IDX;
			  else if (S_ISREG (sn_statf.st_mode))
			    {
			      if (sn_statf.st_mode & S_IXUGO)
				xindx = EXEC_IDX;
			      else
				xindx = REG_IDX;
			    }
#ifdef S_ISMPC
			  else if (S_ISMPC (sn_statf.st_mode))
			    xindx = MPC_IDX;
#endif
#ifdef S_ISFIFO
			  else if (S_ISFIFO (sn_statf.st_mode))
			    xindx = FIFO_IDX;
#endif
#ifdef S_ISSOCK
			  else if (S_ISSOCK (sn_statf.st_mode))
			    xindx = SOCK_IDX;
#endif
			  else if (S_ISCHR (sn_statf.st_mode))
			    xindx = CHR_IDX;
			  else if (S_ISBLK (sn_statf.st_mode))
			    xindx = BLK_IDX;
			  else
			    xindx = UNKN_IDX;
			}
		      lsize = readlink (narray[i], pth, PATH_MAX + 1);
		      pth[lsize] = '\0';
		      if (isatty (TTY) && opt.color)
			set_color (NULL, 0, COLOR_OFF);
		      printf (" -> ");
		      if (isatty (TTY) && opt.color)
			set_color (file_type[xindx].string,
				   file_type[xindx].len, COLOR_ON);
		      printf ("%-*s", nx, pth);
		      free (pth);
		    }
#endif /* ifdef S_ISLNK */
		}
	      else
		{
		  printf ("%s/", path);
		  if (isatty (TTY) && opt.color)
		    set_color (file_type[pindx].string,
			       file_type[pindx].len, COLOR_ON);
		  printf ("%s", narray[i]);
		  x = mcols;
		}
	    }
	  else
	    {
	      x = 0;
	      if (isatty (TTY) && opt.color)
		set_color (NULL, 0, COLOR_OFF);
	      if (opt.find)
		{
		  if (!isatty (TTY))
		    printf ("%c", '\0');
		  else
		    NEWLINE (rows_put);
		}
	      else
		NEWLINE (rows_put);
	    }
	  if (x)
	    ++i;
	  if (narray[i] == NULL)
	    {
	      if (x)
		{
		  if (isatty (TTY) && opt.color)
		    set_color (NULL, 0, COLOR_OFF);
		  if (opt.find)
		    {
		      if (!isatty (TTY))
			printf ("%c", '\0');
		      else
			NEWLINE (rows_put);
		    }
		  else
		    NEWLINE (rows_put);
		}
	      break;		/* exit for */
	    }
	}			/* end for */
      if (opt.search == TOP_DOWN && pindx == DIR_IDX)
	{
#if DEBUG_VERSION
	  trace_flow (TRACE_OUT, 0, 0, "list_names", NULL);
#endif /* DEBUG_VERSION */
	  return;
	}
      for (i = 0; narray[i] != NULL; free (narray[i]), ++i);
      free (narray);
    }
#if DEBUG_VERSION
  trace_flow (TRACE_OUT, 0, 0, "list_names", NULL);
#endif /* DEBUG_VERSION */
  return;
}

/* Set color according to action */
static void
set_color (const char *string, int length, int action)
{
  putchar ('\033');
  putchar ('[');
  switch (action)
    {
    default:
    case 0:			/* No color */
      putchar ('0');
      break;
    case 1:			/* Set color */
      for (; length > 0; --length, putchar (*(string++)));
      break;
    case 2:			/* Reset to defaults */
      break;
    }
  putchar ('m');
  return;
}

/* compare full filename */
static int
compare_full_name (const char **name1, const char **name2)
{
  return opt.reverse ? strcmp (*name2, *name1) : strcmp (*name1, *name2);
}

/* compare base filename. The base filename is everything that
 * precedes the final '.' in the filename. If neither filename
 * contains a '.' the filenames themselves are compared. 
 */
static int
compare_base_name (const char **name1, const char **name2)
{
  int result;
  const char *tname1;
  const char *tname2;
  char *tmp1;
  char *tmp2;
  char *t_ptr = NULL;

  if (opt.reverse)
    {
      tname1 = *name2;
      tname2 = *name1;
    }
  else
    {
      tname1 = *name1;
      tname2 = *name2;
    }
  tmp1 = strcpy (xmalloc (strlen (*name1) + 1), tname1);
  tmp2 = strcpy (xmalloc (strlen (*name2) + 1), tname2);

  /* If the '.' found starts a filename; do nothing.
     Otherwise truncate the filename. */
  t_ptr = strrchr (tmp1, '.');
  if (t_ptr)
    {
      if (t_ptr != &tmp1[0])
	*t_ptr = '\0';
    }
  t_ptr = strrchr (tmp2, '.');
  if (t_ptr)
    {
      if (t_ptr != &tmp2[0])
	*t_ptr = '\0';
    }
  result = strcmp (tmp1, tmp2);
  free (tmp1);
  free (tmp2);
  if (result)
    return result;
  else
    return strcmp (tname1, tname2);
}

/* Compare file extensions.  Files with no extension are `smallest'.
   If extensions are the same, compare by filenames instead. */
static int
compare_ext (const char **name1, const char **name2)
{
  int result;
  const char *tname1;
  const char *tname2;
  const char *tmp1;
  const char *tmp2;
  char *t_ptr = NULL;

  if (opt.reverse)
    {
      tname1 = *name2;
      tname2 = *name1;
    }
  else
    {
      tname1 = *name1;
      tname2 = *name2;
    }
  tmp1 = strrchr (tname1, '.');
  tmp2 = strrchr (tname2, '.');

  if (!tmp1 && !tmp2)
    return strcmp (tname1, tname2);

  /* If the dot found starts a filename, file does not have an extension */
  if (!tmp1 || !tmp2)
    {
      if (tmp1)
	{
	  t_ptr = strrchr (tname1, '.');
	  if (t_ptr == &tmp1[0])
	    tmp2 = tname2;
	}
      else
	{
	  t_ptr = strrchr (tname2, '.');
	  if (t_ptr == &tmp2[0])
	    tmp1 = tname1;
	}
    }
  if (!tmp1)
    return -1;
  if (!tmp2)
    return 1;
  result = strcmp (tmp1, tmp2);
  if (!result)
    return strcmp (tname1, tname2);
  return result;
}

/* compare file size. If sizes are the same, compare by fullnames instead. */
static int
compare_size (const char **name1, const char **name2)
{
  struct stat f1;
  struct stat f2;

  SAFE_LSTAT (*name1, &f1);
  SAFE_LSTAT (*name2, &f2);

  if (f1.st_size > f2.st_size)
    {
      if (opt.reverse)
	return -1;
      else
	return 1;
    }
  else if (f1.st_size < f2.st_size)
    {
      if (opt.reverse)
	return 1;
      else
	return -1;
    }
  return compare_full_name (name1, name2);
}

/* compare file slack. If slack is equal, compare by filesize instead. */
static int
compare_slack (const char **name1, const char **name2)
{
  struct stat f1;
  struct stat f2;
  float f1_slack;
  float f2_slack;

  SAFE_LSTAT (*name1, &f1);
  SAFE_LSTAT (*name2, &f2);
  f1_slack = slack (f1.st_size, f1.st_blocks, f1.st_blksize);
  f2_slack = slack (f2.st_size, f2.st_blocks, f2.st_blksize);

  if (f1_slack > f2_slack)
    {
      if (opt.reverse)
	return -1;
      else
	return 1;
    }
  else if (f1_slack < f2_slack)
    {
      if (opt.reverse)
	return 1;
      else
	return -1;
    }
  return compare_size (name1, name2);
}

static int
check_rows (int rp)
{
  int c;

  if (opt.page && isatty (TTY))
    {
      if (rp >= rows - 2)
	{
	  printf ("Continue? ");
	  fflush (stdout);
	  c = getchar ();
	  c = tolower (c);
	  if (c == 'y' || c == '\n')
	    {
	      putchar ('\n');
	      return 0;
	    }
	  else
	    exit (EXIT_SUCCESS);
	}
      else
	return ++rp;
    }
  return 0;
}

/* Set or unset option 'c' according to the value of swt.
 * swt = 0 unset option, swt = 1 set option.
 */
static int
set_option (int c, int swt, int fatal)
{
  register int i;
  register int n;
  register int m;
  int types_arg_error;
  char *token;
  char *arg_list;

  types_arg_error = 0;
  switch (c)
    {
    case 0:			/* For long options */
      break;
    case 'a':			/* list all files */
      opt.all = swt;
      opt.odf = !swt;
      break;
    case 'e':			/* list empty directories */
      opt.empty = swt;
      break;
    case 'f':			/* find entries, output raw data */
      opt.find = swt;
      opt.names = swt;
      opt.recurse = swt;
      if (opt.search == BEST)	/* if direction not forced set to BOTTOM_UP */
	opt.search = BOTTOM_UP;
      break;
    case 'l':			/* list names in long format */
      opt.long_format = swt;
      if (swt)
	opt.names = swt;
      break;
#ifdef S_ISLNK
    case 'L':			/* Dereference symbolic links */
      opt.dref = swt;
      if (swt)
	opt.names = swt;
      break;
#endif
    case 'k':			/* Display blocks in 1k byte blocks */
      opt.kilo = swt;
      opt.pcbs = !swt;
      break;
    case 'n':			/* List file names */
      opt.names = swt;
      break;
    case 'o':			/* only list directories that start with '.' */
      opt.odf = swt;
      if (swt)
	opt.all = !swt;
      break;
    case 'p':			/* POSIXLY_CORRECT block size */
      opt.pcbs = swt;
      opt.kilo = !swt;
      break;
    case 'r':			/* reverse sort order */
      opt.reverse = swt;
      break;
    case 'R':			/* search all sub-directories */
      opt.recurse = swt;
      if (optarg)
	{
	  for (n = 0; optarg[n] != '\0';
	       optarg[n] = tolower (optarg[n]), ++n);
	  if (!n)
	    error (fatal, 0, "Option -- recurse, argument missing\n");
	  if (!strncmp ("top-down", optarg, n))
	    opt.search = TOP_DOWN;	/* Search direction */
	  else if (!strncmp ("bottom-up", optarg, n))
	    opt.search = BOTTOM_UP;
	  else
	    error (fatal, 0,
		   "Option -- recurse, invalid argument - %s\n", optarg);
	}
      break;
    case 10:			/* Sort option */
      if (swt)
	{
	  sort_type = FULLNAME;	/* sort fullnames (default) */
	  opt.names = swt;	/* Turn on list names option */
	}
      else
	{
	  sort_type = NONE;	/* Do not sort */
	  break;
	}
      if (optarg)
	{
	  for (n = 0; optarg[n] != '\0';
	       optarg[n] = tolower (optarg[n]), ++n);
	  if (!strncmp ("fullname", optarg, n))
	    sort_type = FULLNAME;	/* sort fullnames */
	  else if (!strncmp ("basename", optarg, n))
	    sort_type = BASENAME;	/* sort basenames */
	  else if (!strncmp ("extension", optarg, n))
	    sort_type = EXT;	/* sort extensions */
	  else if (!strncmp ("size", optarg, n))
	    sort_type = SIZE;	/* sort by size */
	  else if (!strncmp ("slack", optarg, n))
	    sort_type = SLACK;	/* sort by slack */
	  else if (!strncmp ("none", optarg, n))
	    sort_type = NONE;	/* Do not sort */
	  else
	    error (fatal, 0,
		   "Option -- sort, invalid argument - %s\n", optarg);
	}
      break;
    case 11:			/* Debug option */
      if (swt)
	opt.Debug = swt;
      else
	{
	  opt.Debug = !swt;
	  break;
	}
      if (optarg)
	{
	  for (n = 0; optarg[n] != '\0';
	       optarg[n] = tolower (optarg[n]), ++n);
	  if (!n)
	    error (fatal, 0, "Option -- debug, argument missing\n");
	  if (!strncmp ("all", optarg, n))
	    {
	      opt.names = swt;	/* Turn on list names option */
	      opt.tr_flow = TR_ALL;	/* trace all */
	    }
	  else if (!strncmp ("recursive", optarg, n))
	    opt.tr_flow = TR_RECUR;	/* trace recursive */
	  else if (!strncmp ("none", optarg, n))
	    opt.tr_flow = TR_NONE;	/* no trace */
	  else
	    error (fatal, 0,
		   "Option -- debug, invalid argument - %s\n", optarg);
	}
      break;
    case 't':			/* File types */
      /* Turn defaults on or off.
       * Note: Since this option is used to exclude certain filetypes
       * from the search, a value of '0' in swt turns on the defaults.
       * This in-effect unsets this option.
       */

      opt.bdev = opt.cdev = opt.dir = opt.exec
	= opt.reg = opt.hlink = opt.slink = opt.olink
	= opt.mux = opt.pipe = opt.sock = opt.unknown = !swt;

      if (!swt)
	break;			/* can't have arguments if turning defaults on. */

      arg_list = xmalloc (strlen (optarg) + 2);
      strcpy (arg_list, ",");
      strcat (arg_list, optarg);
      for (n = 0; arg_list[n] != '\0';
	   arg_list[n] = tolower (arg_list[n]), ++n);
      for (token = strrchr (arg_list, ','); token;
	   token = strrchr (arg_list, ','))
	{
	  *token++ = '\0';
	  m = strlen (token);
	  for (i = 0, n = 0; i < TOT_TYPES; i++)
	    {
	      if (strncmp (file_type[i].arg, token, m))
		continue;
	      n = 1;
	      switch (i)
		{
		case BLK_IDX:	/* Include block devices */
		  opt.bdev = swt;
		  break;
		case CHR_IDX:	/* Include character devices */
		  opt.cdev = swt;
		  break;
		case DIR_IDX:	/* Include sub-directories (if any) */
		  opt.dir = swt;
		  break;
		case EXEC_IDX:	/* Include  executable file(s) */
		  opt.exec = swt;
		  break;
		case REG_IDX:	/* Include  non-executable file(s) */
		  opt.reg = swt;
		  break;
		case HRD_IDX:	/* Include hard link(s) */
		  opt.hlink = swt;
		  break;
#ifdef S_ISLNK
		case SYM_IDX:	/* Include all symbolic link(s) */
		  opt.slink = swt;
		  opt.olink = swt;
		  break;
		case SYMO_IDX:	/* only orphaned symbolic link(s) */
		  opt.olink = swt;
		  break;
#endif
#ifdef S_ISMPC
		case MPC_IDX:	/* Include multiplexors */
		  opt.mux = swt;
		  break;
#endif
#ifdef S_ISFIFO
		case FIFO_IDX:	/* Include FIFO(s) (named pipes) */
		  opt.pipe = swt;
		  break;
#endif
#ifdef S_ISSOCK
		case SOCK_IDX:	/* Include socket(s) */
		  opt.sock = swt;
		  break;
#endif
		case UNKN_IDX:	/* Include unknown files */
		  opt.unknown = swt;
		  break;
		}
	      break;
	    }
	  /* If fatal = 0, we must be processing a types option in an
	   * environment variable. Don't print error message. */
	  if (n <= 0)
	    {
	      if (fatal)
		error (fatal, 0,
		       "Option -- types, invalid argument - %s\n", token);
	      types_arg_error = 1;
	    }
	}
      free (arg_list);
      if (n <= 0 && !types_arg_error)
	error (fatal, 0, "Option -- types, requires an argument\n");
      break;
    case 'v':			/* list filenames vertically */
      opt.vertical = swt;
      opt.names = swt;
      break;
    case '?':
      exit (EXIT_FAILURE);
    default:
      if (fatal)
	{
	  error (0, 0, "Invalid option -- %c\n", c);
	  help (EXIT_FAILURE);
	}
    }
  if (types_arg_error)
    return 0;
  return 1;
}

/* print help screen */
static void
help (int status)
{
  int i;
  int m;
  int n;
  int indent = 27;

  printf ("usage: %s [OPTIONS] [PATH] [FILE]...\n\n", prog_name);

  puts (" -a,  --all              include names starting with '.' in search");
  puts (" -e,  --empty-dir        list empty directories during recursive search");
  puts (" -f,  --find             do a simple recursive search");
  puts (" -k,  --kilobytes        display blocks in 1k byte blocks");
  puts (" -l,  --long-format      list names in long format");
#ifdef S_ISLNK
  puts (" -L,  --dereference      dereference symbolic links");
#endif
  puts (" -n,  --names            list names of files found");
  puts (" -o,  --only-dot-files   only search for files that start with '.'");
  puts (" -p,  --posix-block-size display blocks in 512 byte blocks");
  puts (" -r,  --reverse          reverse order while sorting");
  puts (" -R,  --recurse[=OPTION] recursive search [=top-down or bottom-up]");
  puts (" -v,  --vertical         display filenames vertically");
  puts ("      --color            display in color if a tty");
#if DEBUG_VERSION
  puts ("      --debug[=OPTION]   [=none, all or recursive]");
#endif
  puts ("      --help             display this help and exit");
  puts ("      --page             display results a page at a time");
  puts ("      --sort[=OPTION]    [=fullname, basename, extension, size, slack or none]");
  puts ("      --types=TYPES      search for: (all if option not used)");

  printf ("%*s", indent, " ");
  for (i = 0, m = 0, n = indent; i < TOT_TYPES; i++)
    {
      m = strlen (file_type[i].arg);
      if ((n + m + 3) < (columns - 4))
	{
	  printf ("%s, ", file_type[i].arg);
	  n += m;
	}
      else
	{
	  putchar ('\n');
	  printf ("%*s", indent, " ");
	  printf ("%s, ", file_type[i].arg);
	  n = indent;
	  n += m;
	}
    }
  puts ("\b\b.");

  puts ("      --version          display version information and exit");
  exit (status);
}
