/*****************************************************************************/
/*									     */
/*				   HEXWIN.CC				     */
/*									     */
/* (C) 1995-96	Ullrich von Bassewitz					     */
/*		Wacholderweg 14						     */
/*		D-70597 Stuttgart					     */
/* EMail:	uz@ibb.schwaben.com					     */
/*									     */
/*****************************************************************************/



// $Id$
//
// $Log$
//
//



#include <stdio.h>

#include "chartype.h"
#include "filecoll.h"
#include "strparse.h"
#include "stdmsg.h"
#include "stdmenue.h"
#include "menuedit.h"
#include "filepath.h"
#include "progutil.h"

#include "hexobjid.h"
#include "hexmsg.h"
#include "hexopt.h"
#include "hexwin.h"



// Register the HexedWin class
LINK (HexedWin, ID_HexedWin);



/*****************************************************************************/
/*			      Message constants				     */
/*****************************************************************************/



const u16 msModeHex		= MSGBASE_HEXWIN +  0;
const u16 msModeASCII		= MSGBASE_HEXWIN +  1;
const u16 msModeShow		= MSGBASE_HEXWIN +  2;
const u16 msModeReadonly	= MSGBASE_HEXWIN +  3;
const u16 msModeKey		= MSGBASE_HEXWIN +  4;
const u16 msStatError		= MSGBASE_HEXWIN +  5;
const u16 msRegular		= MSGBASE_HEXWIN +  6;
const u16 msOpenError		= MSGBASE_HEXWIN +  7;
const u16 msSaveChanges		= MSGBASE_HEXWIN +  8;
const u16 msStringNotFound	= MSGBASE_HEXWIN +  9;
const u16 msInvalidEscSeq	= MSGBASE_HEXWIN + 10;
const u16 msSearch		= MSGBASE_HEXWIN + 11;
const u16 msGoto		= MSGBASE_HEXWIN + 12;
const u16 msFilePos		= MSGBASE_HEXWIN + 13;



/*****************************************************************************/
/*				     Data				     */
/*****************************************************************************/



// The default mode if the file is not read only
EditMode	 DefaultEditMode	= emHex;



/*****************************************************************************/
/*				class HexedWin				     */
/*****************************************************************************/



void HexedWin::Init ()
// Initialize the window - is called from the constructors
{
    // Lock the window
    Lock ();

    // Stat the file
    FileInfo FI (Filename);
    if (FI.Error != 0) {
	ErrorMsg (FormatStr (LoadAppMsg (msStatError).GetStr (), Filename.GetStr ()));
	return;
    }

    // The file must be a regular file
    if (!FI.IsReg ()) {
	ErrorMsg (FormatStr (LoadAppMsg (msRegular).GetStr (), Filename.GetStr ()));
	return;
    }

    // Remember the file size
    Filesize = FI.Size;

    // Try to open the file for reading and writing
    F = fopen (Filename.GetStr (), "r+b");
    if (F == NULL) {
	// Ok, maybe we have no write access. Try opening the file
	// for reading only
	F = fopen (Filename.GetStr (), "rb");
	if (F == NULL) {
	    // No chance - print an error message and exit
	    OpenError (Filename);
	    return;
	}
	// Remember that we are in read only mode
	Mode = emReadonly;
    }

    // If the mode is not Readonly (which cannot be changed), and if not
    // prohibited by the user, create a shadow copy of the file
    if (Mode != emReadonly && HexOpt->CreateShadow ()) {

	// Get a temporary name
	ShadowFile = TempName ();

	// Open the target file
	FILE* Target = fopen (ShadowFile.GetStr (), "w+b");
	if (Target == NULL) {
	    OpenError (ShadowFile);
	    return;
	}

	// Copy the file
	CopyFile (F, Target);

	// Close the original and use the target instead
	fclose (F);
	F = Target;

	// Remember that the file is shadowed
	HasShadowFile = 1;

    }

    // Set the cursor position to the current value to update all needed data
    SetCursorOffset (CursorOffset);

    // Set the filename as the header
    SetHeader (' ' + Filename + ' ');

    // Set the footer according to the mode
    SetModeFooter ();

    // Draw the window interior
    DrawInterior ();

    // Clear the error flag
    ErrorFlag = 0;

    // Unlock the window
    Unlock ();
}



HexedWin::HexedWin (const Rect& Bounds, const String& Name):
    ItemWindow (Bounds, wfFramed | wfCanMove | wfCanResize, paBlue, 0),
    Filename (Name),
    StartOffset (0),
    CursorOffset (0),
    LowNibble (0),
    Mode (DefaultEditMode),
    SmallSize (OBounds),
    F (NULL),
    ErrorFlag (1),
    PageBuf (NULL),
    PageBufFill (0),
    PageBufDirty (0),
    FileIsModified (0),
    HasShadowFile (0)
// Construct a HexedWin instance. Name is the name of the file to edit
{
    Init ();
}



HexedWin::HexedWin (StreamableInit):
    ItemWindow (Empty),
    PageBuf (NULL),
    PageBufFill (0),
    PageBufDirty (0),
    FileIsModified (0),
    HasShadowFile (0)
// Create an empty hexed window
{
}



HexedWin::~HexedWin ()
// Delete a HexedWin instance, close the file etc.
{
    // Write out the buffer and close the file
    if (F != NULL) {

	// Write the buffer to disk if dirty
	WritePageBuf ();

	// Close the file
	fclose (F);

	// If we have a shadow file, remove it
	if (HasShadowFile) {
	    remove (ShadowFile.GetStr ());
	}

    }

    // Delete the buffer
    delete [] PageBuf;
}



void HexedWin::Store (Stream& S) const
// Store the hexed window into a stream
{
    // Store derived stuff
    ItemWindow::Store (S);

    // Store special HexedWin stuff
    S << Filename << StartOffset << CursorOffset << LowNibble << (u16) Mode
      << SmallSize << RelFilePos;
}



void HexedWin::Load (Stream& S)
// Load the window from a stream
{
    // Load derived stuff
    ItemWindow::Load (S);

    // Load special HexedWin stuff
    u16 TmpMode;
    S >> Filename >> StartOffset >> CursorOffset >> LowNibble >> TmpMode
      >> SmallSize >> RelFilePos;
    Mode = (EditMode) TmpMode;

    // Initialize the window
    Init ();
}



u16 HexedWin::StreamableID () const
// Return the stream ID of the window
{
    return ID_HexedWin;
}



Streamable* HexedWin::Build ()
// Build an empty hexed window
{
    return new HexedWin (Empty);
}



void HexedWin::OpenError (const String& Name)
// Print an error message "Could not open XXXX"
{
    ErrorMsg (FormatStr (LoadAppMsg (msOpenError).GetStr (), Name.GetStr ()));
}



void HexedWin::CopyFile (FILE* Source, FILE* Target)
// Copy a file from Source to Target
{
    // Allocate buffer memory
    const unsigned BufSize = 8192;
    char* Buffer = new char [BufSize];

    // Tell the user to be patient...
    Window* Win = PleaseWaitWindow ();

    // Seek to file start
    rewind (Source);
    rewind (Target);

    // Copy the original to the target
    u32 Size = Filesize;
    while (Size) {
	unsigned Bytes = Size > BufSize ? BufSize : Size;
	fread (Buffer, Bytes, 1, Source);
	fwrite (Buffer, Bytes, 1, Target);
	Size -= Bytes;
    }

    // Delete the window
    delete Win;

    // Delete the buffer memory
    delete [] Buffer;
}



void HexedWin::SetPageBufDirty ()
// Set the page buf dirty flag. The function checks if this is the first
// time that the file has been changed and updates the footer (where a
// "file is changed" mark is display) if so.
{
    if (FileIsModified == 0 && PageBufDirty == 0) {
	// This is the first time, we have any changes
	PageBufDirty = 1;
	DrawFooter ();
    } else {
	PageBufDirty = 1;
    }
}



void HexedWin::SetCursorOffset (u32 NewOffset)
// Set a new cursor position in the file recalculating all data that
// depends on this position
{
    // Set the new position
    CursorOffset = NewOffset;

    // Calculate the relative position of the upper left corner from the start
    // in percent. Use floating point to avoid overflows and use a rounding
    // strategy or otherwise we will never get more than 99% since
    // Max (CursorOffset) == Filesize-1.
    // Beware: There may be zero length files!
    RelFilePos = 0;
    if (Filesize != 0) {

	double Offset = CursorOffset;
	double Size   = Filesize;
	double Pos    = (Offset / Size) * 100.0 + 0.5;

	// Adjust for rounding errors
	if (Pos < 0.0) {
	    Pos = 0.0;
	} else if (Pos > 100.0) {
	    Pos = 100.0;
	}

	// Remember the new value
	RelFilePos = (int) Pos;

    }

    // Redraw the footer to show the new file position
    DrawFooter ();
}



void HexedWin::ReadPageBuf ()
// Read the page buffer
{
    u32 BytesLeft = Filesize - StartOffset;
    PageBufFill = BytesLeft > PageBufSize ? PageBufSize : BytesLeft;

    if (PageBufFill) {
	fseek (F, StartOffset, SEEK_SET);
	fread (PageBuf, PageBufFill, 1, F);
    }
}



void HexedWin::WritePageBuf ()
// Write the page buffer
{
    if (PageBufDirty) {
	fseek (F, StartOffset, SEEK_SET);
	fwrite (PageBuf, PageBufFill, 1, F);
	PageBufDirty = 0;
	FileIsModified = 1;
    }
}



String HexedWin::GetPosFooterText ()
// Return the text that is displayed in the lower left corner of the
// frame.
{
    // Return the formatted string, add a '*' in front if the file has changes
    if (PageBufDirty || FileIsModified) {
	return FormatStr (" *%d%% ", RelFilePos);
    } else {
	return FormatStr (" %d%% ", RelFilePos);
    }
}



void HexedWin::SetModeFooter ()
// Set the footer string according to the mode
{
    // Get the correct message number
    unsigned MsgNum;
    switch (Mode) {

	case emHex:
	    MsgNum = msModeHex;
	    break;

	case emASCII:
	    MsgNum = msModeASCII;
	    break;

	case emShow:
	    MsgNum = msModeShow;
	    break;

	case emReadonly:
	    MsgNum = msModeReadonly;
	    break;

	default:
	    FAIL ("HexedWin::SetModeFooter: Unknown Mode value");
	    MsgNum = 1000000;	// Never executed - make gcc happy
	    break;

    }

    // Set the footer
    SetFooter (LoadAppMsg (MsgNum));
}



void HexedWin::DrawFooter ()
// Draw the footer of the window frame. The function exits immidiately if
// the window is not framed.
// The footer is positioned at the lower right of the frame, there is
// exactly one horizontal frame element to the right of the footer text.
// This function overrides Window::Footer and places a position mark
// on the left side of the frame
{
    // Lock window output
    Lock ();

    // For simplicity use the derived function, then redraw the part
    // that changes. There will be no screen output anyway.
    ItemWindow::DrawFooter ();

    // Assume that the window is big enough, don't care about overwriting
    // something. If the user does not want garbage, she has to resize the
    // window.

    // Get the needed frame attribute
    u16 AttrChar = GetFrameAttr ();

    // Write out the text
    int X = 2;
    int Y = OBounds.YSize () - 1;
    String S = GetPosFooterText ();
    int Len = S.Len ();
    for (int I = 0; I < Len; I++) {
	WriteChar (X++, Y, Pal->BuildAttr (AttrChar, S [I]));
    }

    // Unlock the window, allow screen updates
    Unlock ();
}



void HexedWin::DrawLine (unsigned Y)
// Draw one line
{
    String Line (IXSize ());

    // Lock the window
    Lock ();

    unsigned Ofs = Y * BytesPerRow;

    // Address
    Line = FormatStr (" %06X   ", StartOffset + Ofs);

    // Hex bytes
    for (unsigned I = 0; I < BytesPerRow; I++) {
	if (Ofs + I < PageBufFill) {
	    Line += FormatStr ("%02X ", PageBuf [Ofs + I]);
	} else {
	    Line += "-- ";
	}
    }

    // some space
    Line += "  ";

    // ASCII chars
    for (I = 0; I < BytesPerRow; I++) {
	// Check for end of file
	if (Ofs + I >= PageBufFill) {
	    break;
	}
	unsigned char C = PageBuf [Ofs + I];
	if (C < 32) {
	    Line += '.';
	} else if (C < 127 || HexOpt->ShowExtAscii ()) {
	    Line += C;
	} else {
	    Line += '.';
	}
    }

    // Pad the line to fill the complete window
    Line.Pad (String::Right, IXSize ());

    // Print the row
    Write (0, Y, Line);

    // Unlock the window
    Unlock ();
}



void HexedWin::InitData ()
// Recalculate needed data, resize buffers
{
    // Create bytes per row from the window size
    BytesPerRow = (IXSize () + 1 - 2 - 6 - 3 - 3) / 4;
			//		       ^ space between address and hex
			//		    ^ space between hex and ascii
			//	       ^ address
			//	   ^ Left/Right one space
			//     ^ magic

    // Create a new buffer page
    PageBufSize = BytesPerRow * IYSize ();
    PageBuf = new unsigned char [PageBufSize];

    // If the cursor is now outside the window, correct StartOffset and/or
    // CursorOffset
    if (CursorOffset >= StartOffset + PageBufSize) {
	StartOffset = CursorOffset - PageBufSize + 1;
    }
}



void HexedWin::DrawInterior ()
// Draw the window interior, recalculating all data first
{
    // Write out the buffer to disk if dirty, then free the buffer
    WritePageBuf ();
    delete [] PageBuf;

    // Recalculate the needed data, resize buffers
    InitData ();

    // Read in a new buffer page
    ReadPageBuf ();

    // Now draw the page
    DrawPage ();
}



void HexedWin::DrawPage ()
// Draw the complete page
{
    // Lock the window
    Lock ();

    // Write out each row
    for (int Y = 0; Y <= MaxY (); Y++) {
	DrawLine (Y);
    }

    // Draw the cursor
    ShowCursor ();

    // Draw the footer to update the file position
    DrawFooter ();

    // Unlock the window
    Unlock ();
}



void HexedWin::DrawCursor (unsigned Attr)
// Draw the cursor in the given attribute
{
    unsigned PageOffset = CursorOffset - StartOffset;
    unsigned CursX = PageOffset % BytesPerRow;
    unsigned CursY = PageOffset / BytesPerRow;

    // Change the attribute of the hex value
    unsigned X;
    X = 1 + 6 + 3 + 3 * CursX;
    unsigned Count = 2;
    if (Mode == emHex) {
	// Nibble mode
	Count = 1;
	if (LowNibble) {
	    X++;
	}
    }
    ChangeAttr (X, CursY, Count, Attr);

    // Change the attribute of the character representation
    X = 1 + 6 + 3 + 3 * BytesPerRow + 2 + CursX;
    ChangeAttr (X, CursY, 1, Attr);
}



void HexedWin::ShowCursor ()
// Draw a visible cursor
{
    DrawCursor (atTextInvers);
}



void HexedWin::HideCursor ()
// Make the cursor invisible
{
    DrawCursor (atTextNormal);
}



void HexedWin::CursorUp (unsigned Bytes)
{
    // Do the cursor movement
    if (CursorOffset >= Bytes) {

	// Hide the cursor
	HideCursor ();

	// Move the cursor
	SetCursorOffset (CursorOffset - Bytes);

	// Move the page pointer if necessary
	if (CursorOffset < StartOffset) {

	    // Write the buffer to disk if dirty
	    WritePageBuf ();

	    // Correct the offset by one row
	    StartOffset -= BytesPerRow;

	    // Read the new buffer
	    ReadPageBuf ();

	    // Lock the window
	    Lock ();

	    // Scroll the window down
	    ScrollDown ();

	    // Redraw the first line
	    DrawLine (0);

	    // Unlock the window
	    Unlock ();

	}

	// Show the cursor
	ShowCursor ();

	// Redraw the footer to show the new file position
	DrawFooter ();
    }
}



void HexedWin::CursorDown (unsigned Bytes)
{
    // Do the cursor movement
    if (CursorOffset + Bytes < Filesize) {

	// Hide the cursor
	HideCursor ();

	// Move the cursor
	SetCursorOffset (CursorOffset + Bytes);

	// Move the page pointer if necessary
	if (CursorOffset >= StartOffset + BytesPerRow * IYSize ()) {

	    // Write the buffer to disk if dirty
	    WritePageBuf ();

	    // Correct the offset by one row
	    StartOffset += BytesPerRow;

	    // Read the new buffer
	    ReadPageBuf ();

	    // Lock the window
	    Lock ();

	    // Scroll the window up
	    ScrollUp ();

	    // Redraw the first line
	    DrawLine (MaxY ());

	    // Unlock the window
	    Unlock ();

	}

	// Show the cursor
	ShowCursor ();

	// Redraw the footer to show the new file position
	DrawFooter ();
    }
}



void HexedWin::PgUp ()
{
    if (StartOffset > 0) {
	// Write the current buffer to disk
	WritePageBuf ();

	// Do the page up
	unsigned SkipCount = BytesPerRow * (IYSize () - 1);
	if (StartOffset > SkipCount) {
	    SetCursorOffset (CursorOffset - SkipCount);
	    StartOffset  -= SkipCount;
	} else {
	    SetCursorOffset (CursorOffset - StartOffset);
	    StartOffset = 0;
	}

	// Read the new buffer contents
	ReadPageBuf ();

	// Display the page
	DrawPage ();

    }
}



void HexedWin::PgDn ()
{
    // Calculate bytes to skip
    unsigned SkipCount = BytesPerRow * (IYSize () - 1);

    //
    if (StartOffset + SkipCount < Filesize) {
	// Write the current buffer to disk
	WritePageBuf ();

	// Do the page down
	SetCursorOffset (CursorOffset + SkipCount);
	StartOffset  += SkipCount;

	// Read the new buffer contents
	ReadPageBuf ();

	// Display the page
	DrawPage ();

    }
}



void HexedWin::Home ()
{
    WritePageBuf ();
    StartOffset = 0;
    SetCursorOffset (0);
    ReadPageBuf ();
    DrawPage ();
}



void HexedWin::End ()
{
    // Calculate new start position
    unsigned PageSize = IYSize () * BytesPerRow;
    if (Filesize < PageSize) {
	// We are already at the end
	return;
    }
    unsigned NewPos = Filesize - PageSize;

    if (StartOffset < NewPos) {

	// Write the current buffer to disk
	WritePageBuf ();

	// Move start offset and cursor offset
	SetCursorOffset (Filesize - 1);
	StartOffset = NewPos;

	// Read the new buffer contents
	ReadPageBuf ();

	// Display the page
	DrawPage ();

    }
}



void HexedWin::Up ()
{
    // Do the cursor movement
    CursorUp (BytesPerRow);
}



void HexedWin::Down ()
{
    // Do the cursor movement
    CursorDown (BytesPerRow);
}



void HexedWin::Left ()
{
    if (Mode == emHex) {
	// Nibble mode is on
	if (CursorOffset > 0 || LowNibble != 0) {
	    HideCursor ();
	    if (LowNibble) {
		LowNibble = 0;
		ShowCursor ();
	    } else {
		LowNibble = 1;
		CursorUp (1);
	    }
	}
    } else {
	// No nibble mode
	CursorUp (1);
    }
}



void HexedWin::Right ()
{
    if (Mode == emHex) {
	// Nibble mode is on
	if (CursorOffset < Filesize - 1 || LowNibble == 0) {
	    HideCursor ();
	    if (!LowNibble) {
		LowNibble = 1;
		ShowCursor ();
	    } else {
		LowNibble = 0;
		CursorDown (1);
	    }
	}
    } else {
	// No nibble mode
	CursorDown (1);
    }
}



void HexedWin::ToggleMode ()
// Toggle the edit mode
{
    HideCursor ();
    switch (Mode) {

	case emHex:
	    Mode = emASCII;
	    SetFooter (LoadAppMsg (msModeASCII));
	    break;

	case emASCII:
	    Mode = emShow;
	    SetFooter (LoadAppMsg (msModeShow));
	    break;

	case emShow:
	    Mode = emHex;
	    SetFooter (LoadAppMsg (msModeHex));
	    break;

	case emReadonly:
	    // Make gcc happy
	    break;

    }
    ShowCursor ();
}



static unsigned Num (unsigned C)
{
    if (C >= '0' && C <= '9') {
	return C - '0';
    } else if (C >= 'A' && C <= 'F') {
	return C - 'A' + 10;
    } else {
	return C - 'a' + 10;
    }
}



void HexedWin::InsertChar (Key K)
// Insert a key
{
    // Calculate the buffer offset and check if the offset is valid
    unsigned Offset = CursorOffset - StartOffset;
    unsigned char* P = &PageBuf [Offset];
    if (Offset >= PageBufFill) {
	return;
    }

    // Insert the character
    if (Mode == emHex && IsXDigit (K)) {

	// Hex mode
	if (LowNibble) {
	    *P = (*P & 0xF0) + Num (K);
	} else {
	    *P = (*P & 0x0F) + (Num (K) << 4);
	}
	SetPageBufDirty ();
	DrawLine (Offset / BytesPerRow);
	Right ();

    } else if (Mode == emASCII) {

	// ASCII Mode
	*P = (unsigned char) K;
	SetPageBufDirty ();
	DrawLine (Offset / BytesPerRow);
	Right ();

    }
}



unsigned HexedWin::MinXSize () const
// Return the minimum window size
{
    return 1 + 1 + 6 + 3 + 3 * 4 + 3 + 1;
}



unsigned HexedWin::MinYSize () const
// Return the minimum window size
{
    return 4;
}



void HexedWin::Zoom ()
// Zoom the window in or out
{
    // Calculate the maximum size
    Rect BigSize = Background->OuterBounds ();
    BigSize.Grow (0, -1);

    // If the window is already big, zoom out, otherwise zoom in
    if (OBounds == BigSize) {
	// Zoom out
	Resize (SmallSize);
    } else {
	// Remember current size, zoom in
	SmallSize = OBounds;
	Resize (BigSize);
    }
}



void HexedWin::Save ()
// Save the file if it is modified
{
    // Write back the buffer if it's dirty
    WritePageBuf ();

    // Saving only needed if we have changes and if we have a shadow file
    if (FileIsModified && HasShadowFile) {

	// Copy the file back
	FILE* Target = fopen (Filename.GetStr (), "wb");
	if (Target == NULL) {
	    OpenError (Filename);
	    return;
	}
	CopyFile (F, Target);
	fclose (Target);

	// File is no longer modified
	FileIsModified = 0;

	// Re-read the page buffer
	ReadPageBuf ();

    }
}



u32 HexedWin::FileSearch (u32 Start, const char* S, unsigned Len)
// Search for the string S beginning from the offset Start up to the end of
// the file. The function returns the position a copy of S is found, or -1
// if no copy has been found. The string S may contain null bytes.
// The function uses a very slow aproach for searching - but since the
// limiting factor are the disk reads, I did not bother to make it faster.
{
    // The given string cannot be empty
    PRECONDITION (Len > 0);

    // Write any changes from the page buffer to the file
    WritePageBuf ();

    // Seek to the start offset
    fseek (F, Start, SEEK_SET);

    // Allocate a search buffer
    const unsigned BufSize = 8192;
    unsigned char* Buf = new unsigned char [BufSize];

    // Calculate the remaining bytes until file end and the bytes to search
    u32 BytesTilEnd = Filesize - Start;
    u32 BytesToSearch = HexOpt->SearchWraps () ? Filesize+Len-1 : BytesTilEnd;

    // Searching...
    while (BytesToSearch > 0) {

	// Set up for an empty buffer
	unsigned BufFill = 0;
	unsigned BufOffs = 0;

	// Inner search loop
	while (BytesTilEnd >= Len && BytesToSearch > 0) {

	    // Calculate amount of bytes that are not searched til now
	    unsigned Remaining = BufFill - BufOffs;

	    // Refill the buffer. Copy the remaining bytes to the
	    // buffer start, then reload the rest of the buffer
	    memmove (Buf, &Buf [BufOffs], Remaining);
	    unsigned BytesToRead = BufSize - Remaining;
	    if (BytesToRead > BytesTilEnd) {
		BytesToRead = BytesTilEnd;
	    }
	    if (BytesToRead > BytesToSearch) {
		BytesToRead = BytesToSearch;
	    }
	    fread (&Buf [Remaining], BytesToRead, 1, F);

	    // Adjust the counters and offsets
	    Start	  += BytesToRead;	    // Current file pos
	    BytesTilEnd   -= BytesToRead;
	    BytesToSearch -= BytesToRead;
	    BufFill = Remaining + BytesToRead;
	    BufOffs = 0;
	    Remaining = BufFill;

	    // Search
	    while (Remaining >= Len) {

		if (memcmp (&Buf [BufOffs], S, Len) == 0) {
		    // Found the string, store the offset in Start, then
		    // break out. Remember: Start currently contains the
		    // offset of the end of the buffer
		    Start -= BufFill - BufOffs;
		    goto ExitPoint;
		}

		// Next offset
		BufOffs++;
		Remaining--;
	    }

	}

	// Check if the search is over or if we have a wrap. Beware: There may
	// be a chunk in the buffer that's unused because it's length is less
	// than the length of the search string!
	if (BytesToSearch > BytesTilEnd) {

	    // Stuff left, must be a wrap.
	    BytesToSearch -= BytesTilEnd;
	    BytesTilEnd = Filesize;
	    Start = 0;
	    rewind (F);
	}

    }

    // If we get here, we did not find the search string
    Start = (u32) -1;

ExitPoint:
    // Free the allocated buffers
    delete [] Buf;

    // Return the offset found
    return Start;
}



static char HexVal (char Leading, char Trailing)
// Convert the characters Leading and Trailing to a hex value
{
    if (Leading >= 'A') {
	Leading -= 'A' - 10;
    } else {
	Leading -= '0';
    }
    if (Trailing >= 'A') {
	Trailing -= 'A' - 10;
    } else {
	Trailing -= '0';
    }

    return Leading * 16 + Trailing;
}



static unsigned CreateBinStr (const String& S, char* T, unsigned Size)
// Create a binary string from S by expanding control sequences.
{
    // Get the length of S
    unsigned StrLen = S.Len ();

    //
    unsigned BinLen = 0;
    unsigned I = 0;
    while (I < StrLen && BinLen < Size) {

	char C = S [I++];

	if (C == '\\') {
	    // OOPS - escape char
	    C = S [I++];
	    char C1;
	    switch (C) {

		case '\\':
		    T [BinLen++] = '\\';
		    break;

		case 't':
		    T [BinLen++] = 0x09;
		    break;

		case 'n':
		    T [BinLen++] = 0x0d;
		    break;

		case 'r':
		    T [BinLen++] = 0x0a;
		    break;

		case 'x':
		    // Hex digit
		    C = NLSUpCase (S [I++]);
		    if (IsXDigit (C)) {
			C1 = NLSUpCase (S [I++]);
			if (IsXDigit (C1)) {
			    T [BinLen++] = HexVal (C, C1);
			} else {
			    ErrorMsg (LoadAppMsg (msInvalidEscSeq));
			    return 0;
			}
		    } else {
			ErrorMsg (LoadAppMsg (msInvalidEscSeq));
			return 0;
		    }
		    break;

		default:
		    ErrorMsg (LoadAppMsg (msInvalidEscSeq));
		    return 0;

	    }

	} else {
	    // no escape char, take it literally
	    T [BinLen++] = C;
	}
    }

    // Return the string length
    return BinLen;
}



void HexedWin::Goto (u32 NewOffs)
// Go to a new cursor offset. The function tries to center the new cursor
// position in the window.
{
    // Write out the buffer in case it's dirty
    WritePageBuf ();

    // Try to position the search string in the center of the window
    unsigned HalfWindow = (BytesPerRow * IYSize ()) / 2;
    if (NewOffs >= HalfWindow) {
	StartOffset = NewOffs - HalfWindow;
    } else {
	StartOffset = 0;
    }
    SetCursorOffset (NewOffs);

    // Read data from the new location
    ReadPageBuf ();

    // Redraw the page
    DrawPage ();
}



void HexedWin::Search ()
// Request a search string from the user and search for this string. Position
// the cursor to the string found.
{
    static String LastSearchString;

    // Load the menue
    Menue* M = (Menue*) LoadResource ("@HEXWIN.SearchStringWindow");

    // Get a pointer to the search string item
    TextEdit* TE = (TextEdit*) M->ForcedItemWithID (1);

    // Set the old value
    TE->SetValue (LastSearchString);

    // Get the new name
    int Abort;
    TE->Edit (Abort);
    if (Abort) {
	delete M;
	return;
    }

    // Retrieve the new name
    LastSearchString = M->GetStringValue (1);

    // Delete the filename window
    delete M;

    // Convert the search string
    char BinStr [256];
    unsigned BinLen = CreateBinStr (LastSearchString, BinStr, sizeof (BinStr));
    if (BinLen == 0) {
	// error
	return;
    }

    // Search for the string
    u32 NewOffs = FileSearch (CursorOffset + 1, BinStr, BinLen);

    // If the string has not been found, tell the user, otherwise change the
    // file position to the string found
    if (NewOffs == (u32) -1) {

	InformationMsg (LoadAppMsg (msStringNotFound));

    } else {

	Goto (NewOffs);

    }
}



void HexedWin::Goto ()
// Set a new position in the file
{
    // Load the menue
    Menue* M = (Menue*) LoadResource ("@HEXWIN.GotoInputWindow");

    // Get a pointer to the search string item
    TextEdit* TE = (TextEdit*) M->ForcedItemWithID (1);

    // Set the old value
    TE->SetValue (FormatStr ("0x%X", CursorOffset));

    // Get the new name
    int Abort;
    TE->Edit (Abort);
    if (Abort) {
	delete M;
	return;
    }

    // Retrieve the new name
    String S = M->GetStringValue (1);

    // Delete the input window
    delete M;

    // Convert the string to a number
    unsigned Base = 10;
    if (S.Len () > 2 && S [0] == '0' && S [1] == 'x') {
	Base = 16;
	S.Del (0, 2);
    }
    StringParser SP (S);
    u32 NewPos;
    unsigned Res = SP.GetU32 (NewPos, Base);

    // Check for errors
    if (Res != 0) {
	// Invalid input string
	ErrorMsg (SP.GetMsg (Res));
	return;
    }

    // Check for the max file length
    if (NewPos >= Filesize) {
	ErrorMsg (LoadAppMsg (msFilePos));
	return;
    }

    // Go to the new position
    Goto (NewPos);

}



Key HexedWin::Browse ()
// Browse the file
{
    // Remember the old window state and activate the window
    unsigned OldState = GetState ();
    Activate ();

    // Create the new status line
    String Line = CreateStatusLine (siClose | siZoom);
    if (Mode != emReadonly) {
	Line += GetKeyName3 (kbMetaM);
	Line += LoadAppMsg (msModeKey);
    }
    Line += GetKeyName3 (kbMetaS);
    Line += LoadAppMsg (msSearch);
    Line += GetKeyName3 (kbMetaP);
    Line += LoadAppMsg (msGoto);
    PushStatusLine (Line);

    // User loop
    Key K;
    int Done = 0;
    do {

	// Get a key...
	K = KbdGet ();

	// ... and handle that key
	switch (K) {

	    case vkHome:
		Home ();
		break;

	    case vkEnd:
		End ();
		break;

	    case vkUp:
		Up ();
		break;

	    case vkDown:
		Down ();
		break;

	    case vkLeft:
		Left ();
		break;

	    case vkRight:
		Right ();
		break;

	    case vkPgUp:
		PgUp ();
		break;

	    case vkPgDn:
		PgDn ();
		break;

	    case kbMetaM:
		ToggleMode ();
		break;

	    case vkResize:
		MoveResize ();
		break;

	    case vkZoom:
		Zoom ();
		break;

	    case kbMetaS:
		Search ();
		break;

	    case kbMetaP:
		Goto ();
		break;

	    case vkAbort:
		Done = 1;
		break;

	    default:
		if (K >= 0x20 && K < 127) {
		    InsertChar (K);
		} else if (KeyIsRegistered (K)) {
		    Done = 1;
		}
		break;

	}
    } while (!Done);

    // Restore the old status line
    PopStatusLine ();

    // Restore the old window state
    SetState (OldState);

    // Return the abort key
    return K;
}



int HexedWin::CanClose ()
// Return true if closing the window is allowed
{
    // Write out the current buffer, in case it's dirty
    WritePageBuf ();

    // If the file has changes and there is a shadow file, ask if the changes
    // should be saved, otherwise closing is allowed
    if (FileIsModified && HasShadowFile) {

	// Split up the filename, don't show the path when asking
	String Dir, Name, Ext;
	FSplit (Filename, Dir, Name, Ext);
	Name += Ext;

	// Ask if the changes should be saved
	String AskStr = FormatStr (LoadAppMsg (msSaveChanges).GetStr (), Name.GetStr ());
	switch (AskYesNo (AskStr)) {

	    case arNo:
		// Discard the changes
		FileIsModified = 0;
		return 1;

	    case arYes:
		// Save the file
		Save ();

		// If the file still has changes, something went wrong
		return FileIsModified == 0;

	    default:
		// Abort, do not allow closing
		return 0;

	}

    } else {

	// No changes, window may be closed
	return 1;

    }
}



void HexedWin::ScreenSizeChanged (const Rect&)
// Called when the screen got another resolution. NewScreen is the new
// screen size.
{
    // Check if the hexed window bounds are outside the screen area now.
    // If so, and if the minimum window size permits, resize the window
    // to be inside the desktop area.
    Rect Desktop = Background->GetDesktop ();
    Rect NewBounds = OBounds;

    if (NewBounds.B.X > Desktop.B.X) {
	NewBounds.B.X = Desktop.B.X;
    }
    if (NewBounds.B.Y > Desktop.B.Y) {
	NewBounds.B.Y = Desktop.B.Y;
    }

    if (NewBounds.XSize () >= MinXSize () && NewBounds.YSize () >= MinYSize ()) {
	Resize (NewBounds);
    }
}
