/************************************************************************/
/* This program synchronizes two directories so that each directory con-*/
/* tains the most current version of duplicate files, and that each dir-*/
/* ectory contains all files that the other directory contains.  The    */
/* files that each directory contains are compared to the other direct- */
/* tory.  If a version of the same file is contained in each directory, */
/* the date & time of the files are compared.  The most current version */
/* is then copied to the other directory.  If the 'recursive' switch is */
/* turned on, then the same process is performed for any subdirectories */
/* as well.  The program is capable of being compiled and run in a DOS/ */
/* Windows environment as well; the proper '#define' must be set, UNIX  */
/* or DOS.                                                              */
/* Written by F. Michael Orr, August, 1998.                             */
/************************************************************************/
/* #define DEBUG */             // UNCOMMENT FOR DEBUG TRACING
#define ERRMSGMAX 81            // MAXIMUM LENGTH OF ERROR MESSAGE

#ifdef UNIX
#define PRTNAMEWIDTH 25         // WIDTH OF NAME IN PRINT LINE
#else
#ifdef DOS
#ifndef NAME_MAX
#define NAME_MAX 128
#endif
#define PRTNAMEWIDTH 15         // WIDTH OF NAME IN PRINT LINE
#endif
#endif

#ifdef UNIX
#ifdef MAX_INPUT
#define CMDLINE MAX_INPUT
#else
#define CMDLINE 1024
#endif
#else
#define CMDLINE 128
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <dirent.h>
#include <ctype.h>
#include <sys/stat.h>
#ifdef UNIX
#include <unistd.h>
#else
#ifdef DOS
#include <dir.h>
#endif
#endif

enum enttype {                  // ENUM OF ENTRY TYPE
    FILEENT,
    DIRENT
};

struct esTag {                  // DIRECTORY ENTRY STRUCTURE
    char *esName;               // ENTRY NAME
    enttype esType;             // ENTRY TYPE
    time_t esTime;              // DATE & TIME OF ENTRY
    unsigned char esMatch;      // FLAG TO INDICATE MATCH ALREADY FOUND
    struct esTag *esNext;       // NEXT ENTRY IN LINKED LIST
};

typedef struct esTag EntryName;

/*
  VARIOUS FLAG VALUES AND THE BYTE TO STORE THEM IN.
*/
#define VERBOSE 0x01
#define RECURSIVE 0x02
#define CHECK 0x04
#define EURODATE 0x08
unsigned char optflag = VERBOSE;    // OPTIONS FLAGS

/*
  FUNCTION PROTOTYPES.
*/
int cmdLineArgs(int, char **, char *, char *);
void displayHelp(char *);
int process(char *, char *);
void deleteDirHash(EntryName **);
EntryName **buildDir(char *, int *);
EntryName *newLLentry(char *, char *, struct stat *);
EntryName *findMatch(char *, EntryName **);
void printLine(EntryName *, EntryName *);
int copyfiles(EntryName *, EntryName *, char *, char *);
int recurseSubDirs(EntryName **, EntryName **,  char *, char *);

int main(int argc, char **argv) {
    char dir1[NAME_MAX],        // 1ST DIRECTORY
        dir2[NAME_MAX];         // 2ND DIRECTORY
    int rc;
    /**/
    if (0 != (rc = cmdLineArgs(argc, argv, dir1, dir2)))
        if (1 == rc)            // HELP WAS REQUESTED; OK
            return(0);
        else
            return(-1);
    /*
      IF VERBOSE MODE ON & ONLY CHECKING, THEN PRINT INFORMATION MSG HERE.
      MUST BE HERE SO THAT IT DOESN'T DISPLAY FOR EACH (POSSIBLE) SUBDIRECTORY
      CALL.
    */
    if ((VERBOSE | CHECK) == (optflag & (VERBOSE | CHECK)))
        printf("Checking only; no copying being performed.\n\n");
    return (process(dir1, dir2));
}                               // END MAIN()


/************************************************************************/
/* This function is the main driver for the processing of the director- */
/* ies.  It accepts the directories as parameters so that it may be     */
/* called recursively.  It returns a return code.                       */
/************************************************************************/
int process(char *indir1, char *indir2) {
    char dirnames[NAME_MAX],
        errmsg[ERRMSGMAX],
        *cwd;                   // HOLDS RETURN FROM GETCWD()
    EntryName **dir1list,
        **dir2list,             // POINTERS TO EACH DIRECTORIES LISTING
        *currentry,             // ENTRY IN THE 1ST DIRECTORY
        *matchentry;            // MATCHING ENTRY IN THE 2ND DIRECTORY
    unsigned char secflag = 1;  // FLAG IF 2ND DIR.'S ENTRIES BEEN PRINTED
    int i,
        dir1cnt,                // # OF ENTRIES IN 1ST DIR.
        dir2cnt,                // # OF ENTRIES IN 2ND DIR.
        first2sec = 0,          // # FILES COPIED FROM 1ST TO 2ND
        sec2first = 0,          // # FILES COPIED FROM 2ND TO 1ST
        ident = 0,              // # OF IDENTICAL FILES
        subdirs1 = 0,           // # OF SUBDIRECTORIES OF 1ST DIR.
        subdirs2 = 0,           // # OF SUBDIRECTORIES OF 2ND DIR.
        numents = 0;            // TOTAL # OF ENTRIES BETWEEN THE TWO
#ifdef DOS
    int drivenum;               // CURRENT DRIVE #
#endif
#ifdef DEBUG
    fprintf(stderr, "Entering process().\n\tindir1=%s\n\tindir2=%s\n", indir1,
            indir2);
#endif
	/*
	  GET CURRENT DIRECTORY, AND IN DOS, THE CURRENT DRIVE, TO SWITCH BACK
	  TO AFTER BUILDING DIRECTORY LINKED LISTS.
	*/
#ifdef UNIX
    if (NULL == (cwd = getcwd(NULL, 0)))
#else
#ifdef DOS
    drivenum = getdisk();
    if (NULL == (cwd = getcwd(NULL, NAME_MAX)))
#endif
#endif
    {
        perror("Could not getcwd()");
        return(-1);
    }                           // END IF ((CWD = GETCWD( ...
    /*
      READ EACH DIRECTORY.
    */
    if (NULL == (dir1list = buildDir(indir1, &dir1cnt))) {
        free(cwd);
        return(-1);
    }                           // END IF ((DIR1LIST = BUILDDIR( ...
	/*
	  CHANGE BACK TO SAME DIRECTORY BEFORE TRYING TO BUILD LINKED LIST OF
	  CONTENTS OF SECOND DIRECTORY.  THIS WILL ALLOW FOR RELATIVE DIRECTORY
	  SPECIFICATION (. AND ..).
	*/
#ifdef DOS
    setdisk(drivenum);
#endif
    if (chdir(cwd) < 0) {
        sprintf(errmsg, "Couldn't change to dir %s", cwd);
        perror(errmsg);
        deleteDirHash(dir1list);
        free(cwd);
        return(-1);
    }                           // END IF (CHDIR(CWD) < 0)
    if (NULL == (dir2list = buildDir(indir2, &dir2cnt))) {
        deleteDirHash(dir1list);
        free(cwd);
        return(-1);
    }                           // END IF ((DIR2LIST = BUILDDIR(...
#ifdef DOS
    setdisk(drivenum);
#endif
    if (chdir(cwd) < 0) {
        sprintf(errmsg, "Couldn't change to dir %s", cwd);
        perror(errmsg);
        deleteDirHash(dir1list);
        deleteDirHash(dir2list);
        free(cwd);
        return(-1);
    }                           // END IF (CHDIR(CWD) < 0)
    /*
      PRINT DIRECTORY NAMES THAT ARE BEING USED IF VERBOSE FLAG IS SET.
    */
    if (optflag & VERBOSE) {
        printf("Directory #1: %s\n", indir1);
        printf("Directory #2: %s\n", indir2);
        if (!dir1cnt && !dir2cnt) {
            printf("\nDirectories empty.\n\n");
            return(0);
        }                       // END IF (!DIR1CNT && !DIR2CNT)
        memset(dirnames, ' ', PRTNAMEWIDTH);
        dirnames[PRTNAMEWIDTH] = 0; // MAKE INTO A STRING
        printf("%s      Directory #1              Directory #2\n", dirnames);
    }                           // IF (OPTFLAG & VERBOSE)
    /*
      PROCESS EACH CHARACTER OF THE ALPHABET.
    */
    for (i = 0; i < 256; i++) {
        if (NULL == (currentry = dir1list[i]))  // IGNORE ENTRIES WITH NOTHING
            continue;                           // IN THEM
        /*
          PROCESS ALL THE FILES ON THE LINKED LIST FOR EACH PARTICULAR
          CHARACTER OF THE ALPHABET.
        */
        while (NULL != currentry) {
            numents++;          // COUNT ALL ELEMENTS OF 1ST DIR
            matchentry = findMatch(currentry->esName, dir2list);
            /*
              PRINT DIRECTORY ENTRIES IF VERBOSE FLAG IS ON.  MAKE SURE THAT
			  IF A MATCH IS FOUND THAT IT IS ALSO A DIRECTORY AND NOT A FILE.
            */
            if (DIRENT == currentry->esType) {
                if (NULL != matchentry) {	// SOME TYPE OF MATCHING DIR FOUND
                    if (DIRENT != matchentry->esType) {
                        fprintf(stderr, "File/directory name conflict for %s"
                                "\n", currentry->esName);
                        deleteDirHash(dir1list);
                        deleteDirHash(dir2list);
                        free(cwd);
                        return(-1);
                    }           // END IF (DIRENT != MATCHENTRY->ESTYPE)
                    ident++;    // PART OF IDENTICAL ENTRIES
                    matchentry->esMatch = 1;  // MARK AS HAVING A MATCH
                } else			// NO MATCHING DIRECTORY FOUND
                    subdirs1++;
                if (optflag & VERBOSE)
                    printLine(currentry, matchentry);
                currentry = currentry->esNext;
                continue;
            }                   // END IF (DIRENT == CURRENTRY->ESTYPE)
            /*
               NOT A DIRECTORY, SO CHECK FILE DATES & TIMES.
            */
            if (NULL == matchentry) {   // FILE IS IN 1ST DIR BUT NOT 2ND
                if (!(optflag & CHECK))
                    if (copyfiles(currentry, NULL, indir1, indir2)) {
                        deleteDirHash(dir1list);
                        deleteDirHash(dir2list);
                        free(cwd);
                        return(-1);
                    }           // END IF (COPYFILES(DIR1LIST, NULL, ...
                if (optflag & VERBOSE)
                    printLine(currentry, NULL);
                first2sec++;
            } else {            // MAKE SURE NAME MATCH IS NOT A DIRECTORY
                if (FILEENT != matchentry->esType) {
                    fprintf(stderr, "File/directory name conflict for %s\n",
                            currentry->esName);
                    deleteDirHash(dir1list);
                    deleteDirHash(dir2list);
                    free(cwd);
                    return(-1);
                }               // END IF (FILEENT != MATCHENTRY->ESTYPE)
				/*
				  MATCH FOUND, AND IS A FILE.  MARK 2ND ENTRY AS HAVING
				  BEEN MATCHED AND FIGURE OUT WHICH DIRECTION TO COPY &
				  WHICH COUNTER TO INCREMENT.
				*/
                matchentry->esMatch = 1;  // MARK 2ND AS HAVING A MATCH IN 1ST
                if (currentry->esTime != matchentry->esTime) {
                    // ONE FILE VERSION IS NEWER THAN ANOTHER
                    if (!(optflag & CHECK))
                        if (copyfiles(currentry, matchentry, indir1, indir2)) {
                            deleteDirHash(dir1list);
                            deleteDirHash(dir2list);
                            free(cwd);
                            return(-1);
                        }       // END IF (COPYFILES(DIR1LIST, ...
                    if (optflag & VERBOSE)
                        printLine(currentry, matchentry);
                    /*
                      NOW THAT COPYING IS DONE, FIGURE OUT WHICH COUNTER
                      TO INCREMENT.
                    */
                    if (currentry->esTime < matchentry->esTime)
                        sec2first++;
                    else
                        first2sec++;
                } else {
                    // IDENTICAL
                    if (optflag & VERBOSE)
                        printLine(currentry, matchentry);
                    ident++;
                }               // END IF (CURRENTRY->ESTIME != ...
            }                   // END IF (NULL == MATCHENTRY ... ELSE
            currentry = currentry->esNext;
        }                       // END WHILE (NULL != CURRENTRY
    }                           // END FOR (I = 0; I < 256; I++)
    /*
      NOW THAT THE ENTIRE 1ST DIRECTORY IS PROCESSED, GO THROUGH THE SECOND
      DIRECTORY, LOOKING FOR ANY ENTRIES THAT HAVEN'T BEEN MATCHED IN THE
      FIRST.
    */
    for (i = 0; i < 256; i++) {
        if (NULL == (currentry = dir2list[i]))  // IGNORE LETTERS WITH NOTHING
            continue;                           // IN THEM
        /*
          PROCESS ALL THE FILES ON THE LINKED LIST FOR EACH PARTICULAR
          CHARACTER OF THE ALPHABET.
        */
        while (NULL != currentry) {
            if (currentry->esMatch) {
                currentry = currentry->esNext; // ALREADY BEEN MATCHED; IGNORE
                continue;
            }                   // END IF (CURRENTRY->ESMATCH)
            /*
              FILE EXISTS IN SECOND DIRECTORY BUT NOT IN FIRST.
            */
            numents++;          // COUNT ALL UNMATCHED ELEMENTS OF 2ND DIR
			if (optflag & VERBOSE)
				if (secflag) { 	// 1ST ENTRY OF 2ND DIR TO BE PRINTED
					printf("\nNext directory's entries follow:\n");
					secflag = 0;
				}              	// END IF (SECFLAG)
            if (DIRENT == currentry->esType) {
                if (optflag & VERBOSE) {
                    subdirs2++; // UNMATCHED SUBDIRECTORY IN 2ND DIR
                    printLine(NULL, currentry);
                }               // END IF (OPTFLAG & VERBOSE)
                currentry = currentry->esNext;
                continue;
            }                   // END IF (DIRENT == CURRENTRY->ESTYPE)
            if (!(optflag & CHECK))
                if (copyfiles(NULL, currentry, indir1, indir2)) {
                    deleteDirHash(dir1list);
                    deleteDirHash(dir2list);
                    free(cwd);
                    return(-1);
                }               // END IF (COPYFILES(NULL, DIR1LIST)
            if (optflag & VERBOSE)
                printLine(NULL, currentry);
            sec2first++;
            currentry = currentry->esNext;
        }                       // END WHILE (NULL != CURRENTRY)
    }                           // END FOR (I = 0; I < 256; I++)
    if (optflag & VERBOSE) {    // PRINT TOTALS IF NEEDED
        printf("\n%d total entr%s\n", numents, 1 == numents ?
               "y" : "ies");
        printf("%d identical entr%s\n", ident, 1 == ident ?
               "y" : "ies");
        printf("%d fil%s %scopied from %s to %s\n", first2sec, 1 ==
               first2sec ? "e" : "es", (optflag & CHECK) ? "would be " : "",
               indir1, indir2);
        printf("%d fil%s %scopied from %s to %s\n", sec2first, 1 ==
               sec2first ? "e" : "es", (optflag & CHECK) ? "would be " : "",
               indir2, indir1);
        printf("%d unmatched subdirector%s in %s\n", subdirs1,
               1 == subdirs1 ? "y" : "ies", indir1);
        printf("%d unmatched subdirector%s in %s\n\n", subdirs2,
               1 == subdirs2 ? "y" : "ies", indir2);
    }                           // END IF (OPTFLAG & VERBOSE)
    if (optflag & RECURSIVE) {
        if (recurseSubDirs(dir1list, dir2list, indir1, indir2)) {
            deleteDirHash(dir1list);
            deleteDirHash(dir2list);
            free(cwd);
            return(-1);
        }                       // END IF (RECURSESUBDIRS(DIR1LIST, ...
    }                           // END IF (OPTFLAG & RECURSIVE)
    /*
      CLEAN UP.
    */
    deleteDirHash(dir1list);
    deleteDirHash(dir2list);
    free(cwd);
#ifdef DEBUG
    fprintf(stderr, "Leaving process()\n");
#endif
    return(0);
}                               // END PROCESS()


/************************************************************************/
/* The following function processes the command line parameters.  It    */
/* receives argc & argv from main(), and pointers to the strings to con-*/
/* tain the names of the two directories to start processing with.  It  */
/* will return a ZRC if all OK, -1 if an invalid command line was re-   */
/* ceived.  This means an invalid option, or the directories were not   */
/* specified.  As a special case, if help was requested, it will return */
/* 1 so that main() knows to exit, but to exit OK.  NOTE: since this    */
/* program must be able to be compiled on a DOS/Windows system, getopt()*/
/* cannot be used.                                                      */
/************************************************************************/
int cmdLineArgs(int argc, char **argv, char *dir1, char *dir2) {
#ifdef UNIX
    char delim='-';				// COMMAND SWITCH DELIMITER
#else
#ifdef DOS
    char delim='/';
#endif
#endif
    int i,
        j,
        len;
    char holdopt;               // HOLDS OPTION PRIOR TO CASE CONVERSION IN
                                // DOS, NOTHING IN UNIX
#ifdef DEBUG
    fprintf(stderr, "Entering cmdLineArgs().\n");
#endif
	/*
	  IF NO COMMAND LINE PARAMETERS WERE RECEIVED, DISPLAY HELP & EXIT.
	*/
    if (argc < 2) {
        displayHelp(argv[0]);
        return(1);
    }                           // END IF (ARGC < 2)
	/*
	  IF ONLY ONE COMMAND LINE PARAMETER RECEIVED, MAKE SURE HELP WAS
	  REQUESTED.
	*/
    if (2 == argc) {
#ifdef DOS
        len = strlen(argv[1]);
        for (i = 0; i < len; i++)
            argv[1][i] = tolower(argv[1][i]);
        if (!strcmp(argv[1], "/h") || !strcmp(argv[1], "/?"))
#else
#ifdef UNIX
        if (!strcmp(argv[1], "-h") || !strcmp(argv[1], "--help"))
#endif
#endif
            {
                displayHelp(argv[0]);
                return(1);
            }                   // END IF (!STRCMP(ARGV[1], ...
    }                           // END IF (2 == ARGC)
	/*
	  MAKE SURE ENOUGH COMMAND LINE PARAMETERS WERE SUPPLIED.
	*/
    if (argc < 3) {
        fprintf(stderr, "Insufficient number of parameters\n");
        displayHelp(argv[0]);
        return(-1);
    }                       // END IF (ARGC < 3)
	/*
	  COPY LAST TWO PARAMETERS TO DIRECTORY NAMES.
	*/
#ifdef DEBUG
    fprintf(stderr, "\tDir #1=%s\n\tDir #2=%s\n", argv[argc - 2],
            argv[argc - 1]);
#endif
    strcpy(dir1, argv[argc - 2]);
    strcpy(dir2, argv[argc - 1]);
	/*
	  PROCESS EACH COMMAND LINE OPTION IN TURN (IF THEY EXIST).
	*/
    for (i = 1; i < (argc - 2); i++) {
#ifdef DEBUG
        fprintf(stderr, "\tOption #%d=%s.\n", i, argv[i]);
#endif
        if (argv[i][0] != delim) {  // MAKE SURE IS AN OPTION
            fprintf(stderr, "Invalid parameter %s\n", argv[i]);
            return(-1);
        }                       // END IF (ARGV[I][0] != DELIM
        if ((len = strlen(argv[i])) < 2) {  // MAKE SURE NOT DELIM ONLY
            fprintf(stderr, "Invalid option; \'%c\' character only\n", delim);
            return(-1);
        }                       // END IF ((LEN = STRLEN(ARGV[I])) < 2)
        for (j = 1; j < len; j++) {
            holdopt = argv[i][j];
#ifdef DOS
            argv[i][j] = tolower(argv[i][j]);
#endif
            switch (argv[i][j]) {
              case 'v':
                  optflag |= VERBOSE;
                  break;
              case 'c':
                  optflag |= CHECK;
                  break;
              case 'r':
                  optflag |= RECURSIVE;
                  break;
              case 'q':
                  optflag &= ~VERBOSE;
                  break;
              case 'e':
                  optflag |= EURODATE;
                  break;
              default:
                  fprintf(stderr, "Invalid option %c%c\n", delim, holdopt);
                  return(-1);
            }                   // END SWITCH(ARGV[I][J]
        }                       // END FOR (J = 1; J < LEN; J++)
    }                           // END FOR (I = 1; I < ARGC; I++)
#ifdef DEBUG
    fprintf(stderr, "Leaving cmdLineArgs()\n");
#endif
    return(0);
}                               // END CMDLINEARGS()


/************************************************************************/
/* The following function builds the linked list of the files contained */
/* in the directory it is passed.  It returns a pointer to the name hash*/
/* for the files, and uses the int pointer it is passed to store the    */
/* number of entries in the directory.  If errors occur, it returns a   */
/* NULL pointer.                                                        */
/************************************************************************/
EntryName **buildDir(char *dirname, int *entrycnt) {
    EntryName **dirlist,		// PTR TO NEW LIST BEING BUILT
        *newentry,				// PTR TO NEW ENTRY BEING BUILT
        *nextentry,				// L.L. TRAVERSAL PTR
        *preventry;				// L.L. TRAVERSAL PTR
    int i;
#ifdef DOS
    int len;					// NEEDED TO XLAT CASE OF FILENAME
#endif
    char errmsg[ERRMSGMAX];
    DIR *dir;
    struct dirent *entry;
    struct stat fileinfo;       // INFO ON FILE TO BE PROCESSED
    /**/
#ifdef DEBUG
    fprintf(stderr, "Entering buildDir(), dirname=%s\n", dirname);
#endif
    /*
      ALLOCATE & INIT THE MEMORY TO HOLD THE DIRECTORY ENTRIES.
    */
    *entrycnt = 0;
    if (NULL == (dirlist = (EntryName **)malloc(256 * sizeof(EntryName *)))) {
        sprintf(errmsg, "Directory struct mem. failure for %s",
                dirname);
        perror(errmsg);
        return(NULL);
    }                           // END IF (NULL == (DIRLIST = (ENTRYNAME **)...
    for (i = 0; i < 256; i++)
        dirlist[i] = NULL;		// INITIALIZE PTR ARRAY
#ifdef DOS
	/*
	  IF NAME CONTAINS A SEPERATE DRIVE LETTER, THEN SWITCH TO IT BEFORE
	  TRYING TO OPEN THE DIRECTORY.
	*/
    if (isalpha(dirname[0]) && (':' == dirname[1]))	// 1ST TWO CHARS ARE 'X:'
        setdisk(toupper(dirname[0]) - 'A');
#endif
    if (NULL == (dir = opendir(dirname))) { // TRY TO OPEN DIRECTORY
        if ((CHECK | RECURSIVE) == (optflag & (CHECK | RECURSIVE)))
            /*
			  OPEN FAILURE IS ALLOWABLE IF RECURSIVELY TRAVERSING A DIRECTORY
			  TREE IN CHECK ONLY MODE.
			*/
            return(dirlist);
        sprintf(errmsg, "Cannot open directory %s", dirname);
        perror(errmsg);
        free(dirlist);
        return(NULL);
    }                           // END IF (NULL == (DIR = OPENDIR(DIRNAME)
    if (chdir(dirname) < 0) {	// CHANGE TO DIR BEFORE READING IT
        sprintf(errmsg, "Cannot chdir to dir %s", dirname);
        perror(errmsg);
        free(dirlist);
        return(NULL);
    }                           // END IF (CHDIR(DIRNAME) < 0)
    /*
      NOW THAT THE DIRECTORY IS OPEN, READ EACH ENTRY AND ADD TO LINKED LIST.
    */
    while ((entry = readdir(dir)) != NULL) {
        if (!strcmp(entry->d_name, ".") || !strcmp(entry->d_name, ".."))
            // IGNORE DIRECTORY POINTER ENTRIES
            continue;
        /*
          GET INFO ON THE FILE.
        */
#ifdef DEBUG
        fprintf(stderr, "\tEntry=%s\n", entry->d_name);
#endif
#ifdef UNIX
        if (lstat(entry->d_name, &fileinfo) < 0)
#else
#ifdef DOS
        if (stat(entry->d_name, &fileinfo) < 0)
#endif
#endif
        {
            sprintf(errmsg, "Cannot get info on file %s", entry->d_name);
            perror(errmsg);
            deleteDirHash(dirlist);
            return(NULL);
        }                       // END IF (STAT(ENTRY->D_NAME, ...
#ifdef UNIX
        /*
          IGNORE ANYTHING BUT REGULAR FILES & DIRECTORIES.
        */
        if (!S_ISREG(fileinfo.st_mode) && !S_ISDIR(fileinfo.st_mode))
            continue;
#endif
#ifdef DOS
        /*
          CONVERT NAME TO LOWERCASE PRIOR TO ADDING TO LINKED LIST.
        */
        len = strlen(entry->d_name);
        for (i = 0; i < len; i++)
            entry->d_name[i] = tolower(entry->d_name[i]);
#endif
        /*
          SEARCH FOR POSITION TO ADD FILE TO LINKED LIST
        */
        nextentry = preventry = dirlist[entry->d_name[0]];
        while (nextentry != NULL)
            if (strcmp(entry->d_name, nextentry->esName) < 0) {
                // INSERT NEW ENTRY IN FRONT OF NEXTENTRY ON L.L.
                if ((newentry = newLLentry(entry->d_name, dirname,
                                           &fileinfo)) == NULL) {
                    deleteDirHash(dirlist);
                    return(NULL);
                }               // END IF ((NEWENTRY = NEWLLENTRY(...
                newentry->esNext = nextentry;
                if (nextentry != dirlist[entry->d_name[0]])
                    preventry->esNext = newentry;
                else            // NEW 1ST ELEMENT ON L.L.
                    dirlist[entry->d_name[0]] = newentry;
                break;
            } else {
                preventry = nextentry;
                nextentry = nextentry->esNext;
            }                   // END IF ((STRCMP(ENTRY->D_NAME, ... ELSE
        /*
          SEE IF LEFT DUE TO END OF L.L. (ADD AT END OF LINKED LIST.
        */
        if (NULL == nextentry) {
            if ((newentry = newLLentry(entry->d_name, dirname, &fileinfo)) ==
                NULL) {
                deleteDirHash(dirlist);
                return(NULL);
            }                   // END IF ((NEWENTRY = NEWLLENTRY(...
            newentry->esNext = NULL;
            if (NULL != preventry)  // NOT 1ST ELEMENT OF LINKED LIST
                preventry->esNext = newentry;
            else                // 1ST ELEMENT ON THE LINKED LIST.
                dirlist[entry->d_name[0]] = newentry;
        }                       // END IF (NULL == NEXTENTRY)
        (*entrycnt)++;          // INCREMENT ENTRY COUNT
    }                           // END WHILE ((ENTRY = READDIR(DIR)) != NULL)
    if (closedir(dir) < 0) {
        sprintf(errmsg, "Error closing directory %s", dirname);
        perror(errmsg);
    }                           // END IF (CLOSEDIR(DIR) < 0)
#ifdef DEBUG
    fprintf(stderr, "Directory in sorted order:\n");
    for (i = 0; i < 256; i++) {
        if (NULL == dirlist[i])
            continue;
        newentry = dirlist[i];
        while (newentry != NULL) {
            fprintf(stderr, "\t%s\n", newentry->esName);
            newentry = newentry->esNext;
        }
    }
    fprintf(stderr, "\nExiting buildDir().\n");
#endif
    return(dirlist);
}                               // END BUILDDIR()


/************************************************************************/
/* The following function allocates a new entry structure, and initial- */
/* izes its fields.  It returns a pointer to the structure on success,  */
/* and a NULL pointer on error.                                         */
/************************************************************************/
EntryName *newLLentry(char *entryname, char *dirname,
                      struct stat *fileinfo) {
    EntryName *newentry;
    char errmsg[ERRMSGMAX];
    /**/
#ifdef DEBUG
    fprintf(stderr, "Entering newLLentry(), entryname=%s\n", entryname);
#endif
    if (NULL == (newentry = (EntryName *)malloc(sizeof(EntryName)))) {
        sprintf(errmsg, "Entry struct mem. error for %s", dirname);
        perror(errmsg);
        return(NULL);
    }                           // END IF (NULL == (NEWENTRY = (ENTRYNAME *)...
    if (NULL == (newentry->esName = (char *) malloc(strlen(entryname) + 1))) {
        sprintf(errmsg, "Entry name mem. error for %s", dirname);
        perror(errmsg);
        free(newentry);
        return(NULL);
    }               			// END IF (NULL == (NEWENTRY = (CHAR *) ...
    /*
      INITIALIZE ALL FIELDS OF THE ENTRY.
    */
    strcpy(newentry->esName, entryname);
    newentry->esMatch = 0;
#ifdef UNIX
    if (S_ISDIR(fileinfo->st_mode))
        newentry->esType = DIRENT;
    else if (S_ISREG(fileinfo->st_mode))
        newentry->esType = FILEENT;
#else
#ifdef DOS
    if (fileinfo->st_mode & S_IFDIR)
        newentry->esType = DIRENT;
    else if (fileinfo->st_mode & S_IFREG)
        newentry->esType = FILEENT;
#endif
#endif
    newentry->esTime = fileinfo->st_mtime;
    newentry->esNext = NULL;
#ifdef DEBUG
    fprintf(stderr, "Exiting newLLentry().\n");
#endif
    return(newentry);
}                               // END NEWLLENTRY()


/************************************************************************/
/* This function copies a file from one directory to another.  It uses  */
/* the builtin copy function of the operating system in order to avoid  */
/* "re-inventing the wheel."  It uses the system() function to call the */
/* "cp" command in UNIX, or the "copy" command in DOS.                  */
/************************************************************************/
int copyfiles(EntryName *file1, EntryName *file2, char *dirname1,
			  char *dirname2) {
    char commandline[CMDLINE];	// COMMAND LINE TO PASS TO SYSTEM()
    unsigned char direction = 0;
    int len,
        rc;
#ifdef UNIX
    char dirdelim='/';
#else
#ifdef DOS
    char dirdelim='\\';
#endif
#endif
#ifdef DEBUG
    fprintf(stderr, "Entering copyFiles()\n\tdir #1=%s\n\tdir #2=%s\n",
            dirname1, dirname2);
#endif
    /**/
    memset(commandline, 0, CMDLINE);
#ifdef UNIX
    strcpy(commandline, "cp -pf \"");	// PUT NAME IN QUOTES FOR SPEC. CHARS
#else
#ifdef DOS
    strcpy(commandline, "copy \"");
#endif
#endif
    /*
      FIGURE OUT WHICH DIRECTION TO COPY FILES IN.
    */
    if (NULL == file1)
        direction = 2;
    else if (NULL == file2)
        direction = 1;
    else if (file1->esTime < file2->esTime)
        direction = 2;
    else if (file1->esTime > file2->esTime)
        direction = 1;
    else if (file1->esTime == file2->esTime)
        direction = 0;
    switch (direction) {
      case 1:
          strcat(commandline, dirname1);
          len = strlen(commandline);
          if (dirdelim != commandline[len - 1])
              commandline[len] = dirdelim;  // APPEND DELIMITER IF NEEDED
          strcat(commandline, file1->esName);
          strcat(commandline, "\" ");
          strcat(commandline, dirname2);
          break;
      case 2:
          strcat(commandline, dirname2);
          len = strlen(commandline);
          if (dirdelim != commandline[len - 1])
              commandline[len] = dirdelim;  // APPEND DELIMITER IF NEEDED
          strcat(commandline, file2->esName);
          strcat(commandline, "\" ");
          strcat(commandline, dirname1);
          break;
      case 0:					// FILES ARE IDENTICAL; DO NOTHING
          return(0);
    }                           // END SWITCH (DIRECTION)
#ifdef DOS
    strcat(commandline, "> NUL:");  // DON'T DISPLAY COPY OUTPUT
#endif
#ifdef DEBUG
    fprintf(stderr, "\tcopyfiles() command line=%s\n", commandline);
#endif
    /*
      NOW THAT THE COMMAND LINE IS BUILT, RUN IT.
    */
    if ((rc = system(commandline)) != 0) {
        if (rc != -1)			// RC == -1 IS SHELL FAILURE
            perror("Unable to run copy command");
        return(-1);
    }                           // END IF ((RC = SYSTEM(...
#ifdef DEBUG
    fprintf(stderr, "Exiting copyfiles().\n");
#endif
    return(0);
}                               // END COPYFILES()


/************************************************************************/
/* This function searches a directory structure for a matching filename */
/* for the name which is passed to it.  Since the structure is built in */
/* sorted order, the search can terminate as soon an entry "greater     */
/* than" the search entry is encountered.                               */
/************************************************************************/
EntryName *findMatch(char *searchname, EntryName **dirlist) {
    EntryName *currentry;
    char tempname[NAME_MAX];
    unsigned char found = 0;        // FLAGS THAT MATCH WAS FOUND
    int srchresult;
#ifdef DOS
    int i,
    len;
#endif
    /**/
#ifdef DEBUG
    fprintf(stderr, "Entering findMatch()\n\tsearchname=%s\n", searchname);
#endif
    strcpy(tempname, searchname);
#ifdef DOS
	/*
	  ENSURE NAME IS IN LOWER CASE PRIOR TO SEARCH.
	*/
    len = strlen(tempname);
    for (i = 0; i < len; i++)
        tempname[i] = tolower(tempname[i]);
#endif
    /*
      START AT BEGINNING OF LIST FOR 1ST CHARACTER OF FILENAME.
    */
    currentry = dirlist[tempname[0]];
    while (currentry != NULL) {
        if (0 == (srchresult = strcmp(tempname, currentry->esName))){
            found = 1;
            break;
        } else if (srchresult < 0)
            break;
        currentry = currentry->esNext;
    }                           // END WHILE (CURRENTRY != NULL)
#ifdef DEBUG
    if (found)
        fprintf(stderr, "\tMatch found.\n");
    else
        fprintf(stderr, "\tMatch not found.\n");
    fprintf(stderr, "Exiting findMatch().\n");
#endif
    if (found)
        return(currentry);
    else
        return(NULL);
}                               // END FINDMATCH()


/************************************************************************/
/* This function prints a line of information about each file in each   */
/* directory.  It pulls the name of the file from either valid entry    */
/* structure, along with the date.  If European date format is desired  */
/* (DD/MM/YYYY), that is accounted for.                                 */
/************************************************************************/
void printLine(EntryName *file1, EntryName *file2) {
    EntryName *validname;       // VALID ENTRY TO WORK FROM
    char printarea[256],
        datetime[21],
        direction[4];
    int len;
    struct tm *filetime;
    /**/
#ifdef DEBUG
    fprintf(stderr, "printLine() called\n");
    if (NULL == file1)
        fprintf(stderr, "\tfile1==NULL\n");
    else
        fprintf(stderr, "\tfile1=%s\n", file1->esName);
    if (NULL == file2)
        fprintf(stderr, "\tfile2==NULL\n");
    else
        fprintf(stderr, "\tfile2=%s\n", file2->esName);
#endif
    memset(printarea, 0, 256);
#ifdef UNIX
	/*
	  ENSURE VALIDNAME POINTS TO A NAME FIELD FROM EITHER STRUCTURE SO THAT
	  A PARTIAL COPY CAN BE DONE IF NAME DOES NOT FIT IN PRINT AREA.
	*/
    if (NULL == file1)
        validname = file2;
    else
        validname = file1;
    if (DIRENT == validname->esType) {
        printarea[0] = '<';
        if (strlen(validname->esName) <= (PRTNAMEWIDTH - 2))
            strcpy(printarea + 1, validname->esName);
        else {
            strncpy(printarea + 1, validname->esName, PRTNAMEWIDTH - 5);
            strcat(printarea, "...");
        }                       // END IF (STRLEN(VALIDNAME-> ... ELSE
    } else
        if (strlen(validname->esName) <= PRTNAMEWIDTH)
            strcpy(printarea, validname->esName);
        else {
            strncpy(printarea, validname->esName, PRTNAMEWIDTH - 3);
            strcat(printarea, "...");
        }                       // END IF (STRLEN(VALIDNAME-> ... ELSE
#else
#ifdef DOS
    if (NULL == file1)
        validname = file2;
    else
        validname = file1;
    if (DIRENT == validname->esType) {
        printarea[0] = '<';
        strcpy(printarea + 1, validname->esName);
    } else
        strcpy(printarea, validname->esName);
#endif
#endif
    /*
      IF LENGTH OF STRING < MAX, FILL DIFFERENCE WITH SPACES.
    */
    len = strlen(printarea);
    if ('<' == printarea[0])
        printarea[len++] = '>';
    if (len < PRTNAMEWIDTH)
        memset(printarea + len, ' ', PRTNAMEWIDTH - len);
    /*
      GET DATE & TIME OF 1ST FILE.
    */
    if (NULL == file1)
        strcpy(datetime, "   Does not exist  ");
    else {
        filetime = localtime(&file1->esTime);
        if (optflag & EURODATE)
            strftime(datetime, 21, "%d/%m/%Y %H:%M:%S", filetime);
        else
            strftime(datetime, 21, "%m/%d/%Y %H:%M:%S", filetime);
    }                           // END IF (NULL == FILE1) ... ELSE
    strcat(printarea, "  ");
    strcat(printarea, datetime);
    /*
      PRINT DIRECTION ARROW.
    */
    if ('<' == printarea[0])    // DIRECTORY PRINT
        if (NULL == file1)
            strcpy(direction, "<==");
        else if (NULL == file2)
            strcpy(direction, "==>");
        else
            strcpy(direction, "   ");
    else if (NULL == file1)     // REGULAR FILE PRINT
        strcpy(direction, "<==");
    else if (NULL == file2)
        strcpy(direction, "==>");
    else if (file1->esTime < file2->esTime)
        strcpy(direction, "<==");
    else if (file1->esTime > file2->esTime)
        strcpy(direction, "==>");
    else if (file1->esTime == file2->esTime)
        strcpy(direction, " = ");
    strcat(printarea, "  ");
    strcat(printarea, direction);
    /*
      GET DATE & TIME OF SECOND FILE.
    */
    if (NULL == file2)
        strcpy(datetime, "   Does not exist  ");
    else {
        filetime = localtime(&file2->esTime);
        if (optflag & EURODATE)
            strftime(datetime, 21, "%d/%m/%Y %H:%M:%S", filetime);
        else
            strftime(datetime, 21, "%m/%d/%Y %H:%M:%S", filetime);
    }                           // END IF (NULL == FILE2) ... ELSE
    strcat(printarea, "  ");
    strcat(printarea, datetime);
    printf("%s\n", printarea);
#ifdef DEBUG
    fprintf(stderr, "Exiting printLine().\n");
#endif
    return;
}                               // END PRINTLINE()


/************************************************************************/
/* The following function deletes an entire directory structure.  It    */
/* traverses the entire array, deleting each linked list structure.  It */
/* then deletes the entire structure.                                   */
/************************************************************************/
void deleteDirHash(EntryName **dirlist) {
    EntryName *nextentry,
        *currentry;
    int i;
    /**/
#ifdef DEBUG
    fprintf(stderr, "Entering deleteDirHash()\n");
#endif
    for (i = 0; i < 256; i++) {
        currentry = dirlist[i];
        while (currentry != NULL) {
            free(currentry->esName);
            nextentry = currentry->esNext;
            free(currentry);
            currentry = nextentry;
        }                       // END WHILE (CURRENTRY != NULL)
    }                           // END FOR (I = 0; I < 256; I++)
    free(dirlist);
#ifdef DEBUG
    fprintf(stderr, "Exiting deleteDirHash().\n");
#endif
    return;
}                               // END DELETEDIRHASH()


/************************************************************************/
/* This function will recursively perform the same functions on any sub-*/
/* directories that may exist.                                          */
/************************************************************************/
int recurseSubDirs(EntryName **dir1list, EntryName **dir2list, char *indir1,
				   char *indir2) {
    EntryName *currentry,
        *matchentry;
    int i,
        len;
    char newfirst[NAME_MAX],
        newsec[NAME_MAX],
        errmsg[ERRMSGMAX];
#ifdef UNIX
    char dirdelim='/';
    struct stat dirinfo;        // NEEDED FOR SAME DIR PERMISSIONS ON MKDIR()
#else
#ifdef DOS
    char dirdelim='\\';
#endif
#endif
#ifdef DEBUG
    fprintf(stderr, "Entering recurseSubDirs()\n");
#endif
    /*
      PROCESS 1ST DIRECTORY STRUCTURE, LOOKING FOR SUBDIRECTORIES.
    */
    for (i = 0; i < 256; i++) {
        if (NULL == dir1list[i])
            continue;
        currentry = dir1list[i];
        while (currentry != NULL) {
            if (FILEENT == currentry->esType) { // IGNORE REGULAR FILES
                currentry = currentry->esNext;
                continue;
            }                   // END IF (FILEENT == CURRENTRY->ESTYPE)
            /*
              WILL NEED THE SUBDIRECTORY NAMES OF EACH DIRECTORY, SO MAKE
              THEM NOW.
            */
            memset(newfirst, 0, NAME_MAX);
            memset(newsec, 0, NAME_MAX);
            strcpy(newfirst, indir1);
            len = strlen(newfirst);
            if (newfirst[len - 1] != dirdelim)
                newfirst[len] = dirdelim;   // APPEND DELIMITER IF NEEDED
            strcat(newfirst, currentry->esName);
            strcpy(newsec, indir2);
            len = strlen(newsec);
            if (newsec[len - 1] != dirdelim)
                newsec[len] = dirdelim; // APPEND DELIMITER IF NEEDED
            strcat(newsec, currentry->esName);
            
            if (NULL == (matchentry = findMatch(currentry->esName,
                                                dir2list))) {
                // NO MATCH FOUND; MUST CREATE DIRECTORY
                if (!(optflag & CHECK)) {
#ifdef UNIX
                    /*
                      GET INFO ON EXISTING DIRECTORY TO MATCH PERMISSIONS.
                    */
                    if (stat(newfirst, &dirinfo) < 0) {
                        sprintf(errmsg, "Couldn't get info on dir %s",
                                newfirst);
                        perror(errmsg);
                        return(-1);
                    }           // END IF (STAT(NEWFIRST, &DIRINFO) < 0)
                    if (mkdir(newsec, S_IRWXU | (dirinfo.st_mode & 07777))
                        < 0)    // MUST HAVE USER RWX NO MATTER WHAT
#else
#ifdef DOS
                    if (mkdir(newsec) < 0)
#endif
#endif
                    {
                        sprintf(errmsg, "Could not create dir %s", newsec);
                        perror(errmsg);
                        return(-1);
                    }               // END IF (MKDIR(...
                    /*
                      PRINT THE NAME OF THE DIRECTORY CREATED IF VERBOSE
                      IS ON.
                    */
                    if (optflag & VERBOSE)
                        printf("%s created\n", newsec);
                }               // END IF (!(OPTFLAG & CHECK))
            } else              // MATCH FOUND
                matchentry->esMatch = 1;    // MARK ALREADY PROCESSED
            /*
              RECURSIVELY CALL PROCESS(), PASSING THE NEW DIRECTORY NAMES.
            */
            if (process(newfirst, newsec))
                return(-1);
            currentry = currentry->esNext;
        }                       // END WHILE (CURRENTRY != NULL)
    }                           // END FOR (I = 0; I < 256; I++)
    /*
      NOW GO THROUGH SECOND STRUCTURE, LOOKING FOR ENTRIES NOT MATCHED IN
      THE FIRST.
    */
    for (i = 0; i < 256; i++) {
        if (NULL == dir2list[i])
            continue;
        currentry = dir2list[i];
        while (currentry != NULL) {
            if (FILEENT == currentry->esType) { // IGNORE REGULAR FILES
                currentry = currentry->esNext;
                continue;
            }                   // END IF (FILEENT == CURRENTRY->ESTYPE)
            if (currentry->esMatch) {   // IGNORE MATCHED ENTRIES
                currentry = currentry->esNext;
                continue;
            }                   // END IF (FILEENT == CURRENTRY->ESTYPE)
            /*
              DIRECTORY FOUND IN SECOND WHICH DOESN'T EXIST IN FIRST;
              CREATE IT.
            */
            memset(newfirst, 0, NAME_MAX);
            memset(newsec, 0, NAME_MAX);
            strcpy(newfirst, indir1);
            len = strlen(newfirst);
            if (newfirst[len - 1] != dirdelim)
                newfirst[len] = dirdelim;   // APPEND DELIMITER IF NEEDED
            strcat(newfirst, currentry->esName);
            
            strcpy(newsec, indir2);
            len = strlen(newsec);
            if (newsec[len - 1] != dirdelim)
                newsec[len] = dirdelim; // APPEND DELIMITER IF NEEDED
            strcat(newsec, currentry->esName);
            if (!(optflag & CHECK)) {
#ifdef UNIX
                /*
                  GET INFO ON EXISTING DIRECTORY TO MATCH PERMISSIONS.
                */
                if (stat(newsec, &dirinfo) < 0) {
                    sprintf(errmsg, "Couldn't get info on dir %s", newsec);
                    perror(errmsg);
                    return(-1);
                }           // END IF (STAT(NEWSEC, &DIRINFO) < 0)
                if (mkdir(newfirst, S_IRWXU | (dirinfo.st_mode & 07777))
                    < 0)    // MUST HAVE USER RWX NO MATTER WHAT
#else
#ifdef DOS
                if (mkdir(newfirst) < 0)
#endif
#endif
                {
                    sprintf(errmsg, "Could not create dir %s", newfirst);
                    perror(errmsg);
                    return(-1);
                }               // END IF (MKDIR(...
                /*
                  PRINT THE NAME OF THE DIRECTORY CREATED IF VERBOSE IS ON.
                */
                if (optflag & VERBOSE)
                    printf("%s created\n", newfirst);
            }                   // END IF (!(OPTFLAG & CHECK))
            /*
              RECURSIVELY CALL PROCESS(), PASSING THE NEW DIRECTORY NAMES.
            */
            if (process(newfirst, newsec))
                return(-1);
            currentry = currentry->esNext;
        }                       // END WHILE (CURRENTRY != NULL)
    }                           // END FOR (I = 0; I < 256; I++)
#ifdef DEBUG
    fprintf(stderr, "Exiting recurseSubDirs()\n");
#endif
    return(0);
}   // END RECURSESUBDIRS()


/************************************************************************/
/* The following function is called when a help message is required.    */
/************************************************************************/
void displayHelp(char *progname) {
#ifdef UNIX
    char delim = '-';
#else
#ifdef DOS
    char delim = '/';
#endif
#endif
#ifdef DEBUG
    fprintf(stderr, "displayHelp() called.\n");
#endif
	printf("\nCopyright 1998, F. Michael Orr\nV1.0, August, 1998.\n");
    printf("This program synchronizes two directories, so that each direct"
           "ory is a mirror\nimage of the other, with the most current versio"
           "n of a file in each directory.\n\nUsage: %s [%ccreqvh] \"dir 1\""
           " \"dir 2\"\nOptions:\n\tc - check only. Do not copy the files ove"
           "r; only display the lists.\n\tr - recursive. Perform the same op"
           "erations for any subdirectories.\n\te - european. Display all da"
           "tes in DD/MM/YYYY format.\n\tq - quiet.  Do not display any mess"
           "ages.\n\tv - verbose.  Display all messages (default).\n\th - he"
           "lp.  This message.\n", progname, delim);
    return;
}                               // END DISPLAYHELP()
