#define VERSION "2.1.0"
/* (2.1.0) = version.release.level */
/*  
how to compile filesplt.c

DOS using TURBO C  
 tcc -mc filesplt.c               

LINUX with dynamicly linked libraries           
 gcc -DDNT="\"`\date`\"" filesplt.c -o filesplt

LINUX with staticly linked libraries 
 gcc -static -DDNT="\"`\date`\"" filesplt.c -o filesplt

OSX              
 att cc -DDNT="\"`\date`\"" filesplt.c -o filesplt

DC/OSX 
 cc -DDNT="\"`\date`\"" filesplt.c -o filesplt

NOTE: it should compile with microsoft c or any type c complier
      with a little tweeking as I tried to keep it as "ANSI" as possible 
      use compact model when compiling filesplt to run under dos
*/
/********************************* FILESPLT.C **********************************
Split large files into smaller more managable files
USAGE: filesplt [-S infile | -J infile] [-F n | -B n] [-O outfile] [-P] [-C file]
-S infile   -split a large file into multiple files
-J infile   -rejoin the files back into the orginal file
-O outfile  -give the outfile a different name (-S or -J option) -optional
-F n        -split the infile into n number of files (-S or -J option)
-B n        -split the infile into n number of bytes (-S option only)
            -either -B or -F option may be used, but not both
-C file     -calculate a crc for the file (stand alone)
-P          -pause between each file (-S or -J option) -optional
EXAMPLE: filesplt -S FILESPLT.EXE -F 4 -P
         filesplt -S FILESPLT.EXE -B 10000
         filesplt -J FILESPLT -O REJOINED
         filesplt -J FILESPLT -F 4
         filesplt -J FILESPLT -P
         filesplt -C SPILT.EXE
An additional file with the extension of .000 contains file parameters
To cat the files together, rename or remove file.000
*******************************************************************************/
#ifdef __TURBOC__
#   ifndef __COMPACT__
#      error "Must be compiled in COMPACT mode"
#   endif
#endif


#include <stdio.h>
#ifdef __MSDOS__
#include <stdlib.h>
#endif
#include <string.h>
#include <ctype.h>
#include <time.h>

#ifndef __MSDOS__
#include <sys/stat.h>
#include <sys/timeb.h>
#endif

#ifdef __MSDOS__
#include <dir.h>
#include <sys\stat.h>
#include <io.h>
#include <alloc.h>
#endif


/* defines for use with way flags variable (16bit unsigned int) */
#define     INFILE          0x0001U
#define     OUTFILE         0x0002U
#define     N_BYTES         0x0004U
#define     N_FILES         0x0008U
#define     SPLIT           0x0010U
#define     JOIN            0x0020U
#define     PAUSE           0x0040U
#define     CRC             0x0080U
/* end of flag defines */

#define     MAX_SPLIT_FILES 999U
#define     BLOCKSIZE       0xFFF0U

/* define unix specific stuff */
#ifndef __MSDOS__
#define PATH_SLASH '/'
/* need to define stuff in dos's <dir.h> for the unix version here */
#define MAXPATH   256
#define MAXFILE   128
#define MAXEXT    0
#endif

/* define dos specific stuff */
#ifdef __MSDOS__
#define PATH_SLASH '\\'
#endif

struct easy {
   char program[MAXFILE + MAXEXT];
   char in_file[MAXPATH];
   char out_file[MAXPATH];
   unsigned long n_files;
   unsigned long n_bytes;
   unsigned flags;
   };

/* setup the CRC tabel */
static const unsigned long crctab[256] = 
{
  0x0LU,
  0x04C11DB7LU, 0x09823B6ELU, 0x0D4326D9LU, 0x130476DCLU, 0x17C56B6BLU,
  0x1A864DB2LU, 0x1E475005LU, 0x2608EDB8LU, 0x22C9F00FLU, 0x2F8AD6D6LU,
  0x2B4BCB61LU, 0x350C9B64LU, 0x31CD86D3LU, 0x3C8EA00ALU, 0x384FBDBDLU,
  0x4C11DB70LU, 0x48D0C6C7LU, 0x4593E01ELU, 0x4152FDA9LU, 0x5F15ADACLU,
  0x5BD4B01BLU, 0x569796C2LU, 0x52568B75LU, 0x6A1936C8LU, 0x6ED82B7FLU,
  0x639B0DA6LU, 0x675A1011LU, 0x791D4014LU, 0x7DDC5DA3LU, 0x709F7B7ALU,
  0x745E66CDLU, 0x9823B6E0LU, 0x9CE2AB57LU, 0x91A18D8ELU, 0x95609039LU,
  0x8B27C03CLU, 0x8FE6DD8BLU, 0x82A5FB52LU, 0x8664E6E5LU, 0xBE2B5B58LU,
  0xBAEA46EFLU, 0xB7A96036LU, 0xB3687D81LU, 0xAD2F2D84LU, 0xA9EE3033LU,
  0xA4AD16EALU, 0xA06C0B5DLU, 0xD4326D90LU, 0xD0F37027LU, 0xDDB056FELU,
  0xD9714B49LU, 0xC7361B4CLU, 0xC3F706FBLU, 0xCEB42022LU, 0xCA753D95LU,
  0xF23A8028LU, 0xF6FB9D9FLU, 0xFBB8BB46LU, 0xFF79A6F1LU, 0xE13EF6F4LU,
  0xE5FFEB43LU, 0xE8BCCD9ALU, 0xEC7DD02DLU, 0x34867077LU, 0x30476DC0LU,
  0x3D044B19LU, 0x39C556AELU, 0x278206ABLU, 0x23431B1CLU, 0x2E003DC5LU,
  0x2AC12072LU, 0x128E9DCFLU, 0x164F8078LU, 0x1B0CA6A1LU, 0x1FCDBB16LU,
  0x018AEB13LU, 0x054BF6A4LU, 0x0808D07DLU, 0x0CC9CDCALU, 0x7897AB07LU,
  0x7C56B6B0LU, 0x71159069LU, 0x75D48DDELU, 0x6B93DDDBLU, 0x6F52C06CLU,
  0x6211E6B5LU, 0x66D0FB02LU, 0x5E9F46BFLU, 0x5A5E5B08LU, 0x571D7DD1LU,
  0x53DC6066LU, 0x4D9B3063LU, 0x495A2DD4LU, 0x44190B0DLU, 0x40D816BALU,
  0xACA5C697LU, 0xA864DB20LU, 0xA527FDF9LU, 0xA1E6E04ELU, 0xBFA1B04BLU,
  0xBB60ADFCLU, 0xB6238B25LU, 0xB2E29692LU, 0x8AAD2B2FLU, 0x8E6C3698LU,
  0x832F1041LU, 0x87EE0DF6LU, 0x99A95DF3LU, 0x9D684044LU, 0x902B669DLU,
  0x94EA7B2ALU, 0xE0B41DE7LU, 0xE4750050LU, 0xE9362689LU, 0xEDF73B3ELU,
  0xF3B06B3BLU, 0xF771768CLU, 0xFA325055LU, 0xFEF34DE2LU, 0xC6BCF05FLU,
  0xC27DEDE8LU, 0xCF3ECB31LU, 0xCBFFD686LU, 0xD5B88683LU, 0xD1799B34LU,
  0xDC3ABDEDLU, 0xD8FBA05ALU, 0x690CE0EELU, 0x6DCDFD59LU, 0x608EDB80LU,
  0x644FC637LU, 0x7A089632LU, 0x7EC98B85LU, 0x738AAD5CLU, 0x774BB0EBLU,
  0x4F040D56LU, 0x4BC510E1LU, 0x46863638LU, 0x42472B8FLU, 0x5C007B8ALU,
  0x58C1663DLU, 0x558240E4LU, 0x51435D53LU, 0x251D3B9ELU, 0x21DC2629LU,
  0x2C9F00F0LU, 0x285E1D47LU, 0x36194D42LU, 0x32D850F5LU, 0x3F9B762CLU,
  0x3B5A6B9BLU, 0x0315D626LU, 0x07D4CB91LU, 0x0A97ED48LU, 0x0E56F0FFLU,
  0x1011A0FALU, 0x14D0BD4DLU, 0x19939B94LU, 0x1D528623LU, 0xF12F560ELU,
  0xF5EE4BB9LU, 0xF8AD6D60LU, 0xFC6C70D7LU, 0xE22B20D2LU, 0xE6EA3D65LU,
  0xEBA91BBCLU, 0xEF68060BLU, 0xD727BBB6LU, 0xD3E6A601LU, 0xDEA580D8LU,
  0xDA649D6FLU, 0xC423CD6ALU, 0xC0E2D0DDLU, 0xCDA1F604LU, 0xC960EBB3LU,
  0xBD3E8D7ELU, 0xB9FF90C9LU, 0xB4BCB610LU, 0xB07DABA7LU, 0xAE3AFBA2LU,
  0xAAFBE615LU, 0xA7B8C0CCLU, 0xA379DD7BLU, 0x9B3660C6LU, 0x9FF77D71LU,
  0x92B45BA8LU, 0x9675461FLU, 0x8832161ALU, 0x8CF30BADLU, 0x81B02D74LU,
  0x857130C3LU, 0x5D8A9099LU, 0x594B8D2ELU, 0x5408ABF7LU, 0x50C9B640LU,
  0x4E8EE645LU, 0x4A4FFBF2LU, 0x470CDD2BLU, 0x43CDC09CLU, 0x7B827D21LU,
  0x7F436096LU, 0x7200464FLU, 0x76C15BF8LU, 0x68860BFDLU, 0x6C47164ALU,
  0x61043093LU, 0x65C52D24LU, 0x119B4BE9LU, 0x155A565ELU, 0x18197087LU,
  0x1CD86D30LU, 0x029F3D35LU, 0x065E2082LU, 0x0B1D065BLU, 0x0FDC1BECLU,
  0x3793A651LU, 0x3352BBE6LU, 0x3E119D3FLU, 0x3AD08088LU, 0x2497D08DLU,
  0x2056CD3ALU, 0x2D15EBE3LU, 0x29D4F654LU, 0xC5A92679LU, 0xC1683BCELU,
  0xCC2B1D17LU, 0xC8EA00A0LU, 0xD6AD50A5LU, 0xD26C4D12LU, 0xDF2F6BCBLU,
  0xDBEE767CLU, 0xE3A1CBC1LU, 0xE760D676LU, 0xEA23F0AFLU, 0xEEE2ED18LU,
  0xF0A5BD1DLU, 0xF464A0AALU, 0xF9278673LU, 0xFDE69BC4LU, 0x89B8FD09LU,
  0x8D79E0BELU, 0x803AC667LU, 0x84FBDBD0LU, 0x9ABC8BD5LU, 0x9E7D9662LU,
  0x933EB0BBLU, 0x97FFAD0CLU, 0xAFB010B1LU, 0xAB710D06LU, 0xA6322BDFLU,
  0xA2F33668LU, 0xBCB4666DLU, 0xB8757BDALU, 0xB5365D03LU, 0xB1F740B4LU
};

void version(struct easy *way);
void command_line_stuff(struct easy *way, char **argv, int argc);
void split(struct easy *way);
void join(struct easy *way);
void usage(struct easy *way);
FILE *openfile(char *filename, char *mode);
FILE *closefile(FILE *file, char *filename);
void cksum (struct easy *way);

void join(struct easy *way)
{
   struct stat buf;
   struct tm *ftm;
   FILE *in, *out;
   char infile[MAXPATH + MAXFILE + MAXEXT];
   static char keyword[20], data[80];
   char string[256];
   int index;
   size_t bytes_read;
   unsigned char *buffer, *cp;
   unsigned long length = 0, crc = 0, bytecount;

   static char filename[MAXFILE + MAXEXT];
   static char filetime[20];
   static char filedate[20];
   unsigned long filesize = 0;
   unsigned long filecrc = 0;
   int filecount = 0;

#ifdef __MSDOS__
   if((buffer = (unsigned char *)farmalloc(BLOCKSIZE)) == NULL)
   {
      fprintf(stderr, "Not enough memory to allocate buffer\n");
      exit(0);
   }
#endif
#ifndef __MSDOS__
   if((buffer = (unsigned char *)malloc(BLOCKSIZE)) == NULL)
   {
      fprintf(stderr, "Not enough memory to allocate buffer\n");
      exit(0);
   }
#endif
   strcpy(infile, way->in_file);

#ifdef __MSDOS__
   /* get rid of the ext at the end of the in file */
   for(index = strlen(infile) - 1;index > 0;index--)
   {
      if(infile[index] == PATH_SLASH)
         break;
      if(infile[index] == '.')
      {
         infile[index] = NULL;
         break;
      }
   }
#endif

   /* try to read the data file.000, if it exist */

   sprintf(way->in_file, "%s.000", infile);
   if(way->flags & PAUSE)
   {
      fflush(stdin);
      fprintf(stderr, "Press the RETURN key to read file %s", way->in_file);
      while(!fgetc(stdin));
   }
   fprintf(stderr, "Reading file %s\n", way->in_file);
/* attempt to open it */
   while((in = openfile(way->in_file,  "r")) == NULL)
   {
      fprintf(stderr, "Data file %s not found.\n"
                      "Would you like to specify a path for "
                      "data file %s (Y/N) ? ", way->in_file, way->in_file);
      fflush(stdin);
      if(toupper(fgetc(stdin)) == 'N')
      {
         fprintf(stderr, "Warning, not using data file %s.000\n"
                         "Can't verify file integrity\n", infile);
         goto let_me_out;
      }
      fprintf(stderr, "Enter complete path and filename for the data file\n");
      fprintf(stderr, "EXAMPLES: %s.000\n"
                      "          %csomepath%c%s.000\n",
                      infile, PATH_SLASH, PATH_SLASH, infile);
#ifdef __MSDOS__
      fprintf(stderr, "          C:%csomepath%c%s.000\n",
                      PATH_SLASH, PATH_SLASH, infile);
#endif
      fflush(stdin);
      fgets(way->in_file, sizeof(way->in_file) - 1, stdin);
#ifdef __TURBOC__
      way->in_file[strlen(way->in_file) -1] = NULL;
#else
      (int)way->in_file[strlen(way->in_file) -1] = NULL;
#endif
   }
/* once it's open then read out whatever we can */
   while(fgets(string, sizeof(string) - 1, in)  != NULL)
   {
#ifdef __TURBOC__
      string[strlen(string) - 1] = NULL;
#else
      (int)string[strlen(string) - 1] = NULL;
#endif
      sscanf(string, "%s %s", keyword, data);
      if(!strcmp(keyword, "Filename"))
      {
         if(way->flags & OUTFILE)
            continue;
         strcpy(filename, data);
         continue;
      }
      if(!strcmp(keyword, "Filesize"))
      {
/*         filesize = (unsigned long)atol(data);*/
filesize =strtoul(data, NULL, 10);
         continue;
      }
      if(!strcmp(keyword, "Filetime"))
      {
         strcpy(filetime, data);
         continue;
      }
      if(!strcmp(keyword, "Filedate"))
      {
         strcpy(filedate, data);
         continue;
      }
      if(!strcmp(keyword, "Filecrc"))
      {
         filecrc =strtoul(data, NULL, 10);
/* filecrc = (unsigned long)atol(data); */
         continue;
      }
      if(!strcmp(keyword, "Filecount"))
      {
/*         filecount = (unsigned)atol(data);*/
filecount = strtol(data, NULL, 10);
         continue;
      }

   }

   /* then close the file */
   if(closefile(in, way->in_file) == NULL)
      exit(0);
   let_me_out: ;
/* is there an outfile name to work with ? */
   if(!(way->flags & OUTFILE))
   {
      if(filename+0 == NULL)
/*      if(filename[0] == NULL)*/
      {
      /* assume the infile name if there is no filename */
         fprintf(stderr, "Warning, unable to determine the file name "
                         "for the out file.\nUsing default %s\n", infile);
         strcpy(way->out_file, infile);
      }
      else
         strcpy(way->out_file, filename);
   }
   /* if there is no file count from the command line and */
   if(!(way->flags & N_FILES))
   {
      /* if the file count from the data file is 0 */
      if(filecount == 0)
      {
      /* assume the max */
         fprintf(stderr, "Warning, unable to determine how may files "
                         "required to join file %s.\nAssuming %d files\n",
                         infile, MAX_SPLIT_FILES);
         filecount = MAX_SPLIT_FILES;
      }

      way->n_files =  filecount;
   }
   /* otherwise use the count on the command line */
   else
      filecount = (int)way->n_files;

   /* and if the outfile exist, do we abort the joining operation ? */
   if(access(way->out_file, 0) == 0)
   {
      fprintf(stderr, "File %s exist\n", way->out_file);
      fprintf(stderr, "Overwrite %s Y/N ?\n", way->out_file);
      fflush(stdin);
      if(toupper(fgetc(stdin)) != 'Y')
         exit(0);
   }


   /* open up the out file */
   if((out = openfile(way->out_file,  "wb")) == NULL)
      exit(0);

   for(filecount = 1;filecount <= way->n_files;filecount++)
   {
      sprintf(way->in_file, "%s.%03d", infile, filecount);

      if(way->flags & PAUSE)
      {
         fflush(stdin);
         fprintf(stderr, "Press the RETURN key to read file %s", way->in_file);
         while(!fgetc(stdin));
      }
      /* and if we were guesing at the number of files (MAX_SPLIT_FILES) */
      if(access(way->in_file, 0) != 0)
      {
         fprintf(stderr, "Warning, file %s not found\n"
                  "Maybe thats all the files there is ?\n"
                  "Warning, unable to verify if filecount is correct.\n",
         way->in_file);
         filecount--;
         way->n_files = filecount;
         break;
      }
      fprintf(stderr, "Reading file %s\n", way->in_file);
      if((in = openfile(way->in_file,  "rb")) == NULL)
         exit(0);
      bytecount = 0;
      while ((bytes_read = fread ((void *)buffer, sizeof(char),
            BLOCKSIZE, in)) > 0)
      {
         /* write the block to the out file */
         fwrite ((void *)buffer, sizeof(char), bytes_read, out);
         bytecount += bytes_read;
         length += bytes_read;
         /* calculate a checksum (crc) */
         cp = buffer;
         {
            while (bytes_read--)
               crc = (crc << 8) ^ crctab[(unsigned)((crc >> 24) ^
                         *(cp++)) & 0xFF];
         }
      }
      /* then close the current split file and go on to the next */
      if(closefile(in, way->in_file) == NULL)
         exit(0);
   }

   if(closefile(out, way->out_file) == NULL)
      exit(0);

   /* set the outfile to the date and time of the orginal file  */
   /* needs some additional work to set the file date and time */

   /* and do the file lengths agree ? */
   stat(way->out_file, &buf);
   if(!filesize)
      fprintf(stderr, "Warning, unable to verify if filesize is correct.\n");
   else
      if(length != filesize)
      {
         fprintf(stderr, "Warning, calculated file size does not match that "
                         "recorded in %s.000\n", infile);
         fprintf(stderr, "calculated %lu vs recorded %lu\n",
                               length, filesize);
      }
   if(length != buf.st_size)
   {
      fprintf(stderr, "Warning, calculated file size does not match the "
                      "actual file size\ncalculated %lu vs actual %lu\n",
                      length, buf.st_size);
   }
   /* and do the check sums argee ? */
   /* do final processing on checksum (crc) */
   while (length > 0)
   {
      crc = (crc << 8) ^ crctab[(unsigned)((crc >> 24) ^ length) & 0xFF];
      length >>= 8;
   }
   crc = ~crc & 0xFFFFFFFFLU;
   if(!filecrc)
      fprintf(stderr, "Warning, unable to verify if crc is correct.\n");
/*   else   */ 
   if(crc != filecrc)
      fprintf(stderr, "Warning, calculated crc does not match that "
                      "recorded in %s.000\n", infile);

/* printf some stuff about the file */
   ftm = localtime(&buf.st_atime);
   fprintf(stderr, "Filename  %s\n"
                   "Filesize  %lu\n"
                   "Filetime  %2d:%02d:%02d\n"
                   "Filedate  %2d/%02d/%02d\n"
                   "Filecrc   %lu\n"
                   "Filecount %lu\n",
                   way->out_file,
                   buf.st_size,
                   ftm->tm_hour, ftm->tm_min, ftm->tm_sec,
                   ftm->tm_mon + 1, ftm->tm_mday, ftm->tm_year,
                   crc,
                   way->n_files);
   version(way);
   return;
}

void main(int argc, char **argv)
{
   struct easy way;

   time_t start_time, end_time, diff_time, hrs, min, sec, timeleftover;
   int index;

   /* start the timer running */
   time(&start_time);

   way.flags = 0;

   /* seperate the program name from the path */
   index = strlen(argv[0]) - 1;
   while((argv[0][index] != PATH_SLASH) & (index != 0))
      index--;
#ifndef  __MSDOS__
   strcpy(way.program, argv[0] + index);
#endif
#ifdef  __MSDOS__
   strcpy(way.program, argv[0] + index + 1);
#endif

#ifdef __MSDOS__
   /* and get rid of the extenstion for the program name */
   index = 0;
   while(way.program[index++] != '.');
   way.program[index - 1] = NULL;
#endif

   /* do we have enough to get started ? */
   if(argc < 2)
      usage(&way);

   /* whats on the command line ? */
   command_line_stuff(&way, argv, argc);

   /* do we calcultae the crc of a file ? */
   if(way.flags & CRC)
      cksum(&way);

   /* or split a file ? */
   if(way.flags & SPLIT)
      split(&way);

   /* or join a file ? */
   if(way.flags & JOIN)
      join(&way);


/* calculate the time required to split/join the file */
/* this is total time including time while paused */
   time(&end_time);
   diff_time = end_time - start_time;
   hrs = diff_time / 3600;
   timeleftover = diff_time % 3600;
   min = timeleftover / 60;
   sec = timeleftover % 60;
   if(hrs)
      fprintf(stderr, "%ld Hours ");
   if(hrs | min)
      fprintf(stderr, "%ld Minutes ");
   fprintf(stderr, "%ld Seconds ", sec);

   /* get rid of the .000 on the end of way.out_file */
#ifdef __TURBOC__
   way.out_file[strlen(way.out_file) -4] = NULL;
#else
   (int)way.out_file[strlen(way.out_file) -4] = NULL;
#endif
   if(way.flags & SPLIT)
      fprintf(stderr, "required to split file %s\n", way.out_file);
   if(way.flags & JOIN)
      fprintf(stderr, "required to join file %s\n", way.out_file);

}

/* take stuff off the command line */
void command_line_stuff(struct easy *way, char **argv, int argc)
{
   int index;
   for(index = 1; index < argc; index++)
   {
      if(argv[index][0] != '-')
         continue;

      toupper(argv[index][0]);
      if(strcmp(argv[index], "-P"))
      {
         if((argv[index + 1][0] == '-' ) | (argv[index + 1][0] == NULL))
         {
            fprintf(stderr, "Options -S, -J, -O requires a filename, options "
            "-F, -B requires a number\n");
            usage(way);
         }
      }
      switch(argv[index][1])
      {
         case 'C':
         case 'c':
            way->flags |= CRC;
            strcpy(way->in_file, argv[++index]);
            break;
         case 'S':
         case 's':
            way->flags |= SPLIT;
            strcpy(way->in_file, argv[++index]);
            break;
         case 'J':
         case 'j':
            way->flags |= JOIN;
            strcpy(way->in_file, argv[++index]);
            break;
         case 'O':
         case 'o':
            way->flags |= OUTFILE;
            strcpy(way->out_file, argv[++index]);
            break;
         case 'F':
         case 'f':
         way->flags |= N_FILES;
            way->n_files = atol(argv[++index]);
            break;
         case 'B':
         case 'b':
            way->flags |= N_BYTES;
            way->n_bytes = atol(argv[++index]);
            break;
         case 'P':
         case 'p':
            way->flags |= PAUSE;
            break;
         default:
            fprintf(stderr, "Unknown option  %s  found on command line\n",
                    argv[index]);
      }

   }

#ifdef __MSDOS__
   /* if the outfile is to a floppy then turn on the pause funciton */
   if(way->out_file[1] == ':')
      switch(way->out_file[0])
      {
         case 'a':
         case 'A':
         case 'b':
         case 'B':
            way->flags |= PAUSE;
            fprintf(stderr, "Output is to drive %c:, starting the PAUSE "
                    "function automatically\n", toupper(way->out_file[0]));
            break;
      }

   /* if the infile is from a floppy then turn on the pause funciton */
   if(way->in_file[1] == ':')
      switch(way->in_file[0])
      {
         case 'a':
         case 'A':
         case 'b':
         case 'B':
            way->flags |= PAUSE;
            fprintf(stderr, "Input is from drive %c:, starting the PAUSE "
                    "function automatically\n", toupper(way->in_file[0]));
            break;
      }
#endif

   /* are both split and join options on the command line? */
   if((way->flags & (SPLIT + JOIN)) == (SPLIT + JOIN))
   {
      fprintf(stderr, "Conflicting -S (split) -J (join) options\n");
      usage(way);
   }

   /* are both the files and bytes options on the command line? */
   if((way->flags & (N_BYTES + N_FILES)) == (N_BYTES + N_FILES))
   {
      fprintf(stderr, "Conflicting -F (files) -B (bytes) options\n");
      usage(way);
   }

   /* if split option then -f or -b option required */
   if(way->flags & SPLIT)
      if(!((way->flags & N_FILES) | (way->flags & N_BYTES)))
      {
         fprintf(stderr, "-F (files) or -B (bytes) option required with "
                 "the -S (split) option\n");
         usage(way);
      }

   /* if no split or join option */
   if(!((way->flags & JOIN) | (way->flags & SPLIT) | (way->flags & CRC)))
   {
      fprintf(stderr, "-S (split) or -J (join) or -C (crc) option required.\n");
      usage(way);
   }
}

void usage(struct easy *way)
{
   printf("Split large files into smaller more managable files\n");
   printf("USAGE: %s [-S infile | -J infile] [-F n | -B n] "
          "[-O outfile] [-P] [-C file]\n", way->program);
   printf("-S infile   -split a large file into multiple files\n"
          "-J infile   -rejoin the files back into the orginal file\n"
          "-O outfile  -give the outfile a different name "
          "(-S or -J option) -optional\n"
          "-F n        -split the infile into n number of files "
          "(-S or -J option)\n"
          "-B n        -split the infile into n number of bytes "
          "(-S option only)\n"
          "            -either -B or -F option may be used, but not both\n"
          "-C file     -calculate a crc for the file (stand alone)\n"
          "-P          -pause between each file (-S or -J option) "
          "-optional\n"
#ifdef __MSDOS__
          "            -specifing either drive A or B as input or output "
          "automatically\n"
          "             forces the pause option\n"
#endif
          );

   printf("EXAMPLE: %s -S FILENAME.EXE -F 4 -P\n"
          "         %s -S FILENAME.EXE -B 10000\n"
          "         %s -J FILENAME -O REJOINED\n"
          "         %s -J FILENAME -F 4\n"
          "         %s -J FILENAME -P\n"
          "         %s -C FILENAME.EXE\n",
   way->program, way->program, way->program,
   way->program, way->program, way->program);
   printf("The file with the extension of .000 contains "
          "file parameters and information\n"
          "To rejoin the files without %s, rename or remove file.000, then\n"
#ifdef __MSDOS__
          "         copy /b FILENAME.00? > FILENAME.EXE\n", way->program);
#else
          "         cat FILENAME.00? > FILENAME.EXE\n", way->program);
#endif
   version(way);
}
void version(struct easy *way)
{
#ifndef __MSDOS__
   printf("%s (Version %s) ", way->program, VERSION);
#endif
#ifdef DNT
   printf("created %s\n", DNT);
#else
   printf("\n");
#endif
#ifdef __MSDOS__
   printf("%s (Version %s) created %s at %s\n", way->program, VERSION,
          __DATE__, __TIME__);
#endif
   exit(0);
}

/* split the file */
void split(struct easy *way)
{
   struct stat buf;
   struct tm *ftm;
   FILE *in, *out;

   int index, filecount;
   size_t block, bytes_read;
   unsigned char *buffer, *cp;
   char outfile[MAXPATH + MAXFILE + MAXEXT], filename[MAXFILE + MAXEXT];
   unsigned long remainder = 0, length = 0, crc = 0, bytecount, filesize;


#ifdef __MSDOS__
   if((buffer = (unsigned char *)farmalloc(BLOCKSIZE)) == NULL)
   {
      fprintf(stderr, "Not enough memory to allocate buffer\n");
      exit(0);
   }
#endif
#ifndef __MSDOS__
   if((buffer = (unsigned char *)malloc(BLOCKSIZE)) == NULL)
   {
      fprintf(stderr, "Not enough memory to allocate buffer\n");
      exit(0);
   }
#endif
   /* seperate the infile name from the path */
   for(index = strlen(way->in_file) - 1; index >= 0; index--)
      if(way->in_file[index] == PATH_SLASH)
      {
         strcpy(filename, way->in_file + (index + 1));
         break;
      }
   if(index < 0)
      strcpy(filename, way->in_file);

   /* if no outfile specified use the infile minus the ext as the outfile */
   if(!(way->flags & OUTFILE))
      strcpy(way->out_file, filename);

#ifdef __MSDOS__
   /* if only the drive is specified as an outfile add the infile minus */
   /* the ext as the outfile */
   if(((way->out_file[1] == ':') & (way->out_file[2] == NULL)) |
      ((way->out_file[1] == ':') & (way->out_file[2] == PATH_SLASH) &
      (way->out_file[3] == NULL)))
      strcat(way->out_file, filename);

   /* get rid of the ext at the end of the outfile */
   for(index = strlen(way->out_file) - 1;index > 0;index--)
   {
      if(way->out_file[index] == PATH_SLASH)
         break;
      if(way->out_file[index] == '.')
      {
         way->out_file[index] = NULL;
         break;
      }
   }
#endif
   strcpy(outfile, way->out_file);

   /* get the file  stats */
   stat(way->in_file, &buf);

   /* get the size of the file */
   filesize = buf.st_size;

   /* open the in file */
   if((in = openfile(way->in_file,  "rb")) == NULL)
      exit(0);

   /* if /B option calculate the number of out files */
   if(way->flags & N_BYTES)
   {
      way->n_files = filesize / way->n_bytes;
      if(filesize % way->n_bytes)
         way->n_files++;
   }

   /* if /F option calculate the number of bytes per file */
   if(way->flags & N_FILES)
   {
      way->n_bytes = (filesize / way->n_files);
      remainder = filesize % way->n_files;
      if(remainder)
         way->n_bytes++;
   }

   /* and if the number of split files exceed the max (MAX_SPLIT_FILES) */
   if(way->n_files > MAX_SPLIT_FILES)
   {
      fprintf(stderr, "Number of out files cannot exceed %u files\n",
              MAX_SPLIT_FILES);
      usage(way);
   }

   fprintf(stderr, "Spliting %s into %ld files\n", filename, way->n_files);
   fprintf(stderr, "Reading file %s   %10lu Bytes\n", filename, filesize);

   for(filecount = 1;filecount <= way->n_files;filecount++)
   {
      /* if doing /F option this makes the byte count of the */
      /* split files come out right (within a byte of each other) */
      if(way->flags & N_FILES)
      {
         if(remainder)
            if(filecount == (remainder + 1))
               way->n_bytes--;
      }
      if(way->flags & PAUSE)
      {
         fflush(stdin);
         fprintf(stderr, "Press the RETURN key to write file %d", filecount);
         while(!fgetc(stdin));
      }
      sprintf(way->out_file, "%s.%03d", outfile, filecount);
      fprintf(stderr, "Writing file %s   ", way->out_file);
      if((out = openfile(way->out_file,  "wb")) == NULL)
         exit(0);

      bytecount = 0;
      if(way->n_bytes < BLOCKSIZE)
         block = (size_t) way->n_bytes;
      else
         block = BLOCKSIZE;
      /* read a block from the in file */
      while ((bytes_read = fread ((void *)buffer, sizeof(char),
            block, in)) > 0)
      {
         /* write the block to the out file */
         fwrite ((void *)buffer, sizeof(char), bytes_read, out);
         bytecount += bytes_read;

         /* calculate a checksum (crc) */
         cp = buffer;
         {
            length += bytes_read;
            while (bytes_read--)
               crc = (crc << 8) ^ crctab[(unsigned)((crc >> 24) ^
                      *(cp++)) & 0xFF];
         }


         if(bytecount == way->n_bytes)
            break;
         if((way->n_bytes - bytecount) < BLOCKSIZE)
            block = (unsigned int) (way->n_bytes - bytecount);

      }

      fprintf(stderr, "%10lu Bytes\n", bytecount);
      /* close file and prepare for the next split file */
      if(closefile(out, way->out_file) == NULL)
         exit(0);
   }
   /* make a quick check on the file sizes */
   if(filesize != length)
      fprintf(stderr, "There is a problem with the file, "
                      "the sizes don't agree\n"
                      "Length = %lu, Filesize = %lu\n",
                       length, filesize);

   /* do final processing on checksum (crc) */
   while (length > 0)
   {
      crc = (crc << 8) ^ crctab[(unsigned)((crc >> 24) ^ length) & 0xFF];
      length >>= 8;
   }
   crc = ~crc & 0xFFFFFFFFLU;

   /* once all the split files have been written we go back */
   /* write the file info stuff into file .000 */
   sprintf(way->out_file, "%s.000", outfile);
   if((out = openfile(way->out_file,  "w")) == NULL)
      exit(0);

   /* get the file time and date to be stored in 1st split file .000 */
/*  ctime(&buf.st_atime);*/
   ftm = localtime(&buf.st_atime);

   fprintf(out, "Filename  %s\n"
                "Filesize  %lu\n"
                "Filetime  %2d:%02d:%02d\n"
                "Filedate  %2d/%02d/%02d\n"
                "Filecrc   %lu\n"
                "Filecount %lu\n",
                filename,
                filesize,
                ftm->tm_hour, ftm->tm_min, ftm->tm_sec,
                ftm->tm_mon + 1, ftm->tm_mday, ftm->tm_year,
                crc,
                way->n_files);
   fprintf(out, "\n%s\n"
           "Contains information about the file that is useful to verify\n"
           "that the rejoined file is the same as the orginal file.\n",
           way->out_file);
   fprintf(out, "Although nice to have but not neccessary, as %s will do\n"
                "it best to rejoin the files, provided the correct number of\n"
                "files is available.\n", way->program);
   fprintf(out, "Alternately the files can be rejoined by cat'ing "
                "if so desired.\n"
                "Rename or delete %s, then cat the remaining\n"
                "files and redirect to %s.\n", way->out_file, filename);
   if(way->n_files < 10)
      fprintf(out, "Example:   cat %s.00? > %s\n", outfile, filename);
   if((way->n_files < 100) && (way->n_files > 10))
      fprintf(out, "Example:   cat %s.0?? > %s\n", outfile, filename);
   if((way->n_files < 1000) && (way->n_files > 100))
      fprintf(out, "Example:   cat %s.??? > %s\n", outfile, filename);
   fprintf(out, "Then calculate the CRC of the newly created file to "
                "verify\n"
                "it was created correctly\n"
                "Example:   %s -c %s\n", way->program, filename);
   closefile(out, way->out_file);
   stat(way->out_file, &buf);
   fprintf(stderr, "Writing file %s   %10lu Bytes\n", way->out_file,
             buf.st_size);
   fprintf(stderr, "Checksum (crc) = %lu (0x%lX)\n", crc, crc);
   version(way);
   return;
}

#ifndef __MSDOS__

/**************** OC_FILES.C ******** unix version ************
**
** example program using openfile() and closefile() 
**
** #include <stdio.h>
** FILE *openfile();
** int closefile();
** main()
** {
**      FILE *in;
**      FILE *out;
**      char in_file[256] = "file_1";
**      char out_file[256] = "file_2";
**
**      in  = openfile(in_file,  "r");
**      out = openfile(out_file,  "w");
**      .
**      .
**      .
**      closefile(in, in_file);
**      closefile(out, out_file);
** }
***************************************************************/


/*********** OPENFILE() **************/
FILE *openfile(char *filename, char *mode)
{
   FILE *file;
   FILE *fopen();
   if((file = fopen(filename, mode)) == NULL)
   {
      fprintf(stderr, "Error opening file %s ", filename);
      perror("");
      fprintf(stderr, "\n");
      return NULL; 
   }
   return(file);   
}   /* end of function openfile() */

/*********** CLOSEFILE() *************/
FILE *closefile(FILE *file, char *filename)
{
   if(fclose(file) == EOF)
   {
      fprintf(stderr,"Error closing file %s ", filename);
      perror("");
      fprintf(stderr, "\n");
      return NULL;
   }      
   return (file);
}   /* end of function closefile() */

/********* END OF OC_FILES.C *********/

#endif
void cksum (struct easy *way)
{
   unsigned char *buffer, *cp;
   unsigned long crc = 0;
   long length = 0;
   long bytes_read;
   register FILE *in;

#ifdef __MSDOS__
   if((buffer = (unsigned char *)farmalloc(BLOCKSIZE)) == NULL)
   {
      fprintf(stderr, "Not enough memory to allocate buffer\n");
      exit(0);
   }
#endif
#ifndef __MSDOS__
   if((buffer = (unsigned char *)malloc(BLOCKSIZE)) == NULL)
   {
      fprintf(stderr, "Not enough memory to allocate buffer\n");
      exit(0);
   }
#endif

   if((in = openfile(way->in_file,  "rb")) == NULL)
      exit(0);

   while ((bytes_read = fread (buffer, 1, BLOCKSIZE, in)) > 0)
   {
      cp = buffer;
      length += bytes_read;
      while (bytes_read--)
      crc = (crc << 8) ^ crctab[(unsigned)((crc >> 24) ^ *(cp++)) & 0xFF];
   }

   if(closefile(in, way->in_file) == NULL)
      exit(0);

   bytes_read = length;
   while (bytes_read > 0)
   {
      crc = (crc << 8) ^ crctab[(unsigned)((crc >> 24) ^ bytes_read) & 0xFF];
      bytes_read >>= 8;
   }

   crc = ~crc & 0xFFFFFFFFLU;

   fprintf(stderr, "%s\nFilecrc %10lu   Filesize%8ld\n", way->in_file, crc, length);
   version(way);
   return;
}

