// main.cc
// $Id: main.cc,v 1.8 2000/03/18 18:42:40 gwiley Exp $
// Glen Wiley, <gwiley@ieee.org>
//
// Copyright (c)2000, Glen Wiley
// 
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
//

#if HAVE_CONFIG_H
# include "config.h"
#endif
#if HAVE_DIRENT_H
# include <dirent.h>
# define NAMLEN(dirent) strlen((dirent)->d_name)
#else
# define dirent direct
# define NAMLEN(dirent) (dirent)->d_namlen
# if HAVE_SYS_NDIR_H
#  include <sys/ndir.h>
# endif
# if HAVE_SYS_DIR_H
#  include <sys/dir.h>
# endif
# if HAVE_NDIR_H
#  include <ndir.h>
# endif
#endif
#include <errno.h>
#include <list>
#if HAVE_SIGNAL_H
# include <signal.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#if HAVE_SYS_STAT_H
# include <sys/stat.h>
#endif
#if HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#if HAVE_STRING_H
# include <string.h>
#else
# if HAVE_STRINGS_H
#  include <strings.h>
# endif
#endif
#include <time.h>
#if HAVE_UNISTD_H
# include <unistd.h>
#endif
#include "filespec.h"
#include "logger.h"
#include "conffile.h"

typedef list<FileSpec *> SpecList;

// all of the configuration files are read from this directory
static const char *DIR_CONF = "/usr/local/etc/filechkd";

// this file will be used to configure the daemon itself
static const char *FN_CONF  = "filechkd.conf";

// suffix of filespec config files, must include the leading '.'
static const char *FILESPEC_SUFF = ".fs";

// default seconds between checks
static const int DFLT_CHECK_FREQ = 120;

// file to write PID to
static const char *DFLT_PIDFILE = "/var/run/filechkd.pid";

// the program name, global b/c logging and output messages may need it
char *g_prg;

// global log object, gets set if user specifies -l on cmd line
Logger *g_log = NULL;

// termination flag - set to true on receipt of terminating signal
bool g_termflag = false;

void usage(void);
void readSpecs(SpecList &flist, const char *dirnm);
void readconf(int argc, char *argv[], int *chkfreq, bool *foreground
 , char **fn_log, char **logsev, char **fn_pid);
void workloop(SpecList &flist, int chkfreq);
bool strtobool(const char *str);

extern "C" void strlower(char *str);
extern "C" int daemonize(void);
extern "C" void sighandler(int sig);

//---------------------------------------- main
int
main(int argc, char *argv[])
{
	pid_t    pid;
	FILE     *fh;
	int      opt;
	int      err;
	int      retval      = 0;
	int      chkfreq     = DFLT_CHECK_FREQ;
	char     *fn_log     = NULL;
	char     *fn_cfg     = NULL;
	char     *logsev     = "warning";
	char *fn_pid         = (char *)DFLT_PIDFILE;
	bool     foreground  = false;
	const char *dir_conf = DIR_CONF;
	SpecList filespecs;
#if HAVE_SIGACTION
	struct sigaction sact;
	sigset_t         sset;
#endif

	g_prg = strrchr(argv[0], '/');
	if(g_prg)
		g_prg--;
	else
		g_prg = argv[0];

	// read a configuration file if there is one to read and
	// populate the arguments from the file - command line
	// options will override these settings

	readconf(argc, argv, &chkfreq, &foreground, &fn_log, &logsev, &fn_pid);

	// now parse the command line properly

	while((opt = getopt(argc, argv, "?hFc:l:L:f:p:")) != EOF)
	{
		switch(opt)
		{
			case 'c':
				dir_conf = optarg;
				break;

			case 'f':
				chkfreq = (int) strtol(optarg, NULL, 10);
				break;

			case 'F':
				foreground = true;
				break;

			case 'l':
				fn_log = optarg;
				break;

			case 'L':
				logsev = optarg;
				break;

			case 'p':
				fn_pid = optarg;
				break;

			case '?':
			case 'h':
			default:
				usage();
				return EINVAL;
				break;
		} // switch(opt)

	} // while((opt = getopt(argc, argv, "")) != EOF)

	// daemonize
	if(!foreground)
	{
		err = daemonize();
		if(err != 0)
		{
			fprintf(stderr, "error daemonizing: %d, %s\n", err, strerror(err));
			exit(1);
		}
	} // if(!foreground)

	// catch some terminating signals
#if HAVE_SIGACTION
	sigemptyset(&sset);
	sact.sa_handler = sighandler;
	sact.sa_mask    = sset;
	sact.sa_flags   = SA_RESTART;
	sigaction(SIGTERM, &sact, NULL);
	sigaction(SIGINT,  &sact, NULL);
#else
	signal(SIGTERM, sighandler);
	signal(SIGINT, sighandler);
#endif

	// we only log if the user specified a log filename
	if(fn_log)
	{
		g_log = new Logger(fn_log);
		g_log->setMinSeverity(logsev);
		g_log->setOptions(Logger::LOGDATE | Logger::LOGPID | Logger::LOGSEVLBL);
	}

	if(g_log)
		g_log->logmsg(Logger::INFO, "starting...");

	// write process id to a file
	fh = fopen(fn_pid, "w+");
	if(fh != NULL)
	{
		fprintf(fh, "%d\n", getpid());
		fchmod(fileno(fh), S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH);
		fclose(fh);
	}
	else
	{
		if(g_log)
			g_log->vlogmsg(Logger::ERROR, "failed to write pid file %s: %s"
			 , fn_pid, strerror(errno));
	}

	readSpecs(filespecs, dir_conf);

	workloop(filespecs, chkfreq);

	if(g_log)
		g_log->logmsg(Logger::INFO, "exiting");

	if(fn_pid)
		unlink(fn_pid);

	return retval;
} // main

//---------------------------------------- usage
void
usage(void)
{
	fprintf(stderr
	 , "filechkd VERSION - Glen Wiley, <gwiley@ieee.org>\n\n"
	   "USAGE: %s [?h] [-c <conf_dir>] [-f <frequency>] [-l <log_file>]\n"
	   "-c <conf_dir>   directory from which to read configuration files\n"
		"-f <frequency>  delay (in seconds) between file scans\n"
		"-l <log_file>   name of file to write logs for this program\n"
		"-L <severity>   minimum severity of messages to write to log file:\n"
		"                debug, info, notice, warning, error, serious, critical\n"
		"-F              run in foreground, do not daemonize\n"
		"-p <pidfile>    write PID to file, default is /var/run/filechkd.pid\n"
	 , g_prg);

	return;
} // usage

//---------------------------------------- readSpecs
// read filespec config files and create a list of filespecs
// all files in "dirnm" that have ".fs" suffix will be read
void
readSpecs(SpecList &flist, const char *dirnm)
{
	DIR      *dir;
	char     *fn;
	char     *suff;
	char     *path = NULL;
	int      l;
	int      ldir;
	int      lpath = 0;
	FileSpec *fs;
	struct   dirent *dent;

	if(g_log)
		g_log->vlogmsg(Logger::DEBUG, "reading file specs from %s", dirnm);

	dir = opendir(dirnm);
	if(dir)
	{
		ldir = strlen(dirnm);
		while((dent = readdir(dir)) != NULL)
		{
			
			fn   = dent->d_name;
			suff = strrchr(fn, '.');
			if(suff == NULL || strcmp(suff, FILESPEC_SUFF) != 0)
				continue;
			
			// if we need more space for the path, then get it
			l  = strlen(fn);
			if(lpath < ldir + l)
			{
				delete[] path;
				lpath = ldir + l + 2;
				path  = new char[lpath];
			}

			// create the filespec object - this will read the spec file too
			sprintf(path, "%s/%s", dirnm, fn);
			fs = new FileSpec(path, g_log);
			if(fs->isValid())
				flist.push_front(fs);
			else
				delete fs;
		} // while((dent = readdir(dir)) != NULL)
	
		delete[] path;
		closedir(dir);
	} // if(dir)
	else
	{
		if(g_log)
			g_log->vlogmsg(Logger::ERROR, "error opening dir %s", dirnm);
	}

	return;
} // readSpecs

//---------------------------------------- workloop
void
workloop(SpecList &flist, int chkfreq)
{
	time_t tm_start;
	double delay;
	SpecList::iterator iter;
	FileSpec *fspec;

	if(g_log)
		g_log->logmsg(Logger::DEBUG, "entering workloop");

	while(!g_termflag)
	{
		tm_start = time(NULL);

		// for each filespec perform a check to see whether it has aged
		// and once it has get it copied

		iter = flist.begin();
		while(iter != flist.end())
		{
			fspec = *iter;

			if(g_log)
				g_log->vlogmsg(Logger::DEBUG, "%s: working filespec"
				 , fspec->getName());

			if(fspec->isStable())
				fspec->copyFile();

			iter++;
		} // while(iter != flist.end())

		// sleep for the remainder of the time we have before our next check
		delay = chkfreq - difftime(time(NULL), tm_start);
		if(delay <= 0)
			continue;

		sleep((int) delay);
	} // while(!g_termflag)

	if(g_log)
		g_log->logmsg(Logger::DEBUG, "exiting workloop");

	return;
} // workloop

//---------------------------------------- sighandler
extern "C" void
sighandler(int sig)
{
	g_termflag = true;

	return;
} // sighandler

//---------------------------------------- readconf
// read daemon configuration file from fn_cfg and value all of
// the arguments passed in
// TODO: regression tests for this function
void
readconf(int argc, char *argv[], int *chkfreq, bool *foreground, char **fn_log
 , char **logsev, char **fn_pid)
{
	ConfFile *cf;
	char     *attr;
	char     *val;
	char     *fn_cfg = NULL;
	int      opt;

	// first see whether a config file is available, if it is we
	// need to read it first because the command line options must
	// override it later

	for(opt=1; opt<argc-1; opt++)
	{
		if(strcmp(argv[opt], "-c") == 0)
		{
			fn_cfg = new char[strlen(argv[opt+1]) + strlen(FN_CONF) + 3];
			sprintf(fn_cfg, "%s/%s", argv[opt+1], FN_CONF);
			if(access(fn_cfg, R_OK) != 0)
			{
				delete[] fn_cfg;
				fn_cfg = NULL;
			}
		}
	}

	// if there was not one specified on the command line then try the
	// default config file

	if(fn_cfg == NULL)
	{
		fn_cfg = new char[strlen(DIR_CONF) + strlen(FN_CONF) + 3];
		sprintf(fn_cfg, "%s/%s", DIR_CONF, FN_CONF);
		if(access(fn_cfg, R_OK) !=  0)
		{
			delete[] fn_cfg;
			fn_cfg = NULL;
		}
	}

	if(fn_cfg)
	{
		cf = new ConfFile(fn_cfg);

		while(cf->nextAttr(&attr, &val) != EOF)
		{
			strlower(attr);
			if(strlen(attr) > 4)
				attr[3] = '\0';

			if(strcmp("fore", attr) == 0)
			{
				*foreground = strtobool(val);
			}
			else if(strcmp("logf", attr) == 0)
			{
				*fn_log = val;
			}
			else if(strcmp("logs", attr) == 0)
			{
				*logsev = val;
			}
			else if(strcmp("pidf", attr) == 0)
			{
				*fn_pid = val;
			}
			else
			{
				fprintf(stderr, "filechkd: unrecognized option in config file: %s"
				 , attr);
				delete[] val;
			}

			delete[] attr;
		} // while(nextAttr(&attr, &val) != EOF)

		delete cf;
	} // if(fn_cfg)

	return;
} // readconf

// main.cc
