// logger.h
// $Id: logger.h,v 1.4 2000/03/07 02:46:39 gwiley Exp $
// Glen Wiley, <gwiley@ieee.org>
//
// Copyright (c)1999,2000 Glen Wiley
// 
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
//
// simple logging class, provides for "wrapping" logs - limits size of
// log file
//
// MT-safe

#ifndef LOGGER_H
#define LOGGER_H 1

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#ifdef HAVE_PTHREAD_H
# include <pthread.h>
# ifndef USETHREADS
#  define USETHREADS 1
# endif
#endif

#ifdef HAVE_STDARG_H
# include <stdarg.h>
# define USE_VARARGS 1
#else
# ifdef HAVE_SYS_VARARGS_H
#  include <sys/varargs.h>
#  define USE_VARARGS 1
# endif
#endif

//---------------------------------------- Logger
class Logger
{
	public :

	enum sevs { CRITICAL = 100,
					SERIOUS  = 90,
					ERROR    = 80,
					WARNING  = 70,
					NOTICE   = 60,
					INFO     = 50,
					DEBUG    = 40 };

	// these are &ed and |ed so they must be powers of 2
	// LOGDATE    prefixes the log entries with the local date/time
	// LOGPID     adds process id to the log entry
	// LOGSEV     writes the severity to the log entry as an integer
	// LOGSEVLBL  writes the severity to the log entry as a string
	// LOGUTCDT   writes UTC time rather than local time to log
	enum opts { LOGDATE   = 1, 
					LOGPID    = 2,
					LOGSEV    = 4,
					LOGSEVLBL = 8,
					LOGUTCDT  = 16 };

	// if mode is specified then the file will be chmod()'d
	Logger(char *fn=NULL, mode_t mode=0);
	~Logger();

	int  getErrno(void);
	int  getErrmsg(char *buf);
	int  getOptions(void);
	int  wouldLog(int severity);

	void setFileName(char *fn);
	void setMaxSize(unsigned long bytes);
	void setTrimSize(unsigned long bytes);
	void setLabel(char *str);
	void setOptions(int opts);
	void addOptions(int opts) { setOptions(opts | getOptions()); };
	void setMinSeverity(int sev);
	void setMinSeverity(const char *sev);
	int  logmsg(int severity, char *msg);
	// this signature is here for backwards compatability
	int  logmsg(char *msg, int severity=INFO)
	{ logmsg(severity, msg); };
#ifdef USE_VARARGS
	int  vlogmsg(int severity, char *fmt, ...);
#endif

	//-------------------- private :

	private :

	static const unsigned long DEF_MAXBYTES;
	static const unsigned long DEF_TRIMBYTES;
	static const int           COPYBUFFSZ;
	static const int           NUMSEVS;
	static const char          *SEVLABELS[];

	char   *vmsgbuf;
	char   *fn_log;
	char   *msg_prev;
	char   *errmsg;
	char   *label;
	int    errno_sys;
	int    options;
	int    minseverity;
	mode_t perms;
	unsigned long   maxbytes,
						 trimbytes;
#ifdef USETHREADS
	pthread_mutex_t mtx_obj;
	pthread_mutex_t mtx_vmsg;
#endif

	// private methods

	int  trimit(int len);
	int  filecopy(int fd_src, char *fn_dst);
	char *tmpfn(char *fn);
	void setError(int num, char *msg);

	Logger(); // don't use
	
}; // Logger

#endif // LOGGER_H

// logger.h
