// logger.cpp
// $Id: logger.cc,v 1.6 2000/03/18 18:42:40 gwiley Exp $
// Glen Wiley, <gwiley@ieee.org>
//
// Copyright (c)1999,2000 Glen Wiley
// 
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
// 
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.
//
// simple logging class, provides for "wrapping" logs - limits size of
// log file
//
// MT-safe

#if HAVE_CONFIG_H
# include "config.h"
#endif

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#if HAVE_STRING_H
# include <string.h>
#else
# if HAVE_STRINGS_H
#  include <strings.h>
# endif
#endif
#include <sys/stat.h>
#include <sys/types.h>
#include <time.h>
#if HAVE_UNISTD_H
# include <unistd.h>
#endif
#include "logger.h"
#include "str.h"

// default maximum size of the log file in bytes before it gets trimmed
const unsigned long Logger::DEF_MAXBYTES  = 100000;

// default number of bytes to trim from the log file once max size is hit
const unsigned long Logger::DEF_TRIMBYTES = 10000;

// size of the buffer used for copying the file
const int           Logger::COPYBUFFSZ    = 4096;

// labels used for printing and setting severity 
const char *Logger::SEVLABELS[] = {"debug", "debug", "debug", "debug", "debug"
 , "info", "notice", "warning", "error", "serious", "critical"};
// number of entries in the severity label list
const int Logger::NUMSEVS = 11;

// number of bytes to allocate for variable length messages
static const int VMSGBUFLEN = 1024;

//---------------------------------------- Logger
Logger::
Logger(char *fn, mode_t mode)
{
	if(fn)
	{
		fn_log = new char[strlen(fn)+1];
		strcpy(fn_log, fn);
	}
	else
		fn_log = NULL;

	maxbytes  = DEF_MAXBYTES;
	trimbytes = DEF_TRIMBYTES;
	errno_sys = 0;
	options   = 0;
	perms     = mode;
	errmsg    = NULL;
	label     = NULL;
	vmsgbuf   = NULL;
	minseverity = -1;

#ifdef USETHREADS
	pthread_mutex_init(&mtx_obj, NULL);
	pthread_mutex_init(&mtx_vmsg, NULL);
#endif
} // Logger

//---------------------------------------- ~Logger
Logger::
~Logger()
{
#ifdef USETHREADS
	pthread_mutex_lock(&mtx_obj);
#endif

	delete[] fn_log;
	delete[] label;
	delete[] vmsgbuf;

#ifdef USETHREADS
	pthread_mutex_unlock(&mtx_obj);
	pthread_mutex_destroy(&mtx_obj);
	pthread_mutex_destroy(&mtx_vmsg);
#endif
} // ~Logger

//---------------------------------------- setFileName
void Logger::
setFileName(char *fn)
{
#ifdef USETHREADS
	if(pthread_mutex_lock(&mtx_obj) != 0)
		return;
#endif

	delete[] fn_log;
	if(fn)
	{
		fn_log = new char[strlen(fn)+1];
		strcpy(fn_log, fn);
	}
	else
		fn_log = NULL;

#ifdef USETHREADS
	pthread_mutex_unlock(&mtx_obj);
#endif

	return;
} // setFileName

//---------------------------------------- setOptions
void Logger::
setOptions(int opt)
{
#ifdef USETHREADS
	pthread_mutex_lock(&mtx_obj);
#endif

	options = opt;

	// ensure that only one of the date options is set
	if(opt == LOGDATE)
		options ^= LOGUTCDT;
	if(opt == LOGUTCDT)
		options ^= LOGDATE;

#ifdef USETHREADS
	pthread_mutex_unlock(&mtx_obj);
#endif

	return;
} // setOptions

//---------------------------------------- getOptions
int Logger::
getOptions(void)
{
#ifdef USETHREADS
	pthread_mutex_lock(&mtx_obj);
#endif

	int retval = options;

#ifdef USETHREADS
	pthread_mutex_unlock(&mtx_obj);
#endif

	return retval;
} // getOptions

//---------------------------------------- setMinSeverity
// set the lowest severity that will be written to the log
// a log message severity must be >= minseverity in order to
// get logged
void Logger::
setMinSeverity(int sev)
{
#ifdef USETHREADS
	pthread_mutex_lock(&mtx_obj);
#endif

	minseverity = sev;

#ifdef USETHREADS
	pthread_mutex_unlock(&mtx_obj);
#endif

	return;
} // setMinSeverity

//---------------------------------------- setMinSeverity (char)
// set the lowest severity that will be written to the log
// compare the first 4 characters (ignoring case) to severity
// labels in order to determine min severity
void Logger::
setMinSeverity(const char *sev)
{
	char sevstr[5];
	int  i;

	if(isdigit(sev[0]))
	{
		i = (int) strtol(sev, NULL, 10);
	}
	else
	{
		strncpy(sevstr, sev, 4);
		sevstr[4] = '\0';
		strlower(sevstr);

		for(i=NUMSEVS-1; i>-1; i--)
		{
			if(strncmp(sevstr, SEVLABELS[i], 4) == 0)
				break;
		}
	}

	if(i > -1)
		setMinSeverity(i * 10);

} // setMinSeverity

//---------------------------------------- setMaxSize
// if bytes == 0 then log will be allowed to grow forever
void Logger::
setMaxSize(unsigned long bytes)
{
#ifdef USETHREADS
	if(pthread_mutex_lock(&mtx_obj) != 0)
		return;
#endif

	maxbytes = bytes;

#ifdef USETHREADS
	pthread_mutex_unlock(&mtx_obj);
#endif

	return;
} // setMaxSize

//---------------------------------------- setTrimSize
// set number of bytes to trim from log file in case of rewrite
// if bytes == 0 then as few lines as possible will be trimmed
void Logger::
setTrimSize(unsigned long bytes)
{
#ifdef USETHREADS
	if(pthread_mutex_lock(&mtx_obj) != 0)
		return;
#endif

	if(bytes < maxbytes)
		trimbytes = bytes;

#ifdef USETHREADS
	pthread_mutex_unlock(&mtx_obj);
#endif

	return;
} // setTrimSize

//---------------------------------------- logmsg
// write message to log file
// write only if severity is >= minseverity
// return 0 on success (or if severity < minseverity)
//        -1 on failure
int Logger::
logmsg(int severity, char *msg)
{
	int    retval = -1,
			 err    = 0,
			 i,
		    len    = 0;
	char   tmstr[30];
	FILE   *fh;
	time_t clock;

#ifdef USETHREADS
	if(pthread_mutex_lock(&mtx_obj) != 0)
		return -1;
#endif
	
	if(minseverity != -1 && severity < minseverity)
	{
#ifdef USETHREADS
		pthread_mutex_unlock(&mtx_obj);
#endif
		return 0;
	}

	len = (label ? strlen(label) : 0) + (options & LOGSEV ? 5 : 0)
	 + (options & LOGSEVLBL ? 10 : 0) + (options & LOGPID ? 6 : 0) 
	 + (options & (LOGDATE | LOGUTCDT) ? 25 : 0)   + (msg ? strlen(msg) : 0);

	if(msg && len)
	{
		// make sure it will be okay to write the message

		if(access(fn_log, R_OK | W_OK) == 0)
			err = trimit(len);

		// write the message to the log file

		if(!err)
		{
			if((fh=fopen(fn_log, "a+")))
			{
				// adjust the permissions first time writing
				if(perms)
				{
					fchmod(fileno(fh), perms);
					perms = 0;
				}

				if(label)
					fprintf(fh, "%s ", label);

				if(options & LOGSEV)
					fprintf(fh, "(%d) ", severity);
				else if(options & LOGSEVLBL)
				{
					if((i=severity / 10) && i < NUMSEVS && SEVLABELS[i])
						fprintf(fh, "(%s) ", SEVLABELS[i]);
					else
						fprintf(fh, "(%d) ", severity);
				}
	
				if(options & LOGPID)
					fprintf(fh, "pid=%d ", getpid());

				if(options & LOGUTCDT)
				{
					time(&clock);
					strftime(tmstr, 30, "%Y/%m/%d:%H:%M:%S UTC", gmtime(&clock));
					fprintf(fh, "%s ", tmstr);
				}

				if(options & LOGDATE)
				{
					time(&clock);
					strftime(tmstr, 30, "%Y/%m/%d:%H:%M:%S %Z", localtime(&clock));
					fprintf(fh, "%s ", tmstr);
				}

				fprintf(fh, "%s\n", msg);

				fclose(fh);
			}
			else
			{
				setError(errno, "fopen(), create or write log file");
				retval = -1;
			}
		}

	} // if msg
	else
		retval = 0;

#ifdef USETHREADS
	pthread_mutex_unlock(&mtx_obj);
#endif

	return retval;
} // logmsg

//---------------------------------------- getErrno
int Logger::
getErrno(void)
{
	int retval;

#ifdef USETHREADS
	if(pthread_mutex_lock(&mtx_obj) != 0)
		return -1;
#endif
	
	retval = errno_sys;

#ifdef USETHREADS
	pthread_mutex_unlock(&mtx_obj);
#endif

	return retval;
} // getErrno

//---------------------------------------- getErrmsg
// return errno_sys, copy error string into buffer
int Logger::
getErrmsg(char *buf)
{
	int retval;

#ifdef USETHREADS
	if(pthread_mutex_lock(&mtx_obj) != 0)
		return -1;
#endif
	
	retval = errno_sys;
	if(buf)
		strcpy(buf, (errmsg ? errmsg : ""));

#ifdef USETHREADS
	pthread_mutex_unlock(&mtx_obj);
#endif

	return retval;
} // getErrmsg

//---------------------------------------- setError
// set object error variables
// THIS FUNC DOES NOT ACQUIRE THE MUTEX
void Logger::
setError(int num, char *msg)
{
	errno_sys = num;
	delete[] errmsg;
	errmsg = NULL;
	if(msg)
	{
		errmsg = new char[strlen(msg) + 1];
		strcpy(errmsg, msg);
	}

	return;
} // setError

//---------------------------------------- trimit (private)
// rewrite the file - trimming bytes from the beginning as needed
// len = minimum number of bytes to trim
// if len == 0 then trim absolute minimum
// return 0 on success, -1 on error
// this function DOES NOT ACQUIRE THE MUTEX - YOU MUST PROTECT IT
int Logger::
trimit(int len)
{
	struct stat buf_stat;
	char   *fn_tmp = NULL;
	int    fd_src,
			 retval  = -1;

	// see whether we need to trim this file

	if(stat(fn_log, &buf_stat) == 0)
	{
		if(maxbytes >= buf_stat.st_size + len)
			return 0;
	}
	else
	{
		setError(errno, "stat() log file");
		return -1;
	}

	// start at offset of maxbytes-trimbytes and find the next end of line 
	// character (where we will begin our copying)

	if((fd_src=open(fn_log, O_RDONLY)) == -1)
		setError(errno, "open() log file for reading");
	else
	{
		if(lseek(fd_src, buf_stat.st_size-trimbytes, SEEK_SET) == -1)
			setError(errno, "lseek() log file");
		else
		{
			if((fn_tmp=tmpfn(fn_log)) != NULL)
			{
				if(filecopy(fd_src, fn_tmp) == 0)
				{
					if(unlink(fn_log))
						setError(errno, "unlink() log file");
					else
					{
						if(rename(fn_tmp, fn_log))
							setError(errno, "rename() temp file");
						else
							retval = 0;
					}

				} // filecopy

				delete[] fn_tmp;

			} // if tmpnam()

			close(fd_src);

		} // if lseek else

	} // if fd_src else

	return retval;
} // trimit

//-------------------------------------- filecopy (private)
// copy file from character following the next end of line character
// to a separate file
// return 0 on success, -1 on failure
int Logger::
filecopy(int fd_src, char *fn_dst)
{
	int   retval = 0,
			done   = 0,
			bytesr;
	FILE  *fh_src = NULL,
			*fh_dst = NULL;
	char  *buff,
			*p      = NULL;
	struct stat stat_s, stat_d;

	buff = new char[COPYBUFFSZ];

	if(fstat(fd_src, &stat_s) != 0)
	{
		setError(errno, "fstat() log file");
		retval = -1;
	}

	if(!retval && stat(fn_dst, &stat_d) == 0)
	{
		if(stat_s.st_dev == stat_d.st_dev && stat_s.st_ino == stat_d.st_ino)
		{
			setError(errno, "log file and dest file are the same");
			retval = EEXIST;
		}
	}

	if(!retval)
	{
	 	if((fh_dst=fopen(fn_dst, "w+")) != NULL)
		{
			// look for next end of line char in source

			done = 0;
			while(!done)
			{
				if((bytesr = read(fd_src, buff, COPYBUFFSZ)) == -1)
				{
					setError(errno, "read() from log file");
					done   = 1;
					retval = -1;
				}
				else
				{
					if((p = (char *) memchr(buff, '\n', bytesr))) 
					{
						done   = 1;
						bytesr = bytesr - (++p - buff);
					}
				}
			} // while !done

			// copy the remaining data to the destination file

			if(!retval)
			{
				if(fwrite(p, 1, bytesr, fh_dst) != bytesr)
				{
					setError(errno, "fwrite() to dest file");
					retval = -1;
				}

				while(bytesr > 0 && !retval)
				{
					if((bytesr = read(fd_src, buff, COPYBUFFSZ)) == -1)
					{
						setError(errno, "read() from log file");
						retval = -1;
					}

					if(fwrite(buff, 1, bytesr, fh_dst) != bytesr)
					{
						setError(errno, "fwrite() to dest file");
						retval = -1;
					}
				}

			} // if !retval

		} // if fh_dst
		else
		{
			setError(errno, "open dest file for writing");
			retval = -1;
		}

		if(fh_dst != NULL)
			fclose(fh_dst);

	} // if !retval

	delete[] buff;

	return retval;
} // filecopy

//---------------------------------------- setLabel
// set label to be prepenede to log messages
void Logger::
setLabel(char *str)
{
#ifdef USETHREADS
	if(pthread_mutex_lock(&mtx_obj) != 0)
		return;
#endif

	delete[] label;
	if(str)
	{
		label = new char[strlen(str)+1];
		strcpy(label, str);
	}
	else
		label = NULL;

#ifdef USETHREADS
	pthread_mutex_unlock(&mtx_obj);
#endif

	return;
} // setMaxSize

//---------------------------------------- tmpfn (private)
// generate unique temp filename with prefix 'fn'
// return allocated file name or NULL on error
char *Logger::
tmpfn(char *fn)
{
	char *newfn = NULL,
		  *suffix;
	int  done   = 0,
		  i      = 0;

	newfn = new char[strlen(fn) + 10];
	strcpy(newfn, (fn ? fn : "/tmp/tmp"));
	suffix = newfn + strlen(newfn);

	while(!done)
	{
		sprintf(suffix, ".tmp.%0.4x", i);
		if(access(newfn, F_OK) != 0 || ++i == 1000)
			done = 1;
	}

	if(i == 1000)
	{
		delete[] newfn;
		newfn = NULL;
		setError(EEXIST, "no temporary file available");
	}

	return newfn;
} // tmpfn

//---------------------------------------- wouldLog
// return 1 if the specified severity would actually log
int Logger::
wouldLog(int severity)
{
#ifdef USETHREADS
	pthread_mutex_lock(&mtx_obj);
#endif

	int retval = 1;

	if(minseverity != -1 && severity < minseverity)
		retval = 0;

#ifdef USETHREADS
	pthread_mutex_unlock(&mtx_obj);
#endif

	return retval;
} // wouldLog

#ifdef USE_VARARGS

//---------------------------------------- vlogmsg
int Logger::
vlogmsg(int severity, char *fmt, ...)
{
	int     retval = 0;
	va_list ap;

#ifdef USETHREADS
	pthread_mutex_lock(&mtx_vmsg);
#endif

	if(vmsgbuf == NULL)
	{
		vmsgbuf = new char[VMSGBUFLEN + 1];
		if(vmsgbuf == NULL)
			return errno;
	}

	va_start(ap, fmt);

#ifdef HAVE_VSNPRINTF
	vsnprintf(vmsgbuf, VMSGBUFLEN, fmt, ap);
	vmsgbuf[VMSGBUFLEN] = '\0';
#else
	vsprintf(vmsgbuf, fmt, ap);
#endif // HAVE_VSNPRINTF

	va_end(ap);

	logmsg(severity, vmsgbuf);

#ifdef USETHREADS
	pthread_mutex_unlock(&mtx_vmsg);
#endif

	return retval;
} // vlogmsg

#endif // USE_VARARGS

// logger.cpp
