// filespec.cc
//
// TODO: track stats on stab/lock files to detect stale locks
// 

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#include <errno.h>
#ifdef HAVE_SYS_TYPES_H
# include <sys/types.h>
#endif
#ifdef HAVE_SYS_STAT_H
# include <sys/stat.h>
#endif
#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif
#ifdef HAVE_GRP_H
#include <grp.h>
#endif
#ifdef HAVE_PWD_H
# include <pwd.h>
#endif
#include <stdlib.h>
#ifdef HAVE_STRING_H
# include <string.h>
#else
# ifdef HAVE_STRINGS_H
#  include <strings.h>
# endif
#endif
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_TIME_H
# include <time.h>
#endif
#include "filespec.h"
#include "conffile.h"
#include "logger.h"

extern "C" int filecopy(const char *fn_src, const char *fn_dst, int setown
 , uid_t uid, gid_t gid, int setmode, mode_t mode);

// default file age minimum
static const int DFLT_AGE_MIN = 120;

//---------------------------------------- FileSpec
FileSpec::
FileSpec(const char *fn_conf, Logger *log)
{
	int err;

	name       = NULL;
	fn_src     = NULL;
	fn_dst     = NULL;
	fn_sta     = NULL;
	fn_lck     = NULL;
	age_min    = DFLT_AGE_MIN;
	uid        = 0;
	gid        = 0;
	mode       = -1;
	nbackups   = 0;
	mail       = NULL;
	isvalid    = false;
	rmsrc      = false;
	rmsta      = false;
	rmlck      = false;
	chgown     = false;
	copysz_src = 0L;
	copytm_src = 0L;
	copysz_dst = 0L;
	copytm_dst = 0L;
	locked     = false;
	this->log  = log;

	// zero out the run-time file check data
	reset();

	err = readConf(fn_conf);
	if(err != 0 || name == NULL || fn_src == NULL || fn_dst == NULL)
	{
		isvalid = false;

		if(log)
			log->vlogmsg(Logger::INFO, "invalid filespec in %s, %s", fn_conf
			 , (err == 0 ? "incomplete spec" : strerror(err)));
	}
	else
	{
		if(log)
		{
			log->vlogmsg(Logger::INFO, "%s: valid filespec read from %s"
			 , name, fn_conf);
		}
		isvalid = true;
	}

} // FileSpec

//---------------------------------------- ~FileSpec
FileSpec::
~FileSpec()
{
	delete[] name;
	delete[] fn_src;
	delete[] fn_dst;
	delete[] mail;
	delete[] fn_lck;
	delete[] fn_sta;

} // ~FileSpec

//---------------------------------------- readConf (private)
// read the configuration file - it is in the form of
// attribute value pairs separated by whitespace, case in attribute
// names is ignored and only the first 4 chars are checked
//  SOURCE <file to check>
//  DEST <file to copy source to>
//  LOCKFILE <if this file exists then consider the source locked/unstable>
//  STABFILE <if this file exists then consider the source stable>
//  RMSOURCE <set to 'true' to remove the source after a confirmed copy>
//  RMLOCK <set to 'true' to remove the lock file after a confirmed copy>
//  RMSTAB <set to 'true' to remove the stab file after a confirmed copy>
//  MAIL <email address to notify>
//  USER <username to chown DEST>
//  GROUP <groupname to chown DEST>
//  MODE  <permissions in octal>
//  NBACKUPS <number of DEST backups to keep, 0 = none>
//  MINAGE   <minimum age of SOURCE to be considered stable>
// 
// return 0 on success, non-zero on format or file errors
// TODO: translate symbolic constants for MODE
// TODO: accept uid/gid in place of names
// TODO: parse mail into multiple mail addresses
int FileSpec::
readConf(const char *fn)
{
	int    retval = 0;
	bool   uidchg = false;
	bool   gidchg = false;
	char   *att   = NULL;
	char   *val   = NULL;
	char   *p;
	struct passwd *pwent = NULL;
	struct group  *grent = NULL;

	if(log)
		log->vlogmsg(Logger::DEBUG, "reading filespec %s", fn);

	uid = getuid();
	gid = getgid();

	ConfFile cf(fn);

	while(cf.hadErrors() == false && cf.nextAttr(&att, &val) != EOF)
	{
		if(strncasecmp(att, "name", 4) == 0)
		{
			name = val;
		}
		if(strncasecmp(att, "sour", 4) == 0)
		{
			fn_src = val;
		}
		else if(strncasecmp(att, "dest", 4) == 0)
		{
			fn_dst = val;
		}
		else if(strncasecmp(att, "lock", 4) == 0)
		{
			fn_lck = val;
		}
		else if(strncasecmp(att, "stab", 4) == 0)
		{
			fn_sta = val;
		}
		else if(strncasecmp(att, "mail", 4) == 0)
		{
			mail = val;
		}
		else if(strncasecmp(att, "user", 4) == 0)
		{
			chgown = true;
			pwent  = getpwnam(val);
			if(pwent != NULL)
			{
				if(uid != pwent->pw_uid)
					uidchg = true;
				uid = pwent->pw_uid;
			}
			else
			{
				if(log)
					log->vlogmsg(Logger::ERROR, "bad user name '%s'", val);
				retval = EINVAL;
			}

			delete[] val;
		}
		else if(strncasecmp(att, "grou", 4) == 0)
		{
			chgown = true;
			grent  = getgrnam(val);
			if(grent != NULL)
			{
				if(gid != grent->gr_gid)
					gidchg = true;
				gid = grent->gr_gid;
			}
			else
			{
				if(log)
					log->vlogmsg(Logger::ERROR, "bad group name '%s'", val);
				retval = EINVAL;
			}

			delete[] val;
		}
		else if(strncasecmp(att, "mode", 4) == 0)
		{
			mode = (int) strtol(val, NULL, 8);
		}
		else if(strncasecmp(att, "nbac", 4) == 0)
		{
			nbackups = (int) strtol(val, NULL, 10);
			delete[] val;
		}
		else if(strncasecmp(att, "mina", 4) == 0)
		{
			age_min = (int) strtol(val, NULL, 10);
			delete[] val;
		}
		else if(strncasecmp(att, "rmso", 4) == 0)
		{
			if(strcasecmp(val, "true") == 0)
				rmsrc = true;
			else
				rmsrc = false;
			delete[] val;
		}
		else if(strncasecmp(att, "rmst", 4) == 0)
		{
			if(strcasecmp(val, "true") == 0)
				rmsta = true;
			else
				rmsta = false;
			delete[] val;
		}
		else if(strncasecmp(att, "rmlo", 4) == 0)
		{
			if(strcasecmp(val, "true") == 0)
				rmlck = true;
			else
				rmlck = false;
			delete[] val;
		}
		else
		{
			if(log)
				log->vlogmsg(Logger::ERROR, "bad attribute name '%s'", att);

			retval = EINVAL;
			delete[] val;
		}

		delete[] att;
	} // while(cf.hadErrors() == false && cf.nextAttr(&att, &val) != EOF)

	if(cf.hadErrors())
	{
		if(log)
			log->vlogmsg(Logger::ERROR, "problems encountered reading filespec, "
			 "possible due to %s", strerror(errno));
	}

	// if we picked up a user id but not a group id then default to
	// that user's primary group
	if(pwent != NULL && uidchg && !gidchg)
		gid = pwent->pw_gid;

	// use the basename of the file with no .fs suffix for the filespec 
	// name if none was specified
	if(retval == 0 && name == NULL)
	{
		p = strrchr(fn, '/');
		if(p && *p)
			p++;
		else
			p = (char *) fn;

		name = new char[strlen(p) + 1];
		if(name == NULL)
		{
			if(log)
				log->vlogmsg(Logger::CRITICAL, "error allocating storage, %s"
				 , strerror(errno));
			retval = errno;
		}
		else
		{
			strcpy(name, p);
			p = strrchr(name, '.');
			if(p && *p)
				*p = '\0';
		}
	}

	return retval;
} // readConf

//---------------------------------------- reset (private)
// reset data collected during stability checks
void FileSpec::
reset(void)
{
	prevtm  = 0L;
	prevsz  = 0L;
} // reset

//---------------------------------------- isStable
// check the current source file against data collected last time
// and return true if the file has aged enough to be considered stable
bool FileSpec::
isStable(void)
{
	bool   isstable = false;
	struct stat statbuf;

	// if a stable flag file was defined and it exists then we are done 
	if(fn_sta)
	{
		if(access(fn_sta, F_OK) == 0)
		{
			if(log)
				log->vlogmsg(Logger::DEBUG
				 , "%s: flag file %s detected - assumed stable"
				 , name, fn_sta);
			return(true);
		}

		if(age_min == 0)
			return(false);
	}

	// if a lock file was defined and it exists then we are done
	// check size and date/time
	if(fn_lck && access(fn_lck, F_OK) == 0)
	{
		if(log)
			log->vlogmsg(Logger::DEBUG
			 , "%s: lock file %s detected - assumed unstable"
			 , name, fn_lck);
		return(false);
	}

	// check the files vital stats
	if(stat(fn_src, &statbuf) != 0)
	{
		reset();
		isstable = false;
		if(log)
			log->vlogmsg(Logger::DEBUG
			 , "%s: unable to stat file", name);
	}
	else
	{
		if(prevtm == statbuf.st_mtime && prevsz == statbuf.st_size
		 && difftime(time(NULL), prevtm) >= age_min)
		{
			isstable = true;
			if(log)
				log->vlogmsg(Logger::DEBUG
				 , "%s: time and size stable", name);
		}
		prevtm = statbuf.st_mtime;
		prevsz = statbuf.st_size;
	}

	// TODO: check using fuser or proc filesystem for open writers

	return isstable;
} // isStable

//---------------------------------------- makeBackups
// handle destination file backups
int FileSpec::
makeBackups(const char *fn)
{
	int i;
	int retval   = 0;
	int last     = -1;
	char *fn_new = NULL;
	char *fn_bk  = NULL;

	if(nbackups > 0)
	{
		fn_new = new char[strlen(fn) + 10];
		if(fn_new == NULL)
		{
			if(log)
				log->vlogmsg(Logger::CRITICAL, "error allocating storage, %s"
				 , strerror(errno));
			return errno;
		}
		fn_bk  = new char[strlen(fn) + 10];
		if(fn_bk == NULL)
		{
			if(log)
				log->vlogmsg(Logger::CRITICAL, "error allocating storage, %s"
				 , strerror(errno));
			return errno;
		}

		// figure out the number of the last available backup
		for(i=0; i<nbackups; i++)
		{
			sprintf(fn_bk, "%s.%d", fn, i);
			if(access(fn_bk, F_OK) == 0)
				last = i;
			else
				break;
		}

		// if backups exist then we need to worry about shuffling them
		if(last > -1)
		{
			// if we have hit the max then remove the oldest backup
			// TODO: detect failure
			if(last == nbackups - 1)
			{
				if(log)
					log->vlogmsg(Logger::INFO, "%s: removing %s", name, fn_bk);

				unlink(fn_bk);
				last--;
			}

			// rename all the existing backups
			for(i=last; i>=0; i--)
			{
				sprintf(fn_bk,  "%s.%d", fn, i);
				sprintf(fn_new, "%s.%d", fn, i+1);

				if(log)
					log->vlogmsg(Logger::INFO, "%s: moving %s to %s", name, fn_bk
					 , fn_new);

				if(rename(fn_bk, fn_new) != 0)
				{
					if(log)
						log->vlogmsg(Logger::ERROR, "%s: rename failed, %s"
						 , name, strerror(errno));
				}
			}
		} // if(last > -1)

		// backup the existing destination file if it exists
		if(access(fn, F_OK) == 0)
		{
			sprintf(fn_new, "%s.0", fn);
			if(log)
				log->vlogmsg(Logger::INFO, "%s: moving %s to %s", name, fn, fn_new);
			if(rename(fn, fn_new) != 0)
			{
				if(log)
					log->vlogmsg(Logger::ERROR, "%s: rename failed, %s"
					 , name, strerror(errno));
			}
		}

		delete[] fn_new;
		delete[] fn_bk;
	} // if(nbackups > 0)

	return retval;
} // makeBackups

//---------------------------------------- copyFile
// execute the copy from source to dest
// and follow any special instructions per config values
// return 0 on success
// TODO: fork to do this in case it takes longer than expected
// TODO: check for system errors and report them properly
int FileSpec::
copyFile(void)
{
	int    err;
	int    retval = 0;
	char   *cmd   = NULL;
	struct stat stat_src;
	struct stat stat_dst;

	// if the dest matches what was previously copied then
	// we have nothing to do
	if(stat(fn_src, &stat_src) == 0)
	{
		if(stat(fn_dst, &stat_dst) == 0)
		{
			if(stat_src.st_size == copysz_src
			 && stat_src.st_mtime == copytm_src
			 && stat_dst.st_size  == copysz_dst
			 && stat_dst.st_mtime == copytm_dst)
			{
				if(log)
					log->vlogmsg(Logger::DEBUG
					 , "%s: copy, source and dest have not changed", name);
				return 0;
			}
		}
	}

	try
	{
		err = locksource();
		if(err != 0)
			throw err;

		// backup existing destination files
		if((err = makeBackups(fn_dst)) != 0)
			throw err;

		// copy the file
		if(log)
			log->vlogmsg(Logger::INFO, "%s: copy %s to %s", name, fn_src, fn_dst);

		// copy the file - this takes care of owner/mode changes
		err = filecopy(fn_src, fn_dst, (chgown ? 1 : 0), uid, gid
		 , (mode != -1 ? 1 : 0), mode);
		if(err != 0)
		{
			if(log)
				log->vlogmsg(Logger::ERROR
				 , "%s: copy, error copying %s to %s, %s"
				 , name, fn_dst, fn_src, strerror(err));
			throw err;
		}

		// stat dest to make sure same length as source
		// TODO: CRC
		if(stat(fn_dst, &stat_dst) != 0 || stat(fn_src, &stat_src) != 0)
		{
			if(log)
				log->vlogmsg(Logger::ERROR
				 , "%s: copy, unable to stat %s or %s after copy"
				 , name, fn_dst, fn_src);
			throw err;
		}

		if(stat_dst.st_size != stat_src.st_size)
		{
			if(log)
				log->vlogmsg(Logger::ERROR
				 , "%s: copy, sizes do not match %s and %s after copy"
				 , name, fn_dst, fn_src);
			throw err;
		}
		else
		{
			copysz_src = stat_src.st_size;
			copytm_src = stat_src.st_mtime;
			copysz_dst = stat_dst.st_size;
			copytm_dst = stat_dst.st_mtime;
		}

		// remove source
		if(rmsrc)
		{
			err = unlink(fn_src);
			if(log)
			{
				if(err == 0)
				{
					log->vlogmsg(Logger::INFO, "%s: unlink %s", name, fn_src);
				}
				else
				{
					log->vlogmsg(Logger::ERROR, "%s: unlink %s failed: %s"
					 , name, fn_src, strerror(errno));
				}
			}
		}

		// remove lock file
		if(rmlck)
		{
			err = unlink(fn_lck);
			if(log)
			{
				if(err == 0)
				{
					log->vlogmsg(Logger::INFO, "%s: unlink %s", name, fn_lck);
				}
				else
				{
					log->vlogmsg(Logger::ERROR, "%s: unlink %s failed: %s"
					 , name, fn_lck, strerror(errno));
				}
			}
		}

		// remove stability flag file
		if(rmsta)
		{
			err = unlink(fn_sta);
			if(log)
			{
				if(err == 0)
					log->vlogmsg(Logger::INFO, "%s: unlink %s", name, fn_sta);
				else
					log->vlogmsg(Logger::ERROR, "%s: unlink %s failed: %s"
					 , name, fn_sta, strerror(errno));
			}
		}

	}
	catch(int ex)
	{
		retval = ex;
	}

	if(locked)
		unlocksource();

	delete[] cmd;
	return retval;
} // copyFile

//---------------------------------------- locksource
// mode == 'l' to lock, 'u' to unlock
// if a lockfile was specified in the filespec then it will be
// used to lock/unlock the file in addition to lockf()
// return 0 on success, errno on error
int FileSpec::
locksource(char mode)
{
	int  retval = 0;
	int  fd;
	char buf[10];

	if((mode == 'u' && !locked) || (mode == 'l' && locked))
	{
		if(log)
			log->vlogmsg(Logger::ERROR, "%s: lock out of sync (%c)", name, mode);
		return EAGAIN;
	}

	// unlink/create lock file
	if(fn_lck)
	{
		if(mode == 'u')
		{
			if(unlink(fn_lck) != 0)
			{
				retval = errno;
				if(log)
					log->vlogmsg(Logger::ERROR, "%s: unlink lock file %s failed, %s"
					 , name, fn_lck, strerror(errno));
			}
			else
			{
				locked = false;
				if(log)
					log->vlogmsg(Logger::DEBUG, "%s: unlink lock file %s"
					 , name, fn_lck);
			}
		}
		else if(mode == 'l')
		{
			fd = open(fn_lck, O_RDWR|O_CREAT|O_EXCL
			 , S_IRUSR|S_IWUSR|S_IRGRP|S_IROTH);
			if(fd == -1)
			{
				if(log)
					log->vlogmsg(Logger::ERROR
					 , "%s: failed to create lock file %s, %s"
					 , name, fn_lck, strerror(errno));
			}
			else
			{
				locked = true;
				sprintf(buf, "%d\n", getpid());
				write(fd, buf, strlen(buf));
				close(fd);
				if(log)
					log->vlogmsg(Logger::DEBUG, "%s: created lock file %s"
					 , name, fn_lck);
			}
		}
		else
			retval = EINVAL;
	} // if(fn_lck)

	// TODO: implement fcntl()/lockf() locking

	return retval;
} // locksource

// filespec.cc
