/*
 * Copyright 1996 by E. Toernig (froese@gmx.de)
 */
/*
    scan directory tree
*/

#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <dirent.h>
#include <string.h>

#include <etlib/generic.h>
#include <etlib/error.h>
#include <etlib/xmalloc.h>
#include <etlib/icmp.h>

#include "mk_ff.h"
#include "hashs.h"
#include "emit.h"



static char path[PATH_LEN];
static char *pathend;
static char *printpath;

static int cwd_fd;
static int nomount;
static umode_t base_dev;


static void
set_path(char *base, int nobase)
{
    if (base == 0 || *base == 0)
	base = ".", nobase = 1;

    pathend = stpcpy(path, base);
    if (pathend[-1] != '/')
	pathend = stpcpy(pathend, "/");
    printpath = nobase ? pathend : path;
}

static char *
add_path(char *file)
{
    char *x;
    
    x = pathend;

    if (pathend[-1] != '/' && *file != '/')
	pathend = stpcpy(pathend, "/");
    pathend = stpcpy(pathend, file);

    return x;
}

static void
del_path(char *old_path)
{
    pathend = old_path;
    *pathend = 0;
}



static int
ent_comp(const void *b, const void *a)
{
    struct entry *e1 = *(struct entry **)a;
    struct entry *e2 = *(struct entry **)b;

    /* directories to the end */
    if (S_ISDIR(e1->mode) != S_ISDIR(e2->mode))
	return S_ISDIR(e1->mode) ? 1 : -1;

    /* 'A' before 'a' before 'B' ... */
    return striicmp(e1->name, e2->name);
}


static struct entry *
get_dir(char *path)
{
    struct entry *dirlist = 0;
    int n = 0;
    struct dirent *de;
    DIR *dir;
    struct stat st[1];

    if (path == 0 || *path == 0)
	path = ".";

    if (test_ignore(path))
	return 0;

    if (dir = opendir(path))
    {
	if (fchdir(dirfd(dir)) != -1) /* dirfd() is system dependant */
	{
	    while (de = readdir(dir))
	    {
		if (de->d_name[0] == '.')
		    if (de->d_name[1] == 0 || (de->d_name[1] == '.' && de->d_name[2] == 0))
			continue;

		if (test_ignore(de->d_name))
		    continue;

		if (lstat(de->d_name, st) == -1)
		    continue;	/* ignore vanished entries */

		if (nomount && st->st_dev != base_dev)
		    continue;

		if (S_ISDIR(st->st_mode) || S_ISREG(st->st_mode))
		{
		    dirlist = alloc_entry(dirlist, de->d_name, st);
		    n++;
		}
	    }
	    if (n > 1)
	    {
		struct entry *tmp[n]; /* GCC feature! */
		struct entry **e;

		for (e = tmp; dirlist; dirlist = dirlist->next)
		    *e++ = dirlist;
		qsort(tmp, n, sizeof(struct entry *), ent_comp);
		for (e = tmp; n--; e++)
		    (*e)->next = dirlist, dirlist = *e;
	    }
	    fchdir(cwd_fd);
	}
	else
	    ioerror(path);

	closedir(dir);
    }
    else
	ioerror(path);

    return dirlist;
}

static void
scan_dir(struct entry *base)
{
    struct entry *dir, *next;

    if (dir = get_dir(path))
    {
	emit_enter_dir(base, path);
	while (dir)
	{
	    char *path_save = add_path(dir->name);

	    if (S_ISDIR(dir->mode))
		scan_dir(dir);
	    else if (not test_ignore(printpath))
		emit_file(dir, printpath);

	    del_path(path_save);

	    next = dir->next;
	    xfree(dir);
	    dir = next;
	}
	emit_leave_dir(base, path);
    }
}

void
process_dir(char *base, int mflg, int nobase)
{
    struct stat st[1];
    struct entry *dir;

    set_path(base, nobase);

    if (stat(path, st) == 0)
    {
	cwd_fd = open(".", O_RDONLY);
	if (cwd_fd != -1)
	{
	    nomount = mflg;
	    base_dev = st->st_dev;
	    dir = alloc_entry(0, path, st);
	    scan_dir(dir);
	    xfree(dir);
	    close(cwd_fd);
	}
	else
	    ioerror(".");
    }
    else
	ioerror(path);
}
