/*
 * Copyright 1996 by E. Toernig (froese@gmx.de)
 */
/*
    Create a table of contents from a directory tree.
*/

#include <sys/stat.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

#include <etlib/generic.h>
#include <etlib/prgname.h>
#include <etlib/error.h>
#include <etlib/xmalloc.h>
#include <etlib/getopt.h>

#include "mk_ff.h"
#include "hashs.h"
#include "emit.h"

struct tm *current_tm = 0;

extern void process_dir(char *dir, int nomount, int nobase);
extern void process_tar(void);
extern void process_cpio(void);

void
usage(FILE *fp)
{
    fprintf(fp, "Usage: %s [-bcefhiImosv] [base-dirs]\n", prgname);
    fprintf(fp, "\t-b\tStrip base-dir from output.\n");
    fprintf(fp, "\t-c str\tSet comment string.\n");
    fprintf(fp, "\t-e days\tDefine 'file expired' warning.\n");
    fprintf(fp, "\t-f fmt\tSelect input format (dir/tar/cpio).\n");
    fprintf(fp, "\t-h\tShow this help.\n");
    fprintf(fp, "\t-i name\tIgnore entries with name 'name'.\n");
    fprintf(fp, "\t-I\tStore inode-numbers in database.\n");
    fprintf(fp, "\t-m\tDescend into mounted filesystems.\n");
    fprintf(fp, "\t-o name\tWrite output into file 'name'.\n");
    fprintf(fp, "\t-s num\tSet security level (%d-%d).\n", S_FIRST, S_LAST);
    fprintf(fp, "\t-v\tShow program version and exit.\n");
    exit(fp != stdout ? 1 : 0);
}

int
main(int argc, char **argv)
{
    char *info = 0;
    char *outfile = 0;
    char *fmt = 0;
    int fields = F_DATE | F_SIZE;
    int sec = -1;
    int nobase = 0;
    int nomount = 1;
    int exp_days = 0;
    time_t t;
    int opt;

    /* setup timezone value */
    time(&t);
    current_tm = localtime(&t);

    while ((opt = getopt(argc, argv, "Ibhmvc:e:f:i:o:s:")) != EOF)
    {
	switch (opt)
	{
	    case 'I':	/* store inode in output file */
		fields |= F_INODE;
		break;
	    case 'b':	/* don't show base-directory in output */
	    	nobase = 1;
		break;
	    case 'c':	/* add comment line */
		info = optarg;
		break;
	    case 'e':	/* expire days */
	    	exp_days = atoi(optarg);
		break;
	    case 'f':	/* data format */
	    	fmt = optarg;
		break;
	    case 'i':	/* ignore name */
		add_ignore(optarg);
		break;
	    case 'm':	/* descend into mounted directories */
	    	nomount = 0;
		break;
	    case 'o':	/* output filename */
		outfile = optarg;
		break;
	    case 's':	/* security level */
		sec = atoi(optarg);
		break;

	    case 'v':
	    	printf("%s v%d.%d --- Copyright 1998 by E. Toernig.\n",
			prgname, VERSION, REVISION);
		exit(0);
	    case 'h':
	    	usage(stdout);
	    default:
		usage(stderr);
	}
    }

    if (fmt == 0 || streq(fmt, "dir"))
    {
	if (sec == -1)
	    sec = S_FULL;

	emit_open(outfile, sec);
	emit_reset(exp_days, fields, info);

	if (optind == argc)
	    process_dir("", nomount, nobase);
	else while (optind < argc)
	    process_dir(argv[optind++], nomount, nobase);

	emit_close();
    }
    else if (streq(fmt, "tar") || streq(fmt, "cpio"))
    {
	if (sec == -1)
	    sec = S_ACCESS;

	if (nobase)
	    fatal("%s-format: -b option invalid.", fmt);
	if (nomount == 0)
	    fatal("%s-format: -m option invalid.", fmt);
	if (fields & F_INODE)
	    fatal("%s-format: -I option invalid.", fmt);
	if (sec != S_NONE && sec != S_ACCESS)
	    fatal("%s-format: only -s%d and -s%d valid.", fmt, S_NONE, S_ACCESS);
	if (optind != argc)
	    fatal("%s-format: '%s ...' not valid. stdin is read.", fmt, argv[optind]);

	emit_open(outfile, sec);
	emit_reset(exp_days, fields, info);
	if (*fmt == 't')
	    process_tar();
	else
	    process_cpio();
	emit_close();
    }
    else
	fatal("%s: unknown format (dir/tar/cpio).", fmt);

    exit(0);
}
