/*
 * Copyright 1996 by E. Toernig (froese@gmx.de)
 */
/*
    Various hashs for uid/gid/name/ignore caching.
*/


#include <sys/types.h>
#include <pwd.h>
#include <grp.h>

#include <etlib/generic.h>
#include <etlib/xmalloc.h>
#include <etlib/hashkey.h>

#include "emit.h"

#define UHASHSIZE 61		/* uid/gid hash size */
#define NHASHSIZE 61		/* name hash size */
#define IHASHSIZE 61		/* ignore hash size */

struct id_cache
{
    struct id_cache *next;
    struct name_cache *name;
    uid_t id;
};

struct name_cache
{
    struct name_cache *next;
    int nid;
    char name[1];
};

struct ignore_cache
{
    struct ignore_cache *next;
    char name[1];
};

static struct id_cache *uid_table[UHASHSIZE];
static struct id_cache *gid_table[UHASHSIZE];
static struct name_cache *name_table[NHASHSIZE];
static struct ignore_cache *ignore_table[IHASHSIZE];
int have_ignore = 0;

static char uid_read = 0;
static char gid_read = 0;



static struct name_cache *
add_name(char *name)
{
    int h = hash_key(name) % NHASHSIZE;
    struct name_cache *p;

    for (p = name_table[h]; p; p = p->next)
	if (streq(p->name, name))
	    return p;
    p = xmalloc(sizeof(*p) + strlen(name));
    p->next = name_table[h];
    name_table[h] = p;
    p->nid = -1;
    strcpy(p->name, name);
    return p;
}

static struct name_cache *
add_id(uid_t id, char *name, struct id_cache **tab)
{
    int h = id % UHASHSIZE;
    struct id_cache *p;
    char buf[32];

    for (p = tab[h]; p; p = p->next)
	if (p->id == id)
	    return p->name;
    p = xmalloc(sizeof(*p));
    p->next = tab[h];
    tab[h] = p;
    p->id = id;
    if (name == 0)
	sprintf(name = buf, "%d", id);
    return p->name = add_name(name);
}

static inline int
name2nid(struct name_cache *p)
{
    if (p->nid == -1)
	p->nid = emit_name(p->name);
    return p->nid;
}



int
make_name(char *name)
{
    return name2nid(add_name(name));
}

int
make_uid(uid_t id)
{
    if (not uid_read)
    {
	struct passwd *pw;

	while (pw = getpwent())
	    add_id(pw->pw_uid, pw->pw_name, uid_table);
	endpwent();
	uid_read = 1;
    }
    return name2nid(add_id(id, 0, uid_table));
}

int
make_gid(gid_t id)
{
    if (not gid_read)
    {
	struct group *gr;

	while (gr = getgrent())
	    add_id(gr->gr_gid, gr->gr_name, gid_table);
	endgrent();
	gid_read = 1;
    }
    return name2nid(add_id(id, 0, gid_table));
}



void
add_ignore(char *str)
{
    int h = hash_key(str) % IHASHSIZE;
    struct ignore_cache *p;

    for (p = ignore_table[h]; p; p = p->next)
	if (streq(p->name, str))
	    return;
    have_ignore++;
    p = xmalloc(sizeof(*p) + strlen(str));
    p->next = ignore_table[h];
    ignore_table[h] = p;
    strcpy(p->name, str);
}

int
test_ignore(char *str)
{
    if (have_ignore)
    {
	int h = hash_key(str) % IHASHSIZE;
	struct ignore_cache *p;

	for (p = ignore_table[h]; p; p = p->next)
	    if (streq(p->name, str))
		return 1;
    }
    return 0;
}
