/*
 * Copyright 1996 by E. Toernig (froese@gmx.de)
 */
#include <stdio.h>
#include <time.h>
#include <string.h>
#include <fcntl.h>
#include <pwd.h>
#include <grp.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <etlib/generic.h>
#include <etlib/getopt.h>
#include <etlib/prgname.h>
#include <etlib/error.h>
#include <etlib/cattr.h>
#include <etlib/xmalloc.h>
#include <etlib/fmatch.h>
#include <etlib/hashkey.h>
#include <etlib/dirname.h>
#include <etlib/basename.h>
#include <etlib/expr.h>
#include <etlib/date.h>

#include "mk_ff.h"
#include "emit.h"

struct file_list
{
    struct file_list *next;
    char *group;
    char *file;
};

struct in_ptr
{
    u8 *ptr;
    u8 *end;
    int fd;
    int eof;
    int access;
    time_t date;
    char *name;
    char *path;
    u8 buf[4096];
};

#define HASH_SIZE	61

struct id_hash
{
    struct id_hash *next;
    u8 flags;
    char name[1];
};

struct file_param
{
    u32 size;		/* size of file in bytes */
    u32 Date;		/* date of file (YYYYMMDD) */
    u32 inode;		/* inode number of file */
    u32 day;		/* day of month of file (1-31) */
    u32 month;		/* month of year of file (1-12) */
    u32 year;		/* year of file (1970-2099) */
    u32 age;		/* age of the file in days */
    u32 kbyte;		/* 1024 */
    u32 mbyte;		/* 1024*1204 */
    u32 date;		/* date of file in days sinze 1.1.1970 */
    char *path;		/* path/name of the file */
    char *db_name;	/* name of the current database */
    char *comment;	/* current comment */
};
#define FILE_PARAM_VARS	"sDidmyakM"

static struct id_hash *passwd_hash[HASH_SIZE];
static u8 *id_perms;
static int id_perms_size;
static char *config_dir;
static time_t current_time;
static u32 current_days;
static int path_strip;

static const u32 mask[] = { 0x0000007f, 0x00003fff, 0x001fffff, 0x0fffffff, 0xffffffff };
static const u32 sign[] = { 0xffffffc0, 0xffffe000, 0xfff00000, 0xf8000000, 0x00000000 };



/*
 * uid/gid management
 */

static void
add_name(char *name, int flags)
{
    int h = hash_key(name) % HASH_SIZE;
    struct id_hash *p;

    for (p = passwd_hash[h]; p; p = p->next)
	if (streq(p->name, name))
	{
	    p->flags |= flags;
	    return;
	}
    p = xmalloc(sizeof(*p) + strlen(name));
    p->next = passwd_hash[h];
    passwd_hash[h] = p;
    p->flags = flags;
    strcpy(p->name, name);
}

static void
add_passwd(void)
{
    uid_t uid = getuid();
    struct passwd *pw;
    struct group *gr;
    char **p;

    while (pw = getpwent())
	if (pw->pw_uid == uid)
	{
	    add_name(pw->pw_name, 1);
	    while (gr = getgrent())
		if (gr->gr_gid == pw->pw_gid)
		    add_name(gr->gr_name, 2);
		else if (p = gr->gr_mem)
		    while (*p)
			if (streq(*p++, pw->pw_name))
			{
			    add_name(gr->gr_name, 2);
			    break;
			}
	    endgrent();
	    break; /* no uid sharing */
	}
    endpwent();
}

static void
new_name(char *name, int id)
{
    static int passwd_read = 0;
    int h = hash_key(name) % HASH_SIZE;
    struct id_hash *p;

    if (not passwd_read)
	add_passwd(), passwd_read = 1;

    if (id_perms_size <= id)
    {
	id_perms_size = (id + 1 + 255) & ~255;
	id_perms = xrealloc(id_perms, id_perms_size);
    }

    for (p = passwd_hash[h]; p; p = p->next)
	if (streq(p->name, name))
	    break;

    id_perms[id] = p ? p->flags : 0;
}



/*
 * buffered file reading.
 */

#define get(inp) (++(inp)->ptr >= (inp)->end ? in_fill_buf((inp)) : 0, *(inp)->ptr)
#define unget(inp) (--(inp)->ptr)

static int
in_open(struct in_ptr *inp, char *name)
{
    char *suff = FF_SUFFIX;
    char *dir = "";
    int l = strlen(name);
    struct stat st[1];

    if (streq(name, "-"))
    {
	strcpy(inp->buf, "<stdin>");
	inp->fd = 0;
    }
    else
    {
	if (l > 2 && memcmp(name + l -  3, suff, 3) == 0)
	    suff = "";

	if (config_dir && name[0] != '/' && (name[0] != '.' || name[1] != '/'))
	    dir = config_dir;

	sprintf(inp->buf, "%s%s%s", dir, name, suff);
	inp->fd = open(inp->buf, O_RDONLY);

	if (inp->fd == -1 && errno == ENOENT)
	{
	    strcpy(inp->buf, name);
	    inp->fd = open(inp->buf, O_RDONLY);
	}
    }
    if (inp->fd != -1)
    {
	inp->date = fstat(inp->fd, st) == 0 ? st->st_mtime : current_time;
	inp->access = inp->fd != 0 ? access(inp->buf, R_OK) == 0 : 1;
	inp->path = xstrdup(inp->buf);
	inp->ptr = inp->buf;
	inp->end = inp->buf;
	inp->name = name;
	inp->eof = 0;
    }

    return inp->fd != -1;
}

static void
in_close(struct in_ptr *inp)
{
    close(inp->fd);
    inp->ptr = inp->buf;
    inp->end = inp->buf;
    inp->fd = -1;
    inp->name = 0;
}

static void
in_fill_buf(struct in_ptr *inp)
{
    int n;

    if (inp->eof)
	fatal("%s: Unexepected end of file.", inp->name);

    n = read(inp->fd, inp->buf, sizeof(inp->buf));

    if (n <= 0)
    {
	if (n < 0)
	    ioerror(inp->name);
	inp->eof = 1;
	inp->buf[n = 0] = 0xff;
    }
    inp->ptr = inp->buf;
    inp->end = inp->buf + n;
}



/*
 * data decoding
 */

static s32
get_long(struct in_ptr *inp)
{
    u32 n, m = 0;

    switch (n = get(inp))
    {
	case 0xf0:
	    m++, n = get(inp);
	case 0xe0 ... 0xef:
	    m++, n = (n << 8) + get(inp);
	case 0xc0 ... 0xdf:
	    m++, n = (n << 8) + get(inp);
	case 0x80 ... 0xbf:
	    m++, n = (n << 8) + get(inp);
	case 0x00 ... 0x7f:
	    n &= mask[m];
	    if (n & sign[m])
		n |= sign[m];
    }
    return n;
}

static u32
get_ulong(struct in_ptr *inp)
{
    u32 n, m = 0;

    switch (n = get(inp))
    {
	case 0xf0:
	    m++, n = get(inp);
	case 0xe0 ... 0xef:
	    m++, n = (n << 8) + get(inp);
	case 0xc0 ... 0xdf:
	    m++, n = (n << 8) + get(inp);
	case 0x80 ... 0xbf:
	    m++, n = (n << 8) + get(inp);
	case 0x00 ... 0x7f:
	    n &= mask[m];
    }
    return n;
}

static char *
get_string(char *buf, struct in_ptr *inp)
{
    buf--;
    while (*++buf = get(inp))
	;
    return buf;
}



/*
 * group managment
 */

static struct file_list **
app_file(struct file_list **f, char *group, char *file)
{
    *f = xmalloc(sizeof(**f));
    (*f)->next = 0;
    (*f)->group = group;
    (*f)->file = file;
    return &(*f)->next;
}

static struct file_list *
load_config_file(char *file)
{
    struct file_list *head = 0;
    struct file_list **tail = &head;
    char *group = 0;
    char buf[256], *x;
    FILE *fp;
    int l;

    if (fp = fopen(file, "r"))
    {
	while (fgets(buf, sizeof(buf), fp))
	{
	    if (x = strchr(buf, '#'))
		*x = 0;
	    for (x = buf; is_space(*x); ++x)
		;
	    l = strlen(x);
	    while (l && is_space(x[l-1]))
		x[--l] = 0;
	    if (l == 0)
		continue;

	    if (*x == '[' && x[l-1] == ']')
	    {
		x[l-1] = 0;
		group = xstrdup(x+1);
		tail = app_file(tail, group, 0);
	    }
	    else if (group)
		tail = app_file(tail, group, xstrdup(x));
	    else
		fatal("%s: Missing group header.", file);
	}
	fclose(fp);
    }
    return head;
}

static void
get_group(struct file_list *conf, struct file_list **head, char *group)
{
    struct file_list *f, **p;
    char *grp_save = 0;
    int all;

    if (group == 0)
	return;

    all = streq(group, "all");

    for (f = conf; f; f = f->next)
	if (f->group && (all || streq(f->group, group)))
	{
	    grp_save = f->group;
	    f->group = 0; /* avoid endless recursion */
	    get_group(conf, head, f->file);
	    f->group = grp_save;
	}

    if (grp_save == 0)
    {
	for (p = head; *p; p = &(*p)->next)
	    if (streq((*p)->file, group))
		break;
	if (*p == 0)
	    app_file(p, 0, group);
    }
}

static void
show_group(struct file_list *conf, char *grp)
{
    struct file_list *f = 0;
    int b, w;

    get_group(conf, &f, grp);

    if (f && f->next == 0 && streq(f->file, grp))
    {
	printf("%10s: %s\n", "<none>", grp);
	xfree(f);
	return;
    }

    b = w = printf("%10s:", grp);
    if (f)
    {
	while (f)
	{
	    struct file_list *n = f->next;
	    if (w + strlen(f->file) > 78)
		w = printf("\n%*s", b, "") - 1;
	    w += printf(" %s", f->file);
	    xfree(f);
	    f = n;
	}
    }
    else
	printf(" <empty>");
    printf("\n");
}

static int
show_groups(struct file_list *conf, int ac, char **av)
{
    struct file_list *f, *g;

    if (ac == 0)
    {
	for (f = conf; f; f = f->next)
	    if (f->file == 0)
	    {
		for (g = conf; g != f; g = g->next)
		    if (g->file == 0 && f->group == g->group)
			break;
		if (g == f)
		    show_group(conf, f->group);
	    }
	show_group(conf, "all");
    }
    else while (ac--)
	show_group(conf, *av++);

    return 0;
}


/*
 * pattern compilation
 */

static void **
make_patterns(char *str, int nocase)
{
    int i;
    char *x;
    void **av;

    for (i = 0, x = str-1; x = strchr(x+1, '|'); ++i)
	;

    av = xcalloc(i + 2, sizeof(void *));

    if (nocase)
	nocase = FMATCH_NOCASE;

    for (i = 0; str; i++)
    {
	if (x = strchr(str, '|'))
	    *x++ = 0;
	av[i] = fmatch_compile(str, FMATCH_AUTOSS|nocase);
	if (av[i] == 0)
	    fatal("Bad pattern '%s'.", str);
	str = x;
    }

    return av;
}



/*
 * Output formatting.
 */

static void
make_date(struct file_param *p)
{
    p->age = current_days - p->date;
    days2dmy(p->date, (int *)&p->day, (int *)&p->month, (int *)&p->year);
    p->Date = p->year * 10000 + p->month * 100 + p->day;
}

static char *
utoa(char *buf, u32 val)
{
    buf += 12;
    *--buf = 0;
    do
	*--buf = to_digit(val % 10), val /= 10;
    while (val);
    return buf;
}

static char *
atou(char *str, int *val)
{
    *val = 0;

    while (is_digit(*str))
	*val = *val * 10 + digitval(*str++);

    return str;
}

static void
out(char *fmt, struct file_param *param)
{
    static char *months[] = { "January", "February", "March", "April", "May",
			      "June", "July", "August", "September", "October",
			      "November", "December" };
    int n;

    while (*fmt)
	switch (*fmt++)
	{
	    default:
	    	putchar(fmt[-1]);
		break;
	    case '%':
	    {
		char buf[32], *arg, *x;
		int width = 0;
		int prec = 0;
		int pad = ' ';

		if (n = *fmt == '-')
		    fmt++;
		if (*fmt == '0')
		    pad = '0';
		while (is_digit(*fmt))
		    width = width * 10 + digitval(*fmt++);
		if (n)
		    width = -width;
		if (*fmt == '.')
		    while (is_digit(*++fmt))
			prec = prec * 10 + digitval(*fmt);
		switch (*fmt++)
		{
		    case 'D':
			while (width > 9)
			    putchar(pad), width--;
			out("%02d.%3m.%02y", param);
			while (width < -9)
			    putchar(' '), width++;
			width = 0;
			arg = "";
			break;
		    case 'd':
			if (param->Date == 0)
			    make_date(param);
			arg = utoa(buf, param->day);
			break;
		    case 'm':
			if (param->Date == 0)
			    make_date(param);
			if (width && width >= -2 && width <= 2)
			    arg = utoa(buf, param->month);
			else
			{
			    arg = strcpy(buf, months[param->month - 1]);
			    if (width == -3 || width == 3)
				buf[3] = 0;
			    if (prec)
				buf[0] = to_lower(buf[0]);
			}
			break;
		    case 'y':
			if (param->Date == 0)
			    make_date(param);
			if (width && width >= -2 && width <= 2)
			    arg = utoa(buf, param->year % 100);
			else
			    arg = utoa(buf, param->year);
			break;
		    case 'a':
			if (param->Date == 0)
			    make_date(param);
			arg = utoa(buf,param->age);
			break;
		    case 's':
			arg = utoa(buf, param->size >> prec);
			break;
		    case 'i':
			arg = utoa(buf, param->inode);
			break;
		    case 'n':
		    case 'f':
			arg = param->db_name;
			break;
		    case 'c':
			arg = param->comment;
			break;
		    case 'p':
			if (prec == 0)
			    prec = path_strip;
			for (arg = param->path; prec-- && (x = strchr(arg, '/')); arg = x + 1)
			    ;
			break;
		    case '%':
		    	arg = "%";
			break;
		    case 0:
			return;
		    default:
			arg = "<?>";
			break;
		}
		if (width <= 0)
		{
		    while (*arg)
			putchar(*arg++), width++;
		    while (width++ < 0)
			putchar(' ');
		}
		else
		{
		    width -= strlen(arg);
		    while (width-- > 0)
			putchar(pad);
		    while (*arg)
			putchar(*arg++);
		}
		break;
	    }
	    case '\\':
	    	switch (*fmt)
		{
		    case '\\':	putchar('\\');	fmt++;	break;
		    case 'n':	putchar('\n');	fmt++;	break;
		    case 'r':	putchar('\r');	fmt++;	break;
		    case 't':	putchar('\t');	fmt++;	break;
		    case 'a':	putchar('\a');	fmt++;	break;
		    case 'e':	putchar('\e');	fmt++;	break;
		    case 'x':
		    	n = 0, fmt++;
			if (is_xdigit(*fmt))
			    n = n * 16 + digitval(*fmt++);
			if (is_xdigit(*fmt))
			    n = n * 16 + digitval(*fmt++);
			putchar(n);
			break;
		    case '0' ... '7':
		    	n = digitval(*fmt++);
			if (is_digit(*fmt) && digitval(*fmt) < 8)
			    n = n * 8 + digitval(*fmt++);
			if (is_digit(*fmt) && digitval(*fmt) < 8)
			    n = n * 8 + digitval(*fmt++);
			putchar(n);
			break;
		    case 0:
		    	return;
		    default:
		    	putchar(*fmt++);
			break;
		}
	}
}



/*
 * the main workhorse
 */

static int
ff(char *file, void **patterns, char *expr, char *outfmt, int nocomments, int quiet)
{
    struct file_param param[1];
    int c, found = 0;
    u32 version, revision, days, flags;
    u32 id, new_comment, private, noperms;
    char comment[PATH_LEN], path[PATH_LEN], buf[256];
    char *comment_end, *path_end;
    struct in_ptr inp[1];
    void **p;

    { /* make gcc shut up */
	flags=0, id=0, new_comment=0, private=0,
	noperms=0, comment_end=0, path_end=0;
    }

    if (in_open(inp, file))
    {
	param->db_name = basename(file, FF_SUFFIX);
	param->path = path;
	param->comment = comment;
	param->kbyte = 1024;
	param->mbyte = 1024 * 1024;

	if ((c = get(inp)) == T_RESET)
	{
	    unget(inp);
	    for (;;)
	    {
		switch (c = get(inp))
		{
		    case T_RESET:
			if (get(inp) != MAGIC_1 || get(inp) != MAGIC_2)
			{
			    error("%s: Magic number not found.", inp->path);
			    break;
			}
			version = get_ulong(inp);
			revision = get_ulong(inp);
			days = get_ulong(inp);
			flags = get_ulong(inp);
			if (version > VERSION || (version == VERSION && revision > REVISION))
			{
			    error("%s: Cannot handle a v%d.%d file.", inp->path, version, revision);
			    break;
			}
			if (days)
			{
			    c = (current_time - inp->date) / (24*60*60);
			    if (c > days)
				error("%s: Database is %d days old!", inp->path, c);
			}
			if ((flags & F_ACCESS) && not inp->access)
			    break;
			path_end = path;
			comment_end = comment;
			*comment_end = 0;
			new_comment = 0;
			param->date = 0;
			param->size = 0;
			param->inode = 0;
			private = 0;
			id = 0;
			noperms = 0;
			if (outfmt == 0)
			    switch (flags & (F_DATE|F_SIZE))
			    {
				case 0:			outfmt = "[%n] %p\n";		break;
				case F_DATE:		outfmt = "%D [%n] %p\n";	break;
				case F_SIZE:		outfmt = "%8s [%n] %p\n";	break;
				case F_DATE|F_SIZE:	outfmt = "%D %8s [%n] %p\n";	break;
			    }
			continue;
		    case T_COMMENT:
			comment_end = get_string(comment_end - get_ulong(inp), inp);
			new_comment = not nocomments;
			continue;
		    case T_NAME:
		    	get_string(buf, inp);
			new_name(buf, id++);
			continue;
		    case T_PUBLIC:
			if (not inp->access)
			    private = not private;
			continue;
		    case T_ENTER_DIR:
		    {
		    	u32 mode = get_ulong(inp); /* mode */
		    	u32 owner = get_ulong(inp); /* owner */
		    	u32 group = get_ulong(inp); /* group */

			if (inp->access)
			    continue;

			if (noperms == 0 && owner < id && group < id)
			{
			    if (id_perms[owner] & 1)
				mode >>= 2;
			    else if (id_perms[group] & 2)
				mode >>= 1;
			    if (mode & 1)
				continue;
			}
			noperms++;
			continue;
		    }
		    case T_LEAVE_DIR:
			if (noperms)
			    --noperms;
		    	continue;
		    case 0xf7 ... 0xfe:
			error("%s: Illegal token $%02x.", inp->path, c);
			break;
		    case 0xff: /* eof */
			break;
		    default:
			unget(inp);
			path_end = get_string(path_end - get_ulong(inp), inp);
			if (flags & F_SIZE)
			    param->size += get_long(inp);
			if (flags & F_DATE)
			    param->date += get_long(inp);
			if (flags & F_INODE)
			    param->inode += get_long(inp);

			if (private || noperms)
			    continue;

			param->Date = 0;

			if (expr)
			{
			    make_date(param);
			    if (vexec_expr(expr, (int *)param) == 0)
				continue;
			}

			for (p = patterns; *p; p++)
			    if (fmatch_exec(*p, path, path_end))
			    {
				if (not quiet)
				{
				    if (new_comment)
				    {
					printf("%s:\n", comment);
					new_comment = 0;
				    }
				    out(outfmt, param);
				}
				found++;
				break;
			    }
			if (found && quiet)
			    break; /* we have an exit status now */
			continue;
		}
		break;
	    }
	}
	else
	    error("%s: Not a data file.", inp->path);
	xfree(param->db_name);
	in_close(inp);
    }
    else
	ioerror(file);
    return found;
}



/*
 * pipe output through a pager
 */

void
pipeless(void)
{
    char *pager = getenv("FF_PAGER") ? : getenv("PAGER") ? : "less -E";
    char *parg[32];
    int fd[2], pid, i = 0;

    if (not isatty(1))
	return;

    pager = xstrdup(pager);

    while (*pager)
    {
	while (is_space(*pager))
	    pager++;
	if (i < 31)
	    parg[i++] = pager;
	while (*pager && not is_space(*pager))
	    pager++;
	if (*pager)
	    *pager++ = 0;
    }
    parg[i++] = 0;

    if (parg[0] == 0)
	return;

    if (pipe(fd) == -1)
	return;

    fflush(stdout);
    fflush(stderr);

    if ((pid = fork()) == -1)
	return;

    if (pid > 0)
    {
	dup2(fd[0], 0);
	close(fd[0]);
	close(fd[1]);
	setuid(getuid());
	setgid(getgid());
	execvp(parg[0], parg);
	fatal("Unable to exec '%s'.", parg[0]);
    }
    else
    {
	dup2(fd[1], 1);
	if (isatty(2))
	    dup2(fd[1], 2);
	close(fd[0]);
	close(fd[1]);
    }
}


/*
 * the rest...
 */

static void
usage(FILE *fp)
{
    fprintf(fp, "Usage: %s [-acefhilosv] [groups/files] pattern[|pattern...]\n", prgname);
    fprintf(fp, "\t-a\tSet default group to 'all'.\n");
    fprintf(fp, "\t-c\tDon't show comments.\n");
    fprintf(fp, "\t-e expr\tNumeric expression to select files.\n");
    fprintf(fp, "\t-f file\tUse alternate group file.\n");
    fprintf(fp, "\t-h\tShow this help.\n");
    fprintf(fp, "\t-i\tPerform case insensitive search.\n");
    fprintf(fp, "\t-l\tList contents of groups.\n");
    fprintf(fp, "\t-o fmt\tDefine output format.\n");
    //fprintf(fp, "\t-p num\tRemove first num dirs from printed path.\n");
    fprintf(fp, "\t-q\tQuiet mode: No output, exit status only.\n");
    fprintf(fp, "\t-s\tStop after first successful file.\n");
    fprintf(fp, "\t-v\tShow program version and exit.\n");
    exit(fp != stdout ? 2 : 0);
}

void
main(int argc, char **argv)
{
    struct file_list *conf, *files = 0;
    void **patterns;
    char *pattern = 0;
    char *expr = 0;
    char *config_name = getenv("FF_GROUPS") ? : FF_GROUPS;
    char *outfmt = getenv("FF_FORMAT");
    char *def_grp = getenv("FF_DEFAULT") ? : "default";
    int stoponmatch = 0;
    int nocase = 0;
    int nocomments = 0;
    int list = 0;
    int quiet = 0;
    int found, opt;

    time(&current_time);
    current_days = current_time / (24 * 60 * 60);

    while ((opt = getopt(argc, argv, "ae:sichvlf:o:p:q")) != EOF)
    {
	switch (opt)
	{
	    case 'a':
	    	def_grp = "all";
		break;
	    case 'e':
		if (expr)
		    fatal("Only one -e expression, please.");
		opt = 2 * strlen(optarg) + 1; /* worst case */
		expr = comp_expr(optarg, FILE_PARAM_VARS, xmalloc(opt), opt);
		if (expr == 0)
		    fatal("Invalid expression '%s'.", optarg);
		break;
	    case 's':
		stoponmatch = 1;
		break;
	    case 'i':
	    	nocase = 1;
		break;
	    case 'c':
	    	nocomments = 1;
		break;
	    case 'o':
	    	outfmt = optarg;
		break;
	    case 'f':
	    	config_name = optarg;
		break;
	    case 'p':
		if (*atou(optarg, &path_strip))
		    fatal("Not a number '%s'.", optarg);
		break;
	    case 'q':
	    	quiet = 1;
		stoponmatch = 1;
		break;
	    case 'l':
	    	list = 1;
		break;
	    case 'v':
	    	printf("%s v%d.%d.%d --- Copyright 1998 by E. Toernig.\n",
			prgname, VERSION, REVISION, SUBREV);
		exit(0);
	    case 'h':
		usage(stdout);
	    default:
		usage(stderr);
	}
    }

    if (not list && optind < argc)
	pattern = argv[--argc];

    config_dir = dirname(config_name, "/");

    conf = load_config_file(config_name);

    if (list)
    {
	pipeless();
	exit(show_groups(conf, argc - optind, argv + optind));
    }

    if (optind == argc)
	get_group(conf, &files, def_grp);
    else while (optind < argc)
	get_group(conf, &files, argv[optind++]);

    if (not files)
	fatal("No files found.");

    if (not pattern)
	usage(stderr);

    patterns = make_patterns(pattern, nocase);

    if (not quiet)
	pipeless();

    for (found = 0; files; files = files->next)
    {
	found += ff(files->file, patterns, expr, outfmt, nocomments, quiet);

	if (found && stoponmatch)
	    break;
    }

    exit(found == 0);
}
