/*
 * Routine to perform parameter substitution.
 * instring is a string containing printf type escapes.
 * outfunc is a function called for each char to be written.
 * outpara is passed to outfunc as the second parameter.
 * The remaining args are parameters access by %p1 - %p9.
 *
 * The following escapes are defined for substituting parameters:
 *
 *	%d	print pop() as in printf
 *	%[0]2d	print pop() like %[0]2d
 *	%[0]3d	print pop() like %[0]3d
 *	%c	print pop() like %c
 *	%s	print pop() like %s
 *	%l	pop() a string and push its length.
 *
 *	%p[1-9]	push ith parm
 *	%P[a-z] set variable
 *	%g[a-z] get variable
 *	%'c'	char constant c
 *	%{nn}	integer constant nn
 *
 *	%+ %- %* %/ %m		arithmetic (%m is mod): push(pop() op pop())
 *	%& %| %^		bit operations:		push(pop() op pop())
 *	%= %> %<		logical operations:	push(pop() op pop())
 *	%A %O			logical and/or:		push(pop() op pop())
 *	%! %~			unary operations	push(op pop())
 *
 *	%? expr %t thenpart %e elsepart %;
 *				if-then-else, %e elsepart is optional.
 *				else-if's are possible ala Algol 68:
 *				%? c1 %t %e c2 %t %e c3 %t %e c4 %t %e %;
 *
 * all other characters are ``self-inserting''.  %% gets % output.
 *
 * padding informations ($<...>) are silently removed.
 *
 */

#include "tinfo.h"

#define STACKS	16
#define push(i) (stack[++top] = (i))
#define pop()	(stack[top--])
#define out(c)	(outfunc((c), outpara))
#define err(x)	"ti_param: " x

char *
ti_param(char *instring, ti_out_t outfunc, void *outpara, ...)
{
    int paras[9];
    int vars[26];
    int stack[STACKS+1], top = 0;
    int c, op;
    char *cp = instring;
    char *xp;

    if (instring == 0)
	return err("null arg");

    if (outfunc == 0)
	return 0;

    for (c = 0; c < 9; c++)
	paras[c] = ((int *)(&outpara + 1))[c];

    while (c = *cp++)
    {
	if (c == '$' && *cp == '<')
	{
	    while (*cp && *cp++ != '>')
		;
	}
	else if (c != '%')
	{
	    out(c);
	}
	else
	{
	    int fmt = 0;

nextc:	    switch (c = *cp++)
	    {
		case '0':
		    fmt |= 0100;
		    goto nextc;
		case '2':
		    fmt |= 0010;
		    goto nextc;
		case '3':
		    fmt |= 0011;
		    goto nextc;
		case 'd':
		    op = pop();

		    if (fmt >= 0111 || op > 99)		/* %03d */
			out('0' + (op/100) % 10);
		    else if (fmt & 0001)		/* %3d */
			out(' ');

		    if (fmt >= 0110 || op > 9)		/* %02d or %03d */
			out('0' + (op/10) % 10);
		    else if (fmt & 0010)		/* %2d or %3d */
			out(' ');

		    out('0' + op % 10);
		    break;

		case 'c':
		    out(pop());
		    break;

		case 'l':
		    xp = (char *) pop();
		    if (xp == 0)
			return err("bad string");
		    push(strlen(xp));
		    break;

		case 's':
		    xp = (char *) pop();
		    if (xp == 0)
			return err("bad string");
		    while (*xp)
			out(*xp++);
		    break;

		/*
		 * %i: shorthand for increment first two parms.
		 * Useful for terminals that start numbering from
		 * one instead of zero (like ANSI terminals).
		 */
		case 'i':
		    paras[0]++;
		    paras[1]++;
		    break;

		/* %pi: push the ith parameter */
		case 'p':
		    c = *cp++ - '1';
		    if (c < 0 || c >= 9)
			return err("bad parameter");
		    push(paras[c]);
		    break;
	    
		/* %Pi: pop from stack into variable i (a-z) */
		case 'P':
		    c = *cp++ - 'a';
		    if (c < 0 || c >= 26)
			return err("bad variable");
		    vars[c] = pop();
		    break;

		/* %gi: push variable i (a-z) */
		case 'g':
		    c = *cp++ - 'a';
		    if (c < 0 || c >= 26)
			return err("bad variable");
		    push(vars[c]);
		    break;

		/* %'c' : character constant */
		case '\'':
		    if (*cp)
			push(*cp++);
		    if (*cp++ != '\'')
			return err("missing closing quote");
		    break;
	    
		/* %{nn} : integer constant.  */
		case '{':
		{
		    int sign = 1;

		    if (*cp == '-')
			cp++, sign = -1;
		    else if (*cp == '+')
			cp++;

		    op = 0;
		    while ((c = *cp++) >= '0' && c <= '9')
			op = 10*op + c - '0';

		    if (c != '}')
			return err("missing closing brace");
		    push(sign * op);
		    break;
		}
	    
		/* binary operators */
		case '+': c=pop(); op=pop(); push(op + c); break;
		case '-': c=pop(); op=pop(); push(op - c); break;
		case '*': c=pop(); op=pop(); push(op * c); break;
		case '/': c=pop(); op=pop(); push(op / c); break;
		case 'm': c=pop(); op=pop(); push(op % c); break; /* %m: mod */
		case '&': c=pop(); op=pop(); push(op & c); break;
		case '|': c=pop(); op=pop(); push(op | c); break;
		case '^': c=pop(); op=pop(); push(op ^ c); break;
		case '=': c=pop(); op=pop(); push(op = c); break;
		case '>': c=pop(); op=pop(); push(op > c); break;
		case '<': c=pop(); op=pop(); push(op < c); break;
		case 'A': c=pop(); op=pop(); push(op && c); break;
		case 'O': c=pop(); op=pop(); push(op || c); break;

		/* Unary operators. */
		case '!': op=pop(); push(!op); break;
		case '~': op=pop(); push(~op); break;
		/* Sorry, no unary minus. minus is binary. */

		/*
		 * If-then-else.  Implemented by a low level hack of
		 * skipping forward until the match is found, counting
		 * nested if-then-elses.
		 */
		case '?':	/* IF - just a marker */
		    break;

		case 't':	/* THEN - branch if false */
		    if (pop())
			break;
		    c = 'e'+'e'-';';
		    /* fall through */
		case 'e':	/* ELSE - branch to ENDIF */
		{
		    int level = 0;
		    int delim = c - 'e' + ';';

		    while (c = *cp++)
			if (c == '%')
			{
			    if ((c = *cp++) == 0)
				break;
			    if ((c == delim || c == ';') && level == 0)
				break;
			    if (c == '?')
				level++;
			    if (c == ';')
				level--;
			}
		    if (c == 0)
			return err("no matching ENDIF");
		}

		case ';':	/* ENDIF - just a marker */
		    break;

		case '\0':
		    return err("incomplete command");

		default:
		    out(c);
		    break;
	    }

	    if (top < 0 || top >= STACKS)
		return err("stack error");
	}
    }
    return 0;
}
