/*
 * Simple expression evaluation.
 *
 * There are two functions. One, comp_expr(), to compile an expression
 * and another, exec_expr(), to compute the value of a compiled
 * expression depending on a variable.
 *
 * The buffer given to comp_expr() must by slightly larger then the
 * original string because numbers lower then 9 are represented as
 * two bytes in compiled form. On the other side, numbers >99 take
 * less bytes and parantheses are completely removed.
 *
 * MAYBEBUG: I think the precences of the operators are messed up.
 *
 * $Id: expr.c,v 1.1 1995/02/21 20:52:44 csserver Exp csserver $
 *
 * $Source: /home/csserver/src/RCS/expr.c,v $
 *
 * $Log: expr.c,v $
 * Revision 1.1  1995/02/21  20:52:44  csserver
 * Initial revision
 *
 */

#include "generic.h"
#include "cattr.h"
#include "expr.h"

#define MAX_DEPTH 64

static u8 *unary(u8 *, int), *binary(u8 *, int);

static u8 *codep, *codee, *varids;



static inline void
store(u8 c)
{
    if (codep < codee)
	*codep++ = c;
}

static void
store_u32(u32 n)
{
    u8 buf[5];
    int cc = 0;

    while (n)
	buf[cc++] = n, n >>= 7;

    store('0' + cc);
    while (cc)
	store(buf[--cc] | 0x80);
}


static u8 *
unary(u8 *str, int depth)
{
    while (is_space(*str))
	++str;

    switch (*str)
    {
	default:
	{
	    int i;
	    u8 *x = "$";

	    if (is_alpha(*str))
		for (i = 0; i < 26 && varids[i]; ++i)
		    if (varids[i] == *str)
		    {
			store('a'+i);
			x = str + 1;
			break;
		    }
	    str = x;
	    break;
	}

	case '-':
	    str = unary(str + 1, depth);
	    store('_');
	    break;

	case '~':
	    str = unary(str + 1, depth);
	    store('~');
	    break;

	case '!':
	    str = unary(str + 1, depth);
	    store('!');
	    break;

	case '(':
	    str = binary(str + 1, depth);
	    if (*str++ != ')')
		str = "$";
	    break;

	case '\'':
	{
	    u32 n = 0;

	    while (*++str && *str != '\'')
		n = (n << 8) + *str;

	    if (*str++ == '\0')
		str = "$";

	    store_u32(n);
	    break;
	}
	case '0' ... '9':
	{
	    u32 n = 0;
	    u32 b = 10;

	    if (*str == '0')
	    {
		b = 8;
		if (*++str == 'x')
		{
		    if (not_xdigit(*++str))
			str = "$";
		    b = 16;
		}
	    }

	    while (is_xdigit(*str) && digitval(*str) < b)
		n = b * n + digitval(*str++);

	    store_u32(n);
	    break;
	}
    }

    while (is_space(*str))
	++str;

    return str;
}



static u8 *
binary(u8 *str, int depth)
{
    static u8 operatorc[]  = "*/%+-A&O|^<>#LGBH==# ?";
    static u8 operator1[]  = "*/%+-&&||^<><<><>==!?:";
    static u8 operator2[]  = "     & |  <>>==  = =   ";
    static u8 precedence[] = "888774a3996655555555210";
//  static u8 precedence[] = "77766392885544444444110";
    u8 stack[16], oc;
    int sp = 0;
    int qc = 0;
    int op;

    for (;;)
    {
	str = unary(str, depth + sp);

	if (is_alpha(*str))	/* special case: "30i" --> "30*i" */
	    op = 0, oc = '*';
	else for (op = 0; oc = operator1[op]; ++op)
	    if (oc == *str)
	    {
		if (operator2[op] == ' ')
		{
		    str += 1;
		    break;
		}
		if (operator2[op] == str[1])
		{
		    str += 2;
		    break;
		}
	    }

	while (sp && precedence[op] <= precedence[stack[sp-1]])
	    if (operatorc[stack[--sp]] != ' ')
		store(operatorc[stack[sp]]);
	    else if (oc == ':')
		break;

	if (oc == '?')
	    qc++;
	else if (oc == ':' && --qc < 0)
	    str = "$";
	else if (oc == '\0')
	    break;

	stack[sp++] = op;

	if (depth + sp >= MAX_DEPTH)
	    str = "$";
    }

    if (qc != 0)
	str = "$";

    return str;
}



int
vexec_expr(u8 *expr, int *vars)
{
    int st[MAX_DEPTH+2];
    int sp = 0;
    int n = 0;

    for (;;)
    {
	switch (*expr++)
	{
	    case '\0':
		break;
	    case '5': n = *expr++ - 0x80;
	    case '4': n = (n << 7) + *expr++ - 0x80;
	    case '3': n = (n << 7) + *expr++ - 0x80;
	    case '2': n = (n << 7) + *expr++ - 0x80;
	    case '1': n = (n << 7) + *expr++ - 0x80;
	    case '0':
		st[sp++] = n;
		n = 0;
		continue;
	    case 'a' ... 'z':
		st[sp++] = vars[expr[-1] - 'a'];
		continue;
	    case '_':
	    	st[sp-1] = - st[sp-1];
		continue;
	    case '~':
	    	st[sp-1] = ~ st[sp-1];
		continue;
	    case '!':
	    	st[sp-1] = ! st[sp-1];
		continue;
	    case '+':
		st[sp-2] += st[sp-1];
		sp--;
		continue;
	    case '-':
		st[sp-2] -= st[sp-1];
		sp--;
		continue;
	    case '*':
		st[sp-2] *= st[sp-1];
		sp--;
		continue;
	    case '/':
		st[sp-2] /= st[sp-1] ? st[sp-1] : 1;
		sp--;
		continue;
	    case '%':
		st[sp-2] %= st[sp-1] ? st[sp-1] : 1;
		sp--;
		continue;
	    case '&':
		st[sp-2] &= st[sp-1];
		sp--;
		continue;
	    case '|':
		st[sp-2] |= st[sp-1];
		sp--;
		continue;
	    case '^':
		st[sp-2] ^= st[sp-1];
		sp--;
		continue;
	    case '<':
		st[sp-2] <<= st[sp-1];
		sp--;
		continue;
	    case '>':
		st[sp-2] >>= st[sp-1];
		sp--;
		continue;
	    case '?':
		st[sp-3] = st[sp-3] ? st[sp-2] : st[sp-1];
		sp--;
		sp--;
		continue;
	    case '=':
		st[sp-2] = st[sp-2] == st[sp-1];
		sp--;
		continue;
	    case '#':
		st[sp-2] = st[sp-2] != st[sp-1];
		sp--;
		continue;
	    case 'B':
		st[sp-2] = st[sp-2] < st[sp-1];
		sp--;
		continue;
	    case 'L':
		st[sp-2] = st[sp-2] <= st[sp-1];
		sp--;
		continue;
	    case 'H':
		st[sp-2] = st[sp-2] > st[sp-1];
		sp--;
		continue;
	    case 'G':
		st[sp-2] = st[sp-2] >= st[sp-1];
		sp--;
		continue;
	    case 'A':
		st[sp-2] = st[sp-2] && st[sp-1];
		sp--;
		continue;
	    case 'O':
		st[sp-2] = st[sp-2] || st[sp-1];
		sp--;
		continue;
	}
	break;
    }
    return st[0];
}



int exec_expr(u8 *expr, ...)
{
    return vexec_expr(expr, (int *)(&expr + 1));
}



u8 *
comp_expr(u8 *str, u8 *vars, u8 *buf, int size)
{
    if (str == 0 || buf == 0 || size == 0)
	return 0;

    varids = vars ? vars : (u8 *)"";
    codep = buf;
    codee = buf + size;

    str = binary(str, 0);

    if (*str != '\0' || codep >= codee)
	return 0;

    store('\0');
    return buf;
}



int
eval_expr(u8 *expr, u8 *vars, ...)
{
    u8 buf[256];

    if (comp_expr(expr, vars, buf, sizeof(buf)))
	return vexec_expr(buf, (int *)(&vars + 1));
    return -1;
}


#ifdef TEST_EXPR
#include <stdio.h>



void
main(int argc, char **argv)
{
    while (--argc)
	printf("%d\n", eval_expr(*++argv, "smhd", 100, 60*100, 60*60*100, 24*60*60*100));
}
#endif /*TEST_EXPR*/
