/*
 * Copyright 1996 by E. Toernig (froese@gmx.de)
 */
/*

    The output format.

    	number encoding: signed (si) or unsigned (ui), depending on context.
	    00-7f		7bit integer
	    80-bf xx		14bit integer
	    c0-df xx xx		21bit integer
	    e0-ef xx xx xx	28bit integer
	    f0    xx xx xx xx	32bit integer

	strings (s):
	    xx xx ... xx 00	zero terminated string

	delta strings (ds): modifies a previous string
	    ui s	remove ui bytes from the end and append s

	tokens:
	    ds si1 si2
		next file
		    si1 is size difference to previous file
		    si2 is date difference to previous file (days)

	    f1 e8 81 ui1 ui2 ui3	(T_RESET)
		reset all state information (first token)
		    e8 magic number (MAGIC_1)
		    81 magic number (MAGIC_2)
		    ui1 version
		    ui2 revision
		    ui3 flags

	    f2 ds			(T_COMMENT)
		modify comment string

	    f3 s			(T_NAME)
		declare string and assign next id-number

	    f4				(T_PUBLIC)
		toggle public/private flag

	    f5 ui1 ui2 ui3		(T_ENTER_DIR)
		enter dir
		   ui1 is mode
		   ui2 is owner id (assign with f3)
		   ui3 is group id (assign with f3)

	    f6				(T_LEAVE_DIR)
		leave dir
*/
		
	    
#include <stdio.h>
#include <time.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>

#include <etlib/generic.h>
#include <etlib/error.h>
#include <etlib/xmalloc.h>

#include "mk_ff.h"
#include "hashs.h"
#include "emit.h"



static char last_path[PATH_LEN];
static char last_comment[PATH_LEN];
static u32 last_date;
static u32 last_size;
static u32 last_inode;
static int priv_nest;
static int name_id;
static int sec_level;
static int out_fields;

static int out_fd;
static u8 out_buf[4096];
static u8 *out_ptr;
static char *out_name;

#define out(c) ((out_ptr >= out_buf + sizeof(out_buf) ? flush_out() : 0), *out_ptr++ = (c))

static void
flush_out()
{
    int n, l = out_ptr - out_buf;

    while (l && (n = write(out_fd, out_ptr - l, l)) != -1)
	l -= n;
    if (l)
	ioerror(out_name);
    out_ptr = out_buf;
}

static void
out_long(long l)
{
    /*
	0xxxxxxx	 7bit int
	10xxxxxx 8x	14bit int
	110xxxxx 16x	21bit int
	1110xxxx 24x	28bit int
	11110000 32x	32bit int
	11110001	special code
	...
	11111111	special code
    */
    if (l >= -(1<<6) && l < (1<<6))
	out(l & 0x7f);
    else if (l >= -(1<<13) && l < (1<<13))
	out(0x80 | (l >> 8) & 0x3f), out(l);
    else if (l >= -(1<<20) && l < (1<<20))
	out(0xc0 | (l >> 16) & 0x1f), out(l>>8), out(l);
    else if (l >= -(1<<27) && l < (1<<27))
	out(0xe0 | (l >> 24) & 0x0f), out(l>>16), out(l>>8), out(l);
    else
	out(0xf0), out(l>>24), out(l>>16), out(l>>8), out(l);
}

static void
out_ulong(u32 l)
{
    /* same as out_long but for unsigned numbers */
    if (l < (1<<7))
	out(l & 0x7f);
    else if (l < (1<<14))
	out(0x80 | (l >> 8) & 0x3f), out(l);
    else if (l < (1<<21))
	out(0xc0 | (l >> 16) & 0x1f), out(l>>8), out(l);
    else if (l < (1<<28))
	out(0xe0 | (l >> 24) & 0x0f), out(l>>16), out(l>>8), out(l);
    else
	out(0xf0), out(l>>24), out(l>>16), out(l>>8), out(l);
}

static void
out_str(char *s)
{
    do
	out(*s);
    while (*s++);
}

static void
out_dstr(char *o, char *n, u8 token)
{
    while (*o && *o == *n)
	o++, n++;

    if (token)
    {
	if (*o == 0 && *n == 0)
	    return;
	out(token);
    }
    out_ulong(strlen(o));
    out_str(n);
    strcpy(o, n);
}


void
emit_open(char *name, int sec)
{
    struct stat st[1];

    if (sec < S_FIRST || sec > S_LAST)
	fatal("Bad security level %d.", sec);

    if (name && not streq(name, "-"))
	out_fd = creat(name, 0600);
    else if (not isatty(1))
	out_fd = 1, name = "<stdout>";
    else
	fatal("Won't write to tty.");

    if (out_fd == -1)
	fatal_ioerror(name);

    if (fstat(out_fd, st) != -1 && S_ISREG(st->st_mode))
    {
	fchmod(out_fd, sec == S_NONE ? 0644 : 0640);

	if (stat("/proc/self/exe", st) != -1)
	    fchown(out_fd, st->st_uid, st->st_gid);
    }

    out_ptr = out_buf;
    out_name = xstrdup(name);
    sec_level = sec;
}

void
emit_close()
{
    flush_out();
    close(out_fd);
    xfree(out_name);
}

void
emit_comment(char *str)
{
    out_dstr(last_comment, str, T_COMMENT);
}

void
emit_reset(int days, int fields, char *info)
{
    if (sec_level == S_ACCESS)
	fields |= F_ACCESS;

    out(T_RESET);
    out(MAGIC_1);
    out(MAGIC_2);
    out_ulong(VERSION);
    out_ulong(REVISION);
    out_ulong(days);
    out_ulong(fields);
    out_fields = fields;
    last_path[0] = 0;
    last_comment[0] = 0;
    last_date = 0;
    last_size = 0;
    last_inode = 0;
    priv_nest = 0;
    name_id = 0;
    if (info && *info)
	emit_comment(info);
}

int
emit_name(char *str)
{
    out(T_NAME);
    out_str(str);
    return name_id++;
}

void
emit_enter_dir(struct entry *dir, char *path)
{
    if (sec_level == S_SIMPLE)
    {
	if (priv_nest || (dir->mode & 005) != 005)
	    priv_nest++;
	if (priv_nest == 1)
	    out(T_PUBLIC);
    }
    else if (sec_level == S_FULL)
    {
	int uid = make_uid(dir->uid);
	int gid = make_uid(dir->gid);
	int mod = 0;

	if ((dir->mode & 0005) == 0005)
	    mod |= 1;
	if ((dir->mode & 0050) == 0050)
	    mod |= 2;
	if ((dir->mode & 0500) == 0500)
	    mod |= 4;

	out(T_ENTER_DIR);
	out_ulong(mod);
	out_ulong(uid);
	out_ulong(gid);
    }
}

void
emit_leave_dir(struct entry *dir, char *path)
{
    if (sec_level == S_SIMPLE)
    {
	if (priv_nest == 1)
	    out(T_PUBLIC);
	if (priv_nest)
	    priv_nest--;
    }
    else if (sec_level == S_FULL)
    {
	out(T_LEAVE_DIR);
    }
}

void
emit_file(struct entry *file, char *path)
{
    out_dstr(last_path, path, 0);
    if (out_fields & F_SIZE)
	out_long(file->size - last_size);
    if (out_fields & F_DATE)
	out_long(file->date - last_date);
    if (out_fields & F_INODE)
	out_long(file->inode - last_inode);
    last_size = file->size;
    last_date = file->date;
    last_inode = file->inode;
}

struct entry *
alloc_entry(struct entry *root, char *name, struct stat *st)
{
    struct entry *e = xmalloc(sizeof(*e) + strlen(name));

    e->next = root;
    e->dev = st->st_dev;
    e->inode = st->st_ino;
    e->mode = st->st_mode;
    e->size = st->st_size;
    e->date = (st->st_mtime - timezone) / (24 * 60 * 60);
    e->uid = st->st_uid;
    e->gid = st->st_gid;
    strcpy(e->name, name);
    return e;
}
