#include	<stdio.h>
#include	<stdlib.h>
#include	<ctype.h>
#include	<unistd.h>
#include	<sys/types.h>
#include	<time.h>
#include	<errno.h>
#include	<fcntl.h>

#define	DEFLEN	(1024 * 1024 * 8)
#define	DEFRD	(1024 * 16)
#define	DEFWR	(1024 * 16)

#define	ulong	unsigned long
#define	uchar	unsigned char

static	char	*dbuff	;
static	int	dblen	= 0	 ;	/* Data buffer length		*/
static	int	dbcnt	= 0	 ;	/* Data buffer count		*/
static	int	dbinsp	= 0	 ;	/* Data buffer insertion point	*/
static	int	dbextp	= 0	 ;	/* Data buffer extraction point	*/
static	int	dbmarg	= 0	 ;
static	int	dbfill	 	 ;	/* Size at which to start dump	*/
static	int	dbdump	= 0	 ;	/* Dump flag			*/
static	int	rdsize	= 0	 ;	/* Maximum read size		*/
static	int	wrsize	= 0	 ;	/* Maximum write size		*/
static	int	verbose	= 0	 ;
static	char	*outfile= NULL	 ;	/* Optional output		*/

static	int	docheck	= 0	 ;	/* Checksum data flag		*/
static	int	chkchrs	= 0	 ;	/* Checksum character count	*/
static	int	chkfor	= 0	 ;	/* Number of chacracters to do	*/
static	ulong	chksum	= 0	 ;	/* Current checksum		*/

static	int	fdin	;
static	int	fdout	;
static	int	rdcnt	= 0	 ;
static	int	wrcnt	= 0	 ;
static	int	bkcnt	= 0	 ;
static	fd_set	mask	;
static	int	numfds	;

#define	USAGE	\
	"usage: acv [--help] [-b#] [-c] [-C] [-m#] [-r#] [-w#] [-ofile] [-v]\n"

/* This is the rotate-right from sum.c (in textutils). The comment	*/
/* there says that this is a 32-bit rotate, but it is clearly 16 bit!	*/
#define ROTR(c)	if ((c) & 01)					\
			(c)   = ((c) >> 1) + 0x8000 ;		\
		else	(c) >>= 1 ;

#define	STRERR	strerror(errno)

/*  help	: Output help summary					*/
/*  (returns)	: void		:					*/

static	int	help ()
{
	printf	(USAGE) ;
	printf	("   --help     This help\n") ;
	printf	("       -b#    Set buffer size (n, nK, nM)\n"  ) ;
	printf	("       -c     Checksum data\n"		) ;
	printf	("       -m#    Set buffer margin (n, nK, nM)\n") ;
	printf	("       -r#    Set read size (n, nK, nM)\n"	) ;
	printf	("       -w#    Set write size (n, nK, nM)\n"	) ;
	printf	("       -ofile Output file\n"			) ;
	printf	("       -v     Verbose - show progress\n"	) ;
}

/*  dumpbuf	: Dump data from buffer					*/
/*  (returns)	: void		:					*/

static	void	dumpbuff ()
{
	int	avail	;
	int	put	;

	/* Calculate how much we can load this time, possibly limited	*/
	/* by the maximum write size.					*/
	if (dbinsp <= dbextp)
		avail	= dblen  - dbextp  ;
	else	avail	= dbinsp - dbextp  ;

	if (avail > wrsize) avail = wrsize ;

	/* Write some stuff out, checking for errors ...		*/
	if ((put = write (fdout, &dbuff[dbextp], avail)) < 0)
	{
#ifdef	_NONBLOCK
		if (errno == EAGAIN) return ;
#endif	/* _NOBBLOCK */
		fprintf	(stderr, "acv: write: %s\n", STRERR) ;
		exit	(1) ;
	}

	/* If we failed to write any output then the next process	*/
	/* along the pipe must have closed it's input.			*/
	if (put == 0)
	{	fprintf	(stderr, "acv: output closed\n") ;
		exit	(1) ;
	}

	/* Update the counts and such, and clear the dump flag if the	*/
	/* amount of data in the buffer falls to zero.			*/
	wrcnt	+= put	;
	dbcnt	-= put	;
	dbextp	+= put	;
	if (dbextp >= dblen) dbextp = 0 ;
	if (dbcnt  ==     0) dbdump = 0 ;
}

/*  xsum	: Add block to checksum					*/
/*  buff	: uchar *	: Data buffer				*/
/*  blen	: int		: Buffer length				*/
/*  (returns)	: void		:					*/

static	void	xsum
	(	uchar	*buff,
		int	blen
	)
{
	/* Checksum until the chkfor count is exhausted or until the	*/
	/* buffer is empty.						*/
	while ((chkfor > 0) && (blen > 0))
	{
		/* Perform the sum command BSD style checksum on the	*/
		/* next data byte, the advance the pointer and update	*/
		/* the checksum character counts.			*/
		ROTR(chksum)	  ;
		chksum	+= *buff  ;
		chksum	&= 0xffff ;

		buff	+= 1	  ;
		blen	-= 1	  ;
		chkchrs	+= 1	  ;
		chkfor	-= 1	  ;
	}
}

/*  loadbuff	: Load data into the buffer				*/
/*  (returns)	: int		: Non-zero to terminate copy		*/

static	int	loadbuff ()
{
	int	avail	;
	int	got	;

	/* Calculate how much data we could read in one go, possibly	*/
	/* limited by the maximum read size.				*/
	if (dbinsp < dbextp)
		avail	= dbextp - dbinsp  ;
	else	avail	= dblen  - dbinsp  ;

	if (avail > rdsize) avail = rdsize ;

	/* Read data, checking for errors ....				*/
	if ((got = read (fdin, &dbuff[dbinsp], avail)) < 0)
	{
#ifdef	_NONBLOCK
		if (errno == EAGAIN) return 0 ;
#endif	/* _NONBLOCK */
		fprintf	(stderr, "acv: read: %s\n", STRERR) ;
		exit	(1) ;
	}

	/* If we read zero bytes then we must have reached the end of	*/
	/* the input. Close our end of the pipe and clear the input	*/
	/* file descriptor; the main loop will detect this.		*/
	if (got == 0)
	{	close	(fdin)	;
		fdin	= -1	;
		return	0	;
	}

	/* If checksumming then do so for the block of data which we	*/
	/* have just read. If the 'chkfor' value drops to zero then	*/
	/* we have finished a pass-two checksum, so stop copying.	*/
	if (docheck > 0)
	{	xsum ((uchar *)(&dbuff[dbinsp]), got) ;
		if (chkfor <= 0)
		{	rdcnt	+= got	;
			dbcnt	+= got	;
			return	1 ;
		}
	}

	/* Update the counts and so forth. If the buffer count passes 	*/
	/* the fill point then turn on dumping.				*/
	rdcnt	+= got	;
	dbcnt	+= got	;
	dbinsp	+= got	;
	if (dbinsp >= dblen) dbinsp = 0 ;
	if (dbcnt  > dbfill)
		if (!dbdump)
		{	dbdump	= 1 ;
			bkcnt  += 1 ;
		}

	return	0 ;
}

/*  numval	: Decode numerical argument				*/
/*  opts	: char **	: Arguement				*/
/*  (returns)	: int		: Value					*/

static	int	numval
	(	char	**opts
	)
{
	int	rv	= strtol (*opts, opts, 0) ;

	if (toupper(**opts) == 'K')
	{	rv	*= 1024 ;
		(*opts)	+= 1	;
		return	rv	;
	}
	if (toupper(**opts) == 'M')
	{	rv	*= 1024 * 1024 ;
		(*opts)	+= 1	;
		return	rv	;
	}

	return	rv ;
}

/*  copydata	: Copy data from source to destination			*/
/*  (returns)	: void		:					*/

static	void	copydata ()
{
	int	copying	= 1 ;

	/* The loop continues so long as there is data, that is, the	*/
	/* input stream is still open or there is data in the buffer.	*/
	while (copying && ((fdin >= 0) || (dbcnt > 0)))
	{
		/* Set the select mask. Input is required so long as	*/
		/* the input stream is still open; output when the dump	*/
		/* flag is set.						*/
		FD_ZERO	(&mask) ;
		if ((fdin  < 0) || dbdump)
			FD_SET (fdout, &mask) ;
		if ((fdin >= 0) && (dbcnt < dblen))
			FD_SET (fdin,  &mask) ;

		if (select (numfds, &mask, NULL, NULL, NULL) < 0)
			if (errno != EINTR)
			{	fprintf	(stderr, "acv: select: %s\n", STRERR) ;
				exit	(1) ;
			}
			else	continue ;

		/* Process output and input according to the mask	*/
		/* returned by select. Note that loading may terminate	*/
		/* the copy (and fix the write count so that verbose	*/
		/* output looks correct).				*/
		if ((fdout >= 0) && FD_ISSET (fdout, &mask))
			dumpbuff () ;
		if ((fdin  >= 0) && FD_ISSET (fdin,  &mask))
			if (loadbuff ())
			{	wrcnt	+= dbcnt ;
				dbcnt	 = 0	 ;
				copying	 = 0	 ;
			}

		if (verbose)
		{	fprintf	(stderr, "r=%8ld w=%8ld p=%8ld b=%d\r",
					 rdcnt,
					 wrcnt,
					 dbcnt,
					 bkcnt) ;
			fflush	(stderr) ;
		}
	}

	if (verbose) fprintf (stderr, "\n") ;
}

/*  main	: Standard entry routine				*/
/*  argc	: int		: Count of arguments			*/
/*  argv	: char **	: Argument vector			*/
/*  (returns)	: int		: Exit code				*/

int	main
	(	int	argc,
		char	**argv
	)
{
	ulong	chkone	;

	argv	+= 1 ;
	argc	-= 1 ;

	while ((argc > 0) && (argv[0][0] == '-'))
	{
		char	*opts	= *argv	;

		argv	+= 1	;
		argc	-= 1	;
		opts	+= 1	;

		if (strcmp (opts, "-help") == 0)
		{	help	()  ;
			exit	(1) ;
		}

		while (*opts) switch (*opts++)
		{
			case 'b' :
				dblen	= numval (&opts) ;
				break	;

			case 'c' :
				docheck	= 1 ;
				break	;

			case 'C' :
				docheck	= 2 ;
				break	;

			case 'm' :
				dbmarg	= numval (&opts) ;
				break	;

			case 'r' :
				rdsize	= numval (&opts) ;
				break	;

			case 'w' :
				wrsize	= numval (&opts) ;
				break	;

			case 'o' :
				outfile	= opts	;
				opts	= ""	;
				break	;

			case 'v' :
				verbose	= 1 ;
				break	;

			default	:
				fprintf	(stderr, USAGE) ;
				exit	(1) ;
		}

	}

	/* Checksum can only be used with a named output file, so check	*/
	/* for this case.						*/
	if ((docheck > 0) && (outfile == NULL))
	{	fprintf	(stderr, "acv: check requires -o option\n") ;
		exit	(1) ;
	}

	/* Check which values have been set, defaulting any that have	*/
	/* not. If there is no specified margin then use 1/8th of the	*/
	/* buffer space.						*/
	if (dblen  ==     0) dblen  = DEFLEN ;
	if (rdsize ==     0) rdsize = DEFRD  ;
	if (wrsize ==     0) wrsize = DEFWR  ;
	if (rdsize  > dblen) rdsize = dblen  ;
	if (wrsize  > dblen) wrsize = dblen  ;

	if ((dbmarg == 0) || (dbmarg >= dblen)) dbmarg = dblen / 8 ;
	dbfill	= dblen - dbmarg ;

	if ((dbuff = malloc (dblen)) == NULL)
	{	fprintf	(stderr, "acv: no memory for buffer\n") ;
		exit	(1) ;
	}

	/* If an output file is specified then attempt to open it and	*/
	/* use it. If not then standard output is used.			*/
	if (outfile != NULL)
	{	if ((fdout = open (outfile, O_WRONLY|O_TRUNC|O_CREAT, 0777)) < 0)
		{	fprintf	(stderr, "acv: %s: %s\n", outfile, STRERR) ;
			exit	(1) ;
		}
	}	
	else	fdout	= fileno(stdout) ;

	fdin	= fileno(stdin ) ;

#ifdef	_NONBLOCK
	if (fcntl (fdin,  F_SETFL, O_NONBLOCK) < 0)
	{	fprintf	(stderr, "acv: non-blocking input: %s\n",  STRERR) ;
		exit	(1) ;
	}
	if (fcntl (fdout, F_SETFL, O_NONBLOCK) < 0)
	{	fprintf	(stderr, "acv: non-blocking output: %s\n", STRERR) ;
		exit	(1) ;
	}
#endif	/* _NONBLOCK */

	chkfor	= 0x7ffffff ;
	numfds	= (fdout > fdin ? fdout : fdin) + 1 ;

	copydata () ;

	if (docheck == 0) return 0 ;

	/* If we are checking then close the input and output streams	*/
	/* and reopen them. The output file becomes the input and the	*/
	/* output goes to /dev/null just to get rid of it.		*/
	if (fdin  >= 0) close (fdin ) ;
	if (fdout >= 0) close (fdout) ;

	if ((fdin = open (outfile, O_RDONLY)) < 0)
	{	fprintf	(stderr, "cat: cannot reopen %s: %s\n", outfile, STRERR) ;
		exit	(1) ;
	}
	if ((fdout = open ("/dev/null", O_WRONLY)) < 0)
	{	fprintf	(stderr, "cat: /dev/null: %s\n", STRERR) ;
		exit	(1) ;
	}

	/* Note the checksum from the real copy pass, the reset so that	*/
	/* we can recalculate it. For the -c checksum option, we sum	*/
	/* over all the characters; for the -C option, we only sum over	*/
	/* the number of characters read on the first pass.		*/
	chkone	= chksum	;
	chksum	= 0		;
	chkfor	= docheck == 2 ? chkchrs : 0x7fffffff ;
	chkchrs	= 0		;

	rdcnt	= 0		;
	wrcnt	= 0		;
	bkcnt	= 0		;

	numfds	= (fdout > fdin ? fdout : fdin) + 1 ;

#ifdef	_NONBLOCK
	if (fcntl (fdin,  F_SETFL, O_NONBLOCK) < 0)
	{	fprintf	(stderr, "acv: non-blocking input: %s\n",  STRERR) ;
		exit	(1) ;
	}
	if (fcntl (fdout, F_SETFL, O_NONBLOCK) < 0)
	{	fprintf	(stderr, "acv: non-blocking output: %s\n", STRERR) ;
		exit	(1) ;
	}
#endif	/* _NONBLOCK */

	/* Copy the data the second time. This will recalculate the	*/
	/* checksum using the data that was output the first time.	*/
	copydata () ;

	if (chkone != chksum)
	{	fprintf	(stderr, "acv: checksum error: %5lu -> %5lu\n",
				 chkone, chksum) ;
		exit	(2) ;
	}

	return	0 ;
}
