/***********************************************************/
/* File:  epsCmd.c                                         */
/*                                                         */
/* Function: send command get ink quantity to printer      */
/*           and print remaining ink quantity on stdout    */
/*                                                         */
/* Author:   Jean-Jacques Sarton jj.sarton@t-online.de     */
/*                                                         */
/* Licence:  This is free software according ro GPL        */
/*           the binary rwLib.o is not free. Reverse       */
/*           engrineering is not allowed, however you may  */
/*           use it in own applications.                   */        
/*                                                         */
/* Changes:   DEC-11.99  removed asking status of LP (this */
/*            may cause problems                           */
/*            open, read and write calls are all timed out */
/*            this will avoid locking                      */
/*                                                         */
/* Version: V1.1                                           */
/***********************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <fcntl.h>
#include <linux/lp.h>
#include <sys/time.h>
#include <unistd.h>

#include "rwLib.h"

#define OPEN_WRITE_TIMEOUT 1000

/* internal functions */

int  rw(int argc,char **argv, void (receive(int hd)));
void send(int,char*);
void receiveIQ(int);
void receiveID(int);

typedef struct {
   char *name;
   char *value;
} opt_s;

/* known options and default value */

opt_s opts[] = {
   {"-dev"    , "lp0" },
   {"-timeout", "500" },
   {"-cmd",     "GetId" },
   {"-debug",   "" },
   {NULL,       NULL  }
};

char         *language; /* at this time no suppport */
int           colors[6]; /* holder for ink quantity send back by the printer */
int           debug  = 0;
char          port[100];
unsigned char ibuf[8092];

/****************************************************************
 * Function: parseArgs
 *           replace defaults in the opts argument structure  
 *
 * input:    int   ac      number of parameters
 *           char *av[]    parameter array from the command line
 *
 * Return:   True if ok, False on error 
 *
 ****************************************************************/

int parseArgs(int ac, char **av)
{
   int i;
   
   while ( ac > 0 )
   {
      if ( strcmp(av[0], "-debug") == 0 )
      {
         debug = 1;
      }
      else
      {
         for ( i = 0; opts[i].name; i++ )
         {
            if ( strcmp(opts[i].name, av[0]) == 0 )
            {
               if ( ac > 1 )
               {
                  ac--;
                  av++;
                  opts[i].value = av[0];
                  break;
               }
               else
               {
                  return 0;
               }
            }
         }
      }
      ac--;
      av++;
   }
   return 1;
}

/****************************************************************
 * Function: main
 *           the main function  
 *
 * input:    int   argc    number of parameters
 *           char *argv[]  parameter array from the command line
 *
 * Return:   0 if all is OK 1 on errors 
 *
 ****************************************************************/

void main(int argc, char *argv[])
{
   language = getenv("LANG");
   if ( language == NULL )
   {
      language = getenv("LC_MESSAGES");
      if ( language == NULL )
      {
         language = getenv("LC_MESSAGES");
         if ( language == NULL )
         {
            language = "en";
         }
      }
   }
   if ( strlen(language) > 2 )
      language[2] = '\0';
   else
      language = "en";

   if (parseArgs(argc, argv))
   {
      int   Argc = 0;
      char *Argv[] = {
        "p", opts[0].value, /* lp0 */
        "s", opts[2].value, /* cmd  */
        "d", opts[1].value, /* delay or reset of printer */
        "r", NULL,
        NULL, NULL
      };

      if ( strcmp(opts[2].value, "CleanHead") == 0 )
      {
         Argv[Argc++] = "p";
         Argv[Argc++] = opts[0].value; 
         Argv[Argc++] = "s";
         Argv[Argc++] = opts[2].value;
         Argv[Argc]   = NULL;
         rw(Argc, Argv, NULL);
      }
      else if ( strcmp(opts[2].value, "PrintPattern") == 0 )
      {
         Argv[Argc++] = "p";
         Argv[Argc++] = opts[0].value;
         Argv[Argc++] = "s";
         Argv[Argc++] = opts[2].value;
         Argv[Argc++] = "d";
         Argv[Argc++] = opts[1].value;
         Argv[Argc++] = "s";
         Argv[Argc++] = "Reset";
         Argv[Argc]   = NULL;
         rw(Argc, Argv, NULL);
      }
      else if ( strcmp(opts[2].value, "GetId") == 0 )
      {
         Argv[Argc++] = "p";
         Argv[Argc++] = opts[0].value;
         Argv[Argc++] = "s";
         Argv[Argc++] = opts[2].value;
         Argv[Argc++] = "d";
         Argv[Argc++] = opts[1].value;
         Argv[Argc++] = "r";
         Argv[Argc]   = NULL;
         rw(Argc, Argv, receiveID);
      }
      else if ( strcmp(opts[2].value, "InkQuantity") == 0 )
      {
         memset(colors, -1, sizeof(colors));
         Argv[Argc++] = "p";
         Argv[Argc++] = opts[0].value;
         Argv[Argc++] = "s";
         Argv[Argc++] = opts[2].value;
         Argv[Argc++] = "d";
         Argv[Argc++] = opts[1].value;
         Argv[Argc++] = "r";
         Argv[Argc]   = NULL;
         rw(Argc, Argv, receiveIQ);

         if ( colors[0] == -1 )
         {
            printf("no answer from /dev/%s\n", opts[0].value);
         }
         else if ( colors[3] == -1 )
         {
            printf("C = %2d%%; M = %2d%%; Y = %2d%%\n",
                    colors[0],colors[1], colors[2]);
         }
         else if( colors[4] == -1 )
         {
            printf("K = %2d%%; C = %2d%%; M = %2d%%; Y = %2d%%\n",
                   colors[0],colors[1], colors[2], colors[3]);
         }
         else
         {
            printf("K = %2d%%; C = %2d%%; M = %2d%%; Y = %2d%%; c = %2d%%; m = %2d%%\n",
                   colors[0],colors[1], colors[2], colors[3],colors[4], colors[5]);
         }
      }  
      else
      {
         exit(1);
      }
   }
   else
   {
      exit(1);
   }
   exit(0);
}

/*********************************************************************/
/* this was taken from the program rw and modified so that the       */
/* control sequences are written from a precompiled module.          */
/* The precompiled module dont use library functions and or system  */
/* calls. This is for avoiding problems due to different             */
/* distributions and DNA restrictions                                */
/* All need to be rewritten in a good manner, but I had not time to  */
/* do this.                                                          */           
/*********************************************************************/


/*********************************************************************/
/* Function:   sigAlarm                                              */
/*                                                                   */
/*             time alarm handler                                    */
/*                                                                   */
/* arguments:  no important                                          */
/*                                                                   */
/*                                                                   */
/* return:    none                                                   */
/*                                                                   */
/*********************************************************************/

void sigAlarm(int code)
{
}

/*********************************************************************/
/* Function:   receiveIQ                                             */
/*                                                                   */
/*              read from the port the ink quantity message          */
/*                                                                   */
/* arguments: int hd  handle to the opened port                      */
/*                                                                   */
/*                                                                   */
/* return:    none                                                   */
/*                                                                   */
/*********************************************************************/

void receiveIQ(int hd)
{
   char hexV[10];
   char *s;
   memset(ibuf,0,sizeof(ibuf));
   memset(hexV,0,sizeof(hexV));
   s = ibuf;

   if (read(hd, ibuf, sizeof(ibuf))>0)
   {
      if ( debug )
      {
         printf("============= received =============\n");
         printf("%s",ibuf);
         printf("\n====================================\n");
      }  
      while (*s )
      {  
         if ( strncmp(s, "IQ:", 3) == 0 )
         {
            s+=3;
            memset(colors, -1, sizeof(colors));
            hexV[0]= s[0];
            hexV[1]= s[1];
            hexV[2]= 0;
            sscanf(hexV, "%x",&colors[0]);
            s +=2;
            
            hexV[0]= s[0];
            hexV[1]= s[1];
            hexV[2]= 0;
            sscanf(hexV, "%x",&colors[1]);
            s +=2;

            hexV[0]= s[0];
            hexV[1]= s[1];
            hexV[2]= 0;
            sscanf(hexV, "%x",&colors[2]);
            s +=2;

            if ( s[0] != ';' )
            {
               hexV[0]= s[0];
               hexV[1]= s[1];
               hexV[2]= 0;
               sscanf(hexV, "%x",&colors[3]);
                s +=2;
 
               if ( s[0] != ';' )
               {
                  hexV[0]= s[0];
                  hexV[1]= s[1];
                  hexV[2]= 0;
                  sscanf(hexV, "%x",&colors[4]);
                  s +=2;

                  if ( s[0] != ';' )
                  {
                     hexV[0]= s[0];
                     hexV[1]= s[1];
                     hexV[2]= 0;
                     sscanf(hexV, "%x",&colors[5]);
                  }
               }
            }
            return;
         }
         else
         {
            s++;
         }
      }   
      memset(ibuf,0,sizeof(ibuf));
      s = ibuf;
   }
}

/*********************************************************************/
/* Function:   receiveIQ                                             */
/*                                                                   */
/*             read from the port the ink quantity message           */
/*                                                                   */
/* arguments: int hd  handle to the opened port                      */
/*                                                                   */
/*                                                                   */
/* return:    none                                                   */
/*                                                                   */
/*********************************************************************/

void receiveID(int hd)
{
   char *s,*t;
   memset(ibuf,0,sizeof(ibuf));
   s = ibuf;

   if (read(hd, ibuf, sizeof(ibuf)) > 0)
   {
      if ( debug )
      {
         printf("============= received =============\n");
         printf("%s",ibuf);
         printf("\n====================================\n");
      }  
      while (*s )
      {  
         if ( strncmp(s, "DES:", 4) == 0 || strncmp(s, "MDL:", 4) == 0 )
         {
            s +=4;
            t = s;
            while(*t && *t != ';')
               t++;
            *t = '\0';
            printf("%s\n",s);
            return;
         }
         else
         {
            s++;
         }
      }
      memset(ibuf,0,sizeof(ibuf));
      s = ibuf;
   }
}

/*********************************************************************/
/* Function:   rw                                                    */
/*                                                                   */
/*             read and write to the device passed by the arg p      */
/*                                                                   */
/* arguments:                                                        */
/*                                                                   */
/* return:    0 if OK 1 on error                                     */
/*                                                                   */
/*********************************************************************/

#define RESET_TIMER(ti,oti) { memset((void*)&ti,0,sizeof(ti)); \
                              memset((void*)&oti,0,sizeof(oti)); \
                              setitimer( ITIMER_REAL ,&ti, &oti); \
                            }

#define SET_TIMER(ti,oti,val) { signal(SIGALRM, sigAlarm); \
                                memset((void*)&ti,0,sizeof(ti)); \
                                memset((void*)&oti,0,sizeof(oti)); \
                                ti.it_value.tv_sec  = val/1000; \
                                ti.it_value.tv_usec = (val%1000)*1000; \
                                setitimer( ITIMER_REAL ,&ti, &oti); \
                              }

int rw(int argc,char **argv, void (receive(int hd)))
{
   int   io  = 0;
   int   len = 0;
   int   i;
   int   ret = 0;
   struct  itimerval ti, oti;
   int   delay;

   *port = 0;
      
   /* main loop for processing the arguments */
   while (argc)
   {
      if (strcmp("p",argv[0]) == 0 )
      {
         argc--;
         argv++;
         
         /* dont allow consecutive handling of ports */
         if ( io )
         {
            close(io);
            if ( debug )
            {
               fprintf(stderr,"port declared twice\n");
            }
            return(1);
         }
         
         /* set path of device */
         sprintf(port,"/dev/%s",argv[0]);


         /* open the device and perform checks */
         /* if the port id locked we will hang */
         if ( debug )
         {
             fprintf(stderr,"try to open %s\n",port);
         }

         /* set wait time and call timer */
         SET_TIMER(ti,oti,OPEN_WRITE_TIMEOUT);
         
         io = open(port,O_RDWR);
         if ( io <= 0 )
         {
            if ( debug )
            {
               fprintf(stderr,"cant open %s\n",port);
            }
            return(1);
         }
         RESET_TIMER(ti,oti);

         if ( debug )
         {
             fprintf(stderr,"opened %s, got handle %d\n",port,io);
         }
      }
      else if (strcmp("s",argv[0]) == 0  )
      {
         /* send arg to the device */
         if ( io == 0 )
         {
            if ( debug )
            {
               fprintf(stderr,"port not given\n");
            }
            exit(1);
         }
         argc--;
         argv++;

         if ( debug )
         {
            fprintf(stderr,"send command %s\n",argv[0]);
         }

         /* set wait time and call timer */
         SET_TIMER(ti,oti,OPEN_WRITE_TIMEOUT);

         if ( SendCmd(io,argv[0], write) < 1 )
         {
            if ( debug )
            {
               fprintf(stderr,"cant write to %s\n",port);
            }
            exit(1);
         }
         RESET_TIMER(ti,oti);
      }
      else if ( strcmp("d",argv[0]) == 0 )
      {
         /* wait for the given time by polling stderr */
         argc--;
         argv++;
     
         delay = atoi(argv[0]);
         fflush(stdout);

         /* set wait time and call timer */
         SET_TIMER(ti,oti,delay);

         /* wait for sigalarm */
         pause();
      }
      else if (strcmp("r",argv[0]) == 0  )
      {
         /* read from port */  
         if ( io == 0 )
         {
            if ( debug )
            {
               fprintf(stderr,"port not given\n");
            }
            exit(1);
         }
         if ( debug )
         {
            fprintf(stderr,"wait %d ms before reading\n",delay);
         }
         SET_TIMER(ti,oti,delay);
         receive(io);
         RESET_TIMER(ti,oti);
      }
      argc--;
      argv++;
   }

   if ( io )
      close(io);
   return(0);
}

/********************************************************************/
/*                                                                  */
/* Copyright (C) JJ Sarton 11999 All Rights Reserved  Confidential  */
/*                                                                  */
/********************************************************************/
