/*

    xfstool - manage floppy mounts/umounts/mkfs and shutdown for users
    Copyright (C) 1994 Torsten Eichner

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    email: 	eichner@rhrk.uni-kl.de
    papermail:	Torsten Eichner, Kampshardtweg 1, D-57587 Birken-Honigsessen, Germany

*/

/*
    Start of file: xfstool.c
*/

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Paned.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Label.h>

#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>

#include <stdio.h>
#include <unistd.h>
#include <string.h>


#include "disk_mount.h"
#include "disk_umount.h"
#include "disk_format.h"
#include "shutdown.h"

#define ERROR(string) { char error[512]; sprintf(error, "[%s:%d]: %s\n", __FILE__, __LINE__, string); XtError(error); }

Pixmap disk_mount, disk_umount, disk_format, shutdown;
Widget w_shell, w_main, w_command1, w_command2, w_command3, w_command4;
XtAppContext context;
int yesno_flag, message_flag;

static void message_cb(Widget w, XtPointer cld, XtPointer cad) {
	message_flag = 0;
	}

static int message(char *string) {
	Widget w_message_shell, w_message_wnd, w_message_text, w_message_ok, w_message_box;
	Dimension width, height;
	Position x, y;

	XtVaGetValues(w_shell, XtNwidth, &width, XtNheight, &height, NULL);
	XtTranslateCoords(w_shell, (Position) width/2, (Position) height/2, &x, &y);

	w_message_shell = XtVaCreatePopupShell("Message", transientShellWidgetClass, w_shell, XtNx, x, XtNy, y, NULL, 0);
	w_message_wnd = XtVaCreateManagedWidget(NULL, panedWidgetClass, w_message_shell, NULL);
	w_message_text = XtVaCreateManagedWidget(NULL, labelWidgetClass, w_message_wnd, XtNshowGrip, False, XtNlabel, string, NULL);
	w_message_box = XtVaCreateManagedWidget(NULL, boxWidgetClass, w_message_wnd, XtNskipAdjust, True, NULL);
	w_message_ok = XtVaCreateManagedWidget(NULL, commandWidgetClass, w_message_box, XtNlabel, "OK", NULL);	

	XtAddCallback(w_message_ok, XtNcallback, message_cb, NULL);

	XtPopup(w_message_shell, XtGrabExclusive);

	message_flag = -1;

	while (message_flag == -1) {
		XEvent event;

		XtAppNextEvent(context, &event);
		XtDispatchEvent(&event);
		}

	XtDestroyWidget(w_message_shell);
	}

static void do_mount(void) {
	int pd[2];
	int pid;
	int status;
	char buffer[512];
	int n;

	if (pipe(pd) != 0)
		ERROR("cannot create pipe");

	mkdir(DIR, 0755);
#ifdef SOUND
        system("play /cd/extras/sound/wav/clink2.wav") ;
#endif


	if ((pid = fork()) == 0) {
		setuid(0);

		close(2);	
		close(pd[0]);

		dup(pd[1]);

		execl(MOUNT, "mount", "-t", FSTYPE, "-o", "nosuid,nodev", FLOPPY, DIR, NULL);

		write(2, "Cannot execute MOUNT", 21);
		exit(-1);
		}

	close(pd[1]);
	waitpid(pid, &status, 0);

	if (status) {
		n = read(pd[0], buffer, 512);
		n = (n < 0) ? 0 : n;
		buffer[n] = '\0';
		message(buffer);
		}
	else
		chown(DIR, getuid(), getgid());

	close(pd[0]);
	}

static void do_umount(int flag) {
	int pd[2];
	int pid;
	int status;
	char buffer[512];
	int n;

	if (pipe(pd) != 0)
		ERROR("cannot create pipe");
#ifdef SOUND
        system("play /cd/extras/sound/wav/clink2.wav") ;
#endif

	if ((pid = fork()) == 0) {
		setuid(0);

		close(2);	
		close(pd[0]);

		dup(pd[1]);

		execl(UMOUNT, "umount", FLOPPY, NULL);

		write(2, "Cannot execute UMOUNT", 22);
		exit(-1);
		}

	close(pd[1]);
	waitpid(pid, &status, 0);

	if (status) {
		n = read(pd[0], buffer, 512);
		n = (n < 0) ? 0 : n;
		buffer[n] = '\0';
		if (!flag)
			message(buffer);
		}
	else
		rmdir(DIR);

	close(pd[0]);
	}

static void do_format(void) {
	int pd[2];
	int pid;
	int status;
	char buffer[512];
	int n;

	do_umount(1);

	if (pipe(pd) != 0)
		ERROR("cannot create pipe");

	if ((pid = fork()) == 0) {
		setuid(0);

		close(2);	
		close(pd[0]);

		dup(pd[1]);

		execl(MKFS, "mkfs", "-c", FLOPPY, NULL);

		write(2, "Cannot execute MKFS", 20);
		exit(-1);
		}

	close(pd[1]);
	waitpid(pid, &status, 0);

	if (status) {
		n = read(pd[0], buffer, 512);
		n = (n < 0) ? 0 : n;
		buffer[n] = '\0';
		message(buffer);
		}

	close(pd[0]);
	}

static void do_shutdown(void) {
	int pd[2];
	int pid;
	int status;
	char buffer[512];
	int n;

	if (pipe(pd) != 0)
		ERROR("cannot create pipe");
#ifdef SOUND
        system("play /cd/extras/sound/wav/mind.wav") ;
#endif

	chdir("/");

	if ((pid = fork()) == 0) {
		setuid(0);
		chdir("/");

		close(0);	
		close(1);
		close(2);
		close(pd[1]);

		setpgrp();

		dup(pd[0]);

		sync();
		sync();
		sync();

		execl(SHUTDOWN, "shutdown", "-r", DELAY, NULL);

		exit(-1);
		}

	close(pd[0]);

	write(pd[1], SHUTDOWNMSG, strlen(SHUTDOWNMSG)+1);
	close(pd[1]);

	waitpid(pid, &status, 0);

	message("Cannot shutdown system!");
	}

static int get_permission(void) {
        char username[512];
        char buffer[512];
        int perm = 0;
        FILE *stream;
        char token1[512];
        char token2[512];
        int i, j;

        if (getpw(getuid(), username) < 0)
                ERROR("cannot get username");

        for (i = 0; i < strlen(username); i++)
                if (username[i] == ':')
                        username[i] = '\0';

        if ((stream = fopen(PERM, "r")) == NULL)
                ERROR("cannot read permission-file");

        while (!feof(stream)) {
                fgets(buffer, 512, stream);

                for (i = 0; (i < strlen(buffer)) && isspace(*(buffer+i)); i++);

                for (j = 0; (i < strlen(buffer)) && !isspace(*(buffer+i)); i++, j++)
                        token1[j] = *(buffer+i);
                token1[j] = '\0';

                for (; (i < strlen(buffer)) && isspace(*(buffer+i)); i++);

                for (j = 0; (i < strlen(buffer)) && !isspace(*(buffer+i)); i++, j++)
                        token2[j] = *(buffer+i);
                token2[j] = '\0';

                if (strcmp(token1, username) == 0)
                        for (i = 0; i < strlen(token2); i++)
                                switch(*(token2+i)) {
                                        case 'f':
                                                perm |= 1;
                                                break;
                                        case 's':
                                                perm |= 2;
                                                break;
                                        default:
                                                ERROR("parse error in permission-file");
                                        }
                }

        fclose(stream);

        return perm;
        }

static void yesno_cb(Widget w, XtPointer cld, XtPointer cad) {
	yesno_flag = (int) cld;
	}

static int yesno(char *string) {
	Widget w_really_shell, w_really_wnd, w_really_text, w_really_yes, w_really_no, w_really_box;
	Dimension width, height;
	Position x, y;

	XtVaGetValues(w_shell, XtNwidth, &width, XtNheight, &height, NULL);
	XtTranslateCoords(w_shell, (Position) width/2, (Position) height/2, &x, &y);

	w_really_shell = XtVaCreatePopupShell("Question", transientShellWidgetClass, w_shell, XtNx, x, XtNy, y, NULL, 0);
	w_really_wnd = XtVaCreateManagedWidget(NULL, panedWidgetClass, w_really_shell, NULL);
	w_really_text = XtVaCreateManagedWidget(NULL, labelWidgetClass, w_really_wnd, XtNshowGrip, False, XtNlabel, string, NULL);
	w_really_box = XtVaCreateManagedWidget(NULL, boxWidgetClass, w_really_wnd, XtNskipAdjust, True, NULL);
	w_really_yes = XtVaCreateManagedWidget(NULL, commandWidgetClass, w_really_box, XtNlabel, "Yes", NULL);	
	w_really_no = XtVaCreateManagedWidget(NULL, commandWidgetClass, w_really_box, XtNlabel, "No", NULL);	

	XtAddCallback(w_really_yes, XtNcallback, yesno_cb, (XtPointer) 1);
	XtAddCallback(w_really_no, XtNcallback, yesno_cb, (XtPointer) 0);

	XtPopup(w_really_shell, XtGrabExclusive);

	yesno_flag = -1;

	while (yesno_flag == -1) {
		XEvent event;

		XtAppNextEvent(context, &event);
		XtDispatchEvent(&event);
		}

	XtDestroyWidget(w_really_shell);

	return yesno_flag;
	}

static void mount_cb(Widget w, XtPointer cld, XtPointer cad) {
	do_mount();
	}

static void umount_cb(Widget w, XtPointer cld, XtPointer cad) {
	do_umount(0);
	}

static void format_cb(Widget w, XtPointer cld, XtPointer cad) {
	if (yesno("Do you really want to format the disk?")) {
		do_format();
		}
	}

static void shutdown_cb(Widget w, XtPointer cld, XtPointer cad) {
	if (yesno("Do you really want to shutdown the computer?")) {
		do_shutdown();
		}
	}

int main(int argc, char *argv[]) {
	int perm;

	chdir("/");
	setpgrp();

	perm = get_permission();

	if (perm == 0)
		ERROR("you are not allowed to use this application!");

	w_shell = XtAppInitialize(&context, "xfstool", NULL, 0, &argc, argv, NULL, NULL, 0);

	w_main = XtVaCreateWidget(NULL, panedWidgetClass, w_shell, NULL);	

	disk_mount = XCreateBitmapFromData(XtDisplay(w_shell), RootWindowOfScreen(XtScreen(w_shell)), disk_mount_bits, disk_mount_width, disk_mount_height);
	w_command1 = XtVaCreateWidget(NULL, commandWidgetClass, w_main, XtNbitmap, disk_mount, XtNmax, 24, XtNmin, 24, XtNshowGrip, False, NULL);
	disk_umount = XCreateBitmapFromData(XtDisplay(w_shell), RootWindowOfScreen(XtScreen(w_shell)), disk_umount_bits, disk_umount_width, disk_umount_height);
	w_command2 = XtVaCreateWidget(NULL, commandWidgetClass, w_main, XtNbitmap, disk_umount, XtNmax, 24, XtNmin, 24, XtNshowGrip, False, NULL);
	disk_format = XCreateBitmapFromData(XtDisplay(w_shell), RootWindowOfScreen(XtScreen(w_shell)), disk_format_bits, disk_format_width, disk_format_height);
	w_command3 = XtVaCreateWidget(NULL, commandWidgetClass, w_main, XtNbitmap, disk_format, XtNmax, 24, XtNmin, 24, XtNshowGrip, False, NULL);
	shutdown = XCreateBitmapFromData(XtDisplay(w_shell), RootWindowOfScreen(XtScreen(w_shell)), shutdown_bits, shutdown_width, shutdown_height);
	w_command4 = XtVaCreateWidget(NULL, commandWidgetClass, w_main, XtNbitmap, shutdown, XtNmax, 24, XtNmin, 24, XtNshowGrip, False, NULL);

	XtAddCallback(w_command1, XtNcallback, mount_cb, NULL);
	XtAddCallback(w_command2, XtNcallback, umount_cb, NULL);
	XtAddCallback(w_command3, XtNcallback, format_cb, NULL);
	XtAddCallback(w_command4, XtNcallback, shutdown_cb, NULL);

	if (perm & 1) {
		XtManageChild(w_command1);
		XtManageChild(w_command2);
		XtManageChild(w_command3);
		}

	if (perm & 2) {
		XtManageChild(w_command4);
		}

	XtManageChild(w_main);

	XtRealizeWidget(w_shell);

	XtAppMainLoop(context);	
	}


/*
    End of file
*/
