/*
** sdir.c
**
** (c) 1997 Stephan Fuhrmann
*/

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>

#include <dirent.h>

#include <string.h>

/* if on SUNOS, use dirent.h instead if non-existing dir.h */
#ifdef SUNOS
#include <sys/dirent.h>
#else
#include <sys/dir.h>
#endif

#include <sys/stat.h>
#include <sys/types.h>

#define VERSION "0.50"

#ifndef S_ISDIR
#define S_ISDIR(m) ((m & S_IFDIR)==S_IFDIR)
#endif

#ifndef S_ISLNK
#define S_ISLNK(m) ((m & S_IFLNK)==S_IFLNK)
#endif

/*#define DEBUG 1*/
#undef DEBUG

#ifdef DEBUG
#define D(x) (x);
#define bug printf
#else
#define D(x) ;
#define bug printf
#endif


/* environment for parseopts */
#define MINUS '-'
#define PLUS '+'
#define FLAGS "Flags"

char *help[]={"?","-?","--?","-help","--help"};

struct parseopts
{
	char *activator1;
	char *activator2;
	int defaultval;
	int minusval;
	int plusval;
	int optsoffset;
	char *description;
};

enum opts {OPT_VERBOSE=0,OPT_LINKS,OPT_PEDANTIC,OPT_NUM};

struct parseopts mypo[]=
{
	{"v","V",0,-1,0,OPT_VERBOSE,"Verbose recursive output"},
	{"l","L",0,-1,0,OPT_LINKS,"Follow links"},
	{"p","P",0,-1,0,OPT_PEDANTIC,"Pedantically report errors"}
};

#define H_PREFIX "sdir v"VERSION" (c) 1997 Stephan Fuhrmann\n\nSynopsis: sdir <flags> dir1 <dir2> ...\n"

/* end of env for parseopts */

#ifndef offsetof
#define offsetof(type,memb) ((size_t) &(((type *) 0L)->memb))
#endif

long recurse (char *dirname,long level,int opts[OPT_NUM]);
char *nicesize(int bytes);
char *mygetcwd(void);

int parseopts (int *argc,char *argv[],int pnum,struct parseopts *po,int opts[OPT_NUM]);

#define PLEN 1024

char path[PLEN];

/* globale Statistik */
int dir;
int file;

int main(int argc,char *argv[])
{
	int cnt;
	long allsize=0;

	int opts[OPT_NUM];

	dir=0;
	file=0;

	if (parseopts(&argc,argv,OPT_NUM,mypo,opts))
	{
		D(bug("argc is %d\n",argc));
		for (cnt=1;cnt<argc;cnt++)
		{
			D(bug("recurs'ing %s.\n",argv[cnt]));
			allsize+=recurse(argv[cnt],0,opts);
		}

		if (dir || file)
			printf ("Calculated size: %ld byte%s (%s) (%d dir%s, %d file%s).\n",
			allsize,allsize==1?"":"s",
			nicesize(allsize),
			dir,dir==1?"":"s",
			file,file==1?"":"s");
	}

	if (argc<2 || ((!dir) && (!file)))
	{
		int forcehelp=0;
		parseopts(&forcehelp,argv,OPT_NUM,mypo,opts);
	}

	return (0);
}

int parseopts (int *argc,char *argv[],int pnum,struct parseopts *po,int opts[OPT_NUM])
{
	int pcnt;
	int cnt;
	int res=1;

	if (*argc==0 || *argc==2)
	{
		int he=0;
		
		if (*argc)
			for(cnt=0;cnt<(sizeof(help)/sizeof(char *));cnt++)
				if (!strcmp(help[cnt],argv[1]))
				{
					he=1;
					break;
				}

		if (he || (!*argc))
		{
			fputs (H_PREFIX,stdout);
			for (cnt=0;cnt<pnum;cnt++)
			{
				if (!po[cnt].activator2)
				{
					printf ("\t%s: (-|+)%s: %s\n",FLAGS,
										po[cnt].activator1,po[cnt].description);
				}
				else
				{
									printf ("\t%s: (-|+)(%s|%s): %s\n",FLAGS,
										po[cnt].activator1,po[cnt].activator2,po[cnt].description);
				}
			}
			return (0);
		}
	}

	/* initialize option array to default values */	
	for (pcnt=0;pcnt<pnum;pcnt++)
	{
		opts[po[pcnt].optsoffset]=po[pcnt].defaultval;

		D(bug("set %d to %d\n",po[pcnt].optsoffset,po[pcnt].defaultval));
	}

	/* parse options */
	for (cnt=1;cnt<*argc;)
	{

		D(bug("Now analyzing %s.\n",argv[cnt]));

		if (*argv[cnt]==MINUS || *argv[cnt]==PLUS)
		{
			char *al=argv[cnt]+1;
			int allen=strlen(al);
			int match=0;

			while (allen>0)
			{
				match=0;
				
				for (pcnt=0;pcnt<pnum;pcnt++)
				{
					int a1,a2,a1l,a2l;

					a1l=strlen(po[cnt].activator1);
					a2l=po[pcnt].activator2 ? strlen(po[cnt].activator2) : 0;
					
					if ((allen < a1l) && (allen < a2l))
						continue;

					a1=strncmp(po[pcnt].activator1,al,a1l);
					/* if no activator set, it doesn't match */
					/* else compare */
					a2=po[pcnt].activator2 ? strncmp(po[pcnt].activator2,al,a2l) : -1;

					if ((!a1) || (!a2))
					{
						opts[po[pcnt].optsoffset]=((*argv[cnt]==MINUS) ? po[pcnt].minusval :po[pcnt].plusval);

						D(bug("option %d found and set to %d.\n",po[pcnt].optsoffset,opts[po[pcnt].optsoffset]));

						match=1;
						
						if (!a1)
						{
							allen-=a1l;
							al+=a1l;
						}
						else
						{
							allen-=a2l;
							al+=a2l;
						}

						break;
					}
				}
				
				if (!match)
				{
					printf ("Error parsing options!\n");
					res=0;
					break;
				}
			}

			if (match)
			{
				/* move other args "over" the parsed option */
				memmove(&argv[cnt],&argv[cnt+1],(sizeof(char *)) * (*argc-(cnt+1)));
				(*argc)--;
			}
			else
				break;
		}
		else /* if PLUS || MINUS */
		{
			cnt++;
		}
	} /* for argc */

	return (res);
}

long recurse (char *dirname,long level,int opts[OPT_NUM])
{
	char *workname;
	DIR *dfd;
	struct stat st;
	struct stat linkst;
	struct stat *act;
	
	long cnt;
	long ret=0;

	if ((workname=mygetcwd ()))
	{
		if (!chdir(dirname))
		{
			if ((dfd=opendir (".")))
			{
				struct dirent *dptr;

				dir++;
		
				while ((dptr=readdir(dfd)))
				{
					int csiz=0;

					act=&st;
					
					if (lstat (dptr->d_name,&st)==-1)
					{
						perror ("sdir/lstat()");
						continue;
					}

					if (S_ISLNK(st.st_mode))
					{
						int ok=0;
						int len;
						
						if ((len=readlink (dptr->d_name,path,PLEN))==-1)
						{
							if (opts[OPT_PEDANTIC] || errno!=ENOENT)
								perror ("sdir/readlink()");

							continue;
						}
						else
						{
							path[len]=0;
							
							if (stat (path,&linkst)==-1)
							{
								/* this fails when links are pointing to nowhere */
								if (opts[OPT_PEDANTIC] || errno!=ENOENT)
								{
									printf ("Error while stat()ing %s:\n",path);
									perror ("sdir/stat()");
								}

								continue;
							}


							act=&linkst;
							ok=S_ISDIR(linkst.st_mode);
						}
						
						if (ok && (!opts[OPT_LINKS]))
						{
							if (opts[OPT_VERBOSE])
								printf ("%s - link skipped\n",dptr->d_name);

							continue;
						}
					}

					if (S_ISDIR(act->st_mode))
					{
						if (strcmp(".",dptr->d_name) && strcmp ("..",dptr->d_name))
						{
							if (opts[OPT_VERBOSE])
								printf("Entering %s (dir)\n",dptr->d_name);

							csiz=recurse (dptr->d_name,level+1,opts);
							ret+=csiz;
						}
					}
					else
					{
						ret+=act->st_size;
						file++;
					}

					if ((!level) || opts[OPT_VERBOSE])
					{
						int nlen=strlen(dptr->d_name);
						
						for (cnt=0;cnt<level;cnt++)
							fputc (' ',stdout);

						printf ("%s%*s(%s) (%s)\n",dptr->d_name,
						40-nlen,"",
						S_ISDIR(act->st_mode)?nicesize(csiz):nicesize(act->st_size),
						S_ISDIR(act->st_mode)?"dir":"file");
					}
				}
		
				closedir (dfd);
			}
			else
				perror ("sdir/opendir()");

			if (chdir (workname))
				perror ("sdir/chdir()");
		}
		else
			perror ("sdir/chdir()");

		free (workname);
	}
	else
		perror ("sdir/mygetcwd()");

	return (ret);
}

#define NLEN 32

#define GMIN 1.0
#define MMIN 1.0
#define KMIN 1.0

char *nicesize(int bytes)
{
	static char nstr[NLEN];

	double kb=((double)bytes)/1024;
	double mb=kb/1024;
	double gb=mb/1024;

	if (gb>=GMIN)
	{
		sprintf (nstr,"%.2fGB",gb);
	}
	else
	if (mb>=MMIN)
	{
		sprintf (nstr,"%.2fMB",mb);		
	}
	else
	if (kb>=KMIN)
	{
		sprintf (nstr,"%.2fKB",kb);
	}
	else
		sprintf (nstr,"%d B",bytes);

	return (nstr);
}

/* universal getcwd for most systems out there */
#define LAMESIZE 16	/* try it in 128 byte steps */
#define MAXSTORE 4096	/* stop if the name is longer than 4 kb
													hope people have shorter paths ;) */
char *mygetcwd(void)
{
	char *result=0;
	char *gres;
	size_t ls=LAMESIZE;	
	int cont=1;

	D(bug("Hello, this is mygetcwd()\n"));

	while (cont && (result=malloc(ls)))
	{
		gres=getcwd(result,ls);

		D(bug("Call to getcwd(%p,%ld)=%p\n",result,ls,gres));

		if (gres)
		{
			break;
		}
		else
		{
			switch (errno)
			{
				/* when size too small, increase and retry */
				case ERANGE:
					ls+=LAMESIZE;
					D(bug("got ERANGE error, retrying with size %ld.\n",ls));
					free (result);
					result=0;
					cont=1;
					break;
				/* when something else occurs, abort */
				case ENOMEM:
				default:
					D(bug("got %ld error, aborting.\n",errno));
					free (result);
					result=0;
					cont=0;
					break;
			}
		}
	}

	if (!result)
		errno=ENOMEM;

	return(result);	
}
