#include <scsilib.h>

#include <getopt.h>

struct option long_options[] =
{
    {"timeout",	    1,  NULL,   't'},
    {"lba",	    1,  NULL,   'a'},
    {"count",	    1,  NULL,   'c'},
    {"pattern",     1,  NULL,   'p'},
    {"PBdata",	    0,	NULL,	'P'},
    {"LBdata",	    0,	NULL,	'L'},
    {"RELADDR",	    0,	NULL,	'R'},
    {"help",	    0,  NULL,	'h'},
    {NULL, 0, NULL, 0}
};

static char *short_options = "t:a:c:p:PLRh";



int main(int argc, char **argv)
{
    struct scsi_fd	*h = NULL;
    char		*dev;
    char		opt;
    int			i;

    void		*data = NULL;

    int	    timeout = ( 1000 * 10 );
    u32	    lba = 0;
    u32	    count = 1;
    u32	    pattern = 0x00000000;
    u32	    flags = 0;
    int	    no_io = 0;

    struct scsi_capacity *cap = NULL;


    while(1) {
	opt = getopt_long (argc, argv,short_options,long_options, &i);
        if(opt == -1)
	    break;

	switch(opt) {
	    case('t'):
		timeout = strtoul(optarg, NULL, 0);
		if(errno) {
		    fprintf(stderr, "Could not parse timeout - %m\n");
		    exit(1);
		}

		timeout *= 1000;

		break;

	    case('a'):
		lba = strtoul(optarg, NULL, 0);
		if(errno) {
		    fprintf(stderr, "Could not parse logicl block address - %m\n");
		    exit(1);
		}

		break;

	    case('c'):
		count = strtoul(optarg, NULL, 0);
		if(errno) {
		    fprintf(stderr, "Could not parse sector count - %m\n");
		    exit(1);
		}

		if(count > 65535) {
		    fprintf(stderr, "Sector count is out of bounds.\n");
		    exit(1);
		}

		break;

	    case('p'):
		pattern = strtoul(optarg, NULL, 0);
		if(errno) {
		    fprintf(stderr, "Could not parse pattern argument- %m\n");
		    exit(1);
		}

		break;

	    case('P'):
		flags |= F_PBDATA;
		break;

	    case('L'):
		flags |= F_LBDATA;
		break;

	    case('R'):
		flags |= F_RELADDR;
		break;

	    case('h'):
		fprintf(stdout, 
"Usage:      read_6 [options] device

options:    --dry-run  -n     Do no I/O on target.
            --timeout=N       Set timout for the format command.
            --lba=N           Set the logical sector address.  [0]
	    --count=N         Set the sector count.  [1]
	    --pattern         Set the pattern word.  [0x00000000]
	    --PBdata          Mark each sector with its physical address.
	    --LBdata          Mark each sector with its logical address.
	    --RELADDR         Set relative address flag.
");
	    exit(1);

	    case('?'):
	    default:
		fprintf(stderr, "Bad option.\n");
		exit(1);
	}
    }

    if(optind == argc) {
	fprintf(stderr, "No device name specified.\n");
	exit(1);
    }

    if(optind < (argc - 1)) {
	fprintf(stderr, "Too many arguments.\n");
	exit(0);
    }


    dev = argv[optind];

    if(!(h = scsi_open_dev(dev, 0, 0))) {
	fprintf(stderr, "Could not open %s - %m\n", argv[1]);
	exit(1);
    }

    scsi_set_timeout(h, timeout);

    if(no_io)
	goto cleanup;


    if(!(cap = scsi_READ_CAPACITY(h, 0, 0))) {
	fprintf(stderr, "Could not read device capacity - %m\n");
	goto cleanup;
    }

    h->sector_size = cap->sector_size;
    h->capacity = cap->capacity;

    if(!(data = malloc(h->sector_size))) {
	fprintf(stderr, "malloc(%d) - %m\n", h->sector_size);
	goto cleanup;
    }

    memset(data, pattern, h->sector_size);

    if(h->sector_size > h->len) {
	if(!(scsi_set_sg_reserve_size(h, h->sector_size))) {
	    fprintf(stderr, "Could not set the kernel buffer size - %m\n");
	    goto cleanup;
	}
    }

    if(!scsi_WRITE_SAME(h, data, lba, count, flags)) {
	if(errno) {
	    fprintf(stderr, "*** WRITE SAME failed - %m\n");
	    goto cleanup;
	}

	fprintf(stderr, "*** WRITE SAME did not complete successfully.\n");
	goto cleanup;
    }

    fprintf(stdout, "Wrote same succeeded.\n");

cleanup:

    if(h->hdr.masked_status == CHECK_CONDITION) {
	scsi_cdb_dump(stderr, (void *)h->hdr.cmdp);
	scsi_sense_dump(stderr, h->hdr.sbp);
    }

    if(cap)
	free(cap);

    if(data)
	free(data);

    if(h)
	scsi_close_dev(h);

    exit(0);
}
