#include <scsilib.h>

#include <getopt.h>

struct option long_options[] =
{
    {"timeout",	    1,  NULL,   't'},
    {"lba",	    1,  NULL,   'a'},
    {"lss",         1,  NULL,   'l'},
    {"RELADDR",	    0,  NULL,   'R'},
    {"help",	    0,  NULL,	'h'},
    {NULL, 0, NULL, 0}
};

static char *short_options = "t:a:l:nh";



int main(int argc, char **argv)
{
    struct scsi_fd  *h = NULL;
    char	    *dev;
    char	    *datafile;
    void	    *data = NULL;
    int		    fd = -1;
    struct stat	    s;
    char	    opt;
    int		    i;

    int	    timeout = ( 1000 * 10 );
    u32	    lba = 0;
    u32	    lss = 0;
    u32	    flags = 0;

    struct scsi_capacity *cap = NULL;


    while(1) {
	opt = getopt_long (argc, argv,short_options,long_options, &i);
        if(opt == -1)
	    break;

	switch(opt) {
	    case('t'):
		timeout = strtoul(optarg, NULL, 0);
		if(errno) {
		    fprintf(stderr, "Could not parse timeout - %m\n");
		    exit(1);
		}

		timeout *= 1000;

		break;

	    case('a'):
		lba = strtoul(optarg, NULL, 0);
		if(errno) {
		    fprintf(stderr, "Could not parse logical block address - %m\n");
		    exit(1);
		}

		break;

	    case('l'):
		lss = strtoul(optarg, NULL, 0);
		if(errno) {
		    fprintf(stderr, "Could not parse long sector size - %m\n");
		    exit(1);
		}

		if(lss > 65535) {   
		    fprintf(stderr, "Long sector size is too big.\n");
		    exit(1);
		}

		break;

	    case('R'):
		flags |= F_RELADDR;
		break;

	    case('h'):
		fprintf(stdout, 
"Usage:      write_long [options] device datafile

options:    --timeout=N       Set timout for the format command.
            --lba=N           Set the logical block address.  [0]
            --lss=N           Set the long sector size.  [probed]
            --RELADDR         Set the relative addressing flag.
");
	    exit(1);

	    case('?'):
	    default:
		fprintf(stderr, "Bad option.\n");
		exit(1);
	}
    }

    if(optind == argc) {
	fprintf(stderr, "No device name specified.\n");
	exit(1);
    }

    dev = argv[optind++];

    if(optind == argc) {
	fprintf(stderr, "No datafile specified.\n");
	exit(1);
    }

    datafile = argv[optind];

    if(optind < (argc - 1)) {
	fprintf(stderr, "Too many arguments.\n");
	exit(0);
    }

    if((fd = open(datafile, O_RDONLY)) == -1) {
	fprintf(stderr, "Could not open %s for reading - %m\n", datafile);
	goto cleanup;
    }

    if(fstat(fd, &s) == -1) {
	fprintf(stderr, "fstat() - %m\n");
	goto cleanup;
    }

    if((data = mmap(NULL, s.st_size, PROT_READ, MAP_SHARED, fd, 0)) == (void *)-1) {
	fprintf(stderr, "Failed to mmap() %s - %m\n", datafile);
	goto cleanup;
    }


    if(!(h = scsi_open_dev(dev, 0, 0))) {
	fprintf(stderr, "Could not open %s - %m\n", argv[1]);
	exit(1);
    }

    scsi_set_timeout(h, timeout);

    if(!lss) {
	if(!(lss = scsi_get_long_sector_size(h, 0))) {
	    fprintf(stderr, "Could not probe long sector size\n");
	    goto cleanup;
	}
    }

    if(lss > h->len) {
	if(!(scsi_set_sg_reserve_size(h, lss))) {
	    fprintf(stderr, "Could not set the kernel buffer size - %m\n");
	    goto cleanup;
	}
    }

    if(!scsi_WRITE_LONG(h, data, lba, lss, flags)) {
	if(errno) {
	    fprintf(stderr, "*** WRITE LONG failed - %m\n");
	    goto cleanup;
	}

	fprintf(stderr, "*** WRITE LONG did not complete successfully.\n");
	goto cleanup;
    }

    fprintf(stdout, "Wrote long sector LBA %d from %s.\n", lba, datafile);


cleanup:

    if(h->hdr.masked_status == CHECK_CONDITION) {
	scsi_cdb_dump(stderr, (void *)h->hdr.cmdp);
	scsi_sense_dump(stderr, h->hdr.sbp);
    }

    if(cap)
	free(cap);

    if(data)
	munmap(data, s.st_size);

    if(fd != -1)
	close(fd);

    if(h)
	scsi_close_dev(h);

    exit(0);
}
