#include <scsilib.h>

#include <getopt.h>

struct option long_options[] =
{
    {"timeout",	    1,  NULL,   't'},
    {"lba",	    1,  NULL,   'a'},
    {"tlen",	    1,  NULL,   'l'},
    {"DPO",	    0,  NULL,   'D'},
    {"BYTCHK",	    0,  NULL,   'B'},
    {"RELADDR",	    0,  NULL,   'R'},
    {"dry-run",	    0,  NULL,	'n'},
    {"help",	    0,  NULL,	'h'},
    {NULL, 0, NULL, 0}
};

static char *short_options = "t:a:l:nh";



int main(int argc, char **argv)
{
    struct scsi_fd  *h = NULL;
    char	    *dev;
    char	    *datafile;
    void	    *data = NULL;
    int		    fd = -1;
    struct stat	    s;
    char	    opt;
    int		    i;

    int	    timeout = ( 1000 * 10 );
    u32	    lba = 0;
    int     tlen = -1;
    u32	    flags = 0;
    int	    no_io = 0;

    struct scsi_capacity *cap = NULL;


    while(1) {
	opt = getopt_long (argc, argv,short_options,long_options, &i);
        if(opt == -1)
	    break;

	switch(opt) {
	    case('t'):
		timeout = strtoul(optarg, NULL, 0);
		if(errno) {
		    fprintf(stderr, "Could not parse timeout - %m\n");
		    exit(1);
		}

		timeout *= 1000;

		break;

	    case('a'):
		lba = strtoul(optarg, NULL, 0);
		if(errno) {
		    fprintf(stderr, "Could not parse logicl block address - %m\n");
		    exit(1);
		}

		break;

	    case('l'):
		tlen = strtoul(optarg, NULL, 0);
		if(errno) {
		    fprintf(stderr, "Could not parse transfer length - %m\n");
		    exit(1);
		}

		if(tlen > 65535) {
		    fprintf(stderr, "Transfer length is out of bounds.\n");
		    exit(1);
		}

		break;

	    case('D'):
		flags |= F_DPO;
		break;

	    case('B'):
		flags |= F_BYTCHK;
		break;

	    case('R'):
		flags |= F_RELADDR;
		break;

	    case('n'):
		no_io = 1;
		break;

	    case('h'):
		fprintf(stdout, 
"Usage:      write_6 [options] device datafile

options:    --dry-run  -n     Do no I/O on target.
            --timeout=N       Set timout for the format command.
            --lba=N           Set the logical block address.  [0]
            --tlen=N          Set the transfer length in sectors. [1]
            --DPO             Set the DPO flag.
            --BYTCHK          Set the byte compare flag.
            --RELADDR         Set the relative addressing flag.
");
	    exit(1);

	    case('?'):
	    default:
		fprintf(stderr, "Bad option.\n");
		exit(1);
	}
    }

    if(optind == argc) {
	fprintf(stderr, "No device name specified.\n");
	exit(1);
    }

    dev = argv[optind++];

    if(optind == argc) {
	fprintf(stderr, "No datafile specified.\n");
	exit(1);
    }

    datafile = argv[optind];

    if(optind < (argc - 1)) {
	fprintf(stderr, "Too many arguments.\n");
	exit(0);
    }

    if((fd = open(datafile, O_RDONLY)) == -1) {
	fprintf(stderr, "Could not open %s for reading - %m\n", datafile);
	goto cleanup;
    }

    if(fstat(fd, &s) == -1) {
	fprintf(stderr, "fstat() - %m\n");
	goto cleanup;
    }

    if((data = mmap(NULL, s.st_size, PROT_READ, MAP_SHARED, fd, 0)) == (void *)-1) {
	fprintf(stderr, "Failed to mmap() %s - %m\n", datafile);
	goto cleanup;
    }


    if(!(h = scsi_open_dev(dev, 0, 0))) {
	fprintf(stderr, "Could not open %s - %m\n", argv[1]);
	exit(1);
    }

    scsi_set_timeout(h, timeout);

    if(no_io)
	goto cleanup;


    if(!(cap = scsi_READ_CAPACITY(h, 0, 0))) {
	fprintf(stderr, "Could not read device capacity - %m\n");
	goto cleanup;
    }

    h->sector_size = cap->sector_size;
    h->capacity = cap->capacity;

    if(tlen == -1) {
	tlen = s.st_size / h->sector_size;

	if(tlen > 65535) {
	    fprintf(stderr, "File size exceeds maximum transfer length.\n");
	    goto cleanup;
	}
    }

    if((tlen * h->sector_size) > s.st_size) {
	fprintf(stderr, "The specified transfer length is larger than the file size.\n");
	goto cleanup;
    }

    if((tlen * h->sector_size) > h->len) {
	if(!(scsi_set_sg_reserve_size(h, ((tlen) ? tlen : 256) * h->sector_size))) {
	    fprintf(stderr, "Could not set the kernel buffer size - %m\n");
	    goto cleanup;
	}
    }

    if(!scsi_WRITE_AND_VERIFY(h, data, lba, tlen, flags)) {
	if(errno) {
	    fprintf(stderr, "*** WRITE AND VERIFY failed - %m\n");
	    goto cleanup;
	}

	fprintf(stderr, "*** WRITE AND VERIFY did not complete successfully.\n");
	goto cleanup;
    }

    fprintf(stdout, "Wrote and verified %d block%s to LBA %d from %s.\n", tlen, ((tlen == 1) ? "" :"s"), lba, datafile);


cleanup:

    if(h->hdr.masked_status == CHECK_CONDITION) {
	scsi_cdb_dump(stderr, (void *)h->hdr.cmdp);
	scsi_sense_dump(stderr, h->hdr.sbp);
    }

    if(cap)
	free(cap);

    if(data)
	munmap(data, s.st_size);

    if(fd != -1)
	close(fd);

    if(h)
	scsi_close_dev(h);

    exit(0);
}
