#include <scsilib.h>

#include <getopt.h>

struct option long_options[] =
{
    {"timeout",	    1,  NULL,   't'},
    {"start",	    1,  NULL,   's'},
    {"end",	    1,  NULL,   'e'},
    {"lss",	    1,  NULL,   'S'},
    {"error",       0,  NULL,   'E'},
    {"CORRCT",	    0,	NULL,	'C'},
    {"dry-run",	    0,  NULL,	'n'},
    {"verbose",     0,  NULL,	'v'},
    {"help",	    0,  NULL,	'h'},
    {NULL, 0, NULL, 0}
};

static char *short_options = "t:s:e:S:E:Cnvh";



int main(int argc, char **argv)
{
    struct scsi_fd	*h = NULL;
    char		*dev;
    char		opt;

    u32			i;

    struct scsi_capacity *cap = NULL;

    void    *buf = NULL;

    int	    timeout = ( 1000 * 10 );
    u32	    start = 0;
    u32	    end = 0;
    size_t  lss = 0;
    int	    err = 0;
    u32	    flags = 0;
    int	    no_io = 0;
    int	    verbose = 0;


    while(1) {
	opt = getopt_long (argc, argv,short_options,long_options, &i);
        if(opt == -1)
	    break;

	switch(opt) {
	    case('t'):
		timeout = strtoul(optarg, NULL, 0);
		if(errno) {
		    fprintf(stderr, "Could not parse timeout - %m\n");
		    exit(1);
		}

		timeout *= 1000;

		break;

	    case('s'):
		start = strtoul(optarg, NULL, 0);
		if(errno) {
		    fprintf(stderr, "Could not parse starting lba - %m\n");
		    exit(1);
		}

		break;

	    case('e'):
		start = strtoul(optarg, NULL, 0);
		if(errno) {
		    fprintf(stderr, "Could not parse ending lba - %m\n");
		    exit(1);
		}

		break;

	    case('S'):
		lss = strtoul(optarg, NULL, 0);
		if(errno) {
		    fprintf(stderr, "Could not parse long sector size - %m\n");
		    exit(1);
		}

		if(lss > 65535) {
		    fprintf(stderr, "Long sector size is out of bounds.\n");
		    exit(1);
		}

		break;

	    case('E'):
		err = 1;
		break;

	    case('C'):
		flags |= F_CORRCT;
		break;

	    case('n'):
		no_io = 1;
		break;

	    case('v'):
		verbose = 1;
		break;

	    case('h'):
		fprintf(stdout, 
"Usage:      read_6 [options] device

options:    --dry-run  -n     Do no I/O on target.
            --timeout=N       Set timeout for SCSI commands.
            --start=N         Set the starting lba.  [0]
            --end=N           Set the ending lba.  [capacity]
            --lss=N           Set the long sector size. [queried]
            --error           Stop on error. [continue]
            --CORRCT          Set the no error correcting flag.
	    --verbose         Be verbose, show progress.\n
");
	    exit(1);

	    case('?'):
	    default:
		fprintf(stderr, "Bad option.\n");
		exit(1);
	}
    }

    if(optind == argc) {
	fprintf(stderr, "No device name specified.\n");
	exit(1);
    }

    if(optind < (argc - 1)) {
	fprintf(stderr, "Too many arguments.\n");
	exit(0);
    }

    dev = argv[optind];

    if(!(h = scsi_open_dev(dev, 0, 0))) {
	fprintf(stderr, "Could not open %s - %m\n", argv[1]);
	exit(1);
    }

    scsi_set_timeout(h, timeout);

    if(no_io)
	goto cleanup;

    if(!lss) {
	lss = scsi_get_long_sector_size(h, 0);

	if(!lss) {
	    fprintf(stderr, "Could not automatically determine the long sector size - %m\n");
	    goto cleanup;
	}

	if(verbose)
	    fprintf(stdout, "Long sector_size = %d\n", lss);
    }

    if(lss > h->len) {
	if(!(scsi_set_sg_reserve_size(h, lss))) {
	    fprintf(stderr, "Could not set the kernel buffer size - %m\n");
	    goto cleanup;
	}
    }

    if(!(cap = scsi_READ_CAPACITY(h, 0, 0))) {
	fprintf(stderr, "Could not read device capacity - %m\n");
	goto cleanup;
    }

    h->sector_size = cap->sector_size;
    h->capacity = cap->capacity;

    if(!(buf = malloc(lss))) {
	fprintf(stderr, "malloc(%d) - %m\n", lss);
	goto cleanup;
    }

    if(!end)
	end = h->capacity - 1;

    if(verbose)
	fprintf(stderr, "Scanning device from 0x%x to 0x%x.\n", start, end);

    for(i = start; i <= end; i++) {

	if(verbose)
	    fprintf(stderr, "Reading %08x\r", i);

	if(!scsi_READ_LONG(h, buf, i, lss, flags)) {
	    if(errno) {
		fprintf(stderr, "*** READ LONG failed - %m\n");
		goto cleanup;
	    }

	    fprintf(stdout, "0x%08x - %s, ", i, scsi_sense_key[SENSE_ERROR(h->hdr.sbp)]);

	    if(SENSE_ASL(h->hdr.sbp))
		fputs(scsi_sense_additional_text(SENSE_ASC(h->hdr.sbp), SENSE_ASCQ(h->hdr.sbp)), stdout);

	    fputc('\n', stdout);

	    if(err)
		goto cleanup;
	}
    }


cleanup:

    if(h->hdr.masked_status == CHECK_CONDITION) {
	scsi_cdb_dump(stderr, (void *)h->hdr.cmdp);
	scsi_sense_dump(stderr, h->hdr.sbp);
    }

    if(buf)
	free(buf);

    if(cap)
	free(cap);

    if(h)
	scsi_close_dev(h);

    exit(0);
}
