#include <scsilib.h>

#include <getopt.h>

struct option long_options[] =
{
	{"timeout", 1,	NULL,	't'},
        {"plist",   0,  NULL,   'p'},
        {"glist",   0,  NULL,   'g'},
        {"format",  1,  NULL,   'f'},
        {"timeout", 1,  NULL,   't'},
        {"help",    0,  NULL,   'h'},
        {NULL, 0, NULL, 0}
};

static char *short_options = "pgf:t:h";



int main(int argc, char **argv)
{
    struct scsi_fd	*h = NULL;
    struct scsi_defect_list_header *d = NULL;
    int			len, i;
    int			flags = 0;
    int			format = ADDR_DESC_PHYSICAL;
    u32			timeout = (10 * 1000);
    char *		dev;
    char		opt;

    while(1) {
	opt = getopt_long (argc, argv,short_options,long_options, &i);
        if(opt == -1)
	    break;

	switch(opt) {
	    case('t'):
		timeout = strtoul(optarg, NULL, 0);
		if(errno) {
		    fprintf(stderr, "Could not parse timeout - %m\n");
		    exit(1);
		}

		timeout *= 1000;

		break;

	    case('p'):
		flags |= F_PLIST;
		break;

	    case('g'):
		flags |= F_GLIST;
		break;

	    case('f'):
		if(strcmp(optarg, "logical") == 0) {
		    format = ADDR_DESC_LOGICAL;
		    break;
		}

		if(strcmp(optarg, "index") == 0) {
		    format = ADDR_DESC_INDEX;
		    break;
		}

		if(strcmp(optarg, "physical") == 0) {
		    format = ADDR_DESC_PHYSICAL;
		    break;
		}

		fprintf(stderr, "Unknown defect list format type %s.\n", optarg);
		exit(1);

	    case('h'):
		fprintf(stderr, "Usage:\n     read_defect_data [-p] [-g] [-f logical|index|physical] device\n\n");
		fprintf(stderr, "Long options:  --plist, read the primary defect list.\n");
		fprintf(stderr, "               --glist, read the grown defect list.\n");
		fprintf(stderr, "               --timeout, set the timeout for the command.\n");
		fprintf(stderr, "               --format arg, set the defect list format.\n");
		fprintf(stderr, "               Valid format types are:\n");
		fprintf(stderr, "                       logical, index, physical\n");

	    default:
		fprintf(stderr, "Bad option.\n");
		exit(1);
	}
    }

    if(optind == argc) {
	fprintf(stderr, "No device name specified.\n");
	exit(1);
    }

    if(optind < (argc - 1)) {
	fprintf(stderr, "Too many arguments.\n");
	exit(0);
    }

    dev = argv[optind];

    if(!(h = scsi_open_dev(dev, 0, 0))) {
	fprintf(stderr, "Could not open %s - %m\n", argv[1]);
	exit(1);
    }

    scsi_set_timeout(h, timeout);

    if(!(d = scsi_READ_DEFECT_DATA(h, NULL, 0, flags, format))) {
	if(errno) {
	    fprintf(stderr, "READ DEFECT DATA phase 1 failed - %m\n");
	    goto cleanup;
	}

	if(h->hdr.masked_status != GOOD)
	    goto cleanup;
    }

    len = d->list_len;

    if(!len) {
	fprintf(stdout, "No defects.\n");
	goto cleanup;
    }

    len += sizeof(struct scsi_defect_list_header);

    free(d);

    if(!(d = scsi_READ_DEFECT_DATA(h, NULL, len, flags, format))) {
	if(errno) {
	    fprintf(stderr, "READ_DEFECT_DATA phase 2 failed - %m\n");
	    goto cleanup;
	}

	if(h->hdr.masked_status != GOOD)
	    goto cleanup;
    }


    scsi_print_defect_list_header(stdout, d, 1);


cleanup:

    if(h->hdr.masked_status == CHECK_CONDITION) {
	scsi_cdb_dump(stderr, (void *)h->hdr.cmdp);
	scsi_sense_dump(stderr, h->hdr.sbp);
    }

    if(d)
	free(d);

    if(h)
	scsi_close_dev(h);

    exit(0);
}
