#include <scsilib.h>

#include <getopt.h>

struct option long_options[] =
{
    {"timeout",	    1,  NULL,   't'},
    {"lba",	    1,  NULL,   'a'},
    {"tlen",	    1,  NULL,   'l'},
    {"DPO",	    0,	NULL,	'D'},
    {"FUA",	    0,	NULL,	'F'},
    {"RELADDR",	    0,	NULL,	'R'},
    {"dump",	    0,	NULL,	'd'},
    {"save",	    1,	NULL,	's'},
    {"dry-run",	    0,  NULL,	'n'},
    {"help",	    0,  NULL,	'h'},
    {NULL, 0, NULL, 0}
};

static char *short_options = "t:a:l:DFRds:nh";



int main(int argc, char **argv)
{
    struct scsi_fd	*h = NULL;
    char		*dev;
    char		opt;
    int			i;

    void		*data = NULL;

    int	    timeout = ( 1000 * 10 );
    u32	    lba = 0;
    u32	    tlen = 1;
    u32	    flags = 0;
    int	    dump = 0;
    char *  filename = NULL;
    int	    no_io = 0;

    struct scsi_capacity *cap = NULL;


    while(1) {
	opt = getopt_long (argc, argv,short_options,long_options, &i);
        if(opt == -1)
	    break;

	switch(opt) {
	    case('t'):
		timeout = strtoul(optarg, NULL, 0);
		if(errno) {
		    fprintf(stderr, "Could not parse timeout - %m\n");
		    exit(1);
		}

		timeout *= 1000;

		break;

	    case('a'):
		lba = strtoul(optarg, NULL, 0);
		if(errno) {
		    fprintf(stderr, "Could not parse logicl block address - %m\n");
		    exit(1);
		}

		break;

	    case('l'):
		tlen = strtoul(optarg, NULL, 0);
		if(errno) {
		    fprintf(stderr, "Could not parse transfer length - %m\n");
		    exit(1);
		}

		if(tlen > 65535) {
		    fprintf(stderr, "Transfer length is out of bounds.\n");
		    exit(1);
		}

		break;

	    case('D'):
		flags |= F_DPO;
		break;

	    case('F'):
		flags |= F_FUA;
		break;

	    case('R'):
		flags |= F_RELADDR;
		break;

	    case('d'):
		dump  = 1;
		break;

	    case('s'):
		filename = optarg;
		break;

	    case('n'):
		no_io = 1;
		break;

	    case('h'):
		fprintf(stdout, 
"Usage:      read_10 [options] device

options:    --dry-run  -n     Do no I/O on target.
            --timeout=N       Set timout for the format command.
            --lba=N           Set the logical block address.  [0]
            --tlen=N          Set the transfer length in sectors. [1]
	    --DPO             Set disable page out flag.
	    --FUA             Set force unit access flag.
	    --RELADDR         Set relative address flag.
            --dump            Dump the sectors in hex format.
            --save=file       Save blocks to file.
");
	    exit(1);

	    case('?'):
	    default:
		fprintf(stderr, "Bad option.\n");
		exit(1);
	}
    }

    if(optind == argc) {
	fprintf(stderr, "No device name specified.\n");
	exit(1);
    }

    if(optind < (argc - 1)) {
	fprintf(stderr, "Too many arguments.\n");
	exit(0);
    }

    dev = argv[optind];

    if(!(h = scsi_open_dev(dev, 0, 0))) {
	fprintf(stderr, "Could not open %s - %m\n", argv[1]);
	exit(1);
    }

    scsi_set_timeout(h, timeout);

    if(no_io)
	goto cleanup;


    if(!(cap = scsi_READ_CAPACITY(h, 0, 0))) {
	fprintf(stderr, "Could not read device capacity - %m\n");
	goto cleanup;
    }

    h->sector_size = cap->sector_size;
    h->capacity = cap->capacity;

    if(!(scsi_set_sg_reserve_size(h, (tlen * h->sector_size)))) {
        fprintf(stderr, "Could not set the kernel buffer size - %m\n");
        goto cleanup;
    }

    if(!(data = scsi_READ_10(h, NULL, lba, tlen, flags))) {
	if(errno) {
	    fprintf(stderr, "*** READ (10) failed - %m\n");
	    goto cleanup;
	}

	fprintf(stderr, "*** READ(10) did not complete successfully.\n");
	goto cleanup;
    }

    fprintf(stdout, "Read %d block%s from %slba %d.\n", tlen, ((tlen == 1) ? "" : "s"), 
			((flags & F_RELADDR) ? "relative " : ""), lba);

    if(filename) {
        int fd = -1;

        if((fd = open(filename, O_WRONLY|O_CREAT|O_EXCL, S_IRUSR|S_IWUSR)) == -1) {
            fprintf(stderr, "Could not open %s - %m\n", filename);
            goto cleanup;
	}

        write(fd, data, (tlen * h->sector_size));

        if(errno)
            fprintf(stderr, "Write to %s failed - %m\n", filename);

        close(fd);
    }

    if(dump)
	hexdump(fileno(stdout), (lba * h->sector_size), data, (tlen * h->sector_size));


cleanup:

    if(h->hdr.masked_status == CHECK_CONDITION) {
	scsi_cdb_dump(stderr, (void *)h->hdr.cmdp);
	scsi_sense_dump(stderr, h->hdr.sbp);
    }

    if(cap)
	free(cap);

    if(data)
	free(data);

    if(h)
	scsi_close_dev(h);

    exit(0);
}
