#include <scsilib.h>

#include <getopt.h>

struct option long_options[] =
{
    {"serial",	    1,	NULL,	'S'},
    {"timeout",	    1,  NULL,   'T'},
    {"longlist",    0,  NULL,   '@'},
    {"ip",	    1,  NULL,   '#'},
    {"ip-file",	    1,  NULL,   ':'},
    {"ip-mod",	    1,  NULL,   '$'},
    {"ip-type",	    1,	NULL,	'|'},
    {"cmplist",	    0,  NULL,   '%'},
    {"VS",	    0,  NULL,   '^'},
    {"Immed",	    0,  NULL,   '&'},
    {"DSP",	    0,  NULL,   '*'},
    {"IP",	    0,  NULL,   '('},
    {"STPF",	    0,  NULL,   ')'},
    {"DCRT",	    0,  NULL,   '_'},
    {"DPRY",	    0,  NULL,   '+'},
    {"FOV",	    0,  NULL,   '{'},
    {"dry-run",	    0,  NULL,	'n'},
    {"help",        0,  NULL,   'h'},
    {NULL, 0, NULL, 0}
};

static char *short_options = "";



int main(int argc, char **argv)
{
    struct scsi_fd	*h = NULL;
    char *		dev;
    char		opt;
    int			i;

    int			flags = 0;
    int			dflags = 0;

    char *  dev_serial = NULL;
    char *  serial = NULL;
    int	    timeout = ( 1000 * 3600 * 2 );
    int	    longlist = 0;
    void *  ip = NULL;
    u32	    ip_len = 0;
/*    char *  ip_file = NULL; */
    int	    ip_type = 0;
    int	    ip_mod = 0;
    int	    no_io = 0;


    while(1) {
	opt = getopt_long(argc, argv,short_options,long_options, &i);
        if(opt == -1)
	    break;

	switch(opt) {
	    case('S'):
		serial = optarg;
		break;

	    case('T'):
		timeout = strtol(optarg, NULL, 0);
		if(errno) {
		    fprintf(stderr, "Could not parse timeout - %m\n");
		    exit(1);
		}

		timeout *= (1000 * 3600);
		break;

	    case('@'):
		flags |= longlist;
		break;

	    case('#'):
		if(!(ip = malloc(sizeof(long)))) {
		    fprintf(stderr, "malloc() - %m\n");
		    exit(1);
		}

		*((unsigned int *) ip) = strtoul(optarg, NULL, 0);
		if(errno) {
		    fprintf(stderr, "Could not parse init pattern - %m\n");
		    exit(0);
		}

		ip_len = sizeof(long);

		flags |= F_FMTDATA;

		break;

	    case(':'):
		fprintf(stderr, "ip-file option is not supported.\n");
		exit(1);

	    case('$'):
		ip_mod = strtol(optarg, NULL, 0);
		if(errno) {
		    fprintf(stderr, "Could not parse ip-mod argument - %m\n");
		    exit(1);
		}

		if(ip_mod < 0 || ip_mod > 3) {
		    fprintf(stderr, "ip-mod value out of range.\n");
		    exit(1);
		}
		break;

	    case('|'):
		ip_type = strtol(optarg, NULL, 0);
		if(errno) {
		    fprintf(stderr, "Could not parse ip-type argument - %m\n");
		    exit(1);
		}

		if(ip_type < 0 || ip_type > 255) {
		    fprintf(stderr, "ip-type value is out of range.\n");
		    exit(1);
		}

		break;

	    case('%'):
		flags |= F_CMPLIST;
		break;

	    case('^'):
		dflags |= DLH_VS;
		flags |= F_FMTDATA;
		break;

	    case('&'):
		dflags |= DLH_IMMED;
		flags |= F_FMTDATA;
		break;

	    case('*'):
		dflags |= DLH_DSP;
		flags |= F_FMTDATA;
		break;

	    case('('):
		dflags |= DLH_IP;
		flags |= F_FMTDATA;
		break;

	    case(')'):
		dflags |= DLH_STPF;
		flags |= F_FMTDATA;
		break;

	    case('_'):
		dflags |= DLH_DCRT;
		flags |= F_FMTDATA;
		break;

	    case('+'):
		dflags |= DLH_DPRY;
		flags |= F_FMTDATA;
		break;

	    case('{'):
		dflags |= DLH_FOV;
		flags |= F_FMTDATA;
		break;

	    case('n'):
		no_io = 1;
		break;

	    case('h'):
		fprintf(stdout, 
"Usage:      format_unit [options] device

options:    --dry-run  -n     Do no I/O on target.
	    --serial          Serial number of device to format.
            --timeout=N       Set timout for the format command (in hours). [2]
            --ip=N            Set long integer as the initalisation pattern.
            --ip-file=file    Use contents of 'file' as the init. pattern.
            --ip-mod=N        Initialisation pattern modifier.
            --ip-type=N       Initialisation pattern type.

Flags:      --longlist  --cmplist  --VS  --Immed  --DSP
            --STPF  --DCRT --DPRY  --FOV  --IP
");
	    exit(1);

	    default:
		fprintf(stderr, "Bad option.\n");
		exit(1);
	}
    }

    if(!serial) {
	fprintf(stderr, "You must supply the serial number of the target.\n");
	exit(1);
    }

    if(optind == argc) {
	fprintf(stderr, "No device name specified.\n");
	exit(1);
    }

    if(optind < (argc - 1)) {
	fprintf(stderr, "Too many arguments.\n");
	exit(1);
    }

    dev = argv[optind];

    if(!(h = scsi_open_dev(dev, 0, 0))) {
	fprintf(stderr, "Could not open %s - %m\n", argv[1]);
	exit(1);
    }

    signal(SIGHUP, SIG_IGN);
    signal(SIGINT, SIG_IGN);
    signal(SIGQUIT, SIG_IGN);
    signal(SIGTERM, SIG_IGN);


    if(no_io)
	goto cleanup;

    if(!(dev_serial = scsi_get_serial_number(h))) {
	fprintf(stderr, "Could not get device serial number - %m");
	goto cleanup;
    }

    if(strcmp(serial, dev_serial) != 0) {
	fprintf(stderr, "*** Serial number mismatch.\n");
	goto cleanup;
    }

    scsi_FORMAT_UNIT(h, timeout, flags, ip, ip_len, ip_type, ip_mod, NULL, 0, 0, dflags);

    if(errno) {
	fprintf(stderr, "*** FORMAT UNIT failed - %m\n");
	goto cleanup;
    }

    if(h->hdr.masked_status != GOOD) {
	fprintf(stderr, "*** FORMAT UNIT did not complete successfully.\n");
	goto cleanup;
    }

    if(dflags & F_IMMED)
	fprintf(stdout, "Format started.\n");
    else
	fprintf(stdout, "Format complete.\n");


cleanup:

    if(h->hdr.masked_status == CHECK_CONDITION) {
	scsi_cdb_dump(stderr, (void *)h->hdr.cmdp);
	scsi_sense_dump(stderr, h->hdr.sbp);
    }

    if(dev_serial)
	free(dev_serial);

    if(ip)
	free(ip);

    if(h)
	scsi_close_dev(h);

    exit(0);
}
