#include <scsilib.h>

#include <getopt.h>

struct option long_options[] =
{
    {"timeout",	    1,  NULL,   't'},
    {"serial",	    1,  NULL,   'S'},
    {"start",	    1,  NULL,   's'},
    {"end",	    1,  NULL,   'e'},
    {"count",	    1,	NULL,   'c'},
    {"DCR",	    0,  NULL,   '!'},
    {"DTE",         0,  NULL,	'@'},
    {"PER",         0,  NULL,	'#'},
    {"EER",         0,  NULL,   '$'},
    {"BYTCHK",      0,  NULL,   '%'},
    {"retry",	    1,  NULL,   '^'},
    {"time-limit",  1,  NULL,   '&'},
    {"verbose",     0,  NULL,	'v'},
    {"help",	    0,  NULL,	'h'},
    {"write-to-media", 0, NULL,	'|'},
    {NULL, 0, NULL, 0}
};

static char *short_options = "t:S:s:e:c:DdPEBRTvhw";


volatile int quit = 0;

void sig_handler(int sig)
{
    quit = 1;

    return;
}



int main(int argc, char **argv)
{
    struct scsi_fd	*h = NULL;
    char		*dev;
    char		opt;

    u32			i;
    
    struct scsi_capacity *cap = NULL;
    struct scsi_mode_param *err1 = NULL, *err2 = NULL, *vfy1 = NULL, *vfy2 = NULL;
    struct scsi_mode_page *err, *vfy;


    struct scsi_reassign_blocks_header *defect = NULL;
    u32	    bad_sector;

    void    *wbuf = NULL;
    size_t  buf_size;
    int	    len;

    int	    pass = 4;
    int	    timeout = ( 1000 * 10 );
    char *  dev_serial = NULL;
    char *  serial = NULL;
    u32	    start = 0;
    u32	    end = 0;
    size_t  scount = 1024;
    int	    verbose = 0;
    int	    DCR = 0;
    int	    DTE = 0;
    int     PER = 0;
    int	    EER = 0;
    int	    BYTCHK = 1;
    int	    destroy = 0;
    u16	    retry = 0;
    u16	    time_limit = 0;
    u32	    error[5] = { 0, 0, 0, 0 };


    while(1) {
	opt = getopt_long (argc, argv,short_options,long_options, &i);
        if(opt == -1)
	    break;

	switch(opt) {
	    case('t'):
		timeout = strtoul(optarg, NULL, 0);
		if(errno) {
		    fprintf(stderr, "Could not parse timeout - %m\n");
		    exit(1);
		}

		timeout *= 1000;
		break;

	    case('S'):
		serial = optarg;
		break;

	    case('s'):
		start = strtoul(optarg, NULL, 0);
		if(errno) {
		    fprintf(stderr, "Could not parse starting lba - %m\n");
		    exit(1);
		}

		break;

	    case('e'):
		end = strtoul(optarg, NULL, 0);
		if(errno) {
		    fprintf(stderr, "Could not parse ending lba - %m\n");
		    exit(1);
		}

		break;

	    case('c'):
		scount = strtoul(optarg, NULL, 0);
		if(errno) {
		    fprintf(stderr, "Could not parse sector count - %m\n");
		    exit(1);
		}

		if(scount > 255) {
		    fprintf(stderr, "Sector size size is out of bounds.\n");
		    exit(1);
		}

		break;

	    case('!'):
		DCR = 1;
		break;

	    case('@'):
		DTE = 1;
		break;

	    case('#'):
		PER = 1;
		break;

	    case('$'):
		EER = 1;
		break;

	    case('%'):
		BYTCHK = 0;
		break;

	    case('^'):
		i = strtoul(optarg, NULL, 0);
		if(errno) {
		    fprintf(stderr, "Could not parse retry count - %m\n");
		    exit(1);
		}

		if(i > 255) {
		    fprintf(stderr, "Retry option is out of bounds.\n");
		    exit(1);
		}

		retry = i;

		break;

	    case('&'):
		i = strtoul(optarg, NULL, 0);
		if(errno) {
		    fprintf(stderr, "Could not parse recovery time-limit - %m\n");
		    exit(1);
		}

		if(i > 65536) {
		    fprintf(stderr, "Recovery time-limit is out of bounds.\n");
		    exit(1);
		}

		time_limit = i;

		break;

	    case('v'):
		verbose = 1;
		break;

	    case('|'):
		destroy = 1;
		break;

	    case('h'):
		fprintf(stdout, 
"Usage:      cert_disk [options] device

options:    --timeout=N       Set timeout for SCSI commands (seconds).
            --serial='xxx'    Supply the serial number of the target.
            --start=N         Set the starting lba.  [0]
            --end=N           Set the ending lba.  [capacity]
            --DCR             Set the disable error code flag.
            --DTE             Set the disable transfer on error flag.
            --PER             Set the post recovered error flag.
            --EER             Set the early error recovery flag.
            --BYTCK           Disable the BYTCHK flag for the VERIFY command.
            --retry=N         Set the read/write/verify retry count.
            --time-limit=N    Set the recovery time limit (msec).
            --count=N         Set the buffer size in sectors. [1024]
	    --verbose         Be verbose, show progress.\n
            --write-to-media  Destructively overwrite the entire disk 4 times.
");
	    exit(1);

	    case('?'):
	    default:
		fprintf(stderr, "Bad option.\n");
		exit(1);
	}
    }

    if(optind == argc) {
	fprintf(stderr, "No device name specified.\n");
	exit(1);
    }

    if(optind < (argc - 1)) {
	fprintf(stderr, "Too many arguments.\n");
	exit(0);
    }

    if(!serial) {
        fprintf(stderr, "You must supply the serial number of the target.\n");
        exit(1);
    }

    if(destroy) {
	fprintf(stdout, "Destroying the contents of the disk in five seconds.\n");

	sleep(5);

	pass = 0;
    }

    signal(SIGTERM, sig_handler);
    signal(SIGINT, sig_handler);
    signal(SIGHUP, sig_handler);

    dev = argv[optind];

    if(!(h = scsi_open_dev(dev, 0, 0))) {
	fprintf(stderr, "Could not open %s - %m\n", dev);
	exit(1);
    }

    scsi_set_timeout(h, timeout);

    if(!(dev_serial = scsi_get_serial_number(h))) {
        fprintf(stderr, "Could not get device serial number - %m");
        goto cleanup;
    }

    if(strcmp(serial, dev_serial) != 0) {
        fprintf(stderr, "*** Serial number mismatch.\n");
        goto cleanup;
    }

    if(!(cap = scsi_READ_CAPACITY(h, 0, 0))) {
	fprintf(stderr, "Could not read device capacity - %m\n");
	goto cleanup;
    }

    h->sector_size = cap->sector_size;
    h->capacity = cap->capacity;

    buf_size = scount * h->sector_size;

    if(buf_size > h->len) {
	if(!(scsi_set_sg_reserve_size(h, buf_size))) {
	    fprintf(stderr, "Could not set the kernel buffer size - %m\n");
	    goto cleanup;
	}
    }

    if(!(wbuf = malloc(buf_size))) {
	fprintf(stderr, "malloc(%d) - %m\n", buf_size);
	goto cleanup;
    }

    if(!(defect = malloc(sizeof(struct scsi_reassign_blocks_header) + sizeof(u32)))) {
	fprintf(stderr, "malloc(%d) - %m\n", sizeof(struct scsi_reassign_blocks_header) + sizeof(u32));
	goto cleanup;
    }

    memset(defect, 0, sizeof(struct scsi_reassign_blocks_header) + sizeof(u32));
    defect->list_len = U16TOS(sizeof(u32));


    if(!end)
	end = h->capacity - 1;

    if((end < start) || (end > (h->capacity - 1))) {
	fprintf(stderr, "Ending lba is out of bounds.\n");
	goto cleanup;
    }

    if(!(err1 = scsi_MODE_SENSE(h, NULL, 0, PAGE_CONTROL_CURRENT, MODE_ERROR_RECOVERY, F_DBD))) {
	fprintf(stderr, "Could not get the read-write error control mode page\n");
	goto cleanup;
    }

    if(!(err2 = scsi_dup_mode_param(err1))) {
	fprintf(stderr, "malloc() - %m\n");
	goto cleanup;
    }

    (u8 *)err =  err2->p + err2->desc_len;

    err->u.error.DCR = DCR;
    err->u.error.DTE = DTE;
    err->u.error.PER = PER;
    err->u.error.EER = EER;
    err->u.error.RC  = 0;
    err->u.error.TB  = 0;
    err->u.error.ARRE = 0;
    err->u.error.AWRE = 0;
    err->u.error.read_retry = retry;
    err->u.error.write_retry = retry;
    err->u.error.recovery_limit = U16TOS(time_limit);

    if(!(scsi_MODE_SELECT(h, err2, F_PF))) {
	fprintf(stderr, "Error setting error recovery mode page - %m\n");
	goto cleanup;
    }


    if(!(vfy1 = scsi_MODE_SENSE(h, NULL, 0, PAGE_CONTROL_CURRENT, MODE_ERROR_RECOVERY, F_DBD))) {
	fprintf(stderr, "Device does not support the verify error mode page\n");
	goto cert;
    }

    if(!(vfy2 = scsi_dup_mode_param(vfy1))) {
	fprintf(stderr, "malloc() - %m\n");
	goto cleanup;
    }

    (u8 *)vfy = vfy2->p + vfy2->desc_len;

    vfy->u.verify.DCR = DCR;
    vfy->u.verify.DTE = DTE;
    vfy->u.verify.PER = PER;
    vfy->u.verify.EER = EER;
    vfy->u.verify.verify_retry = retry;
    vfy->u.verify.verify_recovery_time_limit = U16TOS(time_limit);
    
    if(!(scsi_MODE_SELECT(h, vfy2, F_PF))) {
	fprintf(stderr, "Error setting verify error mode page - %m\n");
	goto cleanup;
    }



cert:
    if(verbose)
	fprintf(stdout, "Certifying device from 0x%x to 0x%x.\n", start, end);

    while(pass < 5) {
	if(verbose)
	    fprintf(stdout, "Pass %d                   \n", pass);

	switch(pass) {
	    case(0):
		memset(wbuf, 0xffffffff, buf_size);
		break;

	    case(1):
		memset(wbuf, 0x55555555, buf_size);
		break;

	    case(2):
		memset(wbuf, 0xcccccccc, buf_size);
		break;

	    case(3):
		memset(wbuf, 0x00000000, buf_size);
		break;
	}

	i = start;

	while(i < end) {
	    if(quit)
		goto cleanup;

	    len = scount;

	    if(verbose) {
		fprintf(stdout, "0x%x\r", i);
		fflush(stdout);
	    }

	    if((i + len) >= end)
		len = end - i;


	    if(pass < 4) {
		if(!(scsi_WRITE_10(h, wbuf, i, len, F_FUA))) {
		    if(errno) {
			fprintf(stderr, "\nscsi_WRITE_10 failed - %m\n");
			goto cleanup;
		    }

		    fprintf(stdout, "WRITE failed [%02x %02x %02x], ", SENSE_KEY(h->hdr.sbp), 
					    SENSE_ASC(h->hdr.sbp), SENSE_ASCQ(h->hdr.sbp));
		    goto do_err;
		}

		if(!(scsi_VERIFY_10(h, wbuf, i, len, (BYTCHK ? F_BYTCHK : 0)))) {
		    if(errno) {
			fprintf(stderr, "\nscsi_VERIFY_10() failed - %m\n");
			goto cleanup;
		    }

		    fprintf(stdout, "VERIFY failed [%02x %02x %02x], ", SENSE_KEY(h->hdr.sbp), 
					    SENSE_ASC(h->hdr.sbp), SENSE_ASCQ(h->hdr.sbp));
		    goto do_err;
		}
	    } else {
		if(!(scsi_READ_10(h, wbuf, i, len, F_FUA))) {
		    if(errno) {
			fprintf(stderr, "\nscsi_READ_10() failed - %m\n");
			goto cleanup;
		    }

		    fprintf(stdout, "READ (10) failed [%02x] %02x %02x], ", SENSE_KEY(h->hdr.sbp), 
					SENSE_ASC(h->hdr.sbp), SENSE_ASCQ(h->hdr.sbp));
		    goto do_err;
		}

	    }

	    i += len;

	    continue;

do_err:
	    if((SENSE_KEY(h->hdr.sbp) == 0x03) && (SENSE_ASC(h->hdr.sbp) == 0x62)) {
		fprintf(stderr, "\nScan head positioning error, retrying\n");
		continue;
	    }

	    if(!(SENSE_KEY(h->hdr.sbp) == MEDIUM_ERROR) &&
		!(SENSE_KEY(h->hdr.sbp) == MISCOMPARE)) {
		fprintf(stderr, "\nnon-medium error, cannot continue\n");
		goto cleanup;
	    }

	    bad_sector = SENSE_INFORMATION(h->hdr.sbp);

	    if((bad_sector < i) || (bad_sector > (i + len))) {
		fprintf(stderr, "\ndevice appears to have put garbage in the information\nfield.  Cannot continue.\n");
		goto cleanup;
	    }

	    defect->descriptor[0] = U32TOS(bad_sector);
	    
	    if(!(scsi_REASSIGN_BLOCKS(h, defect))) {
		fprintf(stderr, "REASSIGN BLOCKS(%08x) failed - %m\n", bad_sector);
		goto cleanup;
	    }

	    if(!(scsi_REZERO_UNIT(h))) {
		fprintf(stderr, "REZERO_UNIT failed - %m\n");
		goto cleanup;
	    }

	    if(verbose)
		fprintf(stdout, "sector 0x%x reassigned.\n", bad_sector);

	    error[pass]++;

	    len = (i + len) - (i + bad_sector);
	    if(!len)
		len = scount;

	    i = bad_sector + 1;
	}

	pass++;
    }


    fprintf(stdout, "Media defect certification complete.\n");
    fprintf(stdout, "Errors by pass: %d %d %d %d %d\n", error[0], error[1], error[2], error[3], error[4]);
    fprintf(stdout, "Total blocks reassigned: %d\n", error[0] + error[1] + error[2] + error[3]);

cleanup:

    if(h->hdr.masked_status == CHECK_CONDITION) {
	scsi_cdb_dump(stderr, (void *)h->hdr.cmdp);
	scsi_sense_dump(stderr, h->hdr.sbp);
    }

    if(err1) {
	if(!(scsi_MODE_SELECT(h, err1, F_PF))) {
	    fprintf(stderr, "Error restoring error recovery mode page.\n");
	}
    }

    if(vfy1) {
	if(!(scsi_MODE_SELECT(h, vfy1, F_PF))) {
	    fprintf(stderr, "Error restoring verify error mode page.\n");
	}
    }

    if(vfy1)
	free(vfy1);

    if(vfy2)
	free(vfy2);

    if(err2)
	free(err2);

    if(err1)
	free(err1);

    if(defect)
	free(defect);

    if(wbuf)
	free(wbuf);

    if(cap)
	free(cap);

    if(dev_serial)
	free(dev_serial);

    if(h)
	scsi_close_dev(h);

    exit(0);
}
