#include <scsilib.h>

#include <getopt.h>

struct option long_options[] =
{
    {"timeout",	    1,  NULL,   't'},
    {"start",	    1,  NULL,   's'},
    {"end",	    1,  NULL,   'e'},
    {"count",	    1,	NULL,   'c'},
    {"verbose",     0,  NULL,	'v'},
    {"help",	    0,  NULL,	'h'},
    {NULL, 0, NULL, 0}
};

static char *short_options = "t:s:e:c:vh";


int main(int argc, char **argv)
{
    struct scsi_fd	*h = NULL;
    char		*dev;
    char		opt;

    u32			i;

    struct scsi_capacity *cap = NULL;
    struct scsi_mode_param *page1 = NULL;
    struct scsi_mode_param *page2 = NULL;
    struct scsi_mode_page *err;
/*
    struct scsi_mode_param_block_desc *desc ;
*/

    void    *wbuf = NULL;
    void    *rbuf = NULL;
    size_t  buf_size;

    int	    timeout = ( 1000 * 10 );
    u32	    start = 0;
    u32	    end = 0;
    size_t  scount = 64;
    u32	    flags = 0;
    int	    verbose = 0;

    struct scsi_cdb *cdb;
    u8	    data[64];


    while(1) {
	opt = getopt_long (argc, argv,short_options,long_options, &i);
        if(opt == -1)
	    break;

	switch(opt) {
	    case('t'):
		timeout = strtoul(optarg, NULL, 0);
		if(errno) {
		    fprintf(stderr, "Could not parse timeout - %m\n");
		    exit(1);
		}

		timeout *= 1000;

		break;

	    case('s'):
		start = strtoul(optarg, NULL, 0);
		if(errno) {
		    fprintf(stderr, "Could not parse starting lba - %m\n");
		    exit(1);
		}

		break;

	    case('e'):
		end = strtoul(optarg, NULL, 0);
		if(errno) {
		    fprintf(stderr, "Could not parse ending lba - %m\n");
		    exit(1);
		}

		break;

	    case('c'):
		scount = strtoul(optarg, NULL, 0);
		if(errno) {
		    fprintf(stderr, "Could not parse sector count - %m\n");
		    exit(1);
		}

		if(scount > 65535) {
		    fprintf(stderr, "Buffer size is out of bounds.\n");
		    exit(1);
		}

		break;

	    case('v'):
		verbose = 1;
		break;

	    case('h'):
		fprintf(stdout, 
"Usage:      read_6 [options] device

options:    --timeout=N       Set timeout for SCSI commands.
            --start=N         Set the starting lba.  [0]
            --end=N           Set the ending lba.  [capacity]
            --count=N         Set the buffer size in sectors. [64]
	    --verbose         Be verbose, show progress.\n
");
	    exit(1);

	    case('?'):
	    default:
		fprintf(stderr, "Bad option.\n");
		exit(1);
	}
    }

    if(optind == argc) {
	fprintf(stderr, "No device name specified.\n");
	exit(1);
    }

    if(optind < (argc - 1)) {
	fprintf(stderr, "Too many arguments.\n");
	exit(0);
    }

    dev = argv[optind];

    if(!(h = scsi_open_dev(dev, 0, 0))) {
	fprintf(stderr, "Could not open %s - %m\n", argv[1]);
	exit(1);
    }

    scsi_set_timeout(h, timeout);

    if(!(cap = scsi_READ_CAPACITY(h, 0, 0))) {
	fprintf(stderr, "Could not read device capacity - %m\n");
	goto cleanup;
    }

    h->sector_size = cap->sector_size;
    h->capacity = cap->capacity;

    buf_size = scount * h->sector_size;

    if(buf_size > h->len) {
	if(!(scsi_set_sg_reserve_size(h, buf_size))) {
	    fprintf(stderr, "Could not set the kernel buffer size - %m\n");
	    goto cleanup;
	}
    }

    if(!(rbuf = malloc(buf_size))) {
	fprintf(stderr, "malloc(%d) - %m\n", buf_size);
	goto cleanup;
    }

    if(!(wbuf = malloc(buf_size))) {
	fprintf(stderr, "malloc(%d) - %m\n", buf_size);
	goto cleanup;
    }

    if(!end)
	end = h->capacity - 1;

    if((end < start) || (end > (h->capacity - 1))) {
	fprintf(stderr, "Ending lba is out of bounds.\n");
	goto cleanup;
    }

    memset(data, 0, 64);

    cdb = (void *)h->hdr.cmdp;

    ZERO_CDB(cdb);

    SET_CDB6_OPCODE(cdb, MODE_SENSE);

    SET_CDB6_DBD(cdb);

    SET_CDB6_PC(cdb, PAGE_CONTROL_CURRENT);

    SET_CDB6_PAGE(cdb, MODE_ERROR_RECOVERY);

    SET_CDB6_ALEN(cdb, 64);

    h->hdr.dxfer_direction = SG_DXFER_FROM_DEV;
    h->hdr.cmd_len         = 6;
    h->hdr.iovec_count     = 0;
    h->hdr.dxfer_len       = 32;
    h->hdr.dxferp          = data;


    if(ioctl(h->fd, SG_IO, &h->hdr) == -1)
        goto cleanup;

    if(h->hdr.masked_status != GOOD)
        goto cleanup;

scsi_cdb_dump(stdout, (void *)h->hdr.cmdp);
hexdump(fileno(stdout), 0, data, 32);

/*
    (u8 *)desc = data + 4;

    fprintf(stdout, "blen %d  bcount %d\n", U24TOS(desc->block_len), U24TOS(desc->block_count));

    desc->block_len = 0;
    desc->block_count = 0;
*/
    (u8 *)err = data + 4;

    err->hdr.PS = 0;

    data[0] = 0;

    err->u.error.DCR = 0;
    err->u.error.DTE = 0;
    err->u.error.PER = 0;
    err->u.error.EER = 0;
    err->u.error.RC  = 0;
    err->u.error.TB  = 0;
    err->u.error.ARRE = 0;
    err->u.error.AWRE = 0;
    err->u.error.read_retry = 0;
    err->u.error.write_retry = 0;
    err->u.error.recovery_limit = U16TOS(1);


    ZERO_CDB(cdb);

    SET_CDB6_OPCODE(cdb, MODE_SELECT);

    SET_CDB6_PF(cdb);

    SET_CDB6_PLEN(cdb, 16);

    h->hdr.dxfer_direction = SG_DXFER_TO_DEV;
    h->hdr.cmd_len         = 6;
    h->hdr.iovec_count     = 0;
    h->hdr.dxfer_len       = 16;
    h->hdr.dxferp          = data;


    if(ioctl(h->fd, SG_IO, &h->hdr) == -1)
        goto cleanup;

scsi_cdb_dump(stdout, (void *)h->hdr.cmdp);
hexdump(fileno(stdout), 0, data, 32);

    if(h->hdr.masked_status != GOOD)
        goto cleanup;

exit(0);
    



    if(verbose)
	fprintf(stdout, "Certifying device from 0x%x to 0x%x.\n", start, end);

    for(i = 0 ; i < 4; i++) {
	if(verbose)
	    fprintf(stdout, "Pass %d\n", i);

	switch(i) {
	    case(0):
		memset(wbuf, 0xffffffff, buf_size);
		break;

	    case(1):
		memset(wbuf, 0x55555555, buf_size);
		break;

	    case(2):
		memset(wbuf, 0xcccccccc, buf_size);
		break;

	    case(3):
		memset(wbuf, 0x00000000, buf_size);
		break;
	}


	for(i = start; i <= end; i += scount) {
	     if(i > end)
		flags = 1;
	}

    }


cleanup:

    if(h->hdr.masked_status == CHECK_CONDITION) {
	scsi_cdb_dump(stderr, (void *)h->hdr.cmdp);
	scsi_sense_dump(stderr, h->hdr.sbp);
    }

    if(page1) {
	err = page1->p;
	
	if(!(scsi_MODE_SELECT(h, page1, F_PF))) {
	    fprintf(stderr, "Error restoring error recovery mode page.\n");
	}
    }

    if(page1)
	free(page1);

    if(page2)
	free(page2);

    if(rbuf)
	free(rbuf);

    if(wbuf)
	free(wbuf);

    if(cap)
	free(cap);

    if(h)
	scsi_close_dev(h);

    exit(0);
}
