/*
    Copyright 2002, 2003, Steve Thompson <stevet010@softhome.net>

    This software is free for non-commercial, non-military use.

    Included with this package, you should have received a file called
    `Licensing' which contains detailed terms of use.

*/
#include <scsilib.h>


/*
    scsi_sense_dump()
 
*/
void	scsi_sense_dump(FILE *fp, void *v)
{
    struct scsi_sense *p = v;

    assert(fp && p);

    fprintf(fp, "Sense data:\n");
    fprintf(fp, "-----------\n");

    if(!p)
	return;

    fprintf(stderr, "%s", scsi_sense_key[p->sense_key]);
    
    if(p->ASL)
	fprintf(stderr, ", %s", scsi_sense_additional_text(p->ASC, p->ASCQ));

    fprintf(fp, "\n\nValid: %1d  Error Code: %02x  Segment : 0x%02x\n",   p->valid, p->error_code, p->segment);
    fprintf(fp, "FMK  : %1d  EOM: %1d  ILI: %1d  Reserved:  0x%1x\n", p->filemark, p->EOM, p->ILI, p->reserved);
    fprintf(fp, "Sense Key: 0x%1d   Information: 0x%08x\n", p->sense_key, p->information);
    fprintf(fp, "ASL: 0x%02x\n", p->ASL);
    if(p->ASL) {
	fprintf(fp, "CSI: 0x%08x    ASC: 0x%02x  ASCQ: 0x%02x\n",        p->CSI, p->ASC, p->ASCQ);
	fprintf(fp, "FRU: 0x%02x       SKSV: %d  SKS: 0x%06x\n",           p->fru_code, p->SKSV, p->SKS);
    } else
	fputc('\n', fp);


/*
    This code can be used if the
    sense buffer is larger than
    16 bytes.
*/
#if 0
    if((p->ASL + 7) > 18) {
	fprintf(fp, "Additional Sense Bytes:\n");

	for(i=0; i < (p->ASL - 18); i++) {
	    if((i % 32) == 0)
		fputc('\n', fp);

	    if(i && ((i % 4) == 0) && (i % 32))
		fputc(' ', fp);

	    fprintf(fp, "%02x ", p->ASB[i]);
	}

	fputs('\n', fp);
    }
#endif

    return;
}



/*
 * scsi_byteswap_sense_data()
 *
 */
void	    scsi_byteswap_sense_data(void *v)
{
    struct scsi_sense *p = v;

    assert(p);

    p->information = STOU32(p->information);

    p->CSI = STOU32(p->CSI);

    return;
}




/*
 * scsi_sense_valid()
 *
 */
int	    scsi_sense_valid(void *v)
{
    struct scsi_sense *p = v;

    assert(p);

    if((p->error_code >> 4) == 7)
	return(1);

    return(0);
}




const char *scsi_sense_key[] = {
    "NO SENSE",
    "RECOVERED ERROR",
    "NOT READY",
    "MEDIUM ERROR",
    "HARDWARE_ERROR",
    "ILLEGAL REQUEST",
    "UNIT ATTENTION",
    "DATA PROTECT",
    "BLANK CHECK",
    "VENDOR SPECIFIC",
    "COPY ABORTED",
    "ABORTED COMMAND",
    "EQUAL",
    "VOLUME OVERFLOW",
    "MISCOMPARE",
    "RESERVED",
    NULL
};



/*

    The following is slightly silly as a reimplementation of what can be found
    in /usr/src/linux/drivers/scsi/constants.[ch], but the kernel is GPL, and
    I don't want to use that.  The codes, names and such come from the SCSI
    spec, and the method used is pretty much optimal, so while some might
    argue that this is a `derivative' of the the version found in the linux
    kernel sources, it is in fact based on table 71 from X3T9.2/375R.  Note
    that I am not attempting to assert copyright over the text of the error
    codes, error descriptions, or anything silly like that.

*/
#define DIRECT		0x0001
#define TAPE		0x0002
#define PRINTER		0x0004
#define PROCESSOR	0x0008
#define WORM		0x0010
#define CDROM		0x0020
#define SCANNER		0x0040
#define OPTICAL		0x0080
#define MEDIUM_CHGR	0x0100
#define COMM		0x0200



#define AFEW	( DIRECT | TAPE | WORM | OPTICAL )
#define ALL	( DIRECT | TAPE | PRINTER | PROCESSOR | WORM | CDROM | SCANNER | OPTICAL | MEDIUM_CHGR | COMM )
#define ALLXC	( DIRECT | TAPE | PRINTER | PROCESSOR | WORM | CDROM | SCANNER | OPTICAL | MEDIUM_CHGR )
#define ALLXM	( DIRECT | TAPE | PRINTER | PROCESSOR | WORM | CDROM | SCANNER | OPTICAL | COMM )
#define ALLXPC	( DIRECT | TAPE | PRINTER | WORM | CDROM | SCANNER | OPTICAL | MEDIUM_CHGR )
#define ALLXP	( DIRECT | TAPE | PRINTER | WORM | CDROM | SCANNER | OPTICAL | MEDIUM_CHGR | COMM )
#define DIROPT	( DIRECT | OPTICAL)
#define DIRTAPE	( DIRECT | TAPE )
#define DLO	( DIRECT | PRINTER | OPTICAL )
#define DTLO	( DIRECT | TAPE | PRINTER | OPTICAL )
#define DTLWRSO	( DIRECT | TAPE | PRINTER | WORM | CDROM | SCANNER | OPTICAL )
#define DTLWSO	( DIRECT | TAPE | PRINTER | WORM | SCANNER | OPTICAL )
#define DTO	( DIRECT | TAPE | OPTICAL )
#define DTWO	( DIRECT | TAPE | WORM | OPTICAL )
#define DTWRO	( DIRECT | TAPE | WORM | CDROM | OPTICAL )
#define DTWRSO	( DIRECT | TAPE | WORM | CDROM | SCANNER | OPTICAL )
#define DTWSO	( DIRECT | TAPE | WORM | SCANNER | OPTICAL )
#define DWO	( DIRECT | WORM | OPTICAL )
#define DWRO	( DIRECT | WORM | CDROM | OPTICAL )
#define MANY	( DIRECT | TAPE | WORM | CDROM | OPTICAL | MEDIUM_CHGR )
#define ROM	( WORM | CDROM | OPTICAL )
#define SOME	( DIRECT | TAPE | WORM | SCANNER | OPTICAL )
#define TAPEOPT	( TAPE | OPTICAL )
#define TAPEPRT	( TAPE | PRINTER )
#define TAPESCN	( TAPE | SCANNER )
#define UNIQ	( DIRECT | WORM | CDROM | OPTICAL | MEDIUM_CHGR )
#define WORMOPT	( WORM | OPTICAL )


struct scsi_asc_ascq scsi_asc_ascq_text[] = {
    { 0x00, 0x00, ALL,		"No additional sense information" },
    { 0x00, 0x01, TAPE,		"Filemark detected" },
    { 0x00, 0x02, TAPESCN,	"End-of-partition/medium detected" },
    { 0x00, 0x03, TAPE,		"Setmark detected" },
    { 0x00, 0x04, TAPESCN,	"Beginning-of-partition/medium detected" },
    { 0x00, 0x05, TAPESCN,	"End-of-data detected" },
    { 0x00, 0x06, ALL,		"I/O process terminated" },
    { 0x00, 0x11, CDROM,	"Audio play operation in progress" },
    { 0x00, 0x12, CDROM,	"Audio play operation paused" },
    { 0x00, 0x13, CDROM,	"Audio play operation successfully completed" },
    { 0x00, 0x14, CDROM,	"Audio play operation stopped due to error" },
    { 0x00, 0x15, CDROM,	"No current audio status to return" },
    { 0x01, 0x00, DWO,		"No index/sector signal" },
    { 0x02, 0x00, UNIQ,		"No seek complete" },
    { 0x03, 0x00, DTLWSO,	"Peripheral device write fault" },
    { 0x03, 0x01, TAPE,		"No write current" },
    { 0x03, 0x02, TAPE,		"Excessive write errors" },
    { 0x04, 0x00, ALL,		"Logical unit is not ready, cause not reportable" },
    { 0x04, 0x01, ALL,		"Logical unit is in process of becoming ready" },
    { 0x04, 0x02, ALL,		"Logical unit not ready, initializing command required" },
    { 0x04, 0x03, ALL,		"Logical unit not ready, manual intervention required" },
    { 0x04, 0x04, DTLO,		"Logical unit not ready, format in progress" },
    { 0x05, 0x00, ALL,		"Logical unit does not respond to selection" },
    { 0x06, 0x00, UNIQ,		"No reference position found" },
    { 0x07, 0x00, ALLXPC,	"Multiple peripheral devices selected" },
    { 0x08, 0x00, ALLXP,	"Logical unit communication failure" },
    { 0x08, 0x01, ALLXP,	"Logical unit communication time-out" },
    { 0x08, 0x02, ALLXP,	"Logical unit communication parity error" },
    { 0x09, 0x00, DTWRO,	"Track following error" },
    { 0x09, 0x01, ROM,		"Tracking servo failure" },
    { 0x09, 0x02, ROM,		"Focus servo failure" },
    { 0x09, 0x03, ROM,		"Spindle servo failure" },
    { 0x0A, 0x00, ALL,		"Error log overflow" },
    { 0x0c, 0x00, TAPESCN,	"Write error" },
    { 0x0c, 0x01, DWO,		"Write error recovered with auto reallocation" },
    { 0x0c, 0x02, DWO,		"Write error - auto reallocation failed" },
    { 0x10, 0x00, DWO,		"ID CRC or ECC error" },
    { 0x11, 0x00, DTWRSO,	"Unrecovered read error" },
    { 0x11, 0x01, SOME,		"Read retries exhausted" },
    { 0x11, 0x02, SOME,		"Error too long to correct" },
    { 0x11, 0x03, DTWSO,	"Multiple read errors" },
    { 0x11, 0x04, DWO,		"Unrecovered read error - auto reallocate failed" },
    { 0x11, 0x05, ROM,		"L-EC uncorrectable error" },
    { 0x11, 0x06, ROM,		"CIRC unrecovered error" },
    { 0x11, 0x07, WORMOPT,	"Data resynchronization error" },
    { 0x11, 0x08, TAPE,		"Incomplete block read" },
    { 0x11, 0x09, TAPE,		"No gap found" },
    { 0x11, 0x0a, DTO,		"Miscorrected error" },
    { 0x11, 0x0b, DWO,		"Unrecovered read error - recommend reassignment" },
    { 0x11, 0x0c, DWO,		"Unrecovered read error - recommend rewrite the data" },
    { 0x12, 0x00, DWO,		"Address mark not found for ID field" },
    { 0x13, 0x00, DWO,		"Address mark not found for data field" },
    { 0x14, 0x00, DTLWRSO,	"Recorded entity not found" },
    { 0x14, 0x01, DTWRO,	"Record not found" },
    { 0x14, 0x02, TAPE,		"Filemark or setmark not found" },
    { 0x14, 0x03, TAPE,		"End-of-data not found" },
    { 0x14, 0x04, TAPE,		"Block sequence error" },
    { 0x15, 0x00, ALLXPC,	"Random positioning error" },
    { 0x15, 0x01, ALLXPC,	"Mechanical positioning error" },
    { 0x15, 0x02, DTWRO,	"Positioning error detected by read of medium" },
    { 0x16, 0x00, DWO,		"Data synchronization mark error" },
    { 0x17, 0x00, DTWRSO,	"Recovered data with no error correction applied" },
    { 0x17, 0x01, DTWRSO,	"Recovered data with retries" },
    { 0x17, 0x02, DTWRO,	"Recovered data with positive head offset" },
    { 0x17, 0x03, DTWRO,	"Recovered data with negative head offset" },
    { 0x17, 0x04, ROM,		"Recovered data with retries and/or CIRC applied" },
    { 0x17, 0x05, DWRO,		"Recovered data using previous sector ID" },
    { 0x17, 0x06, DWO,		"Recovered data without ECC - data auto-reallocated" },
    { 0x17, 0x07, DWO,		"Recovered data without ECC - recommend reassignment" },
    { 0x17, 0x08, DWO,		"Recovered data without ECC - Recommend rewrite" },
    { 0x18, 0x00, DTWRO,	"Recovered data with error correction applied" },
    { 0x18, 0x01, DWRO,		"Recovered data with error correction and retries applied" },
    { 0x18, 0x02, DWRO,		"Recovered data - data auto-reallocated" },
    { 0x18, 0x03, CDROM,	"Recovered data with CIRC" },
    { 0x18, 0x04, CDROM,	"Recovered data with L-EC" },
    { 0x18, 0x05, DWRO,		"Recovered data - recommend reassignment" },
    { 0x18, 0x06, DWRO,		"Recovered data - recommend rewrite" },
    { 0x19, 0x00, DIROPT,	"Defect list error" },
    { 0x19, 0x01, DIROPT,	"Defect list not available" },
    { 0x19, 0x02, DIROPT,	"Defect list error in primary list" },
    { 0x19, 0x03, DIROPT,	"Defect list error in grown list" },
    { 0x1a, 0x00, ALL,		"Parameter list length error" },
    { 0x1b, 0x00, ALL,		"Synchronus data transfer error" },
    { 0x1c, 0x00, DIROPT,	"Defect list not found" },
    { 0x1c, 0x01, DIROPT,	"Primary defect list not found" },
    { 0x1c, 0x02, DIROPT,	"Grown defect list not found" },
    { 0x1d, 0x00, DWO,		"Miscompare during verify operation" },
    { 0x1e, 0x00, DWO,		"Recovered ID with ECC correction" },
    { 0x20, 0x00, ALL,		"Invalid command operation code" },
    { 0x21, 0x00, MANY,		"Logical block address out of range" },
    { 0x21, 0x01, MEDIUM_CHGR,	"Invalid element address" },
    { 0x24, 0x00, ALL,		"Invalid field in CDB" },
    { 0x25, 0x00, ALL,		"Logical unit not supported" },
    { 0x26, 0x00, ALL,		"Invalid field in parameter list" },
    { 0x26, 0x01, ALL,		"Parameter not supported" },
    { 0x26, 0x02, ALL,		"Parameter value invalid" },
    { 0x26, 0x03, ALL,		"Threshold parameters not supported" },
    { 0x27, 0x00, DTWO,		"Write protected" },
    { 0x28, 0x00, ALL,		"Not ready to ready transition, medium may have changed" },
    { 0x28, 0x01, MEDIUM_CHGR,	"Import or export element accessed" },
    { 0x29, 0x00, ALL,		"Power on, reset, or bus device reset occurred" },
    { 0x2a, 0x00, ALLXP,	"Parameters changed" },
    { 0x2a, 0x01, ALLXP,	"Mode parameters changed" },
    { 0x2a, 0x02, ALLXP,	"Log parameters changed" },
    { 0x2b, 0x00, ALLXM,	"Copy cannot execute since host cannot disconnect" },
    { 0x2c, 0x00, ALL,		"Command sequence error" },
    { 0x2c, 0x01, SCANNER,	"Too many windows specified" },
    { 0x2c, 0x02, SCANNER,	"Invalid combination of windows specified" },
    { 0x2d, 0x00, TAPE,		"Overwrite error on update in place" },
    { 0x2f, 0x00, ALL,		"Commands cleared by another initiator" },
    { 0x30, 0x00, MANY,		"Incompatible medium installed" },
    { 0x30, 0x01, DTWRO,	"Cannot read medium - unknown format" },
    { 0x30, 0x02, DTWRO,	"Cannot read medium - incompatible format" },
    { 0x30, 0x03, DIRTAPE,	"Cleaning cartridge installed" },
    { 0x31, 0x00, AFEW,		"Medium format corrupted" },
    { 0x31, 0x01, DLO,		"Format command failed" },
    { 0x32, 0x00, DIRECT,	"Illegal function (should have used 0x20 0x00, 0x24 0x00, 0x26 0x00)" },
    { 0x32, 0x00, DWO,		"No defect spare location available" },
    { 0x32, 0x01, DWO,		"Defect list update failure" },
    { 0x33, 0x00, TAPE,		"Tape length error" },
    { 0x36, 0x00, PRINTER,	"Ribbon, ink, or toner failure" },
    { 0x37, 0x00, ALLXP,	"Rounded parameter" },
    { 0x39, 0x00, ALLXP,	"Saving parameters not supported" },
    { 0x3a, 0x00, ALLXPC,	"Medium not present" },
    { 0x3b, 0x00, TAPEPRT,	"Sequential positioning error" },
    { 0x3b, 0x01, TAPE,		"Tape position error at beginning-of-medium" },
    { 0x3b, 0x02, TAPE,		"Tape position error at end-of-medium" },
    { 0x3b, 0x03, PRINTER,	"Tape or electronic vertical forms unit not ready" },
    { 0x3b, 0x04, PRINTER,	"Slew failure" },
    { 0x3b, 0x05, PRINTER,	"Paper jam" },
    { 0x3b, 0x06, PRINTER,	"Failed to sense top-of-form" },
    { 0x3b, 0x07, PRINTER,	"Failed to sense bottom-of-form" },
    { 0x3b, 0x08, TAPE,		"Reposition error" },
    { 0x3b, 0x09, SCANNER,	"Read past end of medium" },
    { 0x3b, 0x0a, SCANNER,	"Read past beginning of medium" },
    { 0x3b, 0x0B, SCANNER,	"Position past end of medium" },
    { 0x3b, 0x0c, SCANNER,	"Position Past beginning of medium" },
    { 0x3b, 0x0d, MEDIUM_CHGR,	"Medium destination element full" },
    { 0x3b, 0x0e, MEDIUM_CHGR,	"Medium source element empty" },
    { 0x3d, 0x00, ALL,		"Invalid bits in identify message" },
    { 0x3e, 0x00, ALL,		"Logical unit has not self-configured yet" },
    { 0x3f, 0x00, ALL,		"Target operating conditions have changed" },
    { 0x3f, 0x01, ALL,		"Microcode has been changed" },
    { 0x3f, 0x02, ALL,		"Canged operating definition" },
    { 0x3f, 0x03, ALL,		"Inquiry data has changed" },
    { 0x40, 0x00, ALL,		"Diagnostic failure on component 0xxx" },
    { 0x40, 0x00, DIRECT,	"RAM failure (should have used 0x40 0xxx)" },
    { 0x41, 0x00, DIRECT,	"Data path failure (should have used 0x40 0xxx)" },
    { 0x42, 0x00, DIRECT,	"Power-on or self-test failure (should have used 0x40 0xxx)" },
    { 0x43, 0x00, ALL,		"Message error" },
    { 0x44, 0x00, ALL,		"Internal target failure" },
    { 0x45, 0x00, ALL,		"Select or reselect error" },
    { 0x46, 0x00, ALL,		"Unsuccessful soft reset" },
    { 0x47, 0x00, ALL,		"SCSI parity error" },
    { 0x48, 0x00, ALL,		"Initiator detected error message received" },
    { 0x49, 0x00, ALL,		"Invalid message error" },
    { 0x4a, 0x00, ALL,		"Command phase error" },
    { 0x4b, 0x00, ALL,		"Data phase error" },
    { 0x4c, 0x00, ALL,		"Logical unit failed self-configuration" },
    { 0x4e, 0x00, ALL,		"Overlapped commands attempted" },
    { 0x50, 0x00, TAPE,		"Write append error" },
    { 0x50, 0x01, TAPE,		"Write append position error" },
    { 0x50, 0x02, TAPE,		"Position error related to timing" },
    { 0x51, 0x00, TAPEOPT,	"Erase failure" },
    { 0x52, 0x00, TAPE,		"Cartridge fault" },
    { 0x53, 0x00, ALLXP,	"Media load or eject failed" },
    { 0x53, 0x01, TAPE,		"Unload tape failure" },
    { 0x53, 0x02, MANY,		"Medium removal prevented" },
    { 0x54, 0x00, PROCESSOR,	"SCSI to host system interface failure" },
    { 0x55, 0x00, PROCESSOR,	"System resource failure" },
    { 0x57, 0x00, CDROM,	"Unable to recover table-of-contents" },
    { 0x58, 0x00, OPTICAL,	"Generation does not exist" },
    { 0x59, 0x00, OPTICAL,	"Updated block read" },
    { 0x5a, 0x00, ALLXC,	"Operator request or state change input" },
    { 0x5a, 0x01, UNIQ,		"Operator medium removal request" },
    { 0x5a, 0x02, DTWO,		"Operator selected write protect" },
    { 0x5a, 0x03, DTWO,		"Operator selected write permit" },
    { 0x5b, 0x00, ALL,		"Log exception" },
    { 0x5b, 0x01, ALLXC,	"Threshold condition met" },
    { 0x5b, 0x02, ALL,		"Log counter at maximum" },
    { 0x5b, 0x03, ALL,		"Log list codes exhausted" },
    { 0x5c, 0x00, DIROPT,	"RPL status change" },
    { 0x5c, 0x01, DIROPT,	"Spindles synchronized" },
    { 0x5c, 0x02, DIROPT,	"Spindles not synchronized" },
    { 0x60, 0x00, SCANNER,	"Lamp failure" },
    { 0x61, 0x00, SCANNER,	"Video acquisition error" },
    { 0x61, 0x01, SCANNER,	"Unable to acquire video" },
    { 0x61, 0x02, SCANNER,	"Out of focus" },
    { 0x62, 0x00, SCANNER,	"Scan head positioning error" },
    { 0x63, 0x00, CDROM,	"End of user area encountered on this track" },
    { 0x64, 0x00, CDROM,	"Illegal mode for this track" },
    { 0x00, 0x00, 0x00, NULL }
};






char *	scsi_sense_additional_text(u8 asc, u8 ascq)
{
    char *p = NULL;
    u8 i = 0;

    switch(asc) {
	case(0x40):
	    p = "Diagnostic failure on component specified by ASCQ";
	    break;
    
	default:
	    while(scsi_asc_ascq_text[i].text) {
		if(asc == scsi_asc_ascq_text[i].asc && ascq == scsi_asc_ascq_text[i].ascq) {
		    p = scsi_asc_ascq_text[i].text;
		    break;
		}

		i++;
	    }

	    i=0;
	    break;
    }


    if(!p)
	p = "unknown";

    return(p);
}






