/*
    Copyright 2002, 2003, Steve Thompson <stevet010@softhome.net>

    This software is free for non-commercial, non-military use.

    Included with this package, you should have received a file called
    `Licensing' which contains detailed terms of use.

*/

#include <scsilib.h>



char * scsi_addr_desc_type[] = {
    "logical",
    "reserved(1)",
    "reserved(2)",
    "reserved(3)",
    "index",
    "physical",
    "reserved(6)",
    NULL
};


int scsi_addr_desc_len[] = { 4, 1, 1, 1, 8, 8, 1, 1 };
    



void    scsi_byteswap_addr_desc(struct scsi_addr_desc *desc, u8 type)
{
    assert(desc);

    switch(type) {
	case(ADDR_DESC_LOGICAL):
	    desc->u.logical.sector = U32TOS(desc->u.logical.sector);
	    break;

	case(ADDR_DESC_INDEX):
            desc->u.index.cyl = U24TOS(desc->u.index.cyl);
            desc->u.index.offset = U32TOS(desc->u.index.offset);
	    break;

	case(ADDR_DESC_PHYSICAL):
            desc->u.physical.cyl = U24TOS(desc->u.physical.cyl);
            desc->u.physical.sector = U32TOS(desc->u.physical.sector);
	    break;

	default:
	    fprintf(stderr, "%s(): bad address descriptor type.\n", __FUNCTION__);
	    abort();
    }

    return;
}




void scsi_byteswap_defect_list(void *p, u32 len, u8 type)
{
    void *q = p + len;

    assert(p);


    while(p < q) {
	scsi_byteswap_addr_desc(p, type);

        switch(type) {
	    case(ADDR_DESC_LOGICAL):
		p += sizeof(struct scsi_addr_desc_logical);
                break;

            case(ADDR_DESC_INDEX):
		p += sizeof(struct scsi_addr_desc_index);
                break;

            case(ADDR_DESC_PHYSICAL):
                p += sizeof(struct scsi_addr_desc_physical);
                break;

            default:
                fprintf(stderr, "%s(): Unknown defect list format - %d\n", __FUNCTION__, type);
		abort();
	}
    }

    return;
}



/*
    for the READ DEFECT DATA command.

*/
void	scsi_byteswap_defect_list_header(struct scsi_defect_list_header *hdr, u16 len, u8 type)
{
    u32	    dlist_len;

    assert(hdr);

    dlist_len = len - sizeof(struct scsi_defect_list_header);

    hdr->list_len = U16TOS(hdr->list_len);

    if(len > sizeof(struct scsi_defect_list_header))
	scsi_byteswap_defect_list(hdr->defect_list, dlist_len, type);


    return;
}





/*
    scsi_print_defect_list_header()

*/
void	scsi_print_defect_list_header(FILE *s, struct scsi_defect_list_header *hdr, int d)
{
    assert(s && hdr);

    fprintf(s, "Defect List Header:\n-------------------\n");
   
    fprintf(s, "Plist:    %s\nGlist:    %s\nFormat:   %s (0x%x)\nlist_len: 0x%x (%d defects)\n", 
		    BOOLYN(hdr->Plist), BOOLYN(hdr->Glist),
		    scsi_addr_desc_type[hdr->format], hdr->format, hdr->list_len,
		    hdr->list_len ? hdr->list_len / scsi_addr_desc_len[hdr->format] : 0);

    if(d) {
	fputc('\n', s);
	scsi_print_defect_list(s, hdr->defect_list, hdr->list_len, hdr->format);
    }

    return;
}




/*
    for the READ DEFECT DATA (12) command.

*/
void	scsi_byteswap_defect_list_header_12(struct scsi_defect_list_header_12 *hdr, u32 len, u8 type)
{
    u32	    dlist_len;

    assert(hdr);

    dlist_len = len - sizeof(struct scsi_defect_list_header_12);

    hdr->list_len = U32TOS(hdr->list_len);

    if(len > sizeof(struct scsi_defect_list_header_12))
	scsi_byteswap_defect_list(hdr->defect_list, dlist_len, type);


    return;
}


/*
    scsi_print_defect_list_header_12()

*/
void	scsi_print_defect_list_header_12(FILE *s, struct scsi_defect_list_header_12 *hdr, int d)
{
    assert(s && hdr);

    fprintf(s, "Defect List Header (12):\n00000-------------------\n");
   
    fprintf(s, "Plist   : %s\nGlist   : %s\nFormat  : %s (0x%x)\nlist_len: 0x%x (%d defects)\n",
		    BOOLYN(hdr->Plist), BOOLYN(hdr->Glist),
		    scsi_addr_desc_type[hdr->format], hdr->format, hdr->list_len,
		    hdr->list_len ? hdr->list_len / scsi_addr_desc_len[hdr->format] : 0);

    if(d) {
	fputc('\n', s);
	scsi_print_defect_list(s, hdr->defect_list, hdr->list_len, hdr->format);
    }

    return;
}




/*
    scsi_print_format_defect_list_header()

*/
void	scsi_print_format_defect_list_header(FILE *s, struct scsi_format_defect_list_header *hdr)
{
    assert(s && hdr);

    fprintf(s, "Format Defect List Header:\n--------------------------\n");

    fprintf(s, "FOV:   %s\nDPRY:  %s\nDCRT:  %s\nSTPF:  %s\nIP:    %s\nDSP:   %s\nImmed: %s\nVS:    %s\nlen:   0x%x\n", 
	BOOLYN(hdr->FOV), BOOLYN(hdr->DPRY), BOOLYN(hdr->DCRT), BOOLYN(hdr->STPF),
	BOOLYN(hdr->IP), BOOLYN(hdr->DSP), BOOLYN(hdr->Immed), BOOLYN(hdr->VS),
	hdr->list_len);

    return;
}



/*
    scsi_print_defect_list()

*/
void	scsi_print_defect_list(FILE *s, void *p, u32 len, u8 type)
{
    void		    *q;
    struct scsi_addr_desc   *a;
    int			    c;

    assert(p);

    if(!len) {
	fprintf(s, "No defects.\n");
	return;
    }

    c = fprintf(s, "Defect List in ");

    switch(type) {
	case(ADDR_DESC_LOGICAL):
	    c += fprintf(s, "logical block address format:\n");
	    break;
	case(ADDR_DESC_INDEX):
	    c += fprintf(s, "offset from index format, C/H/O:\n");
	    break;
	case(ADDR_DESC_PHYSICAL):
	    c += fprintf(s, "physical C/H/S format:\n");
	    break;
	default:
	    fprintf(s, "an unknown format:\n");
    }

    while(c-- > 0)
	fputc('-', s);

    fputc('\n', s);

    c = 0;
    q = p + len;

    while(p < q) {
	a = p;

	switch(type) {
	    case(ADDR_DESC_LOGICAL):
		fprintf(s, "%08x ", a->u.logical.sector);
		p += sizeof(struct scsi_addr_desc_logical);

		if(++c == 8) {
		    c = 0;
		    fputc('\n', s);
		}

		break;

	    case(ADDR_DESC_INDEX):
		fprintf(s, "%06x %02x %08x ", a->u.index.cyl, a->u.index.head, a->u.index.offset);
		p += sizeof(struct scsi_addr_desc_index);

		if(++c == 3) {
		    c = 0;
		    fputc('\n', s);
		} else {
		    fputs("| ", s);
		}

		break;

	    case(ADDR_DESC_PHYSICAL):
		fprintf(s, "%06x %02x %08x ", a->u.physical.cyl, a->u.physical.head, a->u.physical.sector);
		p += sizeof(struct scsi_addr_desc_physical);

		if(++c == 3) {
		    c = 0;
		    fputc('\n', s);
		} else {
		    fputs("| ", s);
		}

		break;

	    default:
		{
		    char    b[80];
		    int	    r = q - p;
		    int	    l = (r > 16) ? 16 : r;
		    
		    if(hexdump_16(p, l, b) == NULL)
			fprintf(s, "Err dumping hex - %m\n");
		    else
			fprintf(s, "%s\n", b);

		    p += l;
		}
	}
    }

    fputc('\n', s);

    return;
}



int	scsi_write_defect_list(FILE *fp, void *buf, u32 len, u8 format)
{
    void    *p, *q;
    int	    n;
    struct scsi_addr_desc *a;


    assert(buf);

    if(format > 7) {
	errno = EINVAL;
	return(0);
    }

    n = len ? len / scsi_addr_desc_len[format] : 0;

    fprintf(fp, "DEFECT LIST: %d %d", format, n);

    p = buf;
    q = p + len;

    while(p < q) {
	a = p;

	switch(format) {
	    case(ADDR_DESC_INDEX):
		fprintf(fp, "0x%06x 0x%02x 0x%08x\n", a->u.index.cyl, a->u.index.head, a->u.index.offset);
		p += sizeof(struct scsi_addr_desc_index);
		break;

	    case(ADDR_DESC_PHYSICAL):
		fprintf(fp, "0x%06x 0x%02x 0x%08x\n", a->u.physical.cyl, a->u.physical.head, a->u.physical.sector);
		p += sizeof(struct scsi_addr_desc_physical);
		break;

	    case(ADDR_DESC_LOGICAL):
	    default:
		fprintf(fp, "0x%08x\n", a->u.logical.sector);
		p += sizeof(struct scsi_addr_desc_logical);
		break;
	}

	if(errno)
	    return(0);
    }

    fprintf(fp, "END\n");

    return(1);
}

    

int	scsi_read_defect_list(FILE *fp, void *buf, u32 *len)
{
    char    rbuf[81];
    char    *p, *q;

    int	    f, n, i;

    assert(0);

    assert(fp);

    while(1) {
	if(!fgets(rbuf, 81, fp))
	    goto abt;

	if(errno)
	    goto abt;

	if(strncmp(rbuf, "DEFECT LIST: ", 13) != 0)
	    continue;

	break;
    }

    p = index(rbuf, ':') + 2;
    q = index(rbuf, '\n');

    if(!q || q <= p)
	goto abt;

    f = strtol(p, &p, 0);
    if(!p || errno)
	goto abt;

    n = strtol(p, NULL, 0);
    if(errno)
	goto abt;

    for(i = 0; i < n; i++) {
	if(!(fgets(rbuf, 81, fp)))
	    goto abt;
    }

    return(f);

abt:

    return(-1);
}






/*
    scsi_get_long_sector_size()

*/
int	scsi_get_long_sector_size(struct scsi_fd *fd, u32 lba)
{
    char    buf;
    int	    lss = 0;


    scsi_READ_LONG(fd, &buf, lba, 1, F_CORRCT);

    if(SENSE_VALID(fd->hdr.sbp)) {
	lss = 1 - SENSE_INFORMATION(fd->hdr.sbp);

	if(lss < 64 || lss > 65536) {
/*
	    fprintf(stderr, "Sense information field contains junk.\n");
	    scsi_sense_dump(stderr, fd->hdr.sbp);
*/
	    return(0);
	}

	fd->hdr.masked_status = 0;
	SENSE_VALID(fd->hdr.sbp) = 0;

	return(lss);
    }

    fd->hdr.masked_status = 0;
    SENSE_VALID(fd->hdr.sbp) = 0;

    return(0);
}
