/*
    Copyright 2002, 2003, Steve Thompson <stevet010@softhome.net>

    This software is free for non-commercial, non-military use.

    Included with this package, you should have received a file called
    `Licensing' which contains detailed terms of use.

*/

#ifndef SCSI_MODEPAGE_H
#define SCSI_MODEPAGE_H




/*
    MODE PAGES FOR ALL DEVICE TYPES

*/

struct scsi_mode_page_header {
    u8	    page_code:6;
    u8	    reserved:1;
    u8	    PS:1;
    u8	    page_length;
    u8	    b[0];
};



/*
    Disconnect-reconnect page		0x02

*/
#define MODE_DISCONNECT_RECONNECT   0x02

struct scsi_mode_disconnect {
    u8	    buffer_full_ratio;
    u8	    buffer_empty_ratio;
    u16	    bus_inactivity_limit;
    u16	    disconnect_time_limit;
    u16	    connect_time_limit;
    u16	    maximum_burst_size;
    u8	    DTDC:3;
    u8	    Dlmm:1;
    u8	    fair_arbitration:3;
    u8	    EMDP:1;
    u8	    reserved2;
    u16	    first_burst_size;
} __attribute__((packed));


#define DTDC_NONE	0x0
#define DTDC_DTRANSFER	0x1
#define DTDC_SYNCHRONUS	0x3



/*
    Peripheral Device Page		0x09

    Obsolete for SCSI 3.

*/

#define MODE_PERIPHERAL_DEVICE	    0x09

struct scsi_mode_peripheral_device {
    u16	    interface_identifier;
    u8	    reserved1;
    u8	    reserved2;
    u8	    reserved3;
    u8	    reserved4;
    u8	    vendor_specific[0];
} __attribute__((packed));


#define	INTERFACE_SCSI	    0x0000
#define	INTERFACE_STORAGE   0x0001
#define	INTERFACE_ESDI	    0x0002
#define INTERFACE_IPI_2	    0x0003
#define INTERFACE_IPI_3	    0x0004
#define INTERFACE_VENDOR    0xffff




/*
    Control Mode Page			0x0a

*/

#define MODE_CONTROL		    0x0a

struct scsi_mode_control {
    u8	    RLEC:1;
    u8	    GLTSD:1;
    u8	    reserved1:3;
    u8	    tst:3;
    u8	    DQue:1;
    u8	    QErr:2;
    u8	    reserved2:1;
    u8	    queue_algorithm_modifier:4;
    u8	    EAERP:1;
    u8	    UAAERP:1;
    u8	    RAERP:1;
    u8	    SWP:1;
    u8	    UA_INTLCK_CTL:2;
    u8	    RAC:1;
    u8	    TAS:1;
    u8	    autoload_mode:3;
    u8	    reserved4:5;
    u16	    AER_holdoff_period;
    u16	    busy_timeout_period;
    u16	    extended_self_test_completion_time;
} __attribute__((packed));

#define QUEUE_ALGORITHM_MODIFIER_RESTRICTED	0x0
#define QUEUE_ALGORITHM_MODIFIER_UNRESTRICTED	0x1

#define AUTOLOAD_FULL_ACCESS	0x0
#define AUTOLOAD_AUX_ACCESS	0x1
#define AUTOLOAD_NO_ACCESS	0x2



/*
    Power Conditions Page		    0x0d

*/

#define MODE_POWER_CONDITIONS		0x0d 

struct scsi_mode_power_conditions {
    u8	    STANDBY:1;
    u8	    IDLE:1;
    u8	    reserved:6;
    u32	    idle_condition_timer;
    u32	    standby_condition_timer;
} __attribute__((packed));



/*
    Protocol Specific LUN page		    0x18

*/

#define MODE_PROTOCOL_SPECIFIC_LUN	0x18

struct scsi_mode_protocol_specific_lun {
    u8	    protocol_identifier:4;
    u8	    reserved1:4;
    u8	    protocol_data[0];
} __attribute__((packed));


#define SCSI_PROTO_FIBRE    0x0
#define SCSI_PROTO_PARALLEL 0x1
#define SCSI_PROTO_SSA	    0x2
#define SCSI_PROTO_1394	    0x3
#define SCSI_PROTO_RDMA	    0x4
#define SCSI_PROTO_iSCSI    0x5



/*
    Protocol Specific Port page		    0x19

*/

#define MODE_PROTOCOL_SPECIFIC_PORT	0x19

struct scsi_mode_protocol_specific_port {
    u8	    protocol_identifier:4;
    u8	    reserved1:4;
    u8	    protocol_data[0];
} __attribute__((packed));




/*
    Informational Exceptions control page   0x1c

*/

#define MODE_EXCEPTIONS		    0x1c

struct scsi_mode_info_exceptions {
    u8	    LogErr:1;
    u8	    reserved1:1;
    u8	    TEST:1;
    u8	    DExcpt:1;
    u8	    EWasc:1;
    u8	    EBF:1;
    u8	    reserved2:1;
    u8	    PERF:1;
    u8	    MRIE:4;
    u8	    reserved3:4;
    u32	    interval_timer;
    u32	    report_count;
} __attribute__((packed));





/*

	DIRECT ACCESS DEVICES

*/


/*
    Read/Write Error Recovery Page	0x01

*/
#define MODE_ERROR_RECOVERY	    0x01

struct scsi_mode_error_recovery {
    u8	    DCR:1;
    u8	    DTE:1;
    u8	    PER:1;
    u8	    EER:1;
    u8	    RC:1;
    u8	    TB:1;
    u8	    ARRE:1;
    u8	    AWRE:1;
    u8	    read_retry;
    u8	    correction_span;
    u8	    head_offset;
    u8	    data_strobe_offset;
    u8	    reserved1;
    u8	    write_retry;
    u8	    reserved2;
    u16	    recovery_limit;
} __attribute__((packed));


/*
    Format Device Page			0x03

*/

#define MODE_FORMAT_DEVICE	    0x03

struct scsi_mode_format_device {
    u16	    tracks_per_zone;
    u16	    alternate_sectors_per_zone;
    u16	    alternate_tracks_per_zone;
    u16	    alternate_tracks_per_lun;
    u16	    sectors_per_track;
    u16	    bytes_per_sector;
    u16	    interleave;
    u16	    track_skew_factor;
    u16	    cylinder_skew_factor;
    u8	    reserved1:4;
    u8	    SURF:1;
    u8	    RMB:1;
    u8	    HSEC:1;
    u8	    SSEC:1;
    u8	    reserved2;
    u8	    reserved3;
    u8	    reserved4;
} __attribute__((packed));



/*
    Rigid Disk Drive Geometry Page	0x04

*/

#define MODE_RIGID_DISK		    0x04

struct scsi_mode_rigid_disk {
    u16	    cylinders;
    u8	    heads:8;
    u8	    write_precomp_msb;
    u16	    write_precomp_lsb;
    u8	    reduced_current_msb;
    u16	    reduced_current_lsb;
    u16	    step_rate;
    u8	    landing_zone_msb;
    u16	    landing_zone_lsb;
    u8	    RPL:2;
    u8	    reserved1:6;
    u8	    rotation_offset;
    u8	    reserved2;
    u16	    rotation_rate;
    u8	    reserved3;
    u8	    reserved4;
} __attribute__((packed));




/*
    Flexible Disk Page			0x05

*/

#define MODE_FLEXIBLE_DISK	    0x05

struct scsi_mode_flexible_disk {
    u16	    transfer_rate;
    u8	    heads;
    u8	    track_size;
    u16	    sector_size;
    u16	    cylinders;
    u16	    write_precompensation;
    u16	    reduced_write_current;
    u16	    step_rate;
    u8	    step_pulse_width;
    u16	    head_settle_delay;
    u8	    motor_on_delay;
    u8	    motor_off_delay;
    u8	    reserved1:5;
    u8	    MO:1;
    u8	    SSN:1;
    u8	    TRDY:1;
    u8	    write_compensation;
    u8	    head_load_delay;
    u8	    head_unload_delay;
    u8	    pin_2:4;
    u8	    pin_34:4;
    u8	    pin_1:4;
    u8	    pin_4:4;
    u16	    rotation_rate;
    u8	    reserved2;
    u8	    reserved3;
} __attribute__((packed));




/*
    RBC Parameter Page			0x06

*/

#define MODE_RBC_PARAMETER_LIST	    0x06

struct scsi_mode_rbc {
    u8	    WCD:1;
    u8	    reserved1:7;
    u16	    logical_block_size;
    u32	    logical_block_capacity;
    u8	    power_perfomance;
    u8	    LOCKD:1;
    u8	    FORMATD:1;
    u8	    WRITED:1;
    u8	    READD:1;
    u8	    reserved2:4;
    u8	    reserved3;
} __attribute__((packed));




/*
    Veryify error recovery page		0x07

*/

#define MODE_VERIFY_ERROR_RECOVERY  0x07

struct scsi_mode_verify_error {
    u8	    DCR:1;
    u8	    DTE:1;
    u8	    PER:1;
    u8	    EER:1;
    u8	    reserved1:4;
    u8	    verify_retry;
    u8	    verify_correction_span;
    u8	    reserved2;
    u8	    reserved3;
    u8	    reserved4;
    u8	    reserved5;
    u8	    reserved6;
    u16	    verify_recovery_time_limit;
} __attribute__((packed));




/*
    Caching Page			0x08

*/

#define MODE_CACHING		    0x08

struct scsi_mode_caching {
    u8	    RCD:1;
    u8	    MF:1;
    u8	    WCE:1;
    u8	    reserved1:5;
    u8	    write_retention_priority:4;
    u8	    demand_read_retention_priority:4;
    u16	    disable_prefetch_transfer_length;
    u16	    minimum_prefetch;
    u16	    maximum_prefetch;
    u16	    maximum_prefetch_ceiling;
} __attribute__((packed));

#define RETENTION_PRIO_NONE	0x0
#define RETENTION_PRIO_PREFETCH 0x1
#define RETENTION_PRIO_RDWR	0xf




/*
    Notch and Partition Page		0x0b

*/

#define MODE_NOTCH_PARTITION	    0x0c

struct scsi_mode_notch {
    u8	    reserved1:6;
    u8	    LPN:1;
    u8	    ND:1;
    u8	    reserved2;
    u16	    maximum_notches;
    u16	    active_notch;
    u32	    starting_boundary;
    u32	    ending_boundary;
    u64	    notches;
} __attribute__((packed));



/*
    Meduim Types Supported Page		0x0c

*/

#define MODE_MEDIUM_TYPES	    0x0b

struct scsi_mode_medium_types {
    u8	    reserved1;
    u8	    reserved2;
    u8	    medium_1;
    u8	    medium_2;
    u8	    medium_3;
    u8	    medium_4;
} __attribute__((packed));



/*
    XOR Control mode page		0x10

*/

#define MODE_XOR_CONTROL	    0x10

struct scsi_mode_xor_control {
    u8	    reserved1:1;
    u8	    XORDIS:1;
    u8	    reserved2:6;
    u32	    maximum_xor_write_size;
    u32	    reserved3;
    u32	    maximum_regenerate_size;
    u32	    maximum_rebuild_read_size;
    u16	    reserved4;
    u16	    rebuild_delay;
} __attribute__((packed));




/*
    Medium Partition Page 1		0x11

*/

#define MODE_MEDIUM_PARTITION	    0x11
#define MODE_MEDIUM_PARTITION_2	    0x12
#define MODE_MEDIUM_PARTITION_3	    0x13
#define MODE_MEDIUM_PARTITION_4	    0x14

struct scsi_mode_medium_partition {
    u8	    max_additional_partitions;
    u8	    additional_partitions_defined;
    u8	    reserved1:3;
    u8	    PSUM:2;
    u8	    IDP:1;
    u8	    SDP:1;
    u8	    FDP:1;
    u8	    medium_format_recognition;
    u8	    reserved2;
    u8	    reserved3;
} __attribute__((packed));

struct scsi_mode_medium_partition_2 {
    u16	    partition_size_desc[0];
} __attribute__((packed));




/*
    SEQUENTIAL DEVICE MODE PAGES


*/

/*
    Device configuration page

*/

#define MODE_DEVICE_CONFIGURATION   0x10

struct scsi_mode_device_configuration {
    u8	    active_format:4;
    u8	    CAF:1;
    u8	    CAP:1;
    u8	    reserved1:1;
    u8	    active_partition;
    u8	    write_buffer_full_ratio;
    u8	    read_buffer_empty_ratio;
    u16	    write_delay_time;
    u8	    REW:1;
    u8	    RB0:1;
    u8	    SOCF:1;
    u8	    AVC:1;
    u8	    RSmk:1;
    u8	    BIS:1;
    u8	    DBR:1;
    u8	    gap_size;
    u8	    reserved2:3;
    u8	    SEW:1;
    u8	    EEG:1;
    u8	    EOD_defined:3;
    u8	    buffer_size_early_warning_msb;
    u16	    buffer_size_early_warning_lsb;
    u8	    compression_algorithm;
    u8	    reserved3;
} __attribute__((packed));




/*
    Printer Options Page		0x05

*/

#define MODE_PRINTER_CONTROL	    0x05

struct scsi_mode_printer_options {
    u8	    font_identification:7;
    u8	    EVFU:1;
    u8	    AFC:1;
    u8	    SCTE:1;
    u8	    reserved1:2;
    u8	    slew_mode:2;
    u8	    reserved:2;
    u16	    maximum_line_length;
    u8	    EVFU_format_start_char;
    u8	    EVFU_format_end_char;
    u8	    form_slew_options:4;
    u8	    line_slew_options:4;
    u8	    reserved3:4;
    u8	    data_termination_options:4;
    u8	    reserved4;
    u8	    reserved5;
} __attribute__((packed));





#define MODE_CDROM		    0x0d


/*
    CDROM Audio Control Parameters Page		0x0e

*/

#define MODE_CDROM_AUDIO	    0x0e

struct scsi_mode_audio_control {
    u8	    reserved1:1;
    u8	    SOTC:1;
    u8	    Immed:1;
    u8	    reserved2:5;
    u8	    reserved3;
    u8	    reserved4;
    u8	    sec_format:4;
    u8	    reserved5:3;
    u8	    APRVal:1;
    u16	    audio_playback_rate;
    u8	    port_0_chan:4;
    u8	    reserved6:4;
    u8	    port_0_vol;
    u8	    port_1_chan:4;
    u8	    reserved7:4;
    u8	    port_1_vol;
    u8	    port_2_chan:4;
    u8	    reserved8:4;
    u8	    port_2_vol;
    u8	    port_3_chan:4;
    u8	    reserved9:4;
    u8	    port_3_vol;
} __attribute__((packed));



/*

    OPTICAL MEMORY DEVICE PAGES

*/

#define MODE_OPTICAL_MEMORY	    0x06

struct scsi_mode_optical_memory {
    u8	    RUBR:1;
    u8	    reserved1:7;
    u8	    reserved2;
};


/*

	MEDIUM CHANGER

*/




/*
    Mode Sense Page Control Field

+=======-=====================-============+
| Code  |  Type of parameter  | Subclause  |
|-------+---------------------+------------|
|  00b  |  Current values     |  8.2.10.1  |
|  01b  |  Changeable values  |  8.2.10.2  |
|  10b  |  Default values     |  8.2.10.3  |
|  11b  |  Saved values       |  8.2.10.4  |
+==========================================+
*/
#define PAGE_CONTROL_CURRENT    0x0
#define PAGE_CONTROL_CHANGEABLE 0x1
#define PAGE_CONTROL_DEFAULT    0x2
#define PAGE_CONTROL_SAVED      0x3



struct scsi_mode_page {
    struct scsi_mode_page_header		    hdr;
    union {
	/* common pages */
	struct scsi_mode_disconnect		disconnect;
	struct scsi_mode_peripheral_device	peripheral;
	struct scsi_mode_control		control;
	struct scsi_mode_protocol_specific_lun	lun;
	struct scsi_mode_protocol_specific_port	port;
	struct scsi_mode_power_conditions	power;
	struct scsi_mode_info_exceptions	exceptions;
	/* direct access */
	struct scsi_mode_error_recovery		error;
	struct scsi_mode_format_device		format;
	struct scsi_mode_rigid_disk		rigid;
	struct scsi_mode_flexible_disk		flexible;
	struct scsi_mode_rbc			rbc;
	struct scsi_mode_verify_error		verify;
	struct scsi_mode_caching		caching;
	struct scsi_mode_notch			notch;
	struct scsi_mode_medium_types		medium;
	struct scsi_mode_xor_control		xor;
	struct scsi_mode_medium_partition	medium_partition;
	struct scsi_mode_medium_partition_2	medium_partition_2;
	/* sequential */
	struct scsi_mode_device_configuration	device_configuration;
	/* printer */
	struct scsi_mode_printer_options	printer;
	/* CDROM */
	struct scsi_mode_audio_control		audio_control;
	/* Optical */
	struct scsi_mode_optical_memory		optical;
	/* Medium Changer */
	/* Unknown */
	u8					b[0];
    } u;
};



struct scsi_mode_param_block_desc {
    u32	    density_code:8;
    u32	    block_count:24;
    u32	    reserved:8;
    u32	    block_len:24;
};

struct scsi_mode_param {
    u8	    mode_data_len;
    u8	    medium_type;
    u8	    dev_spec_param;
    u8	    desc_len;
    u8	    p[0];
};

struct scsi_mode_param_10 {
    u16	    mode_data_len;
    u8	    medium_type;
    u8	    dev_spec_param;
    u8	    reserved1;
    u8	    reserved2;
    u16	    desc_len;
    u8	    p[0];
};






extern char *scsi_mode_pages[];

int     scsi_unpack_mode_param(struct scsi_mode_param *, void *, void *);
int     scsi_unpack_mode_param_10(struct scsi_mode_param_10 *, void *, void *);

void    scsi_byteswap_mode_param(struct scsi_mode_param *);
void    scsi_byteswap_mode_param_10(struct scsi_mode_param_10 *);
void    scsi_byteswap_mode_param_block_desc(struct scsi_mode_param_block_desc *);
void    scsi_byteswap_mode_page(struct scsi_mode_page *);

struct scsi_mode_param *scsi_dup_mode_param(struct scsi_mode_param *);

#endif

