/*
    Copyright 2002, 2003, Steve Thompson <stevet010@softhome.net>

    This software is free for non-commercial, non-military use.

    Included with this package, you should have received a file called
    `Licensing' which contains detailed terms of use.

*/

#ifndef SCSI_INQUIRY_H
#define SCSI_INQUIRY_H


/*
    The standard defines fields up to 95 bytes.

    The allocation is limited to 256 for now in the expectation that it will
    be increased when it is determined that more is needed for particular
    devices.

    The alternate method is to get the inquiry data with a small length
    specified and then repeat the operation depending on whether there is a
    residual count indicated by the kernel.
*/
#define SCSI_DEFAULT_PAGE_LEN   255





/*
 * INQUIRY data defs
 */
struct scsi1_inquiry_data {
    u8	    peripheral_device_type;

    u8	    device_type_qualifier:7;
    u8	    RMB:1;

    u8	    ANSI:3;
    u8	    ECMA:3;
    u8	    ISO:2;

    u8	    reserved;
    u8	    additional_length;
    u8	    vendor_unique[0];
};


struct scsi2_inquiry_data {
    u8	    peripheral_device_type:5;
    u8	    peripheral_qualifier:3;

    u8	    device_type_modifier:7;
    u8	    RMB:1;

    u8	    ANSI:3;
    u8	    ECMA:3;
    u8	    ISO:2;

    u8	    response_data_format:4;
    u8	    reserved1:2;
    u8	    TrmIOP:1;
    u8	    AENC:1;

    u8	    additional_length;

    u8	    reserved2[2];

    u8	    SftRe:1;
    u8	    CmdQue:1;
    u8	    reserved3:1;
    u8	    Linked:1;
    u8	    Sync:1;
    u8	    WBus16:1;
    u8	    WBus32:1;
    u8	    RelAdr:1;

    u8	    vendor_id[8];
    u8	    product_id[16];
    u8	    product_rev[4];
    u8	    vendor_specific[30];
    u8	    reserved4[40];
    u8	    vendor_data[0];
};


/*
    T10/1236-D rev 18
*/
struct scsi3_inquiry_data {
    u8	    peripheral_device_type:5;
    u8	    peripheral_qualifier:3;

    u8	    device_type_modifier:7;
    u8	    RMB:1;

    u8	    version;

    u8	    response_data_format:4;
    u8	    HiSup:1;
    u8	    NormACA:1;
    u8	    obs0:1;
    u8	    AERC:1;

    u8	    additional_length;

    u8	    reserved2:4;
    u8	    ALUA:2;
    u8	    _3pc:1;
    u8	    SCCS:1;

    u8	    Addr16:1;
    u8	    obs1:1;
    u8	    obs2:1;
    u8	    MChngr:1;
    u8	    MultiP:1;
    u8	    VS1:1;
    u8	    EncServ:1;
    u8	    BQue:1;

    u8	    VS2:1;
    u8	    CmdQue:1;
    u8	    obs3:1;
    u8	    Linked:1;
    u8	    Sync:1;
    u8	    WBus16:1;
    u8	    obs4:1;
    u8	    RelAdr:1;

    u8	    vendor_id[8];
    u8	    product_id[16];
    u8	    product_rev[4];
    u8	    vendor_specific[30];
    u8	    IUS:1;
    u8	    QAS:1;
    u8	    clocking:2;
    u8	    reserved3:4;
    u8	    reserved4[40];
    u8	    vendor_data[0];
};



struct scsi_inquiry {
    union {
	struct scsi1_inquiry_data   s1;
	struct scsi2_inquiry_data   s2;
	struct scsi3_inquiry_data   s3;
	u8			    b[0];
    } u;
} __attribute__ ((packed));



#define INQUIRY_RDF(i)		    ((i)->u.s2.response_data_format)
#define INQUIRY_VENDOR_DATA(i)	    (INQUIRY_RDF(i) ? (i)->u.s2.vendor_data : (i)->u.s1.vendor_data)

#define INQUIRY_DATA_LEN(p)	    ((p)->u.s1.additional_length + 5)



/*
    Peripheral device types
 */
#define TYPE_DIRECT         0x00
#define TYPE_SEQUENTIAL     0x01
#define	TYPE_PRINTER	    0x02
#define TYPE_PROCESSOR      0x03    /* HP scanners use this */
#define TYPE_WORM           0x04    /* Treated as ROM by our system */
#define TYPE_CDROM          0x05
#define TYPE_SCANNER        0x06
#define TYPE_OPTICAL        0x07
#define TYPE_MEDIUM_CHANGER 0x08
#define TYPE_COMMUNICATIONS 0x09
#define TYPE_PREPRESS	    0x0a
#define TYPE_GRAPHICS	    0x0b
#define TYPE_STORAGE	    0x0c
#define TYPE_ENCLOSURE	    0x0d
#define TYPE_RBC	    0x0e
#define TYPE_OCRW	    0x0f
#define TYPE_BRIDGE	    0x10
#define TYPE_OBJECT_STOR    0x11
#define TYPE_NO_LUN         0x1f





void	scsi_inquiry_text_print(FILE *, struct scsi_inquiry *);

extern const char * scsi_peripheral_qualifier[];
extern const char * scsi_peripheral_device_type[];

#endif
