/*
    Copyright 2002, 2003, Steve Thompson <stevet010@softhome.net>

    This software is free for non-commercial, non-military use.

    Included with this package, you should have received a file called
    `Licensing' which contains detailed terms of use.

*/

#ifndef SCSILIB_CDB_H
#define SCSILIB_CDB_H



/*
 * Data structures and macros for 
 * SCSI Command Descriptor Blocks
 *
 */

#define CDB_OPCODE(c)		    ((c)->u.b[0])
#define SET_CDB_OPCODE(c, arg)	    { (c)->u.b[0]  = (arg); }

#define SCSI_OPCODE_GROUP(c)	    (((c) & 0xe0) >> 5)
#define CDB_OPCODE_GROUP(c)	    SCSI_OPCODE_GROUP(CDB_OPCODE(c))



struct scsi_cdb6 {
    u8	    opcode:8;
    u8	    LBA3:5;
    u8	    lun:3;
    u8	    LBA2:8;
    u8	    LBA1:8;
    u8	    TPA;
    u8	    control;
} __attribute__ ((packed));


#define CDB6_OPCODE(c)		CDB_OPCODE(c)
#define SET_CDB6_OPCODE(c, arg)	SET_CDB_OPCODE(c, arg)

#define CDB6_LUN(c)		((c)->u.cdb6.lun)
#define SET_CDB6_LUN(c, arg)	{ (c)->u.cdb6.lun = (arg); }

#define CDB6_LBA(c)		((c)->u.cdb6.LBA1 | ((c)->u.cdb6.LBA2 << 8) | ((c)->u.cdb6.LBA3 << 16))
#define SET_CDB6_LBA(c, arg)	{ (c)->u.cdb6.LBA1 = (((arg) & 0x0000ff));	    \
				  (c)->u.cdb6.LBA2 = (((arg) & 0x00ff00) >> 8);	    \
				  (c)->u.cdb6.LBA3 = (((arg) & 0x1f0000) >> 16); }

#define CDB6_TPA(c)		((c)->u.cdb6.TPA)
#define SET_CDB6_TPA(c, arg)	{ (c)->u.cdb6.TPA = (arg); }

#define CDB6_TLEN(c)		(CDB6_TPA(c) ? : 256)
#define SET_CDB6_TLEN(c, arg)	SET_CDB6_TPA(c, ((arg) == 256) ? 0 : (arg))

#define CDB6_PLEN(c)		CDB6_TPA(c)
#define SET_CDB6_PLEN(c, arg)	SET_CDB6_TPA(c, arg)

#define CDB6_ALEN(c)		CDB6_TPA(c)
#define SET_CDB6_ALEN(c, arg)	SET_CDB6_TPA(c, arg)

#define CDB6_CONTROL(c)		((c)->u.cdb6.control)
#define SET_CDB6_CONTROL(c)	{ (c)->u.cdb6.control = (arg); }



/*
 * Typical Ten byte CDB
 */
struct scsi_cdb10 {
    u8	    opcode;
    u8	    reserved1:5;
    u8	    lun:3;
    u32	    LBA;
    u8	    reserved2;
    u16	    TPA;
    u8	    control;
} __attribute__ ((packed));


#define CDB10_OPCODE(c)		    CDB_OPCODE(c)
#define SET_CDB10_OPCODE(c, arg)    SET_CDB_OPCODE(c, arg)

#define CDB10_LUN(c)		    ((c)->cdb10.lun)
#define SET_CDB10_LUN(c, arg)	    { (c)->u.cdb10.lun = (arg);	}

#define CDB10_LBA(c)		    STOU32((c)->u.cdb10.LBA)
#define SET_CDB10_LBA(c, arg)	    { (c)->u.cdb10.LBA = U32TOS(arg); }

#define CDB10_TPA(c)		    STOU16((c)->u.cdb10.TPA)
#define SET_CDB10_TPA(c, arg)	    { (c)->u.cdb10.TPA = U16TOS(arg); }

#define CDB10_TLEN(c)		    CDB10_TPA(c)
#define SET_CDB10_TLEN(c, arg)	    SET_CDB10_TPA(c, arg)

#define CDB10_PLEN(c)		    CDB10_TPA(c)
#define SET_CDB10_PLEN(c, arg)	    SET_CDB10_TPA(c, arg)

#define CDB10_ALEN(c)		    CDB10_TPA(c)
#define SET_CDB10_ALEN(c, arg)	    SET_CDB10_TPA(c, arg)

#define CDB10_CONTROL(c)	    ((c)->u.cdb10.control)
#define SET_CDB10_CONTROL(c, arg)   { (c)->u.cdb10.control = (arg); }



struct scsi_cdb12 {
    u8	    opcode;
    u8	    reserved1:5;
    u8	    lun:3;
    u32	    LBA;
    u32	    TPA;
    u8	    control;
} __attribute__ ((packed));


#define CDB12_OPCODE(c)		    CDB_OPCODE(c)
#define SET_CDB12_OPCODE(c, arg)    SET_CDB_OPCODE(c, arg)

#define CDB12_LUN(c)		    ((c)->u.cdb12.lun)
#define SET_CDB12_LUN(c, arg)	    { (c)->u.cdb12.lun = (arg);	}

#define CDB12_LBA(c)		    STOU32((c)->u.cdb12.LBA)
#define SET_CDB12_LBA(c, arg)	    { (c)->u.cdb12.LBA = U32TOS(arg); }

#define CDB12_TPA(c)		    STOU32((c)->u.cdb12.TPA)
#define SET_CDB12_TPA(c, arg)	    { (c)->u.cdb12.TPA = U32TOS(arg); }

#define CDB12_TLEN(c)		    CDB12_TPA(c)
#define SET_CDB12_TLEN(c, arg)	    SET_CDB12_TPA(c, arg)

#define CDB12_PLEN(c)		    CDB12_TPA(c)
#define SET_CDB12_PLEN(c, arg)	    SET_CDB12_TPA(c, arg)

#define CDB12_ALEN(c)		    CDB12_TPA(c)
#define SET_CDB12_ALEN(c, arg)	    SET_CDB12_TPA(c, arg)

#define CDB12_CONTROL(c)	    ((c)->u.cdb12.control)
#define SET_CDB12_CONTROL(c, arg)   { (c)->u.cdb12.control = (arg); }




struct scsi_cdb16 {
    u8	    opcode;
    u8	    reserved1:5;
    u8	    lun:3;
    u32	    LBA;
    u32	    CDB;
    u32	    TPA;
    u8	    reserved2;
    u8	    control;
} __attribute__ ((packed));


#define CDB16_OPCODE(c)		CDB_OPCODE(c)
#define SET_CDB16_OPCODE(c, arg) SET_CDB_OPCODE(c, arg)

#define CDB16_LUN(c)		((c)->u.cdb16.lun)
#define SET_CDB16_LUN(c, arg)	{ (c)->u.cdb16.lun = (arg);	    }

#define CDB16_LBA(c)		STOU32((c)->u.cdb16.LBA)
#define SET_CDB16_LBA(c, arg)	{ (c)->u.cdb16.LBA = U32TOS(arg);   }

#define CDB16_CDB(c)		STOU32((c)->u.cdb16.CDB)
#define SET_CDB16_CDB(c, arg)	{ (c)->u.cdb16.CDB = U32TOS(arg);   }

#define CDB16_TPA(c)		STOU32((c)->u.cdb16.TPA)
#define SET_CDB16_TPA(c, arg)	{ (c)->u.cdb16.TPA = U32TOS(arg);   }

#define CDB16_TLEN(c)		(CDB16_TPA(c) ? : 65536)
#define SET_CDB16_TLEN(c, arg)	SET_CDB16_TPA(c, arg)

#define CDB16_PLEN(c)		CDB16_TPA(c)
#define SET_CDB16_PLEN(c, arg)	SET_CDB16_TPA(c, arg)

#define CDB16_ALEN(c)		CDB16_TPA(c)
#define SET_CDB16_ALEN(c, arg)	SET_CDB16_TPA(c, arg)

#define CDB16_CONTROL(c)		((c)->u.cdb16.control)
#define SET_CDB16_CONTROL(c, arg)	{ (c)->u.cdb16.control = (arg); }



/*
 * Miscellaneous macros.
 */
#define ZERO_CDB(c)	    memset((c), 0x00, sizeof(struct scsi_cdb))




/*
    Macros to frob the control byte of the CDB.
 
 */

#define CDB6_V1(c)		TST(CDB6_CONTROL(c), BIT(7))
#define SET_CDB6_V1(c)		{ (c)->u.cdb6.control |= BIT(7); }

#define CDB6_V2(c)		TST(CDB6_CONTROL(c), BIT(6))
#define SET_CDB6_V2(c)		{ (c)->u.cdb6.control |= BIT(6); }

#define CDB6_FLAG(c)		TST(CDB6_CONTROL(c), BIT(1))
#define SET_CDB6_FLAG(c)	{ (c)->u.cdb6.control |= BIT(1); }

#define CDB6_LINK(c)		TST(CDB6_CONTROL(c), BIT(0))
#define SET_CDB6_LINK(c)	{ (c)->u.cdb6.control |= BIT(0); }

#define CDB10_V1(c)		TST(CDB10_CONTROL(c), BIT(7))
#define SET_CDB10_V1(c)		{ (c)->u.cdb10.control |= BIT(7); }

#define CDB10_V2(c)		TST(CDB10_CONTROL(c), BIT(6))
#define SET_CDB10_V2(c)		{ (c)->u.cdb10.control |= BIT(6); }

#define CDB10_FLAG(c)		TST(CDB10_CONTROL(c), BIT(1))
#define SET_CDB10_FLAG(c)	{ (c)->u.cdb10.control |= BIT(1); }

#define CDB10_LINK(c)		TST(CDB10_CONTROL(c), BIT(0))
#define SET_CDB10_LINK(c)	{ (c)->u.cdb10.control |= BIT(0); }

#define CDB12_V1(c)		TST(CDB12_CONTROL(c), BIT(7))
#define SET_CDB12_V1(c)		{ (c)->u.cdb12.control |= BIT(7); }

#define CDB12_V2(c)		TST(CDB12_CONTROL(c), BIT(6))
#define SET_CDB12_V2(c)		{ (c)->u.cdb12.control |= BIT(6); }

#define CDB12_FLAG(c)		TST(CDB12_CONTROL(c), BIT(1))
#define SET_CDB12_FLAG(c)	{ (c)->u.cdb12.control |= BIT(1); }

#define CDB12_LINK(c)		TST(CDB12_CONTROL(c), BIT(0))
#define SET_CDB12_LINK(c)	{ (c)->u.cdb12.control |= BIT(0); }


#define F_LINK	0x1000
#define F_FLAG	0x2000
#define F_V1	0x4000
#define F_V2	0x8000



/*
   
    Miscellaneous macros which a specific to individual commands.  Listed by
    command.
*/


/*
    CHANGE_DEFINITION
*/
#define CDB10_SAVE(c)			TST((c)->u.b[2], 0)
#define SET_CDB10_SAVE(c, arg)		{ (c)->u.b[2] |= BIT(0); }

#define CDB10_DFN(c)			((c)->u.b[3] & 0x7f)
#define SET_CDB10_DFN(c, d)		{ (c)->u.b[3] = (d & 0x7f); }

#define CDB10_PLEN8(c)			((c)->u.b[8])
#define SET_CDB10_PLEN8(c, arg)		{ (c)->u.b[8] = (arg); }


/*
    COPY
*/
#define CDB6_PAD(c)			TST(c->u.b[1], 0)
#define SET_CDB6_PAD(c)			{ (c)->u.b[1] |= BIT(0); }

#define F_PAD	0x0x0001

#define CDB6_PLIST24_LEN(c)		( ((c)->u.b[2] << 16) | ((c)->u.b[3] << 8) | (c)->u.b[4])
#define SET_CDB6_PLIST24_LEN(c, arg)	{ (c)->u.b[2] = (arg & 0x00ff0000) >> 16;   \
					  (c)->u.b[3] = (arg & 0x0000ff00) >> 8;    \
					  (c)->u.b[4] = (arg & 0x000000ff); }

/*
    COPY AND VERIFY
*/
#define CDB10_VBYTCHK(c)		TST(c->u.b[1], 0)
#define SET_CDB10_VBYTCHK(c)		{ (c)->u.b[1] |= BIT(0); }

#define F_VBYTCHK    0x0002

#define CDB10_PLIST_LEN(c)		( ((c)->u.b[3] << 16) | ((c)->u.b[4] << 8) | (c)->u.b[5])
#define SET10_CDB_PLIST_LEN(c, arg)	{ (c)->u.b[3] = (arg & 0x00ff0000) >> 16;   \
					  (c)->u.b[4] = (arg & 0x0000ff00) >> 8;    \
					  (c)->u.b[5] = (arg & 0x000000ff); }


/*
    FORMAT UNIT
*/
#define CDB6_DLIST(c)			((c)->u.b[1] & 0x7)
#define SET_CDB6_DLIST(c, arg)		{ (c)->u.b[1] |= ((arg) & 0x7); };

#define CDB6_CMPLIST(c)			TST((c)->u.b[1], 3)
#define SET_CDB6_CMPLIST(c)		{ (c)->u.b[1] |= BIT(3); }

#define F_CMPLIST   0x0001

#define CDB6_FMTDATA(c)			TST((c)->u.b[1], 4)
#define SET_CDB6_FMTDATA(c)		{ (c)->u.b[1] |= BIT(4); }

#define F_FMTDATA   0x0002

#define CDB6_LONGLIST(c)		TST((c)->u.b[1], 5)
#define SET_CDB6_LONGLIST(c)		{ (c)->u.b[1] |= BIT(5); }

#define F_LONGLIST  0x0004


#define CDB6_INTERLEAVE(c)		(((c)->u.b[3] << 8) | ((c)->u.b[4]))
#define SET_CDB6_INTERLEAVE(c, arg)	{ (c)->u.b[3] = (((arg) & 0xff00) >> 8); \
					  (c)->u.b[4] = ((arg) & 0x00ff); }




/*
    INQUIRY
*/
#define CDB6_EVPD(c)			TST(c->u.b[1], 0)
#define SET_CDB6_EVPD(c)		{ (c)->u.b[1] |= BIT(0); }

#define F_EVPD	0x0001

/* SCSI 3 */
#define CDB6_CMDDT(c)			TST(c->u.b[1], 1)
#define SET_CDB6_CMDDT(c)		{ (c)->u.b[1] |= BIT(1); }

#define F_CMDDT	0x0002

#define CDB6_PAGECODE(c)		((c)->u.b[2])
#define SET_CDB6_PAGECODE(c, arg)	{ (c)->u.b[2] = (arg); }



/*
    MODE SENSE 6
*/
#define CDB6_DBD(c)			TST(c->u.b[1], 3)
#define SET_CDB6_DBD(c)			{ (c)->u.b[1] |= BIT(3); }

#define F_DBD	0x0001

#define CDB6_PC(c)			(((c)->u.b[2] & 0xc0) >> 6)
#define SET_CDB6_PC(c, arg)		{ (c)->u.b[2] |= (((arg) & 0x3) << 6); }

#define CDB6_PAGE(c)			((c)->u.b[2] & 0x3f)
#define SET_CDB6_PAGE(c, arg)		{ (c)->u.b[2] |= ((arg) & 0x3f); }


/*
    MODE SENSE 10
*/
#define CDB10_DBD(c)			CDB6_DBD(c)
#define SET_CDB10_DBD(c)		SET_CDB6_DBD(c)

#define CDB10_PC(c)			CDB6_PC(c)
#define SET_CDB10_PC(c, arg)		SET_CDB6_PC(c, arg)

#define CDB10_PAGE(c)			CDB6_PAGE(c)
#define SET_CDB10_PAGE(c, arg)		SET_CDB6_PAGE(c, arg)


/*
    MODE SELECT 6
*/
#define CDB6_SP(c)			TST(c->u.b[1], 0)
#define SET_CDB6_SP(c)			{ (c)->u.b[1] |= BIT(0); }

#define F_SP	0x0002


#define CDB6_PF(c)			TST(c->u.b[1], 4)
#define SET_CDB6_PF(c)			{ (c)->u.b[1] |= BIT(4); }

#define F_PF	0x0004


/*
    MODE SELECT 10
*/
#define CDB10_SP(c)			CDB6_SP(c)
#define SET_CDB10_SP(c)			SET_CDB6_SP(c);

#define CDB10_PF(c)			CDB6_PF(c)
#define SET_CDB10_PF(c)			SET_CDB6_PF(c)



/*
    LOG_SENSE
*/
#define CDB10_PPC(c)			TST(c->u.b[1], 1)
#define SET_CDB10_PPC(c)		{ (c)->u.b[1] |= BIT(1); }

#define F_PPC	0x0008


#define CDB10_PAGECODE(c)		((c)->u.b[2] & 0xc0)
#define SET_CDB10_PAGECODE(c, arg)	{ (c)->u.b[2] |= ((arg) & 0x3f); }

#define CDB10_PARAM_PTR(c)		(((c)->u.b[5] << 8) | (c)->u.b[6])
#define SET_CDB10_PARAM_PTR(c, arg)	{ (c)->u.b[5] = ((arg) & 0x0000ff00) >> 8; \
					  (c)->u.b[6] = (arg)  & 0x000000ff; }

/*
    LOG_SELECT
*/
#define CDB10_PCR(c)			TST(c->u.b[1], 1)
#define SET_CDB10_PCR(c)		{ (c)->u.b[1] |= BIT(1); }

#define F_PCR	0x0010

#define CDB10_PCL(c)			((c)->u.b[2] >> 6)
#define SET_CDB10_PCL(c)		{ (c)->u.b[2] |= ((c) & 0x03) << 6; }

#define F_PCL	0x0020


/*
    READ BUFFER
    WRITE BUFFER
*/
#define CDB10_MODE(c)			((c)->u.b[1] & 0x03)
#define SET_CDB10_MODE(c, arg)		{ (c)->u.b[2] |= ((arg) & 0x03); }

#define CDB10_ALEN24(c)			( ((c)->u.b[6] << 16) | ((c)->u.b[7] << 8) | (c)->u.b[8])
#define SET_CDB10_ALEN24(c, arg)	{ (c)->u.b[6] = (arg & 0x00ff0000) >> 16;   \
					  (c)->u.b[7] = (arg & 0x0000ff00) >> 8;    \
					  (c)->u.b[8] = (arg & 0x000000ff); }

#define CDB10_BUFF_ID(c)		((c)->u.b[2])
#define SET_CDB10_BUFF_ID(c, arg)	{ (c)->u.b[2] = (arg); }

#define CDB10_BUFF_OFFSET(c)		(((c)->u.b[3] << 16) | ((4)->u.b[7] << 8) | (5)->u.b[8])
#define SET_CDB10_BUFF_OFFSET(c, arg)   { (c)->u.b[3] = (arg & 0x00ff0000) >> 16;   \
					  (c)->u.b[4] = (arg & 0x0000ff00) >> 8;    \
					  (c)->u.b[5] = (arg & 0x000000ff); }


/*
    READ DEFECT DATA
*/
#define CDB10_PLIST(c)			TST((c)->u.b[2], 4)
#define SET_CDB10_PLIST(c)		((c)->u.b[2] |= BIT(4))

#define F_PLIST	0x0001

#define CDB10_GLIST(c)			TST((c)->u.b[2], 3)
#define SET_CDB10_GLIST(c)		((c)->u.b[2] |= BIT(3))

#define F_GLIST	0x0002

#define CDB10_DLFMT(c)			((c)->u.b[2] & 0x7)
#define SET_CDB10_DLFMT(c, arg)		{ (c)->u.b[2] = ((c)->u.b[2] & 0xf8) | (arg & 0x7); }

#define CDB10_DLEN(c)			(((c)->u.b[7] << 8) | ((c)->u.b[8]))
#define SET_CDB10_DLEN(c, arg)		{ (c)->u.b[7] = (arg & 0x0000ff00) >> 8 ; \
					  (c)->u.b[8] = (arg & 0x000000ff) ; }


/*
    READ DEFECT DATA (12)
*/
#define CDB12_PLIST(c)			TST((c)->u.b[2], 4)
#define SET_CDB12_PLIST(c)		((c)->u.b[2] |= BIT(4))

#define CDB12_GLIST(c)			TST((c)->u.b[2], 3)
#define SET_CDB12_GLIST(c)		((c)->u.b[2] |= BIT(3))

#define CDB12_DLFMT(c)			((c)->u.b[2] & 0x7)
#define SET_CDB12_DLFMT(c, arg)		{ (c)->u.b[2] = ((c)->u.b[2] & 0xf8) | (arg & 0x7); }

#define CDB12_DLEN(c)			(((c)->u.b[6] << 24) | ((c)->u.b[7] << 16) | ((c)->u.b.8 <<8) | (c)->u.b[9])
#define SET_CDB12_DLEN(c, arg)		{ (c)->u.b[6] = (arg & 0xff000000) >> 24; \
					  (c)->u.b[7] = (arg & 0x00ff0000) >> 16; \
					  (c)->u.b[8] = (arg & 0x0000ff00) >> 8;  \
					  (c)->u.b[9] = (arg & 0x000000ff) ; }


/*
    SEND DIAGNOSTIC
*/
#define CDB6_SELFTEST(c)		TST(c->u.b[1], 2)
#define SET_CDB6_SELFTEST(c)		{ (c)->u.b[1] |= BIT(2); }

#define F_SELFTEST  0x0001

#define CDB6_DEVOFL(c)			TST(c->u.b[1], 1)
#define SET_CDB6_DEVOFL(c)		{ (c)->u.b[1] |= BIT(1); }

#define F_DEVOFL    0x0002

#define CDB6_UNITOFL(c)			TST(c->u.b[1], 0)
#define SET_CDB6_UNITOFL(c)		{ (c)->u.b[1] |= BIT(0); }

#define F_UNITOFL   0x0004

#define CDB6_PLEN16(c)			(((c)->u.b[3] << 8) | (c)->u.b[4])
#define SET_CDB6_PLEN16(c, arg)		{ (c)->u.b[3] = ((arg) & 0x00ff00) >> 8; \
					  (c)->u.b[4] =  (arg) & 0x0000ff; }


/*
    RECEIVE_DIAGNOSTIC_RESULTS
*/
#define CDB6_ALEN16(c)			(((c)->u.b[3] << 8) | ((c)->u.b.4))
#define SET_CDB6_ALEN16(c, arg)		{ (c)->u.b[3] = ((arg) & 0x00ff00) >> 8; \
					  (c)->u.b[4] =  (arg) & 0x0000ff; }



/*
    READ 10
    WRITE 10
*/
#define CDB10_DPO(c)			TST(c->u.b[1], 4)
#define SET_CDB10_DPO(c)		{ (c)->u.b[1] |= BIT(4); }

#define F_DPO	    0x0001

#define CDB6_FUA(c)			TST(c->u.b[1], 3)
#define SET_CDB10_FUA(c)		{ (c)->u.b[1] |= BIT(3); }

#define F_FUA	    0x0002

#define CDB6_RELADDR(c)			TST(c->u.b[1], 0)
#define SET_CDB10_RELADDR(c)		{ (c)->u.b[1] |= BIT(0); }

#define F_RELADDR   0x0004


/*
    READ 12
    WRITE 12
*/
#define CDB12_DPO(c)			TST(c->u.b[1], 4)
#define SET_CDB12_DPO(c)		{ (c)->u.b[1] |= BIT(4); }

#define CDB12_FUA(c)			TST(c->u.b[1], 3)
#define SET_CDB12_FUA(c)		{ (c)->u.b[1] |= BIT(3); }

#define CDB6_RELADDR(c)			TST(c->u.b[1], 0)
#define SET_CDB12_RELADDR(c)		{ (c)->u.b[1] |= BIT(0); }



/*
    READ CAPACITY
*/
#define CDB6_PMI(c)			TST(c->u.b[8], 0)
#define SET_CDB10_PMI(c)		{ (c)->u.b[8] |= BIT(0); }

#define F_PMI	0x0001


/*
    READ LONG
    WRITE LONG
*/
#define CDB10_CORRCT(c)			TST((c)->u.cdb10.reserved1, 1)
#define SET_CDB10_CORRCT(c)		{ (c)->u.cdb10.reserved1 |= BIT(1); }

#define F_CORRCT    0x0100


/*
    WRITE AND VERIFY
*/
#define CDB10_BYTCHK(c)			TST((c)->u.cdb10.reserved1, 1)
#define SET_CDB10_BYTCHK(c)		{ (c)->u.cdb10.reserved1 |= BIT(1); }

#define F_BYTCHK    0x0008


/*
    WRITE SAME
*/
#define CDB10_PBDATA(c)			TST((c)->u.cdb10.reserved1, 2)
#define SET_CDB10_PBDATA(c)		{ (c)->u.cdb10.reserved1 |= BIT(2); }

#define F_PBDATA    0x0001

#define CDB10_LBDATA(c)			TST((c)->u.cdb10.reserved1, 1)
#define SET_CDB10_LBDATA(c)		{ (c)->u.cdb10.reserved1 |= BIT(1); }

#define F_LBDATA    0x0002



/*
    START STOP UNIT
*/
#define CDB6_SSIMMED(c)			TST((c)->u.b[1], 0)
#define SET_CDB6_SSIMMED(c)		{ (c)->u.b[1] |= BIT(0); }

#define F_IMMED	    0x0001

#define CDB6_START(c)			TST((c)->u.b[4], 0)
#define SET_CDB6_START(c)		{ (c)->u.b[4] |= BIT(0); }

#define F_START	    0x0002
#define F_STOP	    0x0000

#define CDB6_LOEJ(c)			TST((c)->u.b[4], 1)
#define SET_CDB6_LOEJ(c)		{ (c)->u.b[4] |= BIT(1); }

#define F_LOEJ	    0x0004





/*
    A Command Descriptor Block

 */
struct scsi_cdb {
    union {
	struct scsi_cdb6    cdb6;
	struct scsi_cdb10   cdb10;
	struct scsi_cdb12   cdb12;
	struct scsi_cdb16   cdb16;
	u8		    b[16];
    } u;
} __attribute__((packed));





extern const char *scsi_opcode[];



struct scsi_cdb *   scsi_cdb_alloc(u8, u32, u32);
void                scsi_cdb_dump(FILE *, struct scsi_cdb *);
struct scsi_cdb *   scsi_cdb_dup(struct scsi_cdb *);

#endif /* SCSI_CDB_H */


