/*
 * kdfwidget.cpp
 *
 * Copyright (c) 1998 Michael Kropfberger mkropfbe@edu.uni-klu.ac.at
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
 
#include <math.h>

#include <qstring.h>
#include <qmsgbox.h> 
#include <qfile.h>
#include <qtstream.h>
#include <qstring.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qpixmap.h>
#include <qpaintd.h>

#include <kapp.h> 
#include <kmsgbox.h> 
#include <kiconloader.h>
#include "kdfwidget.h"
#include "kdfwidget.moc"


#define BLANK ' '
#define DELIMITER '#'
#define BORDER 5
#define PIX_COLUMN 7
#define FULL_PERCENT 95.0
#define VISIBLE klocale->translate("visible")
#define INVISIBLE klocale->translate("invisible")

#define DEFAULT_FREQ 60
#define DEFAULT_FILEMGR_COMMAND "kfmclient openURL %p"
//#define DEFAULT_FILEMGR_COMMAND "kvt -e mc %p"

static bool GUI;


/***************************************************************************
  * Constructor
**/
KDFWidget::KDFWidget (QWidget * parent, const char *name
                      , bool init)
    : KConfigWidget (parent, name)
{
  //debug("Construct: KDFWidget::KDFWidget");

  popupIfFull=TRUE;
  updateFreq=DEFAULT_FREQ;
  connect(&diskList , SIGNAL(readDFDone() ),
           this, SLOT (updateDFDone()) );

  tabWidths.resize(PIX_COLUMN+1);
  tabHeaders.append(klocale->translate("Icon") );
  tabWidths[0]=24;
  tabHeaders.append(klocale->translate("Device") );
  tabWidths[1]=80;
  tabHeaders.append(klocale->translate("Type") );
  tabWidths[2]=50;
  tabHeaders.append(klocale->translate("Size") );
  tabWidths[3]=50;
  tabHeaders.append(klocale->translate("MountPoint") );
  tabWidths[4]=80;
  tabHeaders.append(klocale->translate("Free") );
  tabWidths[5]=50;
  tabHeaders.append(klocale->translate("Full%") );
  tabWidths[6]=50;
  tabHeaders.append(klocale->translate("UsageBar") );
  tabWidths[7]=100;

  if (init) {
    GUI = FALSE;
  } else
    GUI = TRUE;

  if (GUI)
    {  // inits go here
      this->setMinimumSize(323,200);
      KIconLoader *loader = kapp->getIconLoader();


      settingsBox = new QGroupBox(klocale->translate("Settings"),this);
      CHECK_PTR(settingsBox);
      freqLabel = new QLabel( klocale->translate("update frequency (seconds)")
                              ,settingsBox); CHECK_PTR(freqLabel);
      fileMgrLabel = new QLabel( klocale->translate("FileManager (e.g. kvt -e mc %p)")
                              ,settingsBox); CHECK_PTR(fileMgrLabel);
      freqScroll = new QScrollBar( settingsBox ); CHECK_PTR(freqScroll); 
      freqScroll->setOrientation( QScrollBar::Horizontal );
      freqScroll->setSteps(1,20);
      freqScroll->setRange(0, 180 ); 

      freqLCD = new QLCDNumber( settingsBox ); CHECK_PTR(freqLCD);
      freqLCD->setNumDigits( 3 );
      freqLCD->setSegmentStyle(QLCDNumber::Filled);
      connect(freqScroll, SIGNAL(valueChanged(int)), 
              freqLCD, SLOT(display(int)) );
      connect(freqScroll, SIGNAL(valueChanged(int)), 
              this, SLOT(setUpdateFreq(int)) );
      //      freqScroll->setValue(updateFreq);  // defaultValue
      fileMgrEdit = new QLineEdit( settingsBox ); CHECK_PTR(fileMgrEdit);
      
      showSettings=FALSE;
      settingsBtn=new QPushButton(klocale->translate("Settings"),settingsBox);
      connect(settingsBtn,SIGNAL(clicked()),this,SLOT(settingsBtnClicked()));
      settingsBtn->setToggleButton(TRUE);
      defaultsBtn=new QPushButton(klocale->translate("Defaults"),settingsBox);
      connect(defaultsBtn,SIGNAL(clicked()),this,SLOT(defaultsBtnClicked()));

      QString pcn=parent->className();
      //debug("parent: [%s]",pcn.data());
      if ( pcn == "KDFTopLevel" ) {
         //it is used from KTopLevelWidget "kdf" --> provides a help-btn
         isTopLevel=TRUE;  
      } else 
         isTopLevel=FALSE;

      if (isTopLevel) {
        helpBtn=new QPushButton(klocale->translate("Help"),settingsBox);
        connect(helpBtn,SIGNAL(clicked()),this,SLOT(invokeHTMLHelp()));
     } else
        helpBtn=0; //if

   // normal tabList ****************************************

      tabList = new KTabListBox(this,"tabList",PIX_COLUMN+1,0); 
      CHECK_PTR(tabList);
      tabList->setColumn(0,tabHeaders.at(0)
                         ,tabWidths[0]
                         ,KTabListBox::PixmapColumn);
      for (int i=1;i<PIX_COLUMN;i++)
        tabList->setColumn(i,tabHeaders.at(i)
                         ,tabWidths[i],KTabListBox::TextColumn);
      tabList->setColumn(PIX_COLUMN,tabHeaders.at(PIX_COLUMN)
                         ,tabWidths[PIX_COLUMN]
                         ,KTabListBox::PixmapColumn);
      QPixmap *pix;
      pix = new QPixmap(loader->loadMiniIcon("mini-clock.xpm"));
      tabList->dict().replace("WAIT",pix );

      tabList->setSeparator(BLANK);
      connect(tabList,SIGNAL(headerClicked(int))
             ,this,SLOT(headerClicked(int)) );
      connect(tabList,SIGNAL(popupMenu(int,int))
             ,this,SLOT(popupMenu(int,int)) );
      connect(tabList,SIGNAL(selected(int,int))
             ,this,SLOT(popupMenu(int,int)) );
      connect(tabList,SIGNAL(midClick(int,int))
             ,this,SLOT(popupMenu(int,int)) );
      connect(tabList,SIGNAL(highlighted(int,int))
           ,this,SLOT(popupMenu(int,int)) );
      tabList->dict().setAutoDelete(TRUE);


   // config-tabList ****************************************

      confTabList = new KTabListBox(settingsBox,"confTabList",PIX_COLUMN+1,0); 
      CHECK_PTR(confTabList);
      for (int i=0;i<=PIX_COLUMN;i++)
        confTabList->setColumn(i,tabHeaders.at(i)
                         ,tabWidths[i],KTabListBox::PixmapColumn);
      confTabList->setSeparator(BLANK);
      confTabList->dict().setAutoDelete(TRUE);
      //QPixmap *pix;
      pix = new QPixmap(loader->loadMiniIcon("tick.xpm"));
      confTabList->dict().replace(VISIBLE,pix );
      pix = new QPixmap(loader->loadMiniIcon("delete.xpm"));
      confTabList->dict().replace(INVISIBLE,pix );
    
      QString s;
      for (int i=0;i<=PIX_COLUMN;i++)
         s = s + klocale->translate(VISIBLE) + " ";
      confTabList->appendItem((const char *)s);
      connect(confTabList,SIGNAL(headerClicked(int))
             ,this,SLOT(toggleColumnVisibility(int)) );
      connect(confTabList,SIGNAL(popupMenu(int,int))
             ,this,SLOT(toggleColumnVisibility(int,int)) );
      connect(confTabList,SIGNAL(selected(int,int))
             ,this,SLOT(toggleColumnVisibility(int,int)) );
      connect(confTabList,SIGNAL(midClick(int,int))
             ,this,SLOT(toggleColumnVisibility(int,int)) );
      connect(confTabList,SIGNAL(highlighted(int,int)) 
           ,this,SLOT(toggleColumnVisibility(int,int)) );
 
   }//if GUI

   config = kapp->getConfig();
   loadSettings();
   if (init)
     applySettings();
} // Constructor


/***************************************************************************
  * Destructor
**/
KDFWidget::~KDFWidget() 
{ 
  //debug("DESTRUCT: KDFWidget::~KDFWidget");
  if (GUI) {
  }
}; 

/***************************************************************************
  * is called when the program is exiting
**/
void KDFWidget::closeEvent(QCloseEvent *)
{
  applySettings(); 
  kapp->quit();
};

/***************************************************************************
  * saves KConfig and starts the df-cmd
**/
void KDFWidget::applySettings()
{
  //debug("KDFWidget::applySettings");
 config->setGroup("KDiskFree");
 if (isTopLevel) { // only "kdf" is resizable
   config->writeEntry("Width",width() );
   config->writeEntry("Height",height() );
 }
 config->writeEntry("UpdateFrequency",updateFreq);
 if (GUI) {
    config->writeEntry("FileManagerCommand",fileMgrEdit->text() );
    for (int i=0;i<=PIX_COLUMN;i++)
       config->writeEntry(tabHeaders.at(i),tabList->columnWidth(i) );
 } else { //init
    config->writeEntry("FileManagerCommand",DEFAULT_FILEMGR_COMMAND );
    for (int i=0;i<=PIX_COLUMN;i++)
       config->writeEntry(tabHeaders.at(i),tabWidths[i] );
 }//if
 config->sync();

 if (GUI)
   updateDF();
}


/***************************************************************************
  * reads the KConfig
**/
void KDFWidget::loadSettings()
{
 //debug("KDFWidget::loadSettings");
 config->setGroup("KDiskFree");
 updateFreq=config->readNumEntry("UpdateFrequency",DEFAULT_FREQ);

 if (GUI) {
    if (isTopLevel) { //only "kdf" can be resized

       int appWidth=config->readNumEntry("Width",width());
       int appHeight=config->readNumEntry("Height",height());
       this->resize(appWidth,appHeight);
    }
    fileMgrEdit->setText(config->readEntry("FileManagerCommand"
                         ,DEFAULT_FILEMGR_COMMAND));
    for (int i=0;i<=PIX_COLUMN;i++) {
       tabList->setColumnWidth(i,config->readNumEntry(tabHeaders.at(i)
                                                  ,tabWidths[i]) );
    if (tabList->columnWidth(i) != 0)
       confTabList->changeItemPart(klocale->translate(VISIBLE),0,i); 
    else
       confTabList->changeItemPart(klocale->translate(INVISIBLE),0,i); 
   
    }//for
  freqScroll->setValue(updateFreq);  // initiates a applySettings() !
  updateDF();
 }//if GUI
}

/***************************************************************************
  * resets all to its defaults
**/
void KDFWidget::defaultsBtnClicked()
{
  //debug("KDFWidget::defaultsBtnClicked");
  freqScroll->setValue(DEFAULT_FREQ);
  fileMgrEdit->setText(DEFAULT_FILEMGR_COMMAND);
  for (int i=0;i<=PIX_COLUMN;i++) {
       tabList->setColumnWidth(i,tabWidths[i]);
    if (tabList->columnWidth(i) != 0)
       confTabList->changeItemPart(klocale->translate(VISIBLE),0,i); 
    else
       confTabList->changeItemPart(klocale->translate(INVISIBLE),0,i); 
  }//for
  this->update();
  updateDF();
}

/***************************************************************************
  * pops up the SettingsBox if the settingsBtn is clicked
**/
void KDFWidget::settingsBtnClicked()
{
  if (showSettings)
     showSettings=FALSE;
  else
     showSettings=TRUE;
  applySettings();
  repaint();
}


/***************************************************************************
  * resets the timer for automatic df-refreshes
**/
void KDFWidget::setUpdateFreq(int freq)
{
  //debug("KDFWidget::setUpdateFreq");
  killTimers(); //kills !all! running timers
  updateFreq=freq;
  if (updateFreq > 0)  //0 sets to NO_AUTO_UPDATE   ;)
    startTimer( updateFreq * 1000 );
  applySettings();
}

/***************************************************************************
  * Update (reread) all disk-dependencies
**/
void KDFWidget::timerEvent(QTimerEvent *) 
{ 
  //debug("KDFWidget::timerEvent");
  updateDF();
};

/***************************************************************************
  * checks fstab & df 
**/
void KDFWidget::updateDF()
{
  readingDF=TRUE;
  diskList.readFSTAB();
  diskList.readDF(); 
};

/***************************************************************************
  * gets the signal when the diskList is complete and up to date
**/
void KDFWidget::updateDFDone()
{
  //debug("KDFWidget::updateDFdone()");
  tabList->setAutoUpdate(FALSE);
  tabList->clear();
  DiskEntry *disk;
  QString s,percS,sizeS;
  KIconLoader *loader = kapp->getIconLoader();
  QPixmap *pix;
  for (disk=diskList.first();disk!=0;disk=diskList.next()) {
       if ( disk->kBSize() > 0 ) {
         percS.sprintf("%2.1f%%",disk->percentFull() );
         sizeS=disk->prettyKBSize();
       } else {
         sizeS=klocale->translate("UNKNOWN");    
         percS=klocale->translate("UNKNOWN");    
       }

       s.sprintf("%s %s %s %s %s %s %s %s"
                          ,(const char *)disk->iconName()
                          ,(const char *)disk->deviceName()
                          ,(const char *)disk->fsType()
                          ,(const char *)sizeS
                          ,(const char *)disk->mountPoint()
                          ,(const char *)disk->prettyKBAvail()
                          ,(const char *)percS
                          ,(const char *)disk->deviceName() );
       //   debug("store [%s]", (const char *)s);
       tabList->appendItem((const char *)s);
       pix = new QPixmap(loader->loadMiniIcon(disk->iconName()));
       tabList->dict().replace(disk->iconName(),pix );
  }
  tabList->setAutoUpdate(TRUE);
  this->update();  //enables the automatic PIX_COLUMN-resize but flickers!
  updatePixmaps();
  readingDF=FALSE;
};
 
/**************************************************************************
  * connected with the tabList-Signal
**/
void KDFWidget::headerClicked(int )  //ignore column
{
  //debug("KDFWidget::headerClicked: %d",column);
}

/**************************************************************************
  * connected with the confTabList-Signal
**/
void KDFWidget::toggleColumnVisibility(int column)
{
  //debug("KDFWidget::toggleColumnVisibility: %d %d",column,tabList->columnWidth(column));

  if(tabList->columnWidth(column) == 0) {
    tabList->setColumnWidth(column,tabWidths[column]);
    confTabList->changeItemPart(klocale->translate(VISIBLE),0,column); 
  } else {
    tabList->setColumnWidth(column,0);
    confTabList->changeItemPart(klocale->translate(INVISIBLE),0,column); 
  }
  //confTabList->unmarkAll();
  this->update();
  updateDF();
}


/**************************************************************************
  * pops up and asks for mount/umount right-clicked device
**/
void KDFWidget::popupMenu(int row,int column)
{
  //debug("KDFWidget::popupMenu: %d:%d",row,column);

  // get the clicked disk
      QString dev=tabList->text(row,1);
      QString mnt=tabList->text(row,4);
      DiskEntry *disk = new DiskEntry(dev);
      disk->setMountPoint(mnt);
      int pos=diskList.find(disk);
      delete disk;
      disk=diskList.at(pos);
  
      
  switch (column) { 
  case 0:   //mount/umount icon
  case 1:   //mount/umount deviceName  
    if (!readingDF) {
      unsigned int ipos=0;
      QString e1, e2;
      while(ipos<tabList->count()) {  //find old entry in tabList
         e1=tabList->text(ipos,1);
	 e2=tabList->text(ipos,4);
	 if (e1==(const char *)disk->deviceName()
	  && (e2==(const char *)disk->mountPoint())) {
                                //entry found
             tabList->markItem(ipos);
             tabList->changeItemColor(yellow,ipos);
             tabList->changeItemPart(klocale->translate("WAIT"),ipos,0);
             tabList->changeItemPart(klocale->translate("MOUNTING!"),ipos,3);
             tabList->changeItemPart(klocale->translate("MOUNTING!"),ipos,5);
         }//if
         ipos++;
      }//while        
  
      if (!disk->toggleMount())
         KMsgBox::message(this,klocale->translate("KDiskFree"),
              disk->lastSysError(),KMsgBox::STOP);
      updateDF();
    }
      break;
  case 4:  //open fileManager on MountPoint
      QString cmdS;
      if ( fileMgrEdit->text() != "" ) {
         cmdS=fileMgrEdit->text();
         int pos=cmdS.find("%p");
         if ( pos > 0 ) {
             cmdS=cmdS.replace(pos,2
                           ,(const char *)disk->mountPoint() );
             cmdS.append(" &");
         } else //no %p found; just add path on the back
             cmdS=cmdS+" "+(const char *)disk->mountPoint()+" &";
         //debug("fileMgr-cmd: [%s]",(const char *)cmdS);
  
	system(cmdS);
       }//if 
      break;
 }//switch
};

/**************************************************************************
  * recalculates and repaints the pixBars
**/
void KDFWidget::updatePixmaps()
{
  //debug("KDFWidget::updatePixmaps");
  tabList->setAutoUpdate(FALSE);
  tabList->unmarkAll();
  for ( DiskEntry *disk=diskList.first(); disk != 0; disk=diskList.next() ) {
    if ( !disk->mounted() ) { 
       unsigned int ipos=0;
       QString e1, e2;
       while(ipos<tabList->count()) {  //find old entry in tabList
          e1=tabList->text(ipos,1);
	  e2=tabList->text(ipos,4);
	  if ((e1==(const char *)disk->deviceName())
	    && (e2==(const char *)disk->mountPoint())) {
              //entry found
	    //tabList->markItem(ipos);
             tabList->changeItemColor(lightGray,ipos);
             tabList->dict().remove((const char *)disk->deviceName());
          }//if
        ipos++;
        }//while        
    }//if not mounted
    else {
      if (disk->percentFull() > FULL_PERCENT) {
       unsigned int ipos=0;
       QString e1, e2;
       while(ipos<tabList->count()) {  //find old entry in tabList
          e1=tabList->text(ipos,1);
	  e2=tabList->text(ipos,4);
	  if ((e1==(const char *)disk->deviceName()) 
	    && (e2==(const char *)disk->mountPoint())) {
                                    //entry found
	    //tabList->markItem(ipos);
             tabList->changeItemColor(red,ipos);
          }//if
        ipos++;
        }//while        
      }//if full over FULL_PERCENT
      
      if (tabList->cellWidth(PIX_COLUMN) > 7) {
        QPixmap *pix;
        pix = new QPixmap(tabList->cellWidth(PIX_COLUMN)-7
                      , tabList->cellHeight(1)-1 );
        CHECK_PTR(pix);
        pix->fill(white );
        QPainter p(pix);
        p.setPen(black);
        p.drawRect(0,0,pix->width(),pix->height());
        if (disk->percentFull() > FULL_PERCENT)
            p.setBrush(red);
        else
            p.setBrush(darkBlue);
        p.setPen(white);
        p.drawRect(1,1
             ,(int)(  ((float)pix->width()-2) 
                     * (disk->percentFull()/100) )
             ,pix->height()-2 );
    
        tabList->dict().replace((const char *)disk->deviceName(), pix);
      }//if usageBar is visible
    }//was mounted
  }//for
  tabList->setAutoUpdate(TRUE);
  tabList->repaint();
  confTabList->unmarkAll();
  confTabList->repaint();
}


/**************************************************************************
  * calculates the sizes and shows all devices
**/
void KDFWidget::paintEvent(QPaintEvent *)
{
  //debug("KDFWidget::paintEvent");
   int tabListHeight=this->height()-settingsBox->height();
   if (!showSettings)
      tabListHeight += freqLCD->y()-3;
   int oldPixColumnWidth=tabList->columnWidth(PIX_COLUMN);
   tabList->setGeometry(0, 0, this->width()-1,tabListHeight);
       //this -1 is necessary....         ^^^^  god knows why?
      //setGeometry also changes the width of last column!!
  tabList->setColumnWidth(PIX_COLUMN,oldPixColumnWidth);
  //debug("PIX_COLUMN_width=%d",tabList->columnWidth(PIX_COLUMN));

  //resizing of PIX_COLUMN *************************************
  if (tabList->columnWidth(PIX_COLUMN) != 0 ){
    int totalWidth=0;
    for (int i=0;i<PIX_COLUMN;i++) //all except the last (pix) column
      totalWidth += tabList->columnWidth(i);
    //adjust size to fit completely into tabList-widget...
    if (tabList->width() > totalWidth+4)
       tabList->setColumnWidth(PIX_COLUMN,tabList->width()-totalWidth-4);
  }//if user hasn't turned of the usageBar

  //set widths to default-values
  for (int i=0;i<=PIX_COLUMN;i++)
      confTabList->setColumnWidth(i,tabWidths[i]);
  //same with confTabList
  int totalWidth=0;
  for (int i=0;i<PIX_COLUMN;i++) //all except the last (pix) column
    totalWidth += confTabList->columnWidth(i);
  //adjust size to fit completely into tabList-widget...
  if (confTabList->width() > totalWidth+4)
     confTabList->setColumnWidth(PIX_COLUMN
                        ,confTabList->width()-totalWidth-4);


  updatePixmaps();
}

/**************************************************************************
  * calculates the sizes of the settings and the device-tabList
**/
void KDFWidget::resizeEvent(QResizeEvent *)
{
  // debug("KDFWidget::resizeEvent  %dx%d",width(),height());
   settingsBox->setGeometry(0,this->height()-170,this->width()-1,170); 

   settingsBtn->setGeometry(BORDER,settingsBox->height()-2*BORDER-32,90,32);
   if (isTopLevel)  //only if executed as "kdf"
     helpBtn->setGeometry(settingsBox->width()-BORDER-settingsBtn->width()
             ,settingsBtn->y(),settingsBtn->width()
             ,settingsBtn->height());

 
   freqLCD->setGeometry(BORDER+settingsBtn->x()+settingsBtn->width()
                  ,settingsBox->height()-2*BORDER-32,52,32);
 {
   int x=freqLCD->x()+freqLCD->width()+BORDER;
   int w=160;
   w=settingsBox->width() - x - BORDER;
   if (isTopLevel)
     w -= helpBtn->width() + BORDER;
   freqScroll->setGeometry(x,settingsBox->height()-2*BORDER-16,w,16);
 }
	    
 // confTabList->setGeometry(0,settingsBox->y(),settingsBox->width()-1,60);
 confTabList->setGeometry(BORDER,3*BORDER
                    ,settingsBox->width()-2*BORDER,60);
 
 defaultsBtn->setGeometry(BORDER
                        ,settingsBtn->y()-32-BORDER,90,32);

  freqLabel->setGeometry(freqLCD->x()+freqLCD->width()+BORDER
                  ,freqLCD->y()
		  ,freqScroll->width()+3
                  ,freqLCD->height()-freqScroll->height() );
  fileMgrLabel->setGeometry(defaultsBtn->x()+defaultsBtn->width()+BORDER
                  ,defaultsBtn->y()-7
                  ,freqScroll->width()+freqLCD->width()+BORDER
                  ,16 );
  fileMgrEdit->setGeometry(fileMgrLabel->x()
                 ,(fileMgrLabel->y()+fileMgrLabel->height())
                 ,fileMgrLabel->width()
                 ,(defaultsBtn->y()+defaultsBtn->height())
                   -(fileMgrLabel->y()+fileMgrLabel->height()) );
 repaint();
}

