/*
 * disks.cpp
 *
 * Copyright (c) 1998 Michael Kropfberger <mkropfbe@edu.uni-klu.ac.at>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <kapp.h>
#include "disks.h"
#include "disks.moc"

/****************************************************/
/********************* DiskEntry ********************/
/****************************************************/

/**
  * Constructor
**/
void DiskEntry::init()
{
  device="";
  type="";
  mountedOn="";
  options="";
  size=0;
  used=0;
  avail=0;
  isMounted=FALSE;

  /*
 // GuessDiskIcon  ****************************************
 connect(this,SIGNAL(deviceNameChanged()), this, SLOT(guessDiskIcon()));
 connect(this,SIGNAL(mountPointChanged()), this, SLOT(guessDiskIcon()));
 connect(this,SIGNAL(mountOptionsChanged()), this, SLOT(guessDiskIcon()));
 connect(this,SIGNAL(fsTypeChanged()), this, SLOT(guessDiskIcon()));
 connect(this,SIGNAL(mountedChanged()), this, SLOT(guessDiskIcon()));
 connect(this,SIGNAL(kBSizeChanged()), this, SLOT(guessDiskIcon()));
 connect(this,SIGNAL(kBUsedChanged()), this, SLOT(guessDiskIcon()));
 connect(this,SIGNAL(kBAvailChanged()), this, SLOT(guessDiskIcon()));
  */

 // BackgroundProcesses ****************************************

 sysProc = new KShellProcess(); CHECK_PTR(sysProc);
 connect( sysProc, SIGNAL(receivedStdout(KProcess *, char *, int) ),
        this, SLOT (receivedSysStdErrOut(KProcess *, char *, int)) );
 connect( sysProc, SIGNAL(receivedStderr(KProcess *, char *, int) ),
        this, SLOT (receivedSysStdErrOut(KProcess *, char *, int)) );
 readingSysStdErrOut=FALSE;


}

DiskEntry::DiskEntry(QObject *parent, const char *name)
 : QObject (parent, name)
{
  init();
}

DiskEntry::DiskEntry(QString deviceName, QObject *parent, const char *name)
 : QObject (parent, name)
{
  init();
  setDeviceName(deviceName);
}
DiskEntry::~DiskEntry()
{
  //debug("DESTRUCT: DiskList"); 
  disconnect(this);
  delete sysProc;
};

int DiskEntry::toggleMount()
{ 
  if (!mounted())
      return mount();
  else
      return umount();
};

int DiskEntry::mount()
{ 
  QString cmdS;
  cmdS.sprintf("%s %s",MOUNT_COMMAND,(const char *)this->deviceName());
  /*
    cmdS.sprintf("%s %s -t%s %s -o%s",MOUNT_COMMAND
                            ,(const char *)disk->deviceName()
                            ,(const char *)disk->fsType()
                            ,(const char *)disk->mountPoint()
                            ,(const char *)disk->mountOptions()  );
  */

  //  debug("mount-cmd: [%s]",(const char *)cmdS);
  return sysCall(cmdS);
};

int DiskEntry::umount()
{
  QString cmdS;
  cmdS.sprintf("%s %s",UMOUNT_COMMAND,(const char *)this->deviceName());
  //debug("umount-cmd: [%s]",(const char *)cmdS);
  return sysCall(cmdS);
};


/***************************************************************************
  * starts a command on the underlying system via /bin/sh
**/
int DiskEntry::sysCall(QString command)
{
  //debug("DiskEntry::sysCall");
  if (readingSysStdErrOut || sysProc->isRunning() )  return -1;

  sysStringErrOut=""; // yet no data received
  sysProc->clearArguments();
  (*sysProc) << (const char *)command;
    if (!sysProc->start( KProcess::Block, KProcess::AllOutput ))
     fatal(klocale->translate("could not execute [%s]"),(const char *)command);
  if (sysProc->exitStatus()!=0) emit sysCallError(this, sysProc->exitStatus());
  return !sysProc->exitStatus();  
};


/***************************************************************************
  * is called, when the Sys-command writes on StdOut or StdErr
**/
void DiskEntry::receivedSysStdErrOut(KProcess *, char *data, int len)
{
  //debug("DiskEntry::receivedSysStdErrOut");
  QString tmp = QString(data,len+1);  // adds a zero-byte 
  sysStringErrOut.append(tmp);
};

void DiskEntry::guessDiskIcon()
{
    // try to be intelligent
    if (-1!=mountPoint().find("cdrom",0,FALSE)) icoName="cdrom";
    else if (-1!=deviceName().find("cdrom",0,FALSE)) icoName="cdrom";
    else if (-1!=deviceName().find("fd",0,FALSE)) {
            if (-1!=deviceName().find("360",0,FALSE)) icoName="5floppy";
            if (-1!=deviceName().find("1200",0,FALSE)) icoName="5floppy";
            else icoName="3floppy";
	 }
    else if (-1!=mountPoint().find("floppy",0,FALSE)) icoName="3floppy";
    else if (-1!=mountPoint().find("zip",0,FALSE)) icoName="zip";
    else if (-1!=fsType().find("nfs",0,FALSE)) icoName="nfs";
    else icoName="harddrive";
    mounted() ? icoName+="_mount.xpm" : icoName+="_unmount.xpm";
    if ( -1==mountOptions().find("user",0,FALSE) )
      icoName.prepend("root_"); // special root icon, a normal user cant mount

    //debug("file is [%s]",icoName.data());
    if ( (!QFile::exists(QString(kapp->kde_icondir()
                              +QString("/mini/")+icoName).data())) 
        && (!QFile::exists(QString(kapp->kde_icondir()
                              +QString("/")+icoName).data())) )
    {
      warning("file [%s] doesn't exist (not even mini)!",icoName.data());
      icoName="unknown.xpm";
    }
    
    //debug("device %s is %s",deviceName().data(),icoName.data());
    
    emit iconNameChanged();
};

QString DiskEntry::prettyPrint(int kBValue) const
{
  QString weight;
  float val=(float)kBValue; // size in KiloByte

  //always go up to MegaByte
  val=val/1024;
  weight="MB";

  if (val>999.0) {  //GigaByte
    val=val/1024;
    weight="GB";
  }//if

  QString ret;
  if (val>100.0)  // e.g. 504MB
    ret.sprintf("%3.0f%s",val,(const char *)weight);
  else
    if (val>10.0) // e.g. 54.7MB
      ret.sprintf("%3.1f%s",val,(const char *)weight);
    else // e.g. 1.44KB
      ret.sprintf("%3.2f%s",val,(const char *)weight);
  return ret;
}

float DiskEntry::percentFull() const
{
   if (size != 0) {
      return 100 - ( ((float)avail / (float)size) * 100 );
   } else {
      return 100;
   }
}

void DiskEntry::setDeviceName(QString deviceName)
{
 device=deviceName; 
 emit deviceNameChanged();
};

void DiskEntry::setMountPoint(QString mountPoint)
{
  mountedOn=mountPoint;
 emit mountPointChanged();
};

void DiskEntry::setMountOptions(QString mountOptions)
{
 options=mountOptions;
 emit mountOptionsChanged();
};

void DiskEntry::setFsType(QString fsType)
{
  type=fsType;
  emit fsTypeChanged();
};

void DiskEntry::setMounted(bool nowMounted) 
{ 
  isMounted=nowMounted;
  emit mountedChanged();
};
  
void DiskEntry::setKBSize(int kb_size)
{
  //debug("DiskEntry::setKBSize(%d)",kb_size);
  size=kb_size;
  emit kBSizeChanged();
};

void DiskEntry::setKBUsed(int kb_used)
{
  //debug("DiskEntry::setKBUsed(%d)",kb_used);
  used=kb_used;
  if ( size < (used+avail) ) {  //adjust kBAvail
     warning("device %s: kBAvail(%d)+*kBUsed(%d) exceeds kBSize(%d)"
           ,(const char *)device,avail,used,size);
     setKBAvail(size-used); 
  }
  emit kBUsedChanged();
};

void DiskEntry::setKBAvail(int kb_avail)
{
  //debug("DiskEntry::setKBAvail(%d)",kb_avail);
  avail=kb_avail;
  if ( size < (used+avail) ) {  //adjust kBUsed
     warning("device %s: *kBAvail(%d)+kBUsed(%d) exceeds kBSize(%d)"
           ,(const char *)device,avail,used,size);
     setKBUsed(size-avail); 
  }
  emit kBAvailChanged();
};


