/*
  isoinfo.c  --  list information of an iso9660 image
     -- inspired by mkisofs by Eric Youngdale --
  Copyright (C) 1999-2001 Steffen Solyga <solyga@absinth.net>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 *	$Id: isoinfo.c,v 1.10 2004-12-23 16:55:44 solyga Exp $
 */

#include	"isoinfo.h"

int
display_help( char* fn ) {
  fprintf( HELP_CHANNEL, "%s v%s (%s): ", fn, VERSION_NUMBER, DATE_OF_LAST_MOD );
  fprintf( HELP_CHANNEL, "List information on an iso9660 image.\n" );
  fprintf( HELP_CHANNEL, "Flowers & bug reports to %s.\n", MY_EMAIL_ADDRESS );
  fprintf( HELP_CHANNEL, "Usage: %s [options] [file]\n", fn );
  fprintf( HELP_CHANNEL, "switches:\n" );
  fprintf( HELP_CHANNEL, "  -A\t list application identifier\n" );
  fprintf( HELP_CHANNEL, "  -b\t list image size in bytes\n" );
  fprintf( HELP_CHANNEL, "  -B\t list image size in blocks (of 2048 bytes)\n" );
  fprintf( HELP_CHANNEL, "  -h\t help, write this info to %s and exit sucessfully\n", HELP_CHANNEL==stdout ? "stdout" : "stderr" );
  fprintf( HELP_CHANNEL, "  -i\t print extended info\n" );
  fprintf( HELP_CHANNEL, "  -m\t list image size in msf\n" );
  fprintf( HELP_CHANNEL, "  -M\t list image size in Mbytes\n" );
  fprintf( HELP_CHANNEL, "  -p\t list preparer identifier\n" );
  fprintf( HELP_CHANNEL, "  -P\t list publisher identifier\n" );
  fprintf( HELP_CHANNEL, "  -s\t same as -b\n" );
  fprintf( HELP_CHANNEL, "  -S\t same as -M\n" );
  fprintf( HELP_CHANNEL, "  -v\t be more talkative\n" );
  fprintf( HELP_CHANNEL, "  -V\t list volume identifier\n" );
  fprintf( HELP_CHANNEL, "controllers:\n" );
  fprintf( HELP_CHANNEL, "  -I\t ID\t list identifier ID, where ID is one of\n" );
  fprintf( HELP_CHANNEL, "    \t   \t SYSI - system\n" );
  fprintf( HELP_CHANNEL, "    \t   \t VOLI - volume      (same as -V)\n" );
  fprintf( HELP_CHANNEL, "    \t   \t VOLS - volume set\n" );
  fprintf( HELP_CHANNEL, "    \t   \t PUBL - publisher   (same as -P)\n" );
  fprintf( HELP_CHANNEL, "    \t   \t PREP - preparer    (same as -p)\n" );
  fprintf( HELP_CHANNEL, "    \t   \t APPI - application (same as -A)\n" );
  fprintf( HELP_CHANNEL, "    \t   \t COPY - copyright\n" );
  fprintf( HELP_CHANNEL, "    \t   \t ABST - abstract\n" );
  fprintf( HELP_CHANNEL, "    \t   \t BIBL - bibliography\n" );
  return( 0 );
}


ssize_t
my_read( int fd, void* buf, size_t nbtr ) {
/* returns number of bytes read or -1 on error */
/* like read(2) but nbr<nbtr only if eof reached */
  ssize_t nbr;
  ssize_t tnbr= 0;
  size_t rem= nbtr;
  unsigned char *p= (unsigned char*)buf;
  do {
    if( (nbr=read(fd,p+tnbr,rem)) == -1 ) return( -1 );
    tnbr+= nbr;
    rem= nbtr - tnbr;
  }
  while( nbr>0 && rem>0 );
  return( tnbr );
}


long
iso_num_4( char* p ) {
  return(   (p[0] & 0xffL)
          | (p[1] & 0xffL) <<  8
          | (p[2] & 0xffL) << 16
          | (p[3] & 0xffL) << 24 );
}


int
iso_num_2( char* p ) {
  return(   (int)((p[0] & 0xffL)
                | (p[1] & 0xffL) <<  8) );
}


int
iso_num_1( char* p ) {
  return( (int)(p[0] & 0xffL) );
}


char*
iso_str( char* p, int n ) {
  static char string[256];
  int i;
  for( i=0; i<n; i++ ) string[i]= p[i];
  string[i--]= '\0';
  while( string[i] == ' ' ) string[i--]= '\0';
  return( string );
}


char*
iso_date( char* p ) { /* 14 chars          "1999015220021" */
  static char date[] = "1999-01-15 22:00:19";
  int i= 0;
  int j= 0;
  int k;
  while( i < 4 ) date[j++] = p[i++];
  for( k=6; k<16; k+=2 ) {
    j++;
    while( i < k ) date[j++] = p[i++];
  }
  return( date );
}


int
fra2min( int f ) {
/* convert frames to minutes of msf */
  f= f < 0 ? -f : f;
  return( f/CD_FRAMES/CD_SECS );
}


int
fra2sec( int f ) {
/* convert frames to seconds of msf */
  f= f < 0 ? -f : f;
  return( (f - fra2min(f)*CD_SECS*CD_FRAMES)/CD_FRAMES );
}


int
fra2fra( int f ) {
/* convert frames to frames of msf */
  f= f < 0 ? -f : f;
  return( f - (fra2min(f)*CD_SECS+fra2sec(f))*CD_FRAMES );
}


char *
basename( const char* filename ) {
  /* from glibc-2.1 */
  char *p = strrchr (filename, '/');
  return p ? p + 1 : (char *) filename;
}


int
main( int argc, char** argv ) {
/*
SOLYGA --------------------
SOLYGA main(argc, argv) isoinfo
SOLYGA list information on an ISO9660 image
SOLYGA Steffen Solyga <solyga@absinth.net>
SOLYGA started      : Thu Jan 14 23:30:23 CET 1999 @beast
*/
  char* fpn= *argv;
  char* pn= basename( fpn );		/* prg_name is Makefile constant */
  int retval= RETVAL_OK;		/* return value */
  int c;				/* option char */
  int in_fd;				/* input fd */
  char stdin_fn[]= "stdin";
  char default_fn[]= DEFAULT_INPUT_FILE;
  char* in_fn= default_fn;		/* input file name */
  char buf[ISO_BLOCK_SIZE];		/* block buffer */
  long blk= -1;				/* block in buf */
  long bc_blk= -1;			/* boot catalog block (set from brvd) */
  long be_blk= -1;			/* boot entry block (set from bc) */
  long blks= 0;				/* number of blocks (set from ipd ) */
  long blk_list[] = { ISO_PD_BLOCK, BOOT_VD_BLOCK, INFO_BLOCK, -1, -1, -2 };
  int bli;				/* block list index */
  int verbose= 0;			/* verbosity level */
  int of_bloc= 0;			/* opt-flag: size in blocks */
  int of_size= 0;			/* opt-flag: size in bytes */
  int of_Size= 0;			/* opt-flag: size in Mbytes */
  int of_SIZE= 0;			/* opt-flag: size in msf */
  int of_sysi= 0;			/* opt-flag: system */
  int of_voli= 0;			/* opt-flag: volume */
  int of_vols= 0;			/* opt-flag: volume set */
  int of_publ= 0;			/* opt-flag: publisher */
  int of_prep= 0;			/* opt-flag: preparer */
  int of_appi= 0;			/* opt-flag: application */
  int of_copy= 0;			/* opt-flag: copyright */
  int of_abst= 0;			/* opt-flag: abstract */
  int of_bibl= 0;			/* opt-flag: bibliography */
  int of_info= 0;			/* opt-flag: ext. info */
  int file= 1;				/* flag: .mkisofsrc format */
  ssize_t nbr;
  struct iso_primary_descriptor ipd;
  struct eltorito_boot_descriptor brvd;
  struct eltorito_validation_entry ve;
  struct eltorito_defaultboot_entry be;
  char info[ISO_BLOCK_SIZE];		/* ext. info (mkisofs cmd-line) */

/* process options */

  while( (c=getopt(argc,argv,"AbBhiI:mMpPsSvV")) != EOF )
    switch( c ) {
      case 'A': /* list application identifier */
        of_appi= 1; file= 0;
        break;
      case 'b': /* list image size in bytes */
        of_size= 1; file= 0;
        break;
      case 'B': /* list image size in blocks */
        of_bloc= 1; file= 0;
        break;
      case 'h': /* display help to HELP_CHANNEL and exit sucessfully */
        display_help( pn );
        retval= RETVAL_OK; goto DIE_NOW;
      case 'i': /* print extended info */
        of_info= 1; file= 0;
        break;
      case 'I': /* display identifier */
        file= 0;
        switch( tolower( *optarg ) ) {
          case 'a':
            if( strlen(optarg) < 2 ) {
              fprintf( ERROR_CHANNEL, "%s: Identifier type `%s' too short.\n",
                       pn, optarg );
              retval= RETVAL_ERROR; goto DIE_NOW;
            }
            switch( tolower(optarg[1]) ) {
              case 'b': of_abst= 1;
                break;
              case 'p': of_appi= 1;
                break;
              default :
                fprintf( ERROR_CHANNEL, "%s: Unknown identifier type `%s'.\n",
                         pn, optarg );
                retval= RETVAL_ERROR; goto DIE_NOW;
                break;
            }
            break;
          case 'b': of_bibl= 1;
            break;
          case 'c': of_copy= 1;
            break;
          case 'p':
            if( strlen(optarg) < 2 ) {
              fprintf( ERROR_CHANNEL, "%s: Identifier type `%s' too short.\n",
                       pn, optarg );
              retval= RETVAL_ERROR; goto DIE_NOW;
            }
            switch( tolower(optarg[1]) ) {
              case 'r': of_prep= 1; 
                break;
              case 'u': of_publ= 1; 
                break;
              default : 
                fprintf( ERROR_CHANNEL, "%s: Unknown identifier type `%s'.\n",
                         pn, optarg );
                retval= RETVAL_ERROR; goto DIE_NOW;
                break;
            }
            break;
          case 's': of_sysi= 1;
            break;
          case 'v':
            if( strlen(optarg) < 4 ) {
              fprintf( ERROR_CHANNEL, "%s: Identifier type `%s' too short.\n",
                       pn, optarg );
              retval= RETVAL_ERROR; goto DIE_NOW;
            }
            switch( tolower(optarg[3]) ) {
              case 'i': of_voli= 1; 
                break;
              case 's': of_vols= 1; 
                break;
              default : 
                fprintf( ERROR_CHANNEL, "%s: Unknown identifier type `%s'.\n",
                         pn, optarg );
                retval= RETVAL_ERROR; goto DIE_NOW;
                break;
            }
            break;
          default:
            fprintf( ERROR_CHANNEL, "%s: Unknown identifier type `%s'.\n",
                     pn, optarg );
            retval= RETVAL_ERROR; goto DIE_NOW;
            break;
        }
        break;
      case 'm': /* list image size in msf */
        of_SIZE= 1; file= 0;
        break;
      case 'M': /* list image size in MBytes */
        of_Size= 1; file= 0;
        break;
      case 'p': /* list preparer identifier  */
        of_prep= 1; file= 0;
        break;
      case 'P': /* list publisher identifier  */
        of_publ= 1; file= 0;
        break;
      case 's': /* list image size in Bytes */
        of_size= 1; file= 0;
        break;
      case 'S': /* list image size in MBytes */
        of_Size= 1; file= 0;
        break;
      case 'v': /* raise verbosity level */
        verbose++;
        break;
      case 'V': /* list volume identifier */
        of_voli= 1; file= 0;
        break;
      case '?': /* refer to help and exit unsuccessfully */
        fprintf( ERROR_CHANNEL, "%s: Try `%s -h' for more information.\n",
                 pn, fpn );
        retval= RETVAL_ERROR; goto DIE_NOW;
      default : /* program error */
        fprintf( ERROR_CHANNEL, "%s: Option bug detected. %s %s.\n",
                 pn, BUG_MSG, MY_EMAIL_ADDRESS );
        retval= RETVAL_BUG; goto DIE_NOW;
    }

/* open input file */

  in_fd= -1;
  if( argc > optind ) {
    if( *argv[optind] == '-' ) {
      in_fn= stdin_fn;
      in_fd= STDIN_FILENO;
    }
    else in_fn= argv[optind];
  }
  if( in_fd != STDIN_FILENO ) {
    if( (in_fd=open(in_fn,O_RDONLY)) == -1 ) {
      fprintf( ERROR_CHANNEL, "%s: Cannot open `%s'. %s.\n",
               pn, in_fn, strerror(errno) );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
  }

/* read blocks given by block list, fill structs */

  for( bli=0; blk_list[bli]>0; bli++ ) {
    long blk_to_read= blk_list[bli];
    int i;
    /* read blk_to_read */
    if( in_fd != STDIN_FILENO ) { /* we can lseek() */
      if( lseek(in_fd,blk_to_read*ISO_BLOCK_SIZE,SEEK_SET) == -1 ) {
        fprintf( ERROR_CHANNEL, "%s: Cannot lseek `%s'. %s.\n",
                 pn, in_fn, strerror(errno) );
        retval= RETVAL_ERROR; goto DIE_NOW;
      }
      if( (nbr=my_read(in_fd,buf,ISO_BLOCK_SIZE)) == -1 ) {
        fprintf( ERROR_CHANNEL, "%s: Cannot read block %ld of `%s'. %s.\n",
                 pn, blk, in_fn, strerror(errno) );
        retval= RETVAL_ERROR; goto DIE_NOW;
      }
      blk= blk_to_read;
    }
    else { /* stdin, read() only instead of lseek() */
      do {
        if( (nbr=my_read(in_fd,buf,ISO_BLOCK_SIZE)) != ISO_BLOCK_SIZE ) {
          fprintf( ERROR_CHANNEL, "%s: Unexpected EOF on stdin.\n", pn );
          retval= RETVAL_ERROR; goto DIE_NOW;
        }
        blk++;
      } while ( blk < blk_to_read );
    }
    if( nbr < ISO_BLOCK_SIZE ) {
      fprintf( ERROR_CHANNEL, "%s: File is too small.\n", pn );
      retval= RETVAL_ERROR; goto DIE_NOW;
    }
    /* fill struct */
    if( blk == ISO_PD_BLOCK ) {
      memcpy( &ipd, buf, sizeof(ipd) );
      blks= iso_num_4(ipd.volume_space_size);
    }
    if( blk == BOOT_VD_BLOCK ) {
      memcpy( &brvd, buf, sizeof(brvd) );
      bc_blk= iso_num_4(brvd.bootcat_ptr);
      /* extend block list with bc_blk (boot catalog block number) */
      for( i=0; blk_list[i]!=-2; i++ ) {
        if( blk_list[i] == bc_blk ) break;	/* already in list */
        if( blk_list[i] == -1 ) blk_list[i]= bc_blk;
      }
    }
    if( blk == INFO_BLOCK ) {
      int i;
      memcpy( info, buf, ISO_BLOCK_SIZE );
      info[ISO_BLOCK_SIZE-1]= '\0';
      for( i=0; i<ISO_BLOCK_SIZE; i++ ) {
        if( info[i] == '\n' ) info[i]= ' ';
        if( info[i]<' ' || (info[i]>'~' && info[i]<'|') ) info[i]= '\0';
      }
    }
    if( blk == bc_blk ) {
      memcpy( &ve, buf, sizeof(ve) );
      memcpy( &be, buf+sizeof(ve), sizeof(be) );
      be_blk= iso_num_4(be.bootoff);
    }
  }

/* close input */

  if( close(in_fd) == -1 ) {
    fprintf( ERROR_CHANNEL, "%s: Cannot close `%s'. %s.\n",
             pn, in_fn, strerror(errno) );
    retval= RETVAL_ERROR; goto DIE_NOW;
  }

/* list info, single infos first */

  if( of_sysi ) {
    if( verbose ) fprintf( STD_CHANNEL, "system id            : " );
    fprintf( STD_CHANNEL, "%s\n", ISO_STR(ipd.system_id) );
  }
  if( of_voli ) {
    if( verbose ) fprintf( STD_CHANNEL, "volume id            : " );
    fprintf( STD_CHANNEL, "%s\n", ISO_STR(ipd.volume_id) );
  }
  if( of_vols ) {
    if( verbose ) fprintf( STD_CHANNEL, "volume set id        : " );
    fprintf( STD_CHANNEL, "%s\n", ISO_STR(ipd.volume_set_id) );
  }
  if( of_publ ) {
    if( verbose ) fprintf( STD_CHANNEL, "publisher id         : " );
    fprintf( STD_CHANNEL, "%s\n", ISO_STR(ipd.publisher_id) );
  }
  if( of_prep ) {
    if( verbose ) fprintf( STD_CHANNEL, "preparer id          : " );
    fprintf( STD_CHANNEL, "%s\n", ISO_STR(ipd.preparer_id) );
  }
  if( of_appi ) {
    if( verbose ) fprintf( STD_CHANNEL, "application id       : " );
    fprintf( STD_CHANNEL, "%s\n", ISO_STR(ipd.application_id) );
  }
  if( of_copy ) {
    if( verbose ) fprintf( STD_CHANNEL, "copyright file id    : " );
    fprintf( STD_CHANNEL, "%s\n", ISO_STR(ipd.copyright_file_id) );
  }
  if( of_abst ) {
    if( verbose ) fprintf( STD_CHANNEL, "abstract file id     : " );
    fprintf( STD_CHANNEL, "%s\n", ISO_STR(ipd.abstract_file_id) );
  }
  if( of_bibl ) {
    if( verbose ) fprintf( STD_CHANNEL, "bibliographic file id: " );
    fprintf( STD_CHANNEL, "%s\n", ISO_STR(ipd.bibliographic_file_id) );
  }
  if( of_bloc ) {
    if( verbose ) fprintf( STD_CHANNEL, "image size\t: " );
    fprintf( STD_CHANNEL, "%lu",
             iso_num_4(ipd.volume_space_size) );
    if( verbose ) fprintf( STD_CHANNEL, " blocks (of 2048 bytes)" );
    fprintf( STD_CHANNEL, "\n" );
  }
  if( of_size ) {
    if( verbose ) fprintf( STD_CHANNEL, "image size\t: " );
    fprintf( STD_CHANNEL, "%lu",
             iso_num_4(ipd.volume_space_size)*ISO_BLOCK_SIZE );
    if( verbose ) fprintf( STD_CHANNEL, " Bytes" );
    fprintf( STD_CHANNEL, "\n" );
  }
  if( of_Size ) {
    if( verbose ) fprintf( STD_CHANNEL, "image size\t: " );
    fprintf( STD_CHANNEL, "%lu",
             iso_num_4(ipd.volume_space_size)*ISO_BLOCK_SIZE/1024/1024 );
    if( verbose ) fprintf( STD_CHANNEL, " MBytes" );
    fprintf( STD_CHANNEL, "\n" );
  }
  if( of_SIZE ) {
    if( verbose ) fprintf( STD_CHANNEL, "image size\t: " );
    fprintf( STD_CHANNEL, "%02d:%02d:%02d",
             fra2min(iso_num_4(ipd.volume_space_size)),
             fra2sec(iso_num_4(ipd.volume_space_size)),
             fra2fra(iso_num_4(ipd.volume_space_size)) );
    if( verbose ) fprintf( STD_CHANNEL, " msf" );
    fprintf( STD_CHANNEL, "\n" );
  }
  if( of_info ) {
    if( verbose ) fprintf( STD_CHANNEL, "extended info: " );
    fprintf( STD_CHANNEL, "%s\n", info );
  }
  if( file ) {
    /* list all info in .mkisofsrc format */
    if( verbose ) fprintf( STD_CHANNEL, "# system identifier (32 bytes)\n" );
    fprintf( STD_CHANNEL, "SYSI=%s\n", ISO_STR(ipd.system_id) );

    if( verbose ) fprintf( STD_CHANNEL, "# volume identifier (32 bytes), overridden by -V\n" );
    fprintf( STD_CHANNEL, "VOLI=%s\n", ISO_STR(ipd.volume_id) );

    if( verbose ) fprintf( STD_CHANNEL, "# volume set identifier (128 bytes)\n" );
    fprintf( STD_CHANNEL, "VOLS=%s\n", ISO_STR(ipd.volume_set_id) );

    if( verbose ) fprintf( STD_CHANNEL, "# publisher identifier (128 bytes), overridden by -P\n" );
    fprintf( STD_CHANNEL, "PUBL=%s\n", ISO_STR(ipd.publisher_id) );

    if( verbose ) fprintf( STD_CHANNEL, "# preparer identifier (128 bytes), overridden by -p\n" );
    fprintf( STD_CHANNEL, "PREP=%s\n", ISO_STR(ipd.preparer_id) );

    if( verbose ) fprintf( STD_CHANNEL, "# application identifier (128 bytes), overridden by -A\n" );
    fprintf( STD_CHANNEL, "APPI=%s\n", ISO_STR(ipd.application_id) );

    if( verbose ) fprintf( STD_CHANNEL, "# copyright file identifier (37 bytes)\n" );
    fprintf( STD_CHANNEL, "COPY=%s\n", ISO_STR(ipd.copyright_file_id) );

    if( verbose ) fprintf( STD_CHANNEL, "# abstract file identifier (37 bytes)\n" );
    fprintf( STD_CHANNEL, "ABST=%s\n", ISO_STR(ipd.abstract_file_id) );

    if( verbose ) fprintf( STD_CHANNEL, "# bibliographic file identifier (37 bytes)\n" );
    fprintf( STD_CHANNEL, "BIBL=%s\n", ISO_STR(ipd.bibliographic_file_id) );

    fprintf( STD_CHANNEL, "# creation date     : %s\n",
             iso_date(ipd.creation_date) );
    fprintf( STD_CHANNEL, "# modification date : %s\n",
             iso_date(ipd.modification_date) );
    fprintf( STD_CHANNEL, "# expiration date   : %s\n",
             iso_date(ipd.expiration_date) );
    fprintf( STD_CHANNEL, "# effective date    : %s\n",
             iso_date(ipd.effective_date) );
    fprintf( STD_CHANNEL, "# volume space size : %ld blocks (of 2048 bytes), ",
             iso_num_4(ipd.volume_space_size) );
    fprintf( STD_CHANNEL, "%02d:%02d:%02d msf\n",
             fra2min(iso_num_4(ipd.volume_space_size)),
             fra2sec(iso_num_4(ipd.volume_space_size)),
             fra2fra(iso_num_4(ipd.volume_space_size)) );
    fprintf( STD_CHANNEL, "# volume space size : %lld Bytes\n",
             iso_num_4(ipd.volume_space_size)*(long long)ISO_BLOCK_SIZE );
    fprintf( STD_CHANNEL, "# volume space size : %ld MBytes\n",
             iso_num_4(ipd.volume_space_size)*ISO_BLOCK_SIZE/1024/1024 );
    if( iso_num_1(ve.key1) == 0x55  && iso_num_1(ve.key2) == 0xaa ) {
      fprintf( STD_CHANNEL, "# boot catalog      : block %ld\n", bc_blk );
      fprintf( STD_CHANNEL, "# boot image        : block %ld\n", be_blk );
      fprintf( STD_CHANNEL, "# boot image size   : %d sectors (of 512 Bytes)\n",
               iso_num_1(be.nsect) );
      fprintf( STD_CHANNEL, "# boot arch         : " );
      switch( iso_num_1(be.arch) ) {
        case EL_TORITO_ARCH_x86:
          fprintf( STD_CHANNEL, "x86\n" );
          break;
        case EL_TORITO_ARCH_PPC:
          fprintf( STD_CHANNEL, "PowerPC\n" );
          break;
        case EL_TORITO_ARCH_MAC:
          fprintf( STD_CHANNEL, "Mac\n" );
          break;
        default:
          fprintf( STD_CHANNEL, "Unknown\n" );
      }
    }
    else {
      fprintf( STD_CHANNEL, "# image not bootable\n" );
    }
    fprintf( STD_CHANNEL, "# extended info     : %s\n", info );
  }

DIE_NOW:
  close( in_fd );
  return( retval );
}
