#include "fdecc.h"

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <mntent.h>
#include <string.h>
#include <sys/ioctl.h>
#include <linux/fd.h>

static void check_not_mounted (char *dev);

int open_dev (char *dev, int wflag)
{
  int fd;

  if (wflag)
    check_not_mounted (dev);

  if (! wflag) {
    fd = open (dev, O_RDONLY);
    if (fd < 0)
      fatal ("can't read %s", dev); }
  else {
    fd = open (dev, O_RDWR);
    if (fd < 0)
      fatal ("can't write %s", dev); }
    
  return fd;
}

int read_sector (int fd, int n, void *buf)
{
  int rv;

  rv = lseek (fd, n * 512, 0);
  if (rv < 0) {
    warning ("lseek to sector %d failed", n);
    return 1; }

  rv = read (fd, buf, 512);
  if (rv == 512)
    return 0;
  memset (buf, 0, 512);
  return 1;
}

int write_sector (int fd, int n, void *buf)
{
  int rv;

  rv = lseek (fd, n * 512, 0);
  if (rv < 0) {
    warning ("lseek to sector %d failed", n);
    return 1; }

  rv = write (fd, buf, 512);
  if (rv != 512) {
    warning ("write sector %d failed", n);
    return 1; }
  else
    return 0;
}

int sync_writes (int fd)
{
  int rv;
  rv = fsync (fd);
  if (rv != 0) {
    warning ("write sync failed");
    return 1; }
  else {
    sleep (1);
    return 0; }
  
}

static void check_not_mounted (char *dev)
{
  FILE *f;
  struct mntent *mnt;

  if ((f = setmntent (MOUNTED, "r")) == NULL)
    return;

  while ((mnt = getmntent (f)))
    if (strcmp (dev, mnt->mnt_fsname) == 0)
      fatal ("%s is mounted.  Please dismount %s first.",
	     mnt->mnt_fsname, mnt->mnt_dir);

  endmntent (f);
}

/* Attempt to determine what 'FORMAT A:' would have done -- in particular,
   how long the FAT is.
   I looked at, among others:

       fdformat 1.8, simtel msdos/dskutl/fdform18.zip
       mkdosfs 0.2, sunsite Linux/system/Filesystems/dosfs/mkdosfs-0.2.tar.gz
       interrup.lst, simtel msdos/info/int34*.zip
       the kernel,  linux/fs/msdos/fat.c

   These sources disagree in minor but important details -- such
   things as what the max cluster number is.  What FORMAT.COM actually
   does is yet another question.  This code gets it right for the
   standard formats up to 1.44 meg, which is all that I can test,
   and there is a chance it's right for some other cases as well. */
   
void get_standard_format (int fd, struct boot_block *boot)
{
  struct floppy_struct floppy;
  float fat_bytesize;
  int fat_sectors, fat_bytes, data_clusters;
  int rv;

  memset (boot, 0, sizeof *boot);

  /* Get the kernel's deductions about what the physical geometry is. */

  rv = ioctl (fd, FDGETPRM, &floppy);
  if (rv != 0) {
    warning ("\
Unable to determine floppy geometry.\n\
Probably you are using a non-floppy.\n\
It will work, but some checks will not be done or may be misleading.");
    return; }

  /* Blam to nearby standard size */
  if (floppy.size <= 720)
    floppy.size = 720;		/* 360/1200 */
  else if (floppy.size <= 1440)
    floppy.size = 1440;		/* 720/1440 */
  else if (floppy.size <= 2400)
    floppy.size = 2400;		/* 1200/1200 */
  else if (floppy.size <= 2880)
    floppy.size = 2880;		/* 1440/1440 */
  else
    floppy.size = 5760;		/* 2880/2880 */

  boot->sectors_per_track = floppy.sect;
  boot->heads_per_cyl = floppy.head;
  if (floppy.size < 65536)
    boot->n_sectors_s = floppy.size;
  else
    boot->n_sectors_l = floppy.size;

  /* These are standard values, I think */

  boot->bytes_per_sector = 512;
  boot->n_reserved_sectors = 1;
  boot->n_fats = 2;

  /* fdformat says: default cluster size depends on DD vs. HD floppy */

  if (boot->sectors_per_track > 11) {
    boot->sectors_per_cluster = 1;
    boot->n_rootdir_entries = 224; }
  else {
    boot->sectors_per_cluster = 2;
    boot->n_rootdir_entries = 112; }

  /* this follows fdformat, more or less.  The size limit for 12-bit FAT
     is from msdos_fs.  The size limit for 16-bit fat is from interrup.lst.
     The cluster size and fat bytesize handling is from mkdosfs. */
     
  fat_bytesize = 1.5;

 again:
  fat_sectors =
    1 + fat_bytesize * floppy.size / (512.0 * boot->sectors_per_cluster);

  for (;;) {
    data_clusters = ((floppy.size
		      - boot->n_reserved_sectors
		      - boot->n_rootdir_entries / 16
		      - boot->n_fats * (fat_sectors - 1))
		     / boot->sectors_per_cluster);

    fat_bytes = 0.5 + fat_bytesize * data_clusters;
    if ((fat_sectors - 1) * 512 >= fat_bytes)
      fat_sectors--;
    else
      break; }

  if (fat_bytesize == 1.5) {
    if (data_clusters > 0xfee) {
      fat_bytesize = 2.0;
      goto again; }}
  else
    if (data_clusters > 0xfff6) {
      boot->sectors_per_cluster *= 2;
      goto again; }

  /* at last, the number we need */

  boot->sectors_per_fat = fat_sectors;
}
