#include "fdecc.h"
#include "rs.h"

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <fcntl.h>
#include <getopt.h>
#include <string.h>

const struct option long_options[] =
{
  { "read",		no_argument, 	 0, 'r' },
  { "write",		no_argument,	 0, 'w' },
  { "undo",		no_argument,	 0, 'u' },
  { "force",		no_argument,	 0, 'f' },
  { "size",		required_argument, 0, 'n' },
  { "sector",		required_argument, 0, 's' },
  { 0 },
};

const char short_options[] = "rwufn:s:";

static int rflag;
static int wflag;
static int uflag;
static int fflag;
static unsigned nflag, sflag1, sflag2;
static char *dev;

static void doit (char *dev, int rflag, int wflag);
static volatile void usage (void);
static void check_boot_block (struct boot_block *boot, struct boot_block *std,
			      int read_err, int nflag, int sflag1, int sflag2);
static void compare_fats (unsigned char data[][512],
			  unsigned char *erasures,
			  int mid_line,
			  int spare_fat_start,
			  int spare_fat_end);
static void undo_fat2 (unsigned char data[][512],
		       int spare_fat_start, int spare_fat_end);
static void write_fat2 (int fd, unsigned char data[][512],
			int spare_fat_start, int spare_fat_end);

int main (int argc, char *argv[])
{
  int t;
  char *tail;
  
  /* Parse options */

  nflag = sflag1 = sflag2 = -1;

  for (;;)
    switch (getopt_long (argc, argv, short_options, long_options, &t)) {
    case -1:
      goto done;

    case 'r':
      rflag = 1;
      break;

    case 'w':
      wflag = 1;
      break;

    case 'u':
      uflag = 1;
      break;

    case 'f':
      fflag = 1;
      break;

    case 's':
      sflag1 = strtoul (optarg, &tail, 0);
      if (*tail++ != ':')
	usage ();
      sflag2 = strtoul (tail, &tail, 0);
      if (*tail || sflag1 >= 65535 || sflag2 >= 65535 || sflag1 >= sflag2)
	usage ();
      break;
      
    case 'n':
      nflag = strtoul (optarg, &tail, 0);
      if (*tail || nflag >= 65535)
	usage ();
      break;

    case '?':
      usage ();

    default:
      abort ();
    }

 done:
  if (optind < argc)
    dev = argv[optind++];

  if (optind != argc
      || ! dev
      || ! (rflag || wflag || uflag)
      || (wflag && (rflag || uflag)))
    usage ();

  rsinit ();
  doit (dev, rflag, wflag);

  return 0;
}

static volatile void usage ()
{
  fprintf (stderr, "\
Usage: fdecc -w /dev/fd0          to write ECC checksums\n\
       fdecc -r /dev/fd0 > temp   to correct bad sectors, output to temp file\n\
       fdecc -u /dev/fd0          undo -- remove ECC checksums\n\
Additional flags:\n\
             -n 2880              specify floppy size, in sectors (1/2 k)\n\
             -s 10:18             specify first and last sectors of spare FAT\n\
             -f                   force the requested action even if\n\
                                  consistency checks don't check.\n");
  exit (1);
}

void fatal (char *fmt, ...)
{
  va_list ap;
  va_start (ap, fmt);
  vfprintf (stderr, fmt, ap);
  fprintf (stderr, "\n");
  va_end (ap);
  exit (1);
}

void warning (char *fmt, ...)
{
  va_list ap;
  va_start (ap, fmt);
  vfprintf (stderr, fmt, ap);
  fprintf (stderr, "\n");
  va_end (ap);
}

static void doit (char *dev, int rflag, int wflag)
{
  struct boot_block boot;
  struct boot_block standard;
  int read_err;
  int fd, n_sectors, fat_length, spare_fat_start, spare_fat_end;
  int n_erasures, n_errs;
  int i, j, n, nmsg, nchk, nerrs;
  unsigned char *erasures;
  unsigned short *map, *chk;
  int *errloc, *newerrs;
  symbol *buf, (*data)[512 / sizeof (symbol)];

  fd = open_dev (dev, wflag | uflag);
  read_err = read_sector (fd, 0, &boot);

  get_standard_format (fd, &standard);
  check_boot_block (&boot, &standard, read_err, nflag, sflag1, sflag2);

  if (nflag == -1)
    n_sectors = boot.n_sectors_s;
  else
    n_sectors = nflag;

  if (sflag1 == -1) {
    fat_length = boot.sectors_per_fat;
    spare_fat_start = boot.n_reserved_sectors + fat_length;
    spare_fat_end = spare_fat_start + fat_length; }
  else {
    fat_length = sflag2 - sflag1;
    spare_fat_start = sflag1;
    spare_fat_end = sflag2 + 1; }

  /* make map of sector number -> message buffer positions.
     positions 0..fat_length-1 are checksums, sectors fatstart..fatend
     positions fat_length..n_sectors-1 are data,
         sectors 0..fatstart,fatend..end */

  map = malloc (n_sectors * sizeof *map);
  chk = malloc (fat_length * sizeof *chk);

  nchk = 0;
  nmsg = fat_length;
  for (n = 0; n < n_sectors; n++) {
    if (n >= spare_fat_start && n < spare_fat_end) {
      chk[nchk] = n;
      map[n] = nchk++; }
    else {
      map[n] = nmsg++; }}
      
  /* read the whole floppy, sector n in data[n], bad sectors erasures[n]==1 */

  fprintf (stderr, "Reading...");
  fflush (stderr);

  data = malloc (n_sectors * sizeof *data);
  erasures = malloc (n_sectors * sizeof *erasures);
  errloc = malloc (fat_length * sizeof *errloc);
  newerrs = malloc (fat_length * sizeof *newerrs);
  n_erasures = 0;

  for (n = 0; n < n_sectors; n++) {
    if (n == 0)
      memcpy (&data[0], &boot, 512);
    else
      read_err = read_sector (fd, n, &data[n]);
    erasures[n] = read_err;

    if (read_err) {
      if (n_erasures == 0)
	fprintf (stderr, "\nBad sectors:");
      fprintf (stderr, " %d", n);
      if (n_erasures < fat_length)
	errloc[n_erasures] = map[n]; 
      n_erasures++; }

    if (n == spare_fat_end) {
      if (uflag && ! rflag) {
	fprintf (stderr, "\nWriting duplicate FAT...");
	fflush (stdout);
	undo_fat2 ((void *) data, spare_fat_start, spare_fat_end);
	write_fat2 (fd, (void *) data, spare_fat_start, spare_fat_end);
	fprintf (stderr, "\nDone.\n");
	return; }
      else
	compare_fats ((void *) data, erasures, 1,
		      spare_fat_start, spare_fat_end); }}

  fprintf (stderr, "\n");

  if (n_erasures) {
    if (! rflag && ! fflag)
      fatal ("\
Floppy already has bad sectors.\n\
It is unfortunately too late for ECC info to do any good.");
    
    if (n_erasures > fat_length)
      fatal ("\
There are %d bad sectors, more than the max of %d.\n\
Unfortunately the correction info provides no help here.",
	     n_erasures, fat_length); }

  /* compute the checksums or corrections */

  buf = malloc (n_sectors * sizeof *buf);

  if (rflag) {
    n_errs = 0;
    for (i = 0; i < 512 / sizeof (symbol); i++) {
      for (j = 0; j < n_sectors; j++)
	buf[map[j]] = data[j][i];
      nerrs = decode_erasures (buf + fat_length, n_sectors - fat_length,
			       buf, fat_length,
			       errloc, n_erasures, newerrs);
      if (nerrs == -1)
	fatal ("\
Global CRC error.\n\
Probably the floppy does not have ECC correction info on it.\n\
Sorry, but fdecc cannot recover the data on this floppy.");

      for (n = 0; n < nerrs; n++) {
	int secno = map[newerrs[n]];
	if (! erasures[secno]) {
	  erasures[secno] = 1;
	  if (n_errs == 0)
	    fprintf (stderr, "Bad sectors not reported by hardware:");
	  fprintf (stderr, " %d", secno);
	  n_errs++; }}
	  
      for (j = 0; j < n_sectors; j++)
	data[j][i] = buf[map[j]]; }
    
    if (n_errs)
      fprintf (stderr, "\n");

    if (n_errs == 0 && n_erasures == 0)
      fprintf (stderr, "No errors.\n");
    else
      fprintf (stderr, "All errors corrected.\n");

    if (uflag)
      undo_fat2 ((void *) data, spare_fat_start, spare_fat_end);

    write (1, data, n_sectors * sizeof *data); }

  else if (wflag) {
    fprintf (stderr, "Writing checksums...");
    fflush (stderr);

    for (i = 0; i < 512 / sizeof (symbol); i++) {
      for (j = 0; j < n_sectors; j++)
	buf[map[j]] = data[j][i];
      encode (buf + fat_length, n_sectors - fat_length, buf, fat_length);
      for (j = 0; j < fat_length; j++)
	data[chk[j]][i] = buf[j]; }

    write_fat2 (fd, (void *) data, spare_fat_start, spare_fat_end);
    fprintf (stderr, "\nDone.\n"); }
}

static void check_boot_block (struct boot_block *boot, struct boot_block *std,
			      int read_err, int nflag, int sflag1, int sflag2)
{
  int std_start, std_end, std_size;
  int boot_start, boot_end, boot_size;

  if (! read_err) {
    if ((boot->magic != 0xaa55 && boot->magic != 0)
	|| boot->bytes_per_sector != 512
	|| boot->n_sectors_s == 0
	|| boot->sectors_per_track == 0
	|| boot->heads_per_cyl == 0
	|| boot->n_sectors_s % (boot->sectors_per_track
				* boot->heads_per_cyl) != 0
	|| ((boot->n_reserved_sectors
	     + boot->sectors_per_fat * boot->n_fats
	     + boot->n_rootdir_entries / 16)
	    > boot->n_sectors_s)) {
      if (! fflag)
	fatal ("Sector 0 contains invalid info -- probably not a DOS floppy.");
      else {
	warning ("Sector 0 contains invalid info.  Ignoring it.");
	read_err = 1; }}}

  if (read_err) {
    if (nflag == -1 || sflag1 == -1 || sflag2 == -1) {
      warning ("Sector 0 is bad.");
      if (std->n_sectors_s == 0)
	fatal ("Unable to proceed without knowing floppy geometry.");
      if (std->n_sectors_s > 1000)
	warning ("Assuming a standard-format %g meg floppy.",
		 std->n_sectors_s / 2000.0);
      else
	warning ("Assuming a standard-format %dk floppy.",
		 std->n_sectors_s / 2); }
    memcpy (boot, std, sizeof *boot); }

  if (boot->n_fats < 2)
    fatal ("Floppy doesn't have a spare FAT to use for ECC checksums.");

  std_size = std->n_sectors_s;
  std_start = std->n_reserved_sectors + std->sectors_per_fat;
  std_end = std_start + std->sectors_per_fat;
  boot_size = boot->n_sectors_s;
  boot_start = boot->n_reserved_sectors + boot->sectors_per_fat;
  boot_end = boot_start + boot->sectors_per_fat;

  if (nflag != -1 || sflag1 != -1 || sflag2 != -1)
    warning ("%dk floppy, %d sectors of correction info",
	     nflag / 2, sflag2 - sflag1 + 1);
  else if (boot_start == std_start
	   && boot_end == std_end && boot_size == std_size)
    warning ("Standard %dk floppy, %d sectors of correction info",
	     boot_size / 2, boot->sectors_per_fat);
  else if (! wflag)
    warning ("Nonstandard %dk floppy, %d sectors of correction info",
	     boot_size / 2, boot->sectors_per_fat);
  else
    warning ("\
Nonstandard %dk floppy, %d sectors of correction info.\n\
If sector 0 goes bad, data cannot be recovered unless you use -n and -s.\n\
The command to recover this floppy, if it loses sector 0, is\n\
    fdecc -r -n %d -s %d:%d\n\
For max safety, you may wish to write that on the floppy.",
	     boot_size / 2, boot->sectors_per_fat,
	     boot_size, boot_start, boot_end - 1);
}

static void compare_fats (unsigned char data[][512],
			  unsigned char *erasures,
			  int mid_line,
			  int spare_fat_start,
			  int spare_fat_end)
{
  int fats_differ;
  int fat_length;
  int i;

  fat_length = spare_fat_end - spare_fat_start;
  fats_differ = 0;
  for (i = spare_fat_start; i < spare_fat_end; i++) {
    if (erasures[i] || erasures[i - fat_length])
      continue;
    if (memcmp (data[i], data[i - fat_length], sizeof data[i]))
      fats_differ = 1; }

  if (fats_differ) {
    if (wflag && ! fflag) {
      if (mid_line)
	fprintf (stderr, "\n");
      fatal ("\
The spare FAT is already in use.  (The main FAT and spare FAT differ.)\n\
Perhaps you have already run fdecc -w on this floppy.\n\
  You can use fdecc -r /dev/fd0 >/dev/null to check.\n\
If you wish to do it again, use the -f flag and new ECC info will be written.\n\
If you have never run fdecc on this floppy do NOT use the -f flag -- it\n\
  will overwrite information that cannot be recovered.\n\
  If this really is a FAT floppy, its format is not standard.  Use -n and -s\n\
  to specify its size and the location of the spare FAT."); }}
  else {
    if (rflag && ! fflag) {
      if (mid_line)
	fprintf (stderr, "\n");
      fatal ("\
The spare FAT does not contain ECC information.\n\
Either you never ran fdecc -w on this floppy, or it was modified afterward.");
    }}
}

static void undo_fat2 (unsigned char data[][512],
		       int spare_fat_start, int spare_fat_end)
{
  int fat_length = spare_fat_end - spare_fat_start;
  int i;

  for (i = spare_fat_start; i < spare_fat_end; i++)
    memcpy (data[i], data[i - fat_length], sizeof data[i]);
}

static void write_fat2 (int fd, unsigned char data[][512],
			int spare_fat_start, int spare_fat_end)
{
  int err, n;

  err = 0;
  for (n = spare_fat_start; n < spare_fat_end; n++)
    err |= write_sector (fd, n, &data[n]);

  err |= sync_writes (fd);
  if (err)
    warning ("\n
Write errors occurred.\n\
At the very least, verify the floppy with  fdecc -r %s >/dev/null .", dev);
}
